#ifndef __AT24COMMON_H
#define __AT24COMMON_H

/* Address pointer is 16 bit. */
#define AT24_FLAG_ADDR16	BIT(7)
/* Does not auto-rollover reads to the next slave address. */
#define AT24_FLAG_NO_RDROL	BIT(1)

/*
 * I2C EEPROMs from most vendors are inexpensive and mostly interchangeable.
 * Differences between different vendor product lines (like Atmel AT24C or
 * MicroChip 24LC, etc) won't much matter for typical read/write access.
 * There are also I2C RAM chips, likewise interchangeable. One example
 * would be the PCF8570, which acts like a 24c02 EEPROM (256 bytes).
 *
 * However, misconfiguration can lose data. "Set 16-bit memory address"
 * to a part with 8-bit addressing will overwrite data. Writing with too
 * big a page size also loses data. And it's not safe to assume that the
 * conventional addresses 0x50..0x57 only hold eeproms; a PCF8563 RTC
 * uses 0x51, for just one example.
 *
 * Accordingly, explicit board-specific configuration data should be used
 * in almost all cases. (One partial exception is an SMBus used to access
 * "SPD" data for DRAM sticks. Those only use 24c02 EEPROMs.)
 *
 * So this driver uses "new style" I2C driver binding, expecting to be
 * told what devices exist. That may be in arch/X/mach-Y/board-Z.c or
 * similar kernel-resident tables; or, configuration data coming from
 * a bootloader.
 *
 * Other than binding model, current differences from "eeprom" driver are
 * that this one handles write access and isn't restricted to 24c02 devices.
 * It also handles larger devices (32 kbit and up) with two-byte addresses,
 * which won't work on pure SMBus systems.
 */

struct at24_client {
    struct i2c_client *client;
    struct regmap *regmap;
};

struct at24_data {
    /*
     * Lock protects against activities from other Linux tasks,
     * but not from changes by other I2C masters.
     */
    struct mutex lock;

    unsigned int write_max;
    unsigned int num_addresses;
    unsigned int offset_adj;

    u32 byte_len;
    u16 page_size;
    u8 flags;

    struct nvmem_device *nvmem;
    struct regulator *vcc_reg;

    void (*read_post)(unsigned int off, char *buf, size_t count);

    /*
     * Some chips tie up multiple I2C addresses; dummy devices reserve
     * them for us, and we'll use them with SMBus calls.
     */
    struct at24_client client[];
};

/*
 * This parameter is to help this driver avoid blocking other drivers out
 * of I2C for potentially troublesome amounts of time. With a 100 kHz I2C
 * clock, one 256 byte read takes about 1/43 second which is excessive;
 * but the 1/170 second it takes at 400 kHz may be quite reasonable; and
 * at 1 MHz (Fm+) a 1/430 second delay could easily be invisible.
 *
 * This value is forced to be a power of two so that writes align on pages.
 */
unsigned int at24_io_limit = 128;

/*
 * Specs often allow 5 msec for a page write, sometimes 20 msec;
 * it's important to recover from write timeouts.
 */
unsigned int at24_write_timeout = 25;

static struct device *at24_base_client_dev(struct at24_data *at24)
{
    return &at24->client[0].client->dev;
}

static size_t at24_adjust_read_count(struct at24_data *at24,
                                     unsigned int offset, size_t count)
{
    unsigned int bits;
    size_t remainder;

    /*
     * In case of multi-address chips that don't rollover reads to
     * the next slave address: truncate the count to the slave boundary,
     * so that the read never straddles slaves.
     */
    if (at24->flags & AT24_FLAG_NO_RDROL) {
        bits = (at24->flags & AT24_FLAG_ADDR16) ? 16 : 8;
        remainder = BIT(bits) - offset;
        if (count > remainder)
            count = remainder;
    }

    if (count > at24_io_limit)
        count = at24_io_limit;

    return count;
}

/*
 * This routine supports chips which consume multiple I2C addresses. It
 * computes the addressing information to be used for a given r/w request.
 * Assumes that sanity checks for offset happened at sysfs-layer.
 *
 * Slave address and byte offset derive from the offset. Always
 * set the byte address; on a multi-master board, another master
 * may have changed the chip's "current" address pointer.
 */
static struct at24_client *at24_translate_offset(struct at24_data *at24,
                                                 unsigned int *offset)
{
    unsigned int i;

    if (at24->flags & AT24_FLAG_ADDR16) {
        i = *offset >> 16;
        *offset &= 0xffff;
    } else {
        i = *offset >> 8;
        *offset &= 0xff;
    }

    return &at24->client[i];
}

static ssize_t at24_regmap_read(struct at24_data *at24, char *buf,
                                unsigned int offset, size_t count)
{
    unsigned long timeout, read_time;
    struct at24_client *at24_client;
    struct i2c_client *client;
    struct regmap *regmap;
    int ret;

    at24_client = at24_translate_offset(at24, &offset);
    regmap = at24_client->regmap;
    client = at24_client->client;
    count = at24_adjust_read_count(at24, offset, count);

    /* adjust offset for mac and serial read ops */
    offset += at24->offset_adj;

    timeout = jiffies + msecs_to_jiffies(at24_write_timeout);
    do {
        /*
         * The timestamp shall be taken before the actual operation
         * to avoid a premature timeout in case of high CPU load.
         */
        read_time = jiffies;

        ret = regmap_bulk_read(regmap, offset, buf, count);
        dev_dbg(&client->dev, "read %zu@%d --> %d (%ld)\n",
                count, offset, ret, jiffies);
        if (!ret)
            return count;

        usleep_range(1000, 1500);
    } while (time_before(read_time, timeout));

    return -ETIMEDOUT;
}

int at24_read(void *priv, unsigned int off, void *val, size_t count)
{
    struct at24_data *at24;
    struct device *dev;
    char *buf = val;
    int i, ret;

    at24 = priv;
    dev = at24_base_client_dev(at24);

    if (unlikely(!count))
        return count;

    if (off + count > at24->byte_len)
        return -EINVAL;

    ret = pm_runtime_get_sync(dev);
    if (ret < 0) {
        pm_runtime_put_noidle(dev);
        return ret;
    }

    /*
     * Read data from chip, protecting against concurrent updates
     * from this host, but not from other I2C masters.
     */
    mutex_lock(&at24->lock);

    for (i = 0; count; i += ret, count -= ret) {
        ret = at24_regmap_read(at24, buf + i, off + i, count);
        if (ret < 0) {
            mutex_unlock(&at24->lock);
            pm_runtime_put(dev);
            return ret;
        }
    }

    mutex_unlock(&at24->lock);

    pm_runtime_put(dev);

    if (unlikely(at24->read_post))
        at24->read_post(off, buf, i);

    return 0;
}

/*
 * Note that if the hardware write-protect pin is pulled high, the whole
 * chip is normally write protected. But there are plenty of product
 * variants here, including OTP fuses and partial chip protect.
 *
 * We only use page mode writes; the alternative is sloooow. These routines
 * write at most one page.
 */

static size_t at24_adjust_write_count(struct at24_data *at24,
                                      unsigned int offset, size_t count)
{
    unsigned int next_page;

    /* write_max is at most a page */
    if (count > at24->write_max)
        count = at24->write_max;

    /* Never roll over backwards, to the start of this page */
    next_page = roundup(offset + 1, at24->page_size);
    if (offset + count > next_page)
        count = next_page - offset;

    return count;
}

static ssize_t at24_regmap_write(struct at24_data *at24, const char *buf,
                                 unsigned int offset, size_t count)
{
    unsigned long timeout, write_time;
    struct at24_client *at24_client;
    struct i2c_client *client;
    struct regmap *regmap;
    int ret;

    at24_client = at24_translate_offset(at24, &offset);
    regmap = at24_client->regmap;
    client = at24_client->client;
    count = at24_adjust_write_count(at24, offset, count);
    timeout = jiffies + msecs_to_jiffies(at24_write_timeout);

    do {
        /*
         * The timestamp shall be taken before the actual operation
         * to avoid a premature timeout in case of high CPU load.
         */
        write_time = jiffies;

        ret = regmap_bulk_write(regmap, offset, buf, count);
        dev_dbg(&client->dev, "write %zu@%d --> %d (%ld)\n",
                count, offset, ret, jiffies);
        if (!ret)
            return count;

        usleep_range(1000, 1500);
    } while (time_before(write_time, timeout));

    return -ETIMEDOUT;
}

int at24_write(void *priv, unsigned int off, void *val, size_t count)
{
    struct at24_data *at24;
    struct device *dev;
    char *buf = val;
    int ret;

    at24 = priv;
    dev = at24_base_client_dev(at24);

    if (unlikely(!count))
        return -EINVAL;

    if (off + count > at24->byte_len)
        return -EINVAL;

    ret = pm_runtime_get_sync(dev);
    if (ret < 0) {
        pm_runtime_put_noidle(dev);
        return ret;
    }

    /*
     * Write data to chip, protecting against concurrent updates
     * from this host, but not from other I2C masters.
     */
    mutex_lock(&at24->lock);

    while (count) {
        ret = at24_regmap_write(at24, buf, off, count);
        if (ret < 0) {
            mutex_unlock(&at24->lock);
            pm_runtime_put(dev);
            return ret;
        }
        buf += ret;
        off += ret;
        count -= ret;
    }

    mutex_unlock(&at24->lock);

    pm_runtime_put(dev);

    return 0;
}

#endif