/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Copyright (c) 2015 Endless Mobile, Inc.
 * Author: Carlo Caione <carlo@endlessm.com>
 */

#ifndef __MESON_PARM_H
#define __MESON_PARM_H

#include <linux/bits.h>
#include <linux/regmap.h>

#define PMASK(width)			GENMASK((width) - 1, 0)
#define SETPMASK(width, shift)		GENMASK((shift) + (width) - 1, shift)
#define CLRPMASK(width, shift)		(~SETPMASK(width, shift))

#define PARM_GET(width, shift, reg)					\
	(((reg) & SETPMASK(width, shift)) >> (shift))
#define PARM_SET(width, shift, reg, val)				\
	(((reg) & CLRPMASK(width, shift)) | ((val) << (shift)))

#define MESON_PARM_APPLICABLE(p)		(!!((p)->width))

struct parm {
	u16	reg_off;
	u8	shift;
	u8	width;
};

static inline unsigned int meson_parm_read(struct regmap *map, struct parm *p)
{
	unsigned int val, ret;

	ret = regmap_read(map, p->reg_off, &val);
	if (ret) {
		pr_err("%s failed, ret = %d\n", __func__, ret);
		return ret;
	}

	return PARM_GET(p->width, p->shift, val);
}

static inline void meson_parm_write(struct regmap *map, struct parm *p,
				    unsigned int val)
{
	unsigned int ret;

	ret = regmap_update_bits(map, p->reg_off, SETPMASK(p->width, p->shift),
				 val << p->shift);
	if (ret)
		pr_err("meson_parm_write failed, ret = %d\n", ret);
}

#endif /* __MESON_PARM_H */

