/*
 * Copyright (C) 2022 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// Thin wrappers around V2_2::hal::ComposerResources related classes that
// return HWC3 error codes and accept HWC3 argument types.

#pragma once

#include "Common.h"
#include <composer-resources/2.2/ComposerResources.h>

#include <memory>
#include <optional>

using ReplacedHandle =
    ::android::hardware::graphics::composer::V2_2
    ::hal::ComposerResources::ReplacedHandle;

namespace aidl::android::hardware::graphics::composer3::impl {
namespace meson {

class ComposerResourceReleaser {
public:
    ComposerResourceReleaser(bool isBuffer) : mReplacedHandle(isBuffer) {}
    virtual ~ComposerResourceReleaser() = default;

    ReplacedHandle* getReplacedHandle() {
        return &mReplacedHandle;
    }

private:
    ReplacedHandle mReplacedHandle;
};

class ComposerResources {
public:
    ComposerResources() = default;

    HWC3::Error init();
    std::unique_ptr<ComposerResourceReleaser> createReleaser(bool isBuffer);

    void clear(::android::hardware::graphics::composer::V2_2::hal::
            ComposerResources::RemoveDisplay removeDisplay);
    bool hasDisplay(int64_t display);
    HWC3::Error addPhysicalDisplay(int64_t display);
    HWC3::Error addVirtualDisplay(int64_t displayId,
            uint32_t outputBufferCacheSize);
    HWC3::Error removeDisplay(int64_t display);
    HWC3::Error setDisplayClientTargetCacheSize(int64_t displayId,
            uint32_t clientTargetCacheSize);
    HWC3::Error getDisplayClientTargetCacheSize(int64_t displayId,
            size_t* outCacheSize);
    HWC3::Error getDisplayOutputBufferCacheSize(int64_t displayId,
            size_t* outCacheSize);

    HWC3::Error addLayer(int64_t displayId, int64_t layerId,
            uint32_t bufferCacheSize);
    HWC3::Error removeLayer(int64_t displayId, int64_t layer);

    void setDisplayMustValidateState(int64_t displayId, bool mustValidate);
    bool mustValidateDisplay(int64_t displayId);
    HWC3::Error getDisplayReadbackBuffer(
            int64_t displayId,
            const aidl::android::hardware::common::NativeHandle& handle,
            buffer_handle_t* outHandle, ComposerResourceReleaser* bufReleaser);
    HWC3::Error getDisplayClientTarget(int64_t displayId, const Buffer& buffer,
            buffer_handle_t* outHandle, ComposerResourceReleaser* bufReleaser);

    HWC3::Error getDisplayClientTarget(int64_t display, uint32_t slot, bool fromCache,
            const native_handle_t* rawHandle,
            const native_handle_t** outBufferHandle,
            ReplacedHandle* outReplacedBuffer);

    HWC3::Error getDisplayOutputBuffer(int64_t displayId, const Buffer& buffer,
            buffer_handle_t* outHandle,
            ComposerResourceReleaser* bufReleaser);

    HWC3::Error getDisplayOutputBuffer(int64_t display, uint32_t slot, bool fromCache,
            const native_handle_t* rawHandle,
            const native_handle_t** outBufferHandle,
            ReplacedHandle* outReplacedBuffer);

    HWC3::Error getLayerBuffer(int64_t displayId, int64_t layerId,
            const Buffer& buffer,
            buffer_handle_t* outBufferHandle,
            ComposerResourceReleaser* bufReleaser);
    HWC3::Error setLayerBufferSlotsToClear(int64_t displayId, int64_t layerId, int32_t slot,
            buffer_handle_t* outHandle, ComposerResourceReleaser* releaser);
    HWC3::Error getLayerSidebandStream(
            int64_t displayId, int64_t layerId,
            const aidl::android::hardware::common::NativeHandle& rawHandle,
            buffer_handle_t* outStreamHandle, ComposerResourceReleaser* bufReleaser);

private:
    std::unique_ptr<::android::hardware::graphics::composer::V2_2::hal::ComposerResources> mImpl;
};

class HandleWrapper {

public:
    HandleWrapper(native_handle_t* handle) {
        mHandle = handle;
    }
    HandleWrapper(const native_handle_t* handle) {
        mHandle = handle;
    }
    const native_handle_t *getNativeHandle() const {
        return mHandle;
    }
    ~HandleWrapper() {
        if (mHandle != nullptr) {
            native_handle_delete(const_cast<native_handle_t*>(mHandle));
        }
    }

private:
    const native_handle_t*  mHandle;
};

}
}  // namespace aidl::android::hardware::graphics::composer3::impl
