/*
 * Copyright (c) 2019 Amlogic, Inc. All rights reserved.
 *
 * This source code is subject to the terms and conditions defined in the
 * file 'LICENSE' which is part of this source code package.
 *
 * Description:
 */
#define ATRACE_TAG ATRACE_TAG_GRAPHICS

#include <misc.h>
#include <DrmTypes.h>
#include <BasicTypes.h>
#include <MesonLog.h>
#include "VdinPostProcessor.h"
#include <utils/Trace.h>

//#define PROCESS_DEBUG 1
//#define POST_FRAME_DEBUG 1
//#define VDIN_CAP_ALWAYS_ON 1

#define DEFAULT_FB_ZORDER (1 + 65)

#define VDIN_BUF_CNT (6)
#define VOUT_BUF_CNT (3)
/*vdin will keep one frame always*/
#define VDIN_CAP_CNT (VDIN_BUF_CNT - 1)

VdinPostProcessor::VdinPostProcessor() {
    mExitThread = true;
    mStat = PROCESSOR_STOP;
    mVoutW = 0;
    mVoutH = 0;
    mVdinBufOnScreen = 0;
    mProcessMode = 0;
    mCapStatus = 0;
    mThread = 0;
}

VdinPostProcessor::~VdinPostProcessor() {
    mPlanes.clear();
}

void VdinPostProcessor::reset() {
    crcvalStatus = true;
    crcVal = 0;
}

void VdinPostProcessor::setType(int type) {
    mType = type;
    Vdin::getInstance().setType(type);
}

void VdinPostProcessor::setScreenSize(int w, int h) {
    Vdin::getInstance().setScreenSize(w, h);
}

int32_t VdinPostProcessor::setVout(
    std::shared_ptr<HwDisplayCrtc> & crtc,
    std::vector<std::shared_ptr<HwDisplayPlane>> & planes,
    int w, int h) {
    MESON_LOGE("vdinpostprocessor setvout");
    mVout = crtc;
    mPlanes = planes;
    auto it = mPlanes.begin();
    for (; it != mPlanes.end(); it ++) {
        if ((*it)->getType() == OSD_PLANE) {
            mDisplayPlane = *it;
            mPlanes.erase(it);
            break;
        }
    }
    MESON_ASSERT(mDisplayPlane != NULL, "plane size should > 0.");

    mVoutW = w;
    mVoutH = h;

    return 0;
}

int32_t VdinPostProcessor::postVout(std::shared_ptr<DrmFramebuffer> fb) {
    ATRACE_CALL();
    if (fb.get() != NULL) {
        drm_rect_t dispFrame = fb->getDisplayFrame();
        display_zoom_info_t osdDisplayFrame;
        osdDisplayFrame.framebuffer_w = fb->mSourceCrop.right - fb->mSourceCrop.left;
        osdDisplayFrame.framebuffer_h = fb->mSourceCrop.bottom - fb->mSourceCrop.top;
        osdDisplayFrame.crtc_display_x = 0;
        osdDisplayFrame.crtc_display_y = 0;
        osdDisplayFrame.crtc_display_w = dispFrame.right - dispFrame.left;
        osdDisplayFrame.crtc_display_h = dispFrame.bottom - dispFrame.top;
        mVout->setDisplayFrame(osdDisplayFrame);

        fb->mFbType = DRM_FB_SCANOUT;
        fb->mBlendMode = DRM_BLEND_MODE_COVERAGE;
        fb->mPlaneAlpha = 1.0f;
        fb->mTransform = 0;
        mDisplayPlane->setPlane(fb, DEFAULT_FB_ZORDER, UNBLANK);
    } else {
        mDisplayPlane->setPlane(NULL, DEFAULT_FB_ZORDER, BLANK_FOR_NO_CONTENT);
    }

    /*blank unused planes.*/
    for (auto it = mPlanes.begin(); it != mPlanes.end(); it ++) {
        (*it)->setPlane(NULL, DEFAULT_FB_ZORDER, BLANK_FOR_NO_CONTENT);
    }

    static int32_t fencefd = -1;
    static nsecs_t post_start;

    fencefd = -1;
    post_start = systemTime(CLOCK_MONOTONIC);
    if (mVout->pageFlip(fencefd) < 0) {
        MESON_LOGE("VdinPostProcessor, page flip failed.");
        return -EIO;
    }

    if (fencefd >= 0) {
#if POST_FRAME_DEBUG
        DrmFence fence(fencefd);
        fence.waitForever("vout2");
        float post_time = (float)(systemTime(CLOCK_MONOTONIC) - post_start)/ 1000000.0;
        if (post_time >= 18.0f)
            MESON_LOGE("last present fence timeout  (%d)(%f)!", fencefd, post_time);
#else
        close(fencefd);
#endif
        fencefd = -1;
    }

    return 0;
}

int32_t VdinPostProcessor::startVdin() {
    int w = 0, h = 0, format = 0;
    Vdin::getInstance().getStreamInfo(w, h, format);
    MESON_ASSERT(format == HAL_PIXEL_FORMAT_RGB_888,
        "Only support HAL_PIXEL_FORMAT_RGB_888");
    Vdin::getInstance().setStreamInfo(format, VDIN_BUF_CNT);
    mVdinBufOnScreen = -1;
    mVdinFbs.clear();

    for (int i = 0;i < VDIN_BUF_CNT;i ++) {
        buffer_handle_t hnd = gralloc_alloc_dma_buf(w, h, format, true, false);
        MESON_ASSERT(hnd != NULL && am_gralloc_get_buffer_fd(hnd) >= 0,
            "alloc vdin buf failed.");
        std::shared_ptr<DrmFramebuffer> buf = std::make_shared<DrmFramebuffer>(hnd, -1);
        buf->setUniqueId(i);
        /*queue buf before start streaming.*/
        Vdin::getInstance().queueBuffer(buf, i);
        mVdinFbs.push_back(buf);
        mVdinHnds.push_back(hnd);
    }
    return Vdin::getInstance().start();
}

int32_t VdinPostProcessor::stopVdin() {
    Vdin::getInstance().stop();
    mVdinFbs.clear();
    for (auto it = mVdinHnds.begin(); it != mVdinHnds.end(); it ++) {
        gralloc_free_dma_buf((native_handle_t * )*it);
    }
    mVdinHnds.clear();

    while (!mVdinQueue.empty()) {
        mVdinQueue.pop();
    }
    mVdinBufOnScreen = -1;
    mLastIndex = -1;
    return 0;
}

int32_t VdinPostProcessor::setFbProcessor(
    std::shared_ptr<FbProcessor> & processor) {
    std::unique_lock<std::mutex> cmdLock(mMutex);

    if (processor == NULL) {
        reset();
    }

    if (mStat == PROCESSOR_START) {
        mReqFbProcessor.push(processor);
        mCmdQ.push(PRESENT_UPDATE_PROCESSOR);
        cmdLock.unlock();
        mCmdCond.notify_one();
    } else {
        mFbProcessor = processor;
        mReqFbProcessor.push(NULL);
    }

    return 0;
}

void VdinPostProcessor::createScreenRecordProcessor() {
    createFbProcessor(FB_COPY_PROCESSOR, mSCapProcessor);
}

void VdinPostProcessor::destroyScreenRecordProcessor() {
    if (mSCapProcessor != NULL) {
        mSCapProcessor->teardown();
        mSCapProcessor.reset();
    }
}

bool VdinPostProcessor::getLatestcapFb(std::shared_ptr<DrmFramebuffer> & capFb) {
    ATRACE_NAME("VdinPostProcessor::getLatestcapFb");
    std::shared_ptr<DrmFramebuffer> infb;
    bool ret = false;
    int copyStatus = -1;
    scapfb = capFb;

    if (mLastIndex >= 0 && mLastIndex < mVdinFbs.size()) {
        infb = mVdinFbs[mLastIndex];
    }

    if (infb == NULL) {
        std::unique_lock<std::mutex> stateLock(mLock);
        mCondition.wait_for(stateLock, std::chrono::milliseconds(500));
        infb = mVdinFbs[mLastIndex];
    }

    if (mSCapProcessor != NULL && scapfb != NULL && infb != NULL) {
        ATRACE_BEGIN("process");
        copyStatus = mSCapProcessor->process(infb,scapfb);
        ATRACE_END();
    }

    scapfb.reset();

    if (copyStatus == 0) {
        ret = true;
    }
    return ret;
}

bool VdinPostProcessor::getScreencapFb(
    std::shared_ptr<DrmFramebuffer> & capFb) {

    bool ret = false;

    scapfb = capFb;

    createFbProcessor(FB_COPY_PROCESSOR, mSCapProcessor);

    present(PRESENT_CAPSCREEN,-1);

    nsecs_t start_time = systemTime(CLOCK_MONOTONIC);
    nsecs_t duration = 0;

    while ( mSCapProcessor != NULL ) {
        duration = systemTime(CLOCK_MONOTONIC)- start_time;
        usleep(5*1000);
        MESON_LOGE("wait to get screencapfb time %" PRId64 "",duration);
    }

    scapfb.reset();
    if (mCapStatus == 0) {
        ret = true;
    }

    return ret;
}

int32_t VdinPostProcessor::start() {
    std::lock_guard<std::mutex> lock(mMutex);
    if (mStat == PROCESSOR_START)
        return 0;

    if (mType == PROCESSOR_FOR_LOOPBACK) {
        if (mVoutHnds.size() == 0) {
            for (int i = 0;i < VOUT_BUF_CNT;i ++) {
                buffer_handle_t hnd = gralloc_alloc_dma_buf(
                    mVoutW, mVoutH, HAL_PIXEL_FORMAT_RGBA_8888, true, false);
                MESON_ASSERT(hnd != NULL && am_gralloc_get_buffer_fd(hnd) >= 0,
                    "alloc vout buf failed.");
                mVoutHnds.push_back(hnd);

                auto buf = std::make_shared<DrmFramebuffer>(hnd, -1);
                buf->setUniqueId(i);
                mVoutQueue.push(buf);
            }
        }
    }

    mStat = PROCESSOR_START;
    mProcessMode = PROCESS_IDLE;

    /*process thread will start later when hwc2display really have output.*/
    return 0;
}

int32_t VdinPostProcessor::stop() {
    std::unique_lock<std::mutex> cmdLock(mMutex);
    if (mStat == PROCESSOR_STOP)
        return 0;

    mExitThread = true;
    cmdLock.unlock();
    mCmdCond.notify_one();

    pthread_join(mThread, NULL);
    mStat = PROCESSOR_STOP;

    while (!mCmdQ.empty()) {
        mCmdQ.pop();
    }
    while(!mReqFbProcessor.empty()) {
        mReqFbProcessor.pop();
    }
    mFbProcessor.reset();

    /*clear queues.*/
    while (!mVoutQueue.empty()) {
        mVoutQueue.pop();
    }
    for (auto it = mVoutHnds.begin(); it != mVoutHnds.end(); it ++) {
        gralloc_free_dma_buf((native_handle_t * )*it);
    }
    mVoutHnds.clear();

    return 0;
}

/*
 * need restart vdinPostProcessor if vout mode changed
 */
int32_t VdinPostProcessor::restart(int w, int h) {
    MESON_LOGD("VdinPostProcessor::restart (%d,%d)", w, h);
    if (mVoutW != w || mVoutH !=h) {
        std::shared_ptr<FbProcessor> backProcessor = mFbProcessor;
        mVoutW = w;
        mVoutH = h;
        stop();
        start();
        setFbProcessor(backProcessor);
    }

    return 0;
}

bool VdinPostProcessor::running() {
    return mStat == PROCESSOR_START;
}

void VdinPostProcessor::setKeystoneConfigs(std::string params) {
    mKeystoneConfigs = params;
}

int32_t VdinPostProcessor::present(int flags, int32_t fence) {
    if (fence >= 0)
        close(fence);

    std::unique_lock<std::mutex> cmdLock(mMutex);
    if (mStat == PROCESSOR_STOP)
        return 0;

#ifdef PROCESS_DEBUG
    MESON_LOGE("VdinPostProcessor::present %d @ %d", flags, mStat);
#endif

    /*First present comes, we need start processor thread.*/
    if (!(flags & PRESENT_BLANK)) {
        if (mExitThread == true && mStat == PROCESSOR_START) {
            int ret = pthread_create(&mThread, NULL, VdinPostProcessor::threadMain, (void *)this);
            MESON_ASSERT(ret == 0, "failed to start VdinFlinger main thread: %s", strerror(ret));
            mExitThread = false;
        }
    }

    if (mCmdQ.size() == 0 || mCmdQ.back() != flags)
        mCmdQ.push(flags);

    cmdLock.unlock();
    mCmdCond.notify_one();
    return 0;
}

void * VdinPostProcessor::threadMain(void * data) {
    MESON_ASSERT(data, "vdin data should not be NULL.");

    struct sched_param param = {0};
    param.sched_priority = 2;
    if (sched_setscheduler(0, SCHED_FIFO | SCHED_RESET_ON_FORK, &param) != 0) {
        ALOGE("Couldn't set SCHED_FIFO: %d", errno);
    }

    VdinPostProcessor * pThis = (VdinPostProcessor *) data;
    pThis->reset();
    if (pThis->mFbProcessor)
        pThis->mFbProcessor->setup();

    pThis->startVdin();
    while (!pThis->mExitThread) {
        pThis->process();
    }
    pThis->stopVdin();

    /*blank vout, for we will read the buffer on screen.*/
    if (pThis->mType == PROCESSOR_FOR_LOOPBACK) {
        pThis->postVout(NULL);
    }

    if (pThis->mFbProcessor)
        pThis->mFbProcessor->teardown();

    pthread_exit(0);
    return NULL;
}

#ifdef POST_FRAME_DEBUG
const int track_frames = 120;
static int frames = 0, skip_frames = 0;
static nsecs_t track_start;
#endif

int32_t VdinPostProcessor::process() {
    ATRACE_BEGIN("PostProcess");
    std::unique_lock<std::mutex> lock(mMutex);
    static int capCnt = 0;
    static int preCmd = -1;
    if (mCmdQ.size() > 0) {
        int cmd = mCmdQ.front();
        preCmd = cmd;
        mCmdQ.pop();
        if (cmd & PRESENT_SIDEBAND) {
            mProcessMode = PROCESS_ALWAYS;
        } else if (cmd & PRESENT_UPDATE_PROCESSOR) {
            if (mFbProcessor != mReqFbProcessor.front()) {
                if (mFbProcessor != NULL)
                    mFbProcessor->teardown();
                mFbProcessor = mReqFbProcessor.front();
                if (mFbProcessor != NULL)
                    mFbProcessor->setup();
            }
            mReqFbProcessor.pop();
        } else if ( cmd & PRESENT_CAPSCREEN) {
            mCapStatus = 1;
            mSCapProcessor->setup() ;
            mProcessMode = PROCESS_ONCE;
            capCnt = VDIN_CAP_CNT;
        } else if ((cmd & PRESENT_BLANK) || (cmd == 0)) {
            mProcessMode = PROCESS_ONCE;
            capCnt = VDIN_CAP_CNT;
        }
    } else {
        if (mProcessMode == PROCESS_ONCE && (preCmd & PRESENT_CAPSCREEN)) {
            mProcessMode = PROCESS_ALWAYS;
            preCmd = -1;
        }
    }

    if (mProcessMode == PROCESS_IDLE) {
        //wait new present cmd.
        MESON_LOGD("In idle, waiting new cmd.");
        mCmdCond.wait(lock);
        ATRACE_END();
        return 0;
    }

#ifdef VDIN_CAP_ALWAYS_ON
    mProcessMode = PROCESS_ALWAYS;
#endif
#ifdef POST_FRAME_DEBUG
    bool debug_cap_always = sys_get_bool_prop("vendor.hwc.cap-always", false);
    if (debug_cap_always == true) {
        mProcessMode = PROCESS_ALWAYS;
    } else if (mProcessMode == PROCESS_ALWAYS) {
        mProcessMode = PROCESS_ONCE;
        capCnt = VDIN_CAP_CNT;
    }
#endif
    lock.unlock();

    if (mProcessMode == PROCESS_ALWAYS) {
        /*always set capCnt, so always queue buf to vdin..*/
        capCnt = VDIN_CAP_CNT;
    }

    int deqVdinBufs = mVdinQueue.size() + (mVdinBufOnScreen >= 0 ? 1 : 0);
#ifdef PROCESS_DEBUG
    MESON_LOGD("VdinPostProcessor processMode(%d) capCnt(%d) vdinkeep (%d)",
        mProcessMode, capCnt, deqVdinBufs);
#endif

    /*all vdin bufs consumed, go to idle.*/
    if (capCnt <= 0 &&  deqVdinBufs == VDIN_CAP_CNT) {
        if (mProcessMode == PROCESS_ONCE) {
            MESON_LOGD("PROCESS_ONCE -> PROCESS_IDLE.");
            mProcessMode = PROCESS_IDLE;
        }
    } else {
        std::shared_ptr<DrmFramebuffer> nullfb;
        std::shared_ptr<DrmFramebuffer> infb;
        std::shared_ptr<DrmFramebuffer> outfb;
        int vdinIdx = -1;
        vdin_vf_info vdinCrc;

        /*Release buf to vdin here, for we may keeped all the buf..*/
        while (capCnt > 0 && mVdinQueue.size() > 0) {
            vdinIdx = mVdinQueue.front();
            Vdin::getInstance().queueBuffer(nullfb, vdinIdx);
#ifdef PROCESS_DEBUG
            MESON_LOGE("Vdin::queue %d", vdinIdx);
#endif
            mVdinQueue.pop();
            capCnt --;
        }

#ifdef POST_FRAME_DEBUG
        if (frames == 0) {
            track_start = systemTime(CLOCK_MONOTONIC);
        }
#endif

        /*read vdin and process.*/
        if (Vdin::getInstance().dequeueBuffer(vdinCrc) == 0) {
            if (crcVal == vdinCrc.crc && vdinCrc.crc != 0) {
                crcvalStatus = false;
            } else {
                crcvalStatus = true;
            }
            crcVal = vdinCrc.crc;
            vdinIdx = vdinCrc.index;
            mLastIndex = vdinCrc.index;
            mCondition.notify_all();
            if (mType == PROCESSOR_FOR_LOOPBACK) {
                MESON_ASSERT(vdinIdx >= 0 && !mVoutQueue.empty(), "idx always >= 0.");
            }
#ifdef PROCESS_DEBUG
            MESON_LOGE("Vdin::dequeue %d", vdinIdx);
#endif

#ifdef POST_FRAME_DEBUG
            frames ++;
            if (frames >= track_frames) {
                if (skip_frames > 0) {
                    nsecs_t elapse_time = systemTime(CLOCK_MONOTONIC) - track_start;
                    float fps = (float)frames * 1000000000.0/elapse_time;
                    MESON_LOGE("VdinPostProcessor: FPS (%f)=(%d, %d)/(%lld)",
                        fps, frames, skip_frames, (long long)elapse_time);
                }
                track_start = 0;
                skip_frames = frames = 0 ;
            }
#endif

            if (vdinIdx >= 0 && vdinIdx < mVdinFbs.size())
                infb = mVdinFbs[vdinIdx];

            if (mType == PROCESSOR_FOR_LOOPBACK) {
                if (mSCapProcessor != NULL && (mCapStatus == 1) && scapfb != NULL) {
                    mCapStatus = mSCapProcessor->process(infb,scapfb);
                    mSCapProcessor->teardown();
                    mSCapProcessor.reset();
                }
            }

            if (mFbProcessor != NULL) {
                /*get output buf, and wait it ready.*/
                bool mKeystoneCoordUpdated = mFbProcessor->updateProcess(mKeystoneConfigs);
                bool mAlwaysProcess = false;
#ifdef HWC_ENABLE_VERTICAL_KEYSTONE
                mAlwaysProcess = true;
#endif
                if (crcvalStatus || mKeystoneCoordUpdated == true || mAlwaysProcess) {
                    outfb = mVoutQueue.front();
                    int releaseFence = outfb->getPrevReleaseFence();
                    if (releaseFence >= 0) {
                        DrmFence fence(releaseFence);
                        fence.wait(3000);
                    }
                    mVoutQueue.pop();

                    /*do processor*/
                    {
                        ATRACE_BEGIN("keystoneProcess");
                        mFbProcessor->process(infb, outfb);
                        ATRACE_END();
                    }
                    /*post outbuf to vout, and return to vout queue.*/
                    postVout(outfb);
                    /*push back vout buf.*/
                    mVoutQueue.push(outfb);
                }
                /*push back last displayed buf*/
                if (mVdinBufOnScreen >= 0) {
                    mVdinQueue.push(mVdinBufOnScreen);
                    mVdinBufOnScreen = -1;
                }
                /*push back vdin buf.*/
                mVdinQueue.push(vdinIdx);
            } else {
                /*null procesor, post vdin buf to vout directly.*/
                if (mType == PROCESSOR_FOR_LOOPBACK && crcvalStatus) {
                    postVout(infb);
                }
                /*push back last displayed buf*/
                if (mVdinBufOnScreen >= 0) {
                    mVdinQueue.push(mVdinBufOnScreen);
                    mVdinBufOnScreen = -1;
                }
                /*keep current display buf.*/
                mVdinBufOnScreen = vdinIdx;
            }
        } else {
            MESON_LOGE("Vdin dequeue failed, still need cap %d", capCnt);
#ifdef POST_FRAME_DEBUG
            skip_frames ++;
#endif
        }
    }

    ATRACE_END();
    return 0;
}

void VdinPostProcessor::dumpPlane(String8 & dumpstr) {
    mDisplayPlane->dump(dumpstr);
}

void VdinPostProcessor::dump(String8 & dumpstr) {
    dumpstr.append("VdinPostProcessor:\n");
    dumpstr.appendFormat("    mVoutW=%d, mVoutH=%d\n", mVoutW, mVoutH);
    Vdin::getInstance().dump(dumpstr);
}
