/*
 * Copyright (c) 2019 Amlogic, Inc. All rights reserved.
 *
 * This source code is subject to the terms and conditions defined in the
 * file 'LICENSE' which is part of this source code package.
 *
 * Description:
 */

#include <pthread.h>
#include <sys/time.h>
#include <stdio.h>
#include <sys/wait.h>
#include <stdio.h>
#include <getopt.h>
#include <string.h>

#include "DisplayAdapter.h"

using meson::DisplayAdapter;
using std::unique_ptr;

static const char* short_option = "lc:g:s:r:G:S:F:dvw:bf:h:P:p:R:t:m:";
static const struct option long_option[] = {
    {"list-modes", no_argument, 0, 'l'},
    {"chang-mode", required_argument, 0, 'c'},
    {"get-property", required_argument, 0, 'g'},
    {"set-property", required_argument, 0, 's'},
    {"raw-cmd", required_argument, 0, 'r'},
    {"dump-display-attribute", no_argument, 0, 'd'},
    {"vsync-timestamp", no_argument, 0, 'v'},
    {"sync-protection",required_argument,0,'P'},
    {"disable-sideband",required_argument,0,'p'},
    {"change-type", required_argument, 0, 't'},
    {"perferred-mode", required_argument, 0, 'm'},
    {"keystone-configs",required_argument,0,'k'},
    {"reverse-display",required_argument,0,'V'},
    {0, 0, 0, 0}
};

static void print_usage(const char* name) {
    printf("Usage: %s [-lcrgs]\n"
            "Get or change the mode setting of the weston drm output.\n"
            "Options:\n"
            "       -l,--list-modes        \tlist connector support modes\n"
            "       -c,--change-mode MODE  \tchange connector current mode, MODE format like:%%dx%%d@%%d width,height,refresh\n"
            "       -d,--dump-display-attribute \tdump all display attribute\n"
            "       -g,--get-display-attribute  \"ATTRI_NAME\"\tget display attribute\n"
            "       -s,--set-display-attribute  \"ATTRI_NAME\"=value\tset display attribute\n"
            "       -G \"[ui-rect|display-mode]\"\tget [logic ui rect|display mode]\n"
            "       -S \"[ui-rect]\"\tset [logic ui rect]\n"
            "                               \t eg: \"Content Protection\" 1\n"
            "       -v, --get-vsyn-timestamp and period \t get primary display vsync timestamp and vsync period\n"
            "       -F, [framerate] \tset framerate for AFR\n"
            "       -w,--set-whiteboard  \t  disable [false|true]white board \n "
            "       -b,--get-whiteboard  \t  get current white board mode\n "
            "       -f,--set-whiteboard display position \t  x and y is the position\n "
            "       -h,--hide the video layer \t  hide the video layer\n "
            "       -p,--disable-sideband  \t  disable [false|true]side band \n "
            "       -P,--sync Protection control \t  sync Protection control\n "
            "       -R,--set-hdr-conversionstrategy \tset [strategy] [fource type]\n"
            "       -t,--change-type TYPE  \tchange connector type, TYPE could be 0|1|2|3, like:\n"
            "                              \t\t0    ----DUMMY\n"
            "                              \t\t1    ----HDMI\n"
            "                              \t\t2    ----PANEL\n"
            "                              \t\t3    ----CVBS\n"
            "       -m,--perferred-mode MODE\tchange perferred mode, MODE format like:%%dx%%d@%%d width,height,refresh\n"
            "       -k,--set-Keystone Correction \t  params is the position\n "
            "       -V,--enable kesytone reverse  \t  params is the reverse type\n "
            "       -r,--raw-cmd           \tsend raw cmd\n", name);
}


int main(int argc, char* argv[]) {
    std::vector<meson::DisplayModeInfo> displayModeList;
    if (argc == 1) {
        print_usage(argv[0]);
        return 0;
    }
#ifndef RECOVERY_MODE
    unique_ptr<DisplayAdapter> client = meson::DisplayAdapterCreateRemote();
    DEBUG_INFO("Start client");
#else
    unique_ptr<DisplayAdapter> client = meson::DisplayAdapterCreateLocal(meson::DisplayAdapter::BackendType::DISPLAY_TYPE_FBDEV);
    DEBUG_INFO("Start recovery client");
#endif
    DisplayAdapter::ConnectorType type = DisplayAdapter::CONN_TYPE_HDMI;


    int opt;
    while ((opt = getopt_long(argc, argv, short_option, long_option, NULL)) != -1) {
        switch (opt) {
            case 'l':
                if (client->getSupportDisplayModes(displayModeList, type)) {
                    for (auto mode : displayModeList) {
                        printf("%s %u %u %u %u %f \n", mode.name.c_str(), mode.dpiX, mode.dpiY, mode.pixelW, mode.pixelH, mode.refreshRate);
                    }
                }
                break;
            case 'c':
                client->setDisplayMode(optarg, type);
                break;
            case 'g':
                {
                    std::string value;
                    client->getDisplayAttribute(optarg, value, type);
                    printf("%s\n", value.c_str());
                }
                break;
            case 's':
                if (optind + 1 > argc) {
                    printf("miss parameter\n");
                    break;
                }
                {
                    client->setDisplayAttribute(optarg, argv[optind], type);
                    printf("set %s to %s", optarg, argv[optind]);
                    std::string value;
                    client->getDisplayAttribute(optarg, value, type);
                    printf(", current value:%s\n", value.c_str());
                }
                optind++;
                break;
            case 'G':
                if (optarg == NULL)
                    break;
                if (0 == memcmp("display-mode", optarg, sizeof("display-mode"))) {
                    std::string mode;
                    client->getDisplayMode(mode, type);
                    printf("%s\n", mode.c_str());
                } else if (0 == memcmp("ui-rect", optarg, sizeof("ui-rect"))) {
                    meson::Rect rect;
                    client->getDisplayRect(rect, type);
                    printf("%s\n", rect.toString().c_str());
                } else {
                    NOTIMPLEMENTED;
                }
                break;
            case 'S':
                if (optarg == NULL)
                    break;
                {
                    if (0 == memcmp("display-mode", optarg, sizeof("display-mode"))) {
                        if (optind + 1 > argc) {
                            DEBUG_INFO("miss parameter");
                            break;
                        }
                        client->setDisplayMode(argv[optind], type);
                        optind++;
                    } else if (0 == memcmp("ui-rect", optarg, sizeof("ui-rect"))) {
                        meson::Rect rect;
                        if (optind + 4 > argc) {
                            DEBUG_INFO("miss rect parameter");
                            break;
                        }
                        rect.x = atoi(argv[optind]);
                        optind++;
                        rect.y = atoi(argv[optind]);
                        optind++;
                        rect.w = atoi(argv[optind]);
                        optind++;
                        rect.h = atoi(argv[optind]);
                        DEBUG_INFO("set ui to (%s)", rect.toString().c_str());
                        client->setDisplayRect(rect, type);
                        optind++;
                    } else {
                        NOTIMPLEMENTED;
                    }
                }
                break;
            case 'r':
                NOTIMPLEMENTED;
                break;
            case 'v':
                int64_t vsyncTimestamp;
                int32_t vsyncPeriod;
                client->getDisplayVsyncAndPeriod(vsyncTimestamp, vsyncPeriod);
                printf("vsyncTimestamp:%" PRId64 " ns Period:%d ns\n",
                        vsyncTimestamp, vsyncPeriod);
                break;
            case 'F':
               {
                   if (optarg == NULL)
                       break;

                   float framerate = atof(optarg);
                   int ret = client->setFrameRate(framerate);
                   printf("setFrameRate  %.2f ret:%d\n",framerate, ret);
               }
               break;
            case 'p':
                 if (optarg == NULL)
                     break;
                  {
                      bool isDisable = false;
                      if (0 == memcmp("true", optarg, sizeof("true"))) {
                            isDisable =true;
                      }
                      client->disableSidebandStream(isDisable);
                      printf("set disableSidebandStream to %s \n", isDisable?"true":"false");
                  }
                break;
            case 'd':
                {
                    Json::Value json;
                    client->dumpDisplayAttribute(json, type);
                    printf("Dump display attribute:\n%s", meson::JsonValue2String(json).c_str());
                }
                break;
            case 'w':
                 if (optarg == NULL)
                     break;
                  {
                      bool mode = false;
                      if (0 == memcmp("true", optarg, sizeof("true"))) {
                            mode =true;
                      }
                      client->setWriteBoardMode(mode);
                      printf("set white board to %s \n", mode ? "true":"false");
                  }
                break;
            case 'b':
                  {
                      bool mode = false;
                      client->getWriteBoardMode(mode);
                      printf("get current white board %s \n", mode ? "true":"false");
                  }
                break;
            case 'f':
                  {
                      printf("set the white board position (%s %s)\n", optarg, argv[optind]);
                      client->setWBDisplayFrame(atoi(optarg), atoi(argv[optind]));
                  }
                break;
            case 'h':
                 if (optarg == NULL)
                     break;
                  {
                      bool mode = false;
                      if (0 == memcmp("true", optarg, sizeof("true"))) {
                            mode =true;
                      }
                      client->hideVideoLayer(mode);
                      printf("hide video layer mode is %s \n", mode ? "true":"false");
                  }
                break;
            case 'P':
                 if (optarg == NULL)
                     break;
                  {
                      bool mode = false;
                      if (0 == memcmp("true", optarg, sizeof("true"))) {
                            mode =true;
                      }
                      client->enableSyncProtection(mode);
                      printf("the sync Protection switch to %s \n", mode ? "true":"false");
                  }
                break;
            case 'R':
                 if (optarg == NULL)
                     break;
                  {
                      client->setHdrConversionStrategy(atoi(optarg),atoi(argv[optind]));
                      printf("setHdrConversionStrategy %s %s \n", optarg, argv[optind]);
                  }
                break;
            case 't':
                if (strspn(optarg, "0123456789") != strlen(optarg))
                {
                    print_usage(argv[0]);
                    break;
                }
                switch (atoi(optarg)) {
                    case DisplayAdapter::CONN_TYPE_DUMMY:
                        type = DisplayAdapter::CONN_TYPE_DUMMY;
                        printf("Connector type changed to CONN_TYPE_DUMMY: %d\n", type);
                        break;
                    case DisplayAdapter::CONN_TYPE_HDMI:
                        type = DisplayAdapter::CONN_TYPE_HDMI;
                        printf("Connector type changed to CONN_TYPE_HDMI: %d\n", type);
                        break;
                    case DisplayAdapter::CONN_TYPE_PANEL:
                        type = DisplayAdapter::CONN_TYPE_PANEL;
                        printf("Connector type changed to CONN_TYPE_PANEL: %d\n", type);
                        break;
                    case DisplayAdapter::CONN_TYPE_CVBS:
                        type = DisplayAdapter::CONN_TYPE_CVBS;
                        printf("Connector type changed to CONN_TYPE_CVBS: %d\n", type);
                        break;
                    default:
                        print_usage(argv[0]);
                        break;
                }
                break;
            case 'm':
                client->setPerferredMode(optarg, type);
                printf("Set perferred mode to %s\n", optarg);
                break;
            case 'k':
                 if (optarg == NULL)
                     break;
                  {
                      printf("set the keystone params (%s)\n", optarg);
                      client->setKeystoneCorrection(optarg);
                  }
                break;
            case 'V':
                 if (optarg == NULL)
                     break;
                  {
                      printf("set the reverse type to  (%d)\n", atoi(optarg));
                      client->setReverseMode(atoi(optarg));
                  }
                break;
            default:
                print_usage(argv[0]);
        }
    };

    DEBUG_INFO("Exit client");
    return 0;
}
