/*
 * Copyright (c) 2017 Amlogic, Inc. All rights reserved.
 *
 * This source code is subject to the terms and conditions defined in the
 * file 'LICENSE' which is part of this source code package.
 *
 * Description:
 */

#ifndef HWC2_LAYER_H
#define HWC2_LAYER_H

#include <deque>
#include <hardware/hwcomposer2.h>

#include <BasicTypes.h>
#include <DrmFramebuffer.h>
#include <UvmDettach.h>
#include <VtConsumer.h>
#include <VtAllocSolidColorBuffer.h>

#define VT_CMD_DISABLE_VIDEO     0x01
#define VT_CMD_GAME_MODE_ENABLE  0x02
#define VT_CMD_GAME_MODE_DISABLE 0x04

#define MESON_WHITE_BOARD_ZORDER 60

class VtDisplayObserver {
public:
    VtDisplayObserver() {};
    virtual ~VtDisplayObserver() {};
    virtual void onFrameAvailable() = 0;
    virtual void askSurfaceFlingerRefresh() = 0;
};

class Hwc2Layer : public DrmFramebuffer {
/*Interfaces for hwc2.0 api.*/
public:
    /*set layer data info, only one of tree functions called.*/
    hwc2_error_t setBuffer(buffer_handle_t buffer, int32_t acquireFence);
    hwc2_error_t setSidebandStream(const native_handle_t* stream,
            std::shared_ptr<VtDisplayObserver> observer);
    hwc2_error_t setColor(hwc_color_t color);

    hwc2_error_t setSourceCrop(hwc_frect_t crop);
    hwc2_error_t setDisplayFrame(hwc_rect_t frame);
    hwc2_error_t setBlendMode(hwc2_blend_mode_t mode);
    hwc2_error_t setPlaneAlpha(float alpha);
    hwc2_error_t setTransform(hwc_transform_t transform);
    hwc2_error_t setVisibleRegion(hwc_region_t visible);
    hwc2_error_t setSurfaceDamage(hwc_region_t damage);
    hwc2_error_t setCompositionType(int32_t type);
    hwc2_error_t setDataspace(android_dataspace_t dataspace);
    hwc2_error_t setZorder(uint32_t z);
    int32_t setPerFrameMetadata(
            uint32_t numElements, const int32_t* /*hw2_per_frame_metadata_key_t*/ keys,
            const float* metadata);
    int32_t setBrightness(float brightness);

/*Extend api.*/
public:
    Hwc2Layer(uint32_t dispId);
    virtual ~Hwc2Layer();

    bool isSecure() { return mSecure;}

    bool isVirtualLayer() { return mIsVirtualLayer;}

    int32_t commitCompType(hwc2_composition_t hwcComp);

    bool isUpdateZorder() { return mUpdateZorder;}
    void clearUpdateFlag();

    void setLayerUpdate(bool update);
    bool getVideoInfoFromUVM(int fb);
    int getVideoType() {return mAMVideoType;}
    uint32_t getVideoTimestamp() {return mVideoDecTimestamp;}

    void setDiProcessorFd(int32_t fd);
    void setDiProcessorFence(int32_t fenceFd);
    int32_t getDiProcessorFence();
    int32_t getBufferFd() override;

    /* video tunnel api */
    bool isVtBuffer() override;
    bool isFbUpdated() override;
    void updateVtBuffer();
    int32_t releaseVtBuffer();
    int32_t releaseVtResource();
    void setPresentTime(nsecs_t expectedPresentTime);
    bool shouldPresentNow(nsecs_t timestamp);
    bool newGameBuffer();
    int32_t getSolidColorBuffer();
    virtual bool haveSolidColorBuffer();
    void vtRefresh();
    int getVideoTunnelId();

    int32_t registerConsumer();
    int32_t unregisterConsumer();
    bool isVtNeedClearFrameOrShowColorBuffer();
    int32_t onVtFrameAvailable(std::vector<std::shared_ptr<VtBufferItem>> & items);
    int32_t onVtFrameDisplayed(int bufferFd, int fenceFd);
    void onVtVideoStatus(vt_video_status_t status);
    void onVtVideoGameMode(int data);
    int32_t getVtVideoStatus();
    void setVtSourceCrop(drm_rect_t & crop);
    void setVtDisplayFrame(drm_rect_t & frame);
    void onNeedShowTempBuffer(vt_video_color_t colorType);
    void onNeedShowTempBufferWithStatus(
            vt_video_color_t colorType, vt_video_status_t status);
    void handleDisplayDisconnect(bool connect);

    void adjustDisplayFrame(display_zoom_info_t & calibrateInfo);
    void adjustDisplayFrameLocked();

public:
    android_dataspace_t mDataSpace;
    int32_t mHwcCompositionType;
    hwc_region_t mVisibleRegion;
    hwc_region_t mDamageRegion;
    drm_rect_t mBackupDisplayFrame;
    std::shared_ptr<DrmFence> mDiProcessorFence;

public:
    // for videotunnel
    class VtContentChangeListener : public VtConsumer::VtContentListener {
    public:
        VtContentChangeListener(Hwc2Layer* layer) : mLayer(layer) {};
        ~VtContentChangeListener() {};

        int32_t onFrameAvailable(std::vector<std::shared_ptr<VtBufferItem>> & items);
        void onVideoStatus(vt_video_status_t status);
        void onVideoGameMode(int data);
        int32_t getVideoStatus();
        void onSourceCropChange(vt_rect_t & crop);
        void onDisplayFrameChange(vt_rect_t & crop);
        void onNeedShowTempBuffer(vt_video_color_t colorType);
        void onNeedShowTempBufferWithStatus(
                vt_video_color_t colorType, vt_video_status_t status);

        void onSetupClient(void* layer);
        void onDisconnectedLocked();
        std::mutex mMutex;
    private:
        Hwc2Layer* mLayer;
    };

protected:
    hwc2_error_t handleDimLayer(buffer_handle_t buffer);
    int32_t releaseVtResourceLocked();
    int32_t releaseVtResourceLocked(bool needDisconnect);
    int32_t releaseVtResourceLocked(bool needDisconnect, bool needKeepLastFrame);
    bool isVtBufferLocked() override;
    int32_t getBufferFdLocked();

    /* for NR */
    int32_t attachUvmBuffer(const int bufferFd);
    int32_t dettachUvmBuffer();
    int32_t collectUvmBuffer(const int bufferFd, const int fence);
    int32_t releaseUvmResource();
    int32_t releaseUvmResourceLock();

    /* solid color buffer for video tunnel layer */
    void freeSolidColorBufferLocked();
    void freeSolidColorBuffer();
    void setVtPrevReleaseFence();

protected:
    bool mUpdateZorder;
    bool mVtRefreshed;

    bool mGameMode;
    bool mVtUpdate;
    bool mNeedReleaseVtResource;
    int mTunnelId;
    int mVtBufferFd;
    int mSolidColorBufferfd;
    int mPreVtBufferFd;
    int mDifd;
    int mAMVideoType;
    uint32_t mVideoDecTimestamp;
    uint32_t mDisplayId;
    int32_t mQueuedFrames;
    int64_t mTimestamp;
    nsecs_t mExpectedPresentTime;
    nsecs_t mPreviousTimestamp;
    std::deque<VtBufferItem> mQueueItems;
    std::shared_ptr<VtConsumer::VtContentListener> mContentListener;
    std::shared_ptr<VtConsumer> mVtConsumer;
    vt_video_status_t mVideoDisplayStatus;
    std::shared_ptr<VtDisplayObserver> mDisplayObserver;
    std::shared_ptr<VtAllocSolidColorBuffer> mAllocSolidColorBufferHandle;

    std::shared_ptr<UvmDettach> mUvmDettach;
    int mPreUvmBufferFd;

    float mBrightness;

    display_zoom_info_t mCalibrateInfo;
};

#endif/*HWC2_LAYER_H*/
