/*
 * Copyright (c) 2019 Amlogic, Inc. All rights reserved.
 *
 * This source code is subject to the terms and conditions defined in the
 * file 'LICENSE' which is part of this source code package.
 *
 * Description:
 */

#include "FixedDisplayPipe.h"
#include <HwcConfig.h>
#include <MesonLog.h>
#include <misc.h>
#include <HwDisplayCrtc.h>
#include "systemcontrol.h"
#include <HwDisplayManager.h>

#define HDMI_HAS_USED_STATE "/sys/class/amhdmitx/amhdmitx0/hdmi_used"

FixedDisplayPipe::FixedDisplayPipe()
    : HwcDisplayPipe() {
}

FixedDisplayPipe::~FixedDisplayPipe() {
    mVdinPostProcessor.reset();
}

int32_t FixedDisplayPipe::getPostProcessor(
    hwc_post_processor_t type,
    std::shared_ptr<HwcPostProcessor> & processor) {
    MESON_ASSERT(type == VDIN_POST_PROCESSOR,
        "only support VDIN_POST_PROCESSOR.");

    if (!mVdinPostProcessor) {
        mVdinPostProcessor = std::make_shared<VdinPostProcessor>();
        mVdinPostProcessor->setType(PROCESSOR_FOR_SCREENRECORD);
    }

    processor = std::dynamic_pointer_cast<HwcPostProcessor>(mVdinPostProcessor);
    return 0;
}

int32_t FixedDisplayPipe::init(
    std::map<uint32_t, std::shared_ptr<HwcDisplay>> & hwcDisps) {
    HwcDisplayPipe::init(hwcDisps);
    return 0;
}

void FixedDisplayPipe::handleEvent(drm_display_event event, int val) {
    if (event == DRM_EVENT_HDMITX_HOTPLUG) {
        std::lock_guard<std::mutex> lock(mMutex);
        mEventState = val;
        bool connected = (val == 0 || val == DRM_EVENT_SUSPEND) ? false : true;
        std::shared_ptr<PipeStat> pipe;
        drm_connector_type_t targetConnector = DRM_MODE_CONNECTOR_Unknown;
        for (auto statIt : mPipeStats) {
            uint32_t connectorType = HwcConfig::getConnectorType((int)statIt.first);
            pipe = statIt.second;

            /* let crct known we are in hotplug event process */
            pipe->modeCrtc->setHotplugStatus(HwDisplayCrtc::HotplugStatus::InHotplugProcess);

            /*update current connector status, now getpipecfg() need
            * read connector status to decide connector.
            */
            pipe->modeConnector->update();
            pipe->modeCrtc->update();

            if (connectorType == HWC_HDMI_CVBS || connectorType == DRM_MODE_CONNECTOR_HDMIA) {
                targetConnector = getConnectorCfg((int)statIt.first);

                MESON_LOGD("handleEvent  DRM_EVENT_HDMITX_HOTPLUG %d VS %d",
                    pipe->cfg.hwcConnectorType, targetConnector);
                if (pipe->cfg.hwcConnectorType != targetConnector) {
                    /*we need latest connector status, and no one will update
                    *connector not bind to crtc, we update here.
                    */
                    std::shared_ptr<HwDisplayConnector> hwConnector;
                    getConnector(targetConnector, hwConnector);
                    hwConnector->update();
                    /*update display pipe.*/
                    updatePipe(pipe);
                }
                pipe->modeCrtc->setHotplugStatus(HwDisplayCrtc::HotplugStatus::Default);
                /*update display mode, workaround now.*/
                initDisplayMode(pipe);
                statIt.second->hwcDisplay->onHotplug(connected);
            } else {
                pipe->modeCrtc->setHotplugStatus(HwDisplayCrtc::HotplugStatus::Default);
            }
        }
    } else {
        HwcDisplayPipe::handleEvent(event, val);
    }
}

int32_t FixedDisplayPipe::getPipeCfg(uint32_t hwcid, PipeCfg & cfg) {
    drm_connector_type_t  connector = getConnectorCfg(hwcid);
    cfg.modePipeIdx = cfg.hwcPipeIdx = DRM_PIPE_VOUT1;
    cfg.hwcPostprocessorType = INVALID_POST_PROCESSOR;
    cfg.modeConnectorType = cfg.hwcConnectorType = connector;
    return 0;
}

drm_connector_type_t FixedDisplayPipe::getConnectorCfg(uint32_t hwcid) {
    drm_connector_type_t  connector = HwcDisplayPipe::getConnectorCfg(hwcid);
    // need switch connector
    if (connector == DRM_MODE_CONNECTOR_Unknown ||
            connector == DRM_MODE_CONNECTOR_TV ||
            connector == DRM_MODE_CONNECTOR_HDMIA) {
        std::shared_ptr<HwDisplayConnector> hwConnector;
        getConnector(DRM_MODE_CONNECTOR_HDMIA, hwConnector);

        switch (mEventState) {
        // plug out
        case DRM_EVENT_DISABLE :
            if (isVMXCertification() && connector != DRM_MODE_CONNECTOR_HDMIA) {
                connector = DRM_MODE_CONNECTOR_TV;
            } else {
                connector = hasDummyConnector() ?
                    DRM_MODE_CONNECTOR_VIRTUAL : DRM_MODE_CONNECTOR_HDMIA;
            }
            break;
        // plug in
        case DRM_EVENT_ENABLE :
            connector = DRM_MODE_CONNECTOR_HDMIA;
            break;
        case DRM_EVENT_SUSPEND:
            if (hwConnector->isConnected()) {
                connector = hasDummyConnector() ?
                    DRM_MODE_CONNECTOR_VIRTUAL : DRM_MODE_CONNECTOR_HDMIA;
            }
            break;
        case DRM_EVENT_RESUME:
        // init
        default:
            if (hwConnector->isConnected()) {
                connector = DRM_MODE_CONNECTOR_HDMIA;
            } else {
                if ((isVMXCertification() || !hasHdmiConnected()) &&
                        connector != DRM_MODE_CONNECTOR_HDMIA) {
                    connector = DRM_MODE_CONNECTOR_TV;
                } else {
                    connector = hasDummyConnector() ?
                        DRM_MODE_CONNECTOR_VIRTUAL : DRM_MODE_CONNECTOR_HDMIA;
                }
            }
        }
    }

    MESON_LOGE("FixedDisplayPipe::getConnectorCfg %d", connector);
    return connector;
}

bool FixedDisplayPipe::hasHdmiConnected() {
    bool ret = sysfs_get_int(HDMI_HAS_USED_STATE, 0) == 1 ? true : false;
    MESON_LOGD("FixedDisplayPipe::hasHdmiConnected:%d", ret);
    return ret;
}

bool FixedDisplayPipe::isVMXCertification() {
    bool ret = sc_get_property_boolean("persist.vendor.sys.vmx", false);
    MESON_LOGD("FixedDisplayPipe::isVMXCertification:%d", ret);
    return ret;
}
