/*
 * Copyright (c) 2019 Amlogic, Inc. All rights reserved.
 *
 * This source code is subject to the terms and conditions defined in the
 * file 'LICENSE' which is part of this source code package.
 *
 * Description:
 */
#define ATRACE_TAG ATRACE_TAG_GRAPHICS

#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/utsname.h>
#include <fcntl.h>
#include <poll.h>
#include <stdlib.h>
#include <string.h>
#include <utils/Trace.h>
#include <HwDisplayManager.h>
#include <HwDisplayCrtc.h>
#include <MesonLog.h>
#include <Vdin.h>


ANDROID_SINGLETON_STATIC_INSTANCE(Vdin)

#define VDIN1_DEV "/dev/vdin1"
#define POLL_TIMEOUT_MS (40)

#define _TM_T 'T'
#define TVIN_IOC_S_VDIN_V4L2START  _IOW(_TM_T, 0x25, struct vdin_v4l2_param_s)
#define TVIN_IOC_S_VDIN_V4L2STOP   _IO(_TM_T, 0x26)

#define TVIN_IOC_S_CANVAS_ADDR  _IOW(_TM_T, 0x4f, struct vdin_set_canvas_s)
#define TVIN_IOC_S_CANVAS_RECOVERY  _IO(_TM_T, 0x0a)

Vdin::Vdin() {
    mDev = open(VDIN1_DEV, O_RDONLY);
    MESON_ASSERT(mDev >= 0, "Vdin device open fail.");
    mStatus = STREAMING_STOP;
    mCanvasCnt = 0;
    mDefFormat = 0;
    memset(&mCapParams, 0, sizeof(mCapParams));
    memset(&mCanvas, 0, sizeof(mCanvas));

    struct utsname buf;
    int major = 0;
    int minor = 0;
    if (!uname(&buf)) {
        if (sscanf(buf.release, "%d.%d", &major, &minor) != 2) {
            major = 0;
        }
    }
    if (major == 0) {
        MESON_LOGV("Can't determine kernel version for access sysfs!");
    }

   mNeedDupFd = true;
   if (major >= 5 && minor >= 15) {
       mNeedDupFd = false;
   }
}

Vdin::~Vdin() {
    stop();
    releaseCanvas();
    close(mDev);
}

//set the vdin1 loopback size
void Vdin::setScreenSize(int w, int h) {
    mRecordHeight = h;
    mRecordWidth = w;
}

//setting the loopback purpose
void Vdin::setType(int type) {
    mVdinType = type;
}

int32_t Vdin::getStreamInfo(int & width, int & height, int & format) {
    /*read current */
    drm_mode_info_t modeInfo;
    auto crtc = getHwDisplayManager()->getCrtcByPipe(DRM_PIPE_VOUT2);
    if (crtc->getMode(modeInfo) != 0) {
        MESON_LOGE("getStreamInfo failed.");
        mCapParams.width = 1920;
        mCapParams.height = 1080;
        mCapParams.fps = 60;
        mCapParams.bit_dep = 8;
        mDefFormat = HAL_PIXEL_FORMAT_RGB_888;
    } else {
#ifdef HWC_ENABLE_VERTICAL_KEYSTONE
        mCapParams.width = modeInfo.pixelH;
        mCapParams.height = modeInfo.pixelW;
#else
        mCapParams.width = modeInfo.pixelW;
        mCapParams.height = modeInfo.pixelH;
#endif
        mCapParams.bit_dep = 8;
        if (mVdinType == PROCESSOR_FOR_SCREENRECORD) {
            mCapParams.dst_width = mRecordWidth;
            mCapParams.dst_height = mRecordHeight;
        }
        mCapParams.fps = (int)modeInfo.refreshRate;
        /*force use RGB888*/
        mDefFormat = HAL_PIXEL_FORMAT_RGB_888;
    }

    if (mVdinType == PROCESSOR_FOR_LOOPBACK) {
        width = mCapParams.width;
        height = mCapParams.height;
    } else if (mVdinType == PROCESSOR_FOR_SCREENRECORD) {
        width = mRecordWidth;
        height = mRecordHeight;
    }

    mCapParams.secure_memory_en = 1;
    mCapParams.bit_order = 1;
    format = mDefFormat;
    return 0;
}

int32_t Vdin::setStreamInfo(int  format, int bufCnt) {
    UNUSED(format);

    if (mStatus != STREAMING_STOP) {
        MESON_LOGE("Cannot setStreamInfo when do streaming.");
        return -EINVAL;
    }

    releaseCanvas();
    createCanvas(bufCnt);
    return 0;
}

void Vdin::createCanvas(int bufCnt) {
    mCanvasCnt = bufCnt;
    for (int i = 0;i < VDIN_CANVAS_MAX_CNT; i++) {
        mCanvas[i].fd = -1;
        mCanvas[i].index = -1;
    }
}

void Vdin::releaseCanvas() {
    if (mCanvasCnt <= 0)
        return ;

    for (int i = 0;i < mCanvasCnt; i++) {
        if (mCanvas[i].fd >= 0)
            close(mCanvas[i].fd);
        mCanvas[i].fd = -1;
    }

    mCanvasCnt = 0;
}

int32_t Vdin::queueBuffer(std::shared_ptr<DrmFramebuffer> & fb, int idx) {
    ATRACE_CALL();
    MESON_ASSERT(idx < VDIN_CANVAS_MAX_CNT, "queueBuffer idx is invalid.");

    if (mStatus == STREAMING_STOP) {
        MESON_ASSERT(fb.get() != NULL, "init queue fb should not be null.");
        mCanvas[idx].index = idx;
        int bufFd = am_gralloc_get_buffer_fd(fb->mBufferHandle);
        if ( bufFd >= 0) {
            if (mNeedDupFd) {
                mCanvas[idx].fd = ::dup(bufFd);
            } else {
                mCanvas[idx].fd = bufFd;
            }
        }
        MESON_LOGD("Vdin::queue new Buffer %d - %d", idx, mCanvas[idx].fd);
    } else {
        /*TODO: cannot queue back specific buffer, just return the last buffer.*/
        ioctl(mDev, TVIN_IOC_S_CANVAS_RECOVERY, &idx);
    }
    return 0;
}

int32_t Vdin::dequeueBuffer(vdin_vf_info & crcinfo) {
    ATRACE_CALL();
    struct pollfd fds[1];
    fds[0].fd = mDev;
    fds[0].events = POLLIN;
    fds[0].revents = 0;

    struct vdin_vf_info vInfo = {
        .index = -1,
        .crc = static_cast<unsigned int>(-1),
        .ready_clock[0] = 0,
        .ready_clock[1] = 0,
        .ready_clock[2] = 0,
    };

    int pollrtn = poll(fds, 1, POLL_TIMEOUT_MS);
    if (pollrtn > 0 && fds[0].revents == POLLIN) {
        int ret = read(mDev, &vInfo, sizeof(struct vdin_vf_info));
        if (ret  > 0) {
            crcinfo.index = vInfo.index;
            crcinfo.crc = vInfo.crc;
            //MESON_LOGD("vdin::dequeueBuffer idx(%d) crcval(0x%x) ",vInfo.index,vInfo.crc);
        }

        /* latency of dequeueBuffer to vdin put */
        nsecs_t dequeueToPut = vInfo.ready_clock[2] - vInfo.ready_clock[1];
        ATRACE_INT64("Vdin:dequeueToPut", dequeueToPut);
    }

    if (vInfo.index < 0)
        return -EFAULT;

    return 0;
}

int32_t Vdin::start() {
    if (mStatus == STREAMING_START)
        return 0;

    MESON_ASSERT(mCanvas != NULL, "Canvas should set before streaming.");

    if (ioctl(mDev, TVIN_IOC_S_CANVAS_ADDR, mCanvas) < 0) {
        int rtn = -errno;
        MESON_ASSERT(0, "TVIN_IOC_S_CANVAS_ADDR failed (%d) ", rtn);
        return rtn;
    }

    /*all fd passed to drv, reset to -1.*/
    if (mVdinType == PROCESSOR_FOR_LOOPBACK) {
        for (int i = 0;i < mCanvasCnt; i++)
            mCanvas[i].fd = -1;
    }

    if (ioctl(mDev, TVIN_IOC_S_VDIN_V4L2START, &mCapParams) < 0) {
        int rtn = -errno;
        MESON_ASSERT(0, "Vdin start ioctl failed (%d) ", rtn);
        return rtn;
    }

    mStatus = STREAMING_START;
    return 0;
}

int32_t Vdin::pause() {
    if (ioctl(mDev, TVIN_IOC_S_VDIN_V4L2STOP, &mCapParams) < 0) {
        int rtn = -errno;
        MESON_ASSERT(0, "Vdin start ioctl failed (%d) ", rtn);
        return rtn;
    }

    mStatus = STREAMING_PAUSE;
    return 0;
}

int32_t Vdin::stop() {
    if (STREAMING_STOP == mStatus)
        return 0;

    if (mStatus == STREAMING_START) {
        pause();
    }

    releaseCanvas();
    mStatus = STREAMING_STOP;
    return 0;
}

void Vdin::dump(String8 & dumpstr) {
    dumpstr.appendFormat("    Vdin mCapParam width=%d, height=%d\n",
            mCapParams.width, mCapParams.height);
}
