/*
 * Copyright (C) 2016, 2018-2022 ARM Limited. All rights reserved.
 *
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * You may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <hardware/gralloc1.h>
#include <mutex>

#include "private_interface_types.h"
#include "buffer.h"
#include "allocator/allocator.h"
#include "allocator/shared_memory/shared_memory.h"
#include "buffer_allocation.h"
#include "gralloc_version.h"
#include "buffer_access.h"

using scoped_mutex = std::lock_guard<std::mutex>;

static std::mutex s_map_lock;

int mali_gralloc_reference_retain(buffer_handle_t handle)
{
	if (private_handle_t::validate(handle) < 0)
	{
		MALI_GRALLOC_LOGE("Registering/Retaining invalid buffer %p, returning error", handle);
		return -EINVAL;
	}

	private_handle_t *hnd = (private_handle_t *)handle;
	scoped_mutex lock(s_map_lock);

	/* Ensure the state is valid for newly registered buffers */
	hnd->base = nullptr;
	if (hnd->allocating_pid != getpid() && hnd->remote_pid != getpid())
	{
		hnd->remote_pid = getpid();
	}

	return 0;
}

int mali_gralloc_reference_release(buffer_handle_t handle)
{
	if (private_handle_t::validate(handle) < 0)
	{
		MALI_GRALLOC_LOGE("unregistering/releasing invalid buffer %p, returning error", handle);
		return -EINVAL;
	}

	private_handle_t *hnd = (private_handle_t *)handle;
	scoped_mutex lock(s_map_lock);

	if (hnd->allocating_pid == getpid())
	{
		mali_unmap_buffer(hnd);

		mali_gralloc_buffer_free(hnd);
		native_handle_delete(const_cast<native_handle_t *>(handle));
	}
	else if (hnd->remote_pid == getpid()) // never unmap buffers that were not imported into this process
	{
		mali_unmap_buffer(hnd);

		/*
		 * Close shared attribute region file descriptor. It might seem strange to "free"
		 * this here since this can happen in a client process, but free here is nothing
		 * but unmapping and closing the duplicated file descriptor. The original shared
		 * fd instance is still open until alloc_device_free() is called. Even sharing
		 * of gralloc buffers within the same process should have fds dup:ed.
		 */
		gralloc_shared_memory_free(hnd->share_attr_fd, hnd->attr_base, hnd->attr_size);
		hnd->share_attr_fd = -1;
		hnd->attr_base = MAP_FAILED;
	}
	else
	{
		MALI_GRALLOC_LOGE("Trying to unregister buffer %p from process %d that was not imported into current process: %d", hnd,
		     hnd->remote_pid, getpid());
	}

	return 0;
}
