/*
 * Copyright (C) 2021 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <android-base/properties.h>
#include <log/log.h>
#include <inttypes.h>
#include "DumpstateUtil.h"

#include "Dumpstate.h"

using android::os::dumpstate::DumpFileToFd;
using android::os::dumpstate::CommandOptions;
using android::os::dumpstate::RunCommandToFd;

namespace aidl {
namespace android {
namespace hardware {
namespace dumpstate {

const uint64_t NANOS_PER_SEC = 1000000000;

uint64_t Nanotime() {
    timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    return static_cast<uint64_t>(ts.tv_sec * NANOS_PER_SEC + ts.tv_nsec);
}

const char kVerboseLoggingProperty[] = "persist.dumpstate.verbose_logging.enabled";

ndk::ScopedAStatus Dumpstate::dumpstateBoard(const std::vector<::ndk::ScopedFileDescriptor>& in_fds,
                                             IDumpstateDevice::DumpstateMode in_mode,
                                             int64_t in_timeoutMillis) {
    (void)in_timeoutMillis;

    if (in_fds.size() < 1) {
        return ndk::ScopedAStatus::fromExceptionCodeWithMessage(EX_ILLEGAL_ARGUMENT,
                                                                "No file descriptor");
    }

    int fd = in_fds[0].get();
    if (fd < 0) {
        return ndk::ScopedAStatus::fromExceptionCodeWithMessage(EX_ILLEGAL_ARGUMENT,
                                                                "Invalid file descriptor");
    }

    switch (in_mode) {
        case IDumpstateDevice::DumpstateMode::FULL:
            return dumpstateBoardImpl(fd, true);

        case IDumpstateDevice::DumpstateMode::DEFAULT:
            return dumpstateBoardImpl(fd, false);

        case IDumpstateDevice::DumpstateMode::INTERACTIVE:
        case IDumpstateDevice::DumpstateMode::REMOTE:
        case IDumpstateDevice::DumpstateMode::WEAR:
        case IDumpstateDevice::DumpstateMode::CONNECTIVITY:
        case IDumpstateDevice::DumpstateMode::WIFI:
        case IDumpstateDevice::DumpstateMode::PROTO:
            return ndk::ScopedAStatus::fromServiceSpecificErrorWithMessage(ERROR_UNSUPPORTED_MODE,
                                                                           "Unsupported mode");

        default:
            return ndk::ScopedAStatus::fromExceptionCodeWithMessage(EX_ILLEGAL_ARGUMENT,
                                                                    "Invalid mode");
    }

    return ndk::ScopedAStatus::ok();
}

ndk::ScopedAStatus Dumpstate::getVerboseLoggingEnabled(bool* _aidl_return) {
    *_aidl_return = getVerboseLoggingEnabledImpl();
    return ndk::ScopedAStatus::ok();
}

ndk::ScopedAStatus Dumpstate::setVerboseLoggingEnabled(bool in_enable) {
    ::android::base::SetProperty(kVerboseLoggingProperty, in_enable ? "true" : "false");
    return ndk::ScopedAStatus::ok();
}

bool Dumpstate::getVerboseLoggingEnabledImpl() {
    return ::android::base::GetBoolProperty(kVerboseLoggingProperty, false);
}

void Dumpstate::dumpstateBoardOfSystem(int fd, int64_t maxtime) {
    (void)maxtime;

    DumpFileToFd(fd, "LITTLE cluster time-in-state", "/sys/devices/system/cpu/cpu0/cpufreq/stats/time_in_state");
    //clock master
    DumpFileToFd(fd, "clkmsr", "/sys/kernel/debug/aml_clkmsr/clkmsr");

    //interrupts
    DumpFileToFd(fd, "INTERRUPTS", "/proc/interrupts");

    //page trace
    DumpFileToFd(fd, "pagetrace", "/proc/pagetrace");

    DumpFileToFd(fd, "rdma_mgr", "/sys/module/rdma_mgr/parameters/reset_count");

    //codec_mm
    DumpFileToFd(fd, "codec_mm config", "/sys/class/codec_mm/config");
    DumpFileToFd(fd, "codec_mm dump", "/sys/class/codec_mm/codec_mm_dump");
    DumpFileToFd(fd, "codec_mm keeper", "/sys/class/codec_mm/codec_mm_keeper_dump");
    DumpFileToFd(fd, "codec_mm scatter", "/sys/class/codec_mm/codec_mm_scatter_dump");
    DumpFileToFd(fd, "codec_mm fastplay enable", "/sys/class/codec_mm/fastplay_enable");
    DumpFileToFd(fd, "codec_mm tvp enable", "/sys/class/codec_mm/tvp_enable");
    DumpFileToFd(fd, "codec_mm tvp_region", "/sys/class/codec_mm/tvp_region");

    //dmabuf_manage
    DumpFileToFd(fd, "dmabuf_manage_config", "/sys/class/dmabuf_manage/dmabuf_manage_config");
    DumpFileToFd(fd, "dmabuf_manage_dump", "/sys/class/dmabuf_manage/dmabuf_manage_dump");

    //resource_mgr
    DumpFileToFd(fd, "resource_mgr ver", "/sys/class/resource_mgr/ver");
    DumpFileToFd(fd, "resource_mgr config", "/sys/class/resource_mgr/config");
    DumpFileToFd(fd, "resource_mgr res", "/sys/class/resource_mgr/res");
    DumpFileToFd(fd, "resource_mgr usage", "/sys/class/resource_mgr/usage");

    return;
}

void Dumpstate::dumpstateBoardOfAudio(int fd, int64_t maxtime) {
    (void)maxtime;

    ALOGI("dumpstateBoardOfAudio() Fd: %d", fd);

    //license decoder efuse check
    DumpFileToFd(fd, "Efuse dolby_enable", "/sys/class/amaudio/dolby_enable");
    DumpFileToFd(fd, "Efuse dts_enable", "/sys/class/amaudio/dts_enable");

    return;
}

void Dumpstate::dumpstateBoardOfDisplay(int fd, int64_t maxtime) {
    char buf[PATH_MAX] = { 0 };
    char path[PATH_MAX] = { 0 };

    snprintf(buf, PATH_MAX, "/proc/self/fd/%d", fd);
    if (readlink(buf, path, PATH_MAX - 1) != -1) {
        ALOGI("dumpstateBoardOfDisplay file %s", path);
        RunCommandToFd(fd, "Display", {"/vendor/bin/dumpstate_display", path},
            CommandOptions::WithTimeout(maxtime).Build());
    }

    return;
}

void Dumpstate::dumpstateBoardOfMedia(int fd, int64_t maxtime) {
    uint64_t start = Nanotime() / NANOS_PER_SEC;
    uint64_t elapsed = 0;
    uint64_t rest = 0;

    if (fd > 0) {
        RunCommandToFd(fd, "Dump Drm info", { "drminfo" });
        RunCommandToFd(fd, "Dump Decoder Status", { "cat", "/sys/class/dec_report/status" });
        DumpFileToFd(fd, "Notify Media Service Event", "/sys/class/resource_mgr/res_report");
        elapsed = Nanotime() / NANOS_PER_SEC - start;
        rest = maxtime - elapsed;
        if (rest > 0)
            sleep(rest);
    }
}

ndk::ScopedAStatus Dumpstate::dumpstateBoardImpl(const int fd, const bool full) {
    uint64_t start = Nanotime() / NANOS_PER_SEC;
    uint64_t elapsed = 0;

    ALOGI("DumpstateDevice::dumpstateBoard() FD: %d start at %" PRId64, fd, start);

    dprintf(fd, "verbose logging: %s\n", getVerboseLoggingEnabledImpl() ? "enabled" : "disabled");
    dprintf(fd, "[%s] %s\n", (full ? "full" : "default"), "Hello, world!");

    // Shows an example on how to use the libdumpstateutil API.
    DumpFileToFd(fd, "cmdline", "/proc/self/cmdline");
    elapsed = Nanotime() / NANOS_PER_SEC - start;
    dumpstateBoardOfSystem(fd, 28 - elapsed);
    elapsed = Nanotime() / NANOS_PER_SEC - start;
    dumpstateBoardOfAudio(fd, 28 - elapsed);
    elapsed = Nanotime() / NANOS_PER_SEC - start;
    dumpstateBoardOfDisplay(fd, 28 - elapsed);
    elapsed = Nanotime() / NANOS_PER_SEC - start;
    dumpstateBoardOfMedia(fd, 28 - elapsed);
    elapsed = Nanotime() / NANOS_PER_SEC;
    ALOGI("DumpstateDevice::dumpstateBoard() elapsed total %" PRId64, elapsed - start);
    return ndk::ScopedAStatus::ok();
}

}  // namespace dumpstate
}  // namespace hardware
}  // namespace android
}  // namespace aidl
