/*
 * Copyright (C) 2019-2024 Amlogic, Inc. All rights reserved.
 *
 * All information contained herein is Amlogic confidential.
 *
 */
#include "lib_cve.h"
#include "sample_comm_cve.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

typedef struct {
    CVE_SRC_IMAGE_T stSrc;
    CVE_DST_IMAGE_T stDstH;
    CVE_DST_IMAGE_T stDstV;
    CVE_SOBEL_CTRL_T stSobelCtrl;
    FILE *pFpSrc;
    FILE *pFpDstH;
    FILE *pFpDstV;
} SAMPLE_CVE_SOBEL_INFO_T;

SAMPLE_CVE_SOBEL_INFO_T stSobelInfo;

static AML_CHAR acMask0[25] = {0, 0, 0, 0,  0, 0, -1, 0, 1, 0, 0, -2, 0,
                               2, 0, 0, -1, 0, 1, 0,  0, 0, 0, 0, 0};

static AML_CHAR acMask1[25] = {-1, -2, 0,  2,  1, -4, -8, 0,  8,  4, -6, -12, 0,
                               12, 6,  -4, -8, 0, 8,  4,  -1, -2, 0, 2,  1};

void SAMPLE_CVE_SobelCaseList(void)
{
    printf("\t\t  0) Output:H&V MT:0.\n");
    printf("\t\t  1) Output:H&V MT:1.\n");
    printf("\t\t  2) Output:H MT:0.\n");
    printf("\t\t  3) Output:H MT:1.\n");
    printf("\t\t  4) Output:V MT:0.\n");
    printf("\t\t  5) Output:V MT:1.\n");
    return;
}

static AML_VOID SAMPLE_CVE_SobelUninit(SAMPLE_CVE_SOBEL_INFO_T *pstSobelInfo)
{
    AML_S32 s32Ret = AML_SUCCESS;
    s32Ret = SAMPLE_COMM_CVE_WriteFile(&pstSobelInfo->stDstH, pstSobelInfo->pFpDstH);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Write Dst H Image failed!\n", s32Ret);
    }
    s32Ret = SAMPLE_COMM_CVE_WriteFile(&pstSobelInfo->stDstV, pstSobelInfo->pFpDstV);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Write Dst V Image failed!\n", s32Ret);
    }

    CVE_PMZ_FREE(pstSobelInfo->stSrc.au64PhyAddr[0], pstSobelInfo->stSrc.au64VirAddr[0]);
    CVE_PMZ_FREE(pstSobelInfo->stDstH.au64PhyAddr[0], pstSobelInfo->stDstH.au64VirAddr[0]);
    CVE_PMZ_FREE(pstSobelInfo->stDstV.au64PhyAddr[0], pstSobelInfo->stDstV.au64VirAddr[0]);
    CVE_CLOSE_FILE(pstSobelInfo->pFpSrc);
    CVE_CLOSE_FILE(pstSobelInfo->pFpDstH);
    CVE_CLOSE_FILE(pstSobelInfo->pFpDstV);
}

static AML_S32 SAMPLE_CVE_SobelInit(SAMPLE_CVE_SOBEL_INFO_T *pstSobelInfo, AML_U32 u32ModelType,
                                    AML_U32 u32Width, AML_U32 u32Height, AML_CHAR *pchSrcFileName,
                                    AML_CHAR *pchDstHFileName, AML_CHAR *pchDstVFileName)
{
    AML_S32 s32Ret = AML_SUCCESS;

    memset(&pstSobelInfo->stSrc, 0, sizeof(CVE_SRC_IMAGE_T));
    memset(&pstSobelInfo->stDstH, 0, sizeof(CVE_DST_IMAGE_T));
    memset(&pstSobelInfo->stDstV, 0, sizeof(CVE_DST_IMAGE_T));

    CVE_OPEN_FILE(pstSobelInfo->pFpSrc, pchSrcFileName, "r+");
    CVE_OPEN_FILE(pstSobelInfo->pFpDstH, pchDstHFileName, "w");
    CVE_OPEN_FILE(pstSobelInfo->pFpDstV, pchDstVFileName, "w");

    if (u32ModelType == 0) {
        memcpy(&pstSobelInfo->stSobelCtrl.as8Mask, acMask0, 25);
    } else {
        memcpy(&pstSobelInfo->stSobelCtrl.as8Mask, acMask1, 25);
    }

    s32Ret =
        SAMPLE_COMM_CVE_CreateImage(&pstSobelInfo->stSrc, CVE_IMAGE_TYPE_U8C1, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Src Image failed!\n", s32Ret);
        goto CVE_SOBEL_EXIT1;
    }

    if (pstSobelInfo->stSobelCtrl.enOutCtrl == CVE_SOBEL_OUT_CTRL_BOTH ||
        pstSobelInfo->stSobelCtrl.enOutCtrl == CVE_SOBEL_OUT_CTRL_HOR) {
        s32Ret = SAMPLE_COMM_CVE_CreateImage(&pstSobelInfo->stDstH, CVE_IMAGE_TYPE_S16C1, u32Width,
                                             u32Height);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("[ERROR][%x] Create Dst Image failed!\n", s32Ret);
            goto CVE_SOBEL_EXIT2;
        }
    }

    if (pstSobelInfo->stSobelCtrl.enOutCtrl == CVE_SOBEL_OUT_CTRL_BOTH ||
        pstSobelInfo->stSobelCtrl.enOutCtrl == CVE_SOBEL_OUT_CTRL_VER) {
        s32Ret = SAMPLE_COMM_CVE_CreateImage(&pstSobelInfo->stDstV, CVE_IMAGE_TYPE_S16C1, u32Width,
                                             u32Height);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("[ERROR][%x] Create Dst Image failed!\n", s32Ret);
            goto CVE_SOBEL_EXIT3;
        }
    }

    s32Ret = SAMPLE_COMM_CVE_ReadFile(&pstSobelInfo->stSrc, pstSobelInfo->pFpSrc);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Read Src Image failed!\n", s32Ret);
        goto CVE_SOBEL_EXIT4;
    }
    goto CVE_SOBEL_EXIT0;

CVE_SOBEL_EXIT4:
    CVE_PMZ_FREE(pstSobelInfo->stDstV.au64PhyAddr[0], pstSobelInfo->stDstV.au64VirAddr[0]);
CVE_SOBEL_EXIT3:
    CVE_PMZ_FREE(pstSobelInfo->stDstH.au64PhyAddr[0], pstSobelInfo->stDstH.au64VirAddr[0]);
CVE_SOBEL_EXIT2:
    CVE_PMZ_FREE(pstSobelInfo->stSrc.au64PhyAddr[0], pstSobelInfo->stSrc.au64VirAddr[0]);
CVE_SOBEL_EXIT1:
    CVE_CLOSE_FILE(pstSobelInfo->pFpSrc);
    CVE_CLOSE_FILE(pstSobelInfo->pFpDstH);
    CVE_CLOSE_FILE(pstSobelInfo->pFpDstV);
CVE_SOBEL_EXIT0:

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_SobelProc(SAMPLE_CVE_SOBEL_INFO_T *pstSobelInfo)
{
    AML_BOOL_E bInstant = AML_FALSE;
    AML_BOOL_E bFinish = AML_FALSE;
    CVE_HANDLE cveHandle;
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = CVE_Sobel(&cveHandle, &pstSobelInfo->stSrc, &pstSobelInfo->stDstH,
                       &pstSobelInfo->stDstV, &pstSobelInfo->stSobelCtrl, bInstant);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Sobel Proc error\n");
        return s32Ret;
    }
    s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Sobel Query error\n");
        return s32Ret;
    }

    return s32Ret;
}

AML_S32 SAMPLE_CVE_SobelCase(AML_U32 u32CaseIndex)
{
    AML_U32 u32ModelType;
    CVE_SOBEL_CTRL_T *pstSobelCtrl = &stSobelInfo.stSobelCtrl;
    AML_U32 u32Width = 352;
    AML_U32 u32Height = 288;
    AML_S32 s32Ret;
    AML_CHAR *SrcName = "/data/cve/00_352x288_SP420.yuv";
    AML_CHAR pFpSrcName[64] = {0};
    AML_CHAR *pFpDstHName = "/data/cve/sobel_h.out";
    AML_CHAR *pFpDstVName = "/data/cve/sobel_v.out";
    stSobelInfo.stSobelCtrl.stCrop.enCrop = AML_FALSE;
    stSobelInfo.stSobelCtrl.stCrop.u16X = 0;
    stSobelInfo.stSobelCtrl.stCrop.u16Y = 0;
    stSobelInfo.stSobelCtrl.stCrop.u16Width = 200;
    stSobelInfo.stSobelCtrl.stCrop.u16Height = 200;

    switch (u32CaseIndex) {
    case 0:
        TEST_PRT("[INFO] SOBEL CASE 0: Output:H&V MT:0 \n");
        u32ModelType = 0;
        pstSobelCtrl->enOutCtrl = CVE_SOBEL_OUT_CTRL_BOTH;
        break;
    case 1:
        TEST_PRT("[INFO] SOBEL CASE 1: Output:H&V MT:1\n");
        u32ModelType = 1;
        pstSobelCtrl->enOutCtrl = CVE_SOBEL_OUT_CTRL_BOTH;
        break;
    case 2:
        TEST_PRT("[INFO] SOBEL CASE 2: Output:H MT:0\n");
        u32ModelType = 0;
        pstSobelCtrl->enOutCtrl = CVE_SOBEL_OUT_CTRL_HOR;
        break;
    case 3:
        TEST_PRT("[INFO] SOBEL CASE 3: Output:H MT:1\n");
        u32ModelType = 1;
        pstSobelCtrl->enOutCtrl = CVE_SOBEL_OUT_CTRL_HOR;
        break;
    case 4:
        TEST_PRT("[INFO] SOBEL CASE 4: Output:V MT:0\n");
        u32ModelType = 0;
        pstSobelCtrl->enOutCtrl = CVE_SOBEL_OUT_CTRL_VER;
        break;
    case 5:
        TEST_PRT("[INFO] SOBEL CASE 5: Output:V MT:1\n");
        u32ModelType = 1;
        pstSobelCtrl->enOutCtrl = CVE_SOBEL_OUT_CTRL_VER;
        break;
    default:
        printf("the sobel case index is invalid! valid index:\n");
        SAMPLE_CVE_SobelCaseList();
        return AML_FAILURE;
    }

    printf("please input src file name:");
    SAMPLE_COMM_CVE_Input_FileName(pFpSrcName, sizeof(pFpSrcName), SrcName);
    printf("src file name:%s\n", pFpSrcName);
    printf("dst file name:%s\n", pFpDstHName);
    printf("dst file name:%s\n", pFpDstVName);

    s32Ret = SAMPLE_CVE_SobelInit(&stSobelInfo, u32ModelType, u32Width, u32Height, pFpSrcName,
                                  pFpDstHName, pFpDstVName);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_SobelInit failed!\n", s32Ret);
        return s32Ret;
    }

    s32Ret = SAMPLE_CVE_SobelProc(&stSobelInfo);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_SobelProc failed!\n", s32Ret);
    }

    SAMPLE_CVE_SobelUninit(&stSobelInfo);
    memset(&stSobelInfo, 0, sizeof(SAMPLE_CVE_SOBEL_INFO_T));

    return s32Ret;
}
