/*
 * Copyright (C) 2019-2024 Amlogic, Inc. All rights reserved.
 *
 * All information contained herein is Amlogic confidential.
 *
 */
#include "lib_cve.h"
#include "sample_comm_cve.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

typedef struct {
    CVE_SRC_IMAGE_T stSrc;
    CVE_DST_IMAGE_T stDst;
    CVE_SRC_MEM_INFO_T stMap;
    CVE_MAP_CTRL_T stMapCtrl;
    FILE *pFpSrc;
    FILE *pFpSrcMap;
    FILE *pFpDst;
} SAMPLE_CVE_MAP_INFO_T;

SAMPLE_CVE_MAP_INFO_T stMapInfo;

void SAMPLE_CVE_MapCaseList(void)
{
    printf("\t\t  0) MapMode: U8.\n");
    printf("\t\t  1) MapMode: S16.\n");
    printf("\t\t  2) MapMode: U16.\n");
    return;
}

static AML_VOID SAMPLE_CVE_MapUninit(SAMPLE_CVE_MAP_INFO_T *pstMapInfo)
{
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = SAMPLE_COMM_CVE_WriteFile(&pstMapInfo->stDst, pstMapInfo->pFpDst);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Write Dst Image failed!\n", s32Ret);
    }

    CVE_PMZ_FREE(pstMapInfo->stSrc.au64PhyAddr[0], pstMapInfo->stSrc.au64VirAddr[0]);
    CVE_PMZ_FREE(pstMapInfo->stDst.au64PhyAddr[0], pstMapInfo->stDst.au64VirAddr[0]);
    CVE_PMZ_FREE(pstMapInfo->stMap.u64PhyAddr, pstMapInfo->stMap.u64VirAddr);
    CVE_CLOSE_FILE(pstMapInfo->pFpSrc);
    CVE_CLOSE_FILE(pstMapInfo->pFpSrcMap);
    CVE_CLOSE_FILE(pstMapInfo->pFpDst);
}

static AML_S32 SAMPLE_CVE_MapInit(SAMPLE_CVE_MAP_INFO_T *pstMapInfo, CVE_IMAGE_TYPE_E enDstType,
                                  AML_U32 u32Width, AML_U32 u32Height, AML_CHAR *pchSrcFileName,
                                  AML_CHAR *pchSrcMapFileName, AML_CHAR *pchDstFileName)
{
    AML_S32 s32Ret = AML_SUCCESS;
    AML_U8 *buffer;
    AML_S16 *buffer1;
    AML_U16 *buffer2;
    AML_U32 temp;
    AML_S32 i;
    AML_S32 s32ReadRet = 0;

    memset(&pstMapInfo->stSrc, 0, sizeof(CVE_SRC_IMAGE_T));
    memset(&pstMapInfo->stMap, 0, sizeof(CVE_SRC_MEM_INFO_T));
    memset(&pstMapInfo->stDst, 0, sizeof(CVE_DST_IMAGE_T));

    CVE_OPEN_FILE(pstMapInfo->pFpSrc, pchSrcFileName, "r+");
    CVE_OPEN_FILE(pstMapInfo->pFpSrcMap, pchSrcMapFileName, "r+");
    CVE_OPEN_FILE(pstMapInfo->pFpDst, pchDstFileName, "w");

    s32Ret =
        SAMPLE_COMM_CVE_CreateImage(&pstMapInfo->stSrc, CVE_IMAGE_TYPE_U8C1, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Src Image failed!\n", s32Ret);
        goto CVE_MAP_EXIT1;
    }

    s32Ret = SAMPLE_COMM_CVE_CreateImage(&pstMapInfo->stDst, enDstType, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Dst Image failed!\n", s32Ret);
        goto CVE_MAP_EXIT2;
    }

    s32Ret = SAMPLE_COMM_CVE_CreateMemInfo(&pstMapInfo->stMap, CVE_HIST_NUM * sizeof(AML_U32));
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Dst Mem failed!\n", s32Ret);
        goto CVE_MAP_EXIT3;
    }

    s32Ret = SAMPLE_COMM_CVE_ReadFile(&pstMapInfo->stSrc, pstMapInfo->pFpSrc);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Read Src2 Image failed!\n", s32Ret);
        goto CVE_MAP_EXIT4;
    }

    switch (pstMapInfo->stMapCtrl.enMode) {
    case CVE_MAP_MODE_U8:
        buffer = (AML_U8 *)(AML_UINTPTR_T)pstMapInfo->stMap.u64VirAddr;
        for (i = 0; i < 256; i++) {
            buffer[i] = i;
        }
        break;
    case CVE_MAP_MODE_S16:
        buffer1 = (AML_S16 *)(AML_UINTPTR_T)pstMapInfo->stMap.u64VirAddr;
        for (i = 0; i < 256; i++) {
            s32ReadRet = fread(&temp, 4, 1, pstMapInfo->pFpSrcMap);
            if (s32ReadRet == 0) {
                TEST_PRT("fread error !!!\n");
                return s32ReadRet;
            }
            buffer1[i] = 0 - ((AML_S16)temp);
        }
        break;
    case CVE_MAP_MODE_U16:
        buffer2 = (AML_U16 *)(AML_UINTPTR_T)pstMapInfo->stMap.u64VirAddr;
        for (i = 0; i < 256; i++) {
            s32ReadRet = fread(&temp, 4, 1, pstMapInfo->pFpSrcMap);
            if (s32ReadRet == 0) {
                TEST_PRT("fread error !!!\n");
                return s32ReadRet;
            }
            buffer2[i] = ((AML_U16)temp);
        }
        break;
    default:
        break;
    }

    goto CVE_MAP_EXIT0;

CVE_MAP_EXIT4:
    CVE_PMZ_FREE(pstMapInfo->stMap.u64PhyAddr, pstMapInfo->stMap.u64VirAddr);
CVE_MAP_EXIT3:
    CVE_PMZ_FREE(pstMapInfo->stDst.au64PhyAddr[0], pstMapInfo->stDst.au64VirAddr[0]);
CVE_MAP_EXIT2:
    CVE_PMZ_FREE(pstMapInfo->stSrc.au64PhyAddr[0], pstMapInfo->stSrc.au64VirAddr[0]);
CVE_MAP_EXIT1:
    CVE_CLOSE_FILE(pstMapInfo->pFpSrc);
    CVE_CLOSE_FILE(pstMapInfo->pFpDst);
CVE_MAP_EXIT0:

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_MapProc(SAMPLE_CVE_MAP_INFO_T *pstMapInfo)
{
    AML_BOOL_E bInstant = AML_FALSE;
    AML_BOOL_E bFinish = AML_FALSE;
    CVE_HANDLE cveHandle;
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = CVE_Map(&cveHandle, &pstMapInfo->stSrc, &pstMapInfo->stMap, &pstMapInfo->stDst,
                     &pstMapInfo->stMapCtrl, bInstant);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Map Proc error\n");
        return s32Ret;
    }
    s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Map Query error\n");
        return s32Ret;
    }

    return s32Ret;
}

AML_S32 SAMPLE_CVE_MapCase(AML_U32 u32CaseIndex)
{
    AML_U32 u32Width = 352;
    AML_U32 u32Height = 288;
    AML_S32 s32Ret;
    CVE_MAP_CTRL_T *pstMapCtrl = &stMapInfo.stMapCtrl;
    CVE_IMAGE_TYPE_E enDstType;
    AML_CHAR *SrcName = "/data/cve/00_352x288_y.yuv";
    AML_CHAR *SrcMapName = "/data/cve/hist.out";
    AML_CHAR pFpSrcName[64] = {0};
    AML_CHAR pFpSrcMapName[64] = {0};
    AML_CHAR *pFpDstName = "/data/cve/Map.out";

    stMapInfo.stMapCtrl.stCrop.enCrop = AML_FALSE;
    stMapInfo.stMapCtrl.stCrop.u16X = 0;
    stMapInfo.stMapCtrl.stCrop.u16Y = 0;
    stMapInfo.stMapCtrl.stCrop.u16Width = 200;
    stMapInfo.stMapCtrl.stCrop.u16Height = 200;

    switch (u32CaseIndex) {
    case 0:
        TEST_PRT("[INFO] Map CASE 0: MapMode: U8\n");
        pstMapCtrl->enMode = CVE_MAP_MODE_U8;
        enDstType = CVE_IMAGE_TYPE_U8C1;
        break;
    case 1:
        TEST_PRT("[INFO] Map CASE 1: MapMode: S16\n");
        pstMapCtrl->enMode = CVE_MAP_MODE_S16;
        enDstType = CVE_IMAGE_TYPE_S16C1;
        break;
    case 2:
        TEST_PRT("[INFO] Map CASE 2: MapMode: U16\n");
        pstMapCtrl->enMode = CVE_MAP_MODE_U16;
        enDstType = CVE_IMAGE_TYPE_U16C1;
        break;
    default:
        printf("the map case index is invalid! valid index:\n");
        SAMPLE_CVE_MapCaseList();
        return AML_FAILURE;
    }

    printf("please input src file name:");
    SAMPLE_COMM_CVE_Input_FileName(pFpSrcName, sizeof(pFpSrcName), SrcName);
    printf("please input src Map file name:");
    SAMPLE_COMM_CVE_Input_FileName(pFpSrcMapName, sizeof(pFpSrcMapName), SrcMapName);
    printf("src file name:%s\n", pFpSrcName);
    printf("src mem file name:%s\n", pFpSrcMapName);
    printf("dst file name:%s\n", pFpDstName);

    s32Ret = SAMPLE_CVE_MapInit(&stMapInfo, enDstType, u32Width, u32Height, pFpSrcName,
                                pFpSrcMapName, pFpDstName);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_MapInit failed!\n", s32Ret);
        return s32Ret;
    }

    s32Ret = SAMPLE_CVE_MapProc(&stMapInfo);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_MapProc failed!\n", s32Ret);
    }

    SAMPLE_CVE_MapUninit(&stMapInfo);
    memset(&stMapInfo, 0, sizeof(SAMPLE_CVE_MAP_INFO_T));

    return s32Ret;
}
