/*
 * Copyright (C) 2019-2024 Amlogic, Inc. All rights reserved.
 *
 * All information contained herein is Amlogic confidential.
 *
 */
#include "lib_cve.h"
#include "sample_comm_cve.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>

typedef struct {
    CVE_SRC_IMAGE_T *pstSrc;
    CVE_DST_MEM_INFO_T *pstDst;
    CVE_HIST_CTRL_T stHistCtrl;
} CVE_HIST_INFO_T;

typedef struct {
    CVE_SRC_IMAGE_T *pstSrc;
    CVE_DST_IMAGE_T *pstDst;
    CVE_EQUALIZE_HIST_CTRL_T stEqualizeHistCtrl;
} CVE_EQUALIZE_HIST_INFO_T;

typedef struct {
    CVE_SRC_IMAGE_T stHistSrc;
    CVE_DST_MEM_INFO_T stHistDst;
    FILE *pFpHistSrc;
    FILE *pFpHistDstMem;

    CVE_DST_IMAGE_T stEqHistDst;
    FILE *pFpEqHistSrcMem;
    FILE *pFpEqHistDst;

    CVE_HIST_INFO_T stHistInfo;
    CVE_EQUALIZE_HIST_INFO_T stEqHistInfo;
} SAMPLE_CVE_HIST_EQHIST_INFO_T;

SAMPLE_CVE_HIST_EQHIST_INFO_T *pstHistEqHistInfo;

void SAMPLE_CVE_HistEqHistCaseList(void)
{
    printf("\t\t  0) Hist + eqhist.\n");
    return;
}

static AML_S32 SAMPLE_CVE_HistOpenFile(AML_CHAR *pchHistSrcFileName)
{
    AML_S32 s32Ret = AML_SUCCESS;

    AML_CHAR *pFpHistMemName = "/data/cve/Hist.out";

    CVE_OPEN_FILE(pstHistEqHistInfo->pFpHistSrc, pchHistSrcFileName, "r+");
    CVE_OPEN_FILE(pstHistEqHistInfo->pFpHistDstMem, pFpHistMemName, "w");

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_HistReadFile(AML_VOID)
{
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = SAMPLE_COMM_CVE_ReadFile(&pstHistEqHistInfo->stHistSrc, pstHistEqHistInfo->pFpHistSrc);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Read Src Image failed!\n", s32Ret);
        goto CVE_EXIT;
    }
CVE_EXIT:
    return s32Ret;
}

static AML_S32 SAMPLE_CVE_HistCloseFile(AML_VOID)
{
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = SAMPLE_COMM_CVE_WriteFile_From_Mem(&pstHistEqHistInfo->stHistDst,
                                                pstHistEqHistInfo->pFpHistDstMem);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Write Hist dst reg  failed!\n", s32Ret);
    }
    CVE_CLOSE_FILE(pstHistEqHistInfo->pFpHistDstMem);

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_EqHistOpenFile(AML_VOID)
{
    AML_S32 s32Ret = AML_SUCCESS;
    AML_CHAR *pFpEqHistSrcMemName = "/data/cve/EqHist_Mem.in";
    AML_CHAR *pFpEqHistName = "/data/cve/EqHist.out";

    CVE_OPEN_FILE(pstHistEqHistInfo->pFpEqHistSrcMem, pFpEqHistSrcMemName, "r+");
    CVE_OPEN_FILE(pstHistEqHistInfo->pFpEqHistDst, pFpEqHistName, "w");
    s32Ret =
        SAMPLE_COMM_CVE_ReadFile_On_Mem(&pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stMem,
                                        pstHistEqHistInfo->pFpEqHistSrcMem);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Read EqHist Src mem failed!\n", s32Ret);
    }

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_EqHistCloseFile(AML_VOID)
{
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret =
        SAMPLE_COMM_CVE_WriteFile(&pstHistEqHistInfo->stEqHistDst, pstHistEqHistInfo->pFpEqHistDst);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Write EqHist dst Image failed!\n", s32Ret);
    }
    CVE_CLOSE_FILE(pstHistEqHistInfo->pFpEqHistDst);
    return s32Ret;
}

static AML_VOID SAMPLE_CVE_Hist_EqHist_UnInit(AML_VOID)
{

    CVE_PMZ_FREE(pstHistEqHistInfo->stHistSrc.au64PhyAddr[0],
                 pstHistEqHistInfo->stHistSrc.au64VirAddr[0]);
    CVE_PMZ_FREE(pstHistEqHistInfo->stHistDst.u64PhyAddr, pstHistEqHistInfo->stHistDst.u64VirAddr);
    CVE_PMZ_FREE(pstHistEqHistInfo->stEqHistDst.au64PhyAddr[0],
                 pstHistEqHistInfo->stEqHistDst.au64VirAddr[0]);
    CVE_PMZ_FREE(pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stMem.u64PhyAddr,
                 pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stMem.u64VirAddr);

    free(pstHistEqHistInfo);
}

static AML_S32 SAMPLE_CVE_Hist_EqHist_Init(AML_VOID)
{
    AML_S32 s32Ret = AML_SUCCESS;

    pstHistEqHistInfo->stHistInfo.pstSrc = &pstHistEqHistInfo->stHistSrc;
    pstHistEqHistInfo->stHistInfo.pstDst = &pstHistEqHistInfo->stHistDst;

    pstHistEqHistInfo->stEqHistInfo.pstSrc = &pstHistEqHistInfo->stHistSrc;
    pstHistEqHistInfo->stEqHistInfo.pstDst = &pstHistEqHistInfo->stEqHistDst;

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_HistInit(CVE_HIST_INFO_T *pstHistInfo, AML_U32 u32Width,
                                   AML_U32 u32Height)
{
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret =
        SAMPLE_COMM_CVE_CreateImage(pstHistInfo->pstSrc, CVE_IMAGE_TYPE_U8C1, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Src Image failed!\n", s32Ret);
        goto CVE_EXIT;
    }

    s32Ret = SAMPLE_COMM_CVE_CreateMemInfo(pstHistInfo->pstDst, CVE_HIST_NUM * sizeof(AML_U32) * 2);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Dst Mem failed!\n", s32Ret);
        goto CVE_EXIT;
    }

CVE_EXIT:

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_HistProc(CVE_HIST_INFO_T *pstHistInfo)
{
    AML_BOOL_E bInstant = AML_FALSE;
    AML_BOOL_E bFinish = AML_FALSE;
    CVE_HANDLE cveHandle;
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = CVE_Hist(&cveHandle, pstHistInfo->pstSrc, pstHistInfo->pstDst,
                      &pstHistInfo->stHistCtrl, bInstant);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Hist Proc error\n");
        return s32Ret;
    }
    s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Hist Query error\n");
        return s32Ret;
    }

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_EqHistInit(CVE_EQUALIZE_HIST_INFO_T *pstEqHistInfo, AML_U32 u32Width,
                                     AML_U32 u32Height)
{
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = SAMPLE_COMM_CVE_CreateImage(pstEqHistInfo->pstDst, CVE_IMAGE_TYPE_U8C1, u32Width,
                                         u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Dst Image failed!\n", s32Ret);
        goto CVE_EXIT;
    }
    s32Ret = SAMPLE_COMM_CVE_CreateMemInfo(&pstEqHistInfo->stEqualizeHistCtrl.stMem,
                                           sizeof(CVE_EQUALIZE_HIST_CTRL_MEM_T));
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Src Mem failed!\n", s32Ret);
        goto CVE_EXIT;
    }

CVE_EXIT:
    return s32Ret;
}

static AML_S32 SAMPLE_CVE_EqHistProc(CVE_EQUALIZE_HIST_INFO_T *pstEqHistInfo)
{
    AML_BOOL_E bInstant = AML_FALSE;
    AML_BOOL_E bFinish = AML_FALSE;
    CVE_HANDLE cveHandle;
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = CVE_EqualizeHist(&cveHandle, pstEqHistInfo->pstSrc, pstEqHistInfo->pstDst,
                              &pstEqHistInfo->stEqualizeHistCtrl, bInstant);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE EqHist Proc error\n");
        return s32Ret;
    }
    s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE EqHist Query error\n");
        return s32Ret;
    }

    return s32Ret;
}

AML_S32 SAMPLE_CVE_TEST_Time_Proc(AML_VOID)
{
    AML_BOOL_E bInstant = AML_FALSE;
    AML_BOOL_E bFinish = AML_FALSE;
    CVE_HANDLE cveHandle;
    AML_S32 s32Ret = AML_SUCCESS;
    AML_S32 i;
    CVE_HIST_INFO_T *pstHistInfo = &pstHistEqHistInfo->stHistInfo;
    CVE_EQUALIZE_HIST_INFO_T *pstEqHistInfo = &pstHistEqHistInfo->stEqHistInfo;
    struct timeval time;
    AML_S32 start_op_time = 0, end_op_time = 0, all_op_time = 0;
    CVE_DATA_T stDmaSrc;
    CVE_DATA_T stDmaDst;
    CVE_DMA_CTRL_T stDmaCtrl;

    stDmaCtrl.enMode = CVE_DMA_MODE_DIRECT_COPY;
    stDmaCtrl.stCrop.enCrop = AML_FALSE;
    stDmaSrc.dmafd = pstEqHistInfo->pstSrc->dmafd;
    stDmaSrc.u64VirAddr = pstEqHistInfo->pstSrc->au64VirAddr[0];
    stDmaSrc.u64PhyAddr = pstEqHistInfo->pstSrc->au64PhyAddr[0];
    stDmaSrc.u32Width = pstEqHistInfo->pstSrc->u32Width;
    stDmaSrc.u32Height = pstEqHistInfo->pstSrc->u32Height;
    stDmaSrc.u32Stride = pstEqHistInfo->pstSrc->au32Stride[0];
    stDmaDst.dmafd = pstEqHistInfo->pstDst->dmafd;
    stDmaDst.u64VirAddr = pstEqHistInfo->pstDst->au64VirAddr[0];
    stDmaDst.u64PhyAddr = pstEqHistInfo->pstDst->au64PhyAddr[0];
    stDmaDst.u32Width = pstEqHistInfo->pstDst->u32Width;
    stDmaDst.u32Height = pstEqHistInfo->pstDst->u32Height;
    stDmaDst.u32Stride = pstEqHistInfo->pstDst->au32Stride[0];

    for (i = 0; i < 400; i++) {
        gettimeofday(&time, NULL);
        start_op_time = time.tv_sec * 1000000 + time.tv_usec;

        s32Ret = CVE_Hist(&cveHandle, pstHistInfo->pstSrc, pstHistInfo->pstDst,
                          &pstHistInfo->stHistCtrl, bInstant);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("CVE Hist Proc error\n");
            return s32Ret;
        }
        s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("CVE EqHist Query error\n");
            return s32Ret;
        }
        s32Ret = CVE_DMA(&cveHandle, &stDmaSrc, &stDmaDst, &stDmaCtrl, bInstant);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("CVE DMA Proc error\n");
            return s32Ret;
        }
        s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("CVE EqHist Query error\n");
            return s32Ret;
        }

        s32Ret = CVE_EqualizeHist(&cveHandle, pstEqHistInfo->pstSrc, pstEqHistInfo->pstDst,
                                  &pstEqHistInfo->stEqualizeHistCtrl, bInstant);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("CVE EqHist Proc error\n");
            return s32Ret;
        }
        s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("CVE EqHist Query error\n");
            return s32Ret;
        }

        s32Ret = CVE_DMA(&cveHandle, &stDmaSrc, &stDmaDst, &stDmaCtrl, bInstant);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("CVE DMA Proc error\n");
            return s32Ret;
        }
        s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
        if (s32Ret != AML_SUCCESS) {
            TEST_PRT("CVE EqHist Query error\n");
            return s32Ret;
        }
        gettimeofday(&time, NULL);
        end_op_time = time.tv_sec * 1000000 + time.tv_usec;
        all_op_time += end_op_time - start_op_time;
    }

    s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE EqHist Query error\n");
        return s32Ret;
    }
    printf("%s %d op used_time:%d us\n", __FUNCTION__, __LINE__, (all_op_time) / i);
    return s32Ret;
}

AML_S32 SAMPLE_CVE_Hist_EqHist_Case(AML_U32 u32CaseIndex)
{
    AML_U32 u32Width = 720;
    AML_U32 u32Height = 576;
    AML_S32 s32Ret;
    AML_CHAR *pFpSrcName = "/data/cve/penguinx.yuv";

    pstHistEqHistInfo = malloc(sizeof(SAMPLE_CVE_HIST_EQHIST_INFO_T));
    pstHistEqHistInfo->stHistInfo.stHistCtrl.stCrop.enCrop = AML_FALSE;
    pstHistEqHistInfo->stHistInfo.stHistCtrl.stCrop.u16X = 0;
    pstHistEqHistInfo->stHistInfo.stHistCtrl.stCrop.u16Y = 0;
    pstHistEqHistInfo->stHistInfo.stHistCtrl.stCrop.u16Width = 60;
    pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stCrop.u16Height = 60;
    pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stCrop.enCrop = AML_FALSE;
    pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stCrop.u16X = 0;
    pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stCrop.u16Y = 0;
    pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stCrop.u16Width = 60;
    pstHistEqHistInfo->stEqHistInfo.stEqualizeHistCtrl.stCrop.u16Height = 60;

    if (pstHistEqHistInfo == NULL) {
        s32Ret = AML_FAILURE;
        TEST_PRT("malloc memory error\n");
        return s32Ret;
    }
    memset(pstHistEqHistInfo, 0, sizeof(SAMPLE_CVE_HIST_EQHIST_INFO_T));

    switch (u32CaseIndex) {
    case 0:
        TEST_PRT("[INFO] Hist + EqHist.\n");
        break;
    default:
        printf("the thresh case index is invalid! valid index:\n");
        SAMPLE_CVE_HistEqHistCaseList();
        return AML_FAILURE;
    }

    printf("src file name:%s\n", pFpSrcName);

    s32Ret = SAMPLE_CVE_Hist_EqHist_Init();
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_Hist_Map_Init failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_HistInit(&pstHistEqHistInfo->stHistInfo, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_HistInit failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_HistOpenFile(pFpSrcName);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_HistOpenFile failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_HistReadFile();
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_HistReadFile failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_HistProc(&pstHistEqHistInfo->stHistInfo);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_HistProc failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_HistCloseFile();
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_HistCloseFile failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_EqHistInit(&pstHistEqHistInfo->stEqHistInfo, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_EqHistInit failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_EqHistOpenFile();
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_EqHistOpenFile failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_EqHistProc(&pstHistEqHistInfo->stEqHistInfo);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_EqHistProc failed!\n", s32Ret);
        goto Exit_Func;
    }
    s32Ret = SAMPLE_CVE_EqHistCloseFile();
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_EqHistCloseFile failed!\n", s32Ret);
        goto Exit_Func;
    }

    s32Ret = SAMPLE_CVE_TEST_Time_Proc();
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_TEST_Time_Proc failed!\n", s32Ret);
        goto Exit_Func;
    }

Exit_Func:
    SAMPLE_CVE_Hist_EqHist_UnInit();

    return s32Ret;
}
