/*
 * Copyright (C) 2019-2024 Amlogic, Inc. All rights reserved.
 *
 * All information contained herein is Amlogic confidential.
 *
 */
#include "lib_cve.h"
#include "sample_comm_cve.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

typedef struct {
    CVE_SRC_IMAGE_T stSrc;
    CVE_DST_IMAGE_T stDst;
    CVE_FILTER_CTRL_T stFltCtrl;
    FILE *pFpSrc;
    FILE *pFpDst;
} SAMPLE_CVE_FILTER_INFO_T;

SAMPLE_CVE_FILTER_INFO_T stFilterInfo;

static AML_CHAR acMask0[25] = {0, 0, 0, 0, 0, 0, 1, 2, 1, 0, 0, 2, 4,
                               2, 0, 0, 1, 2, 1, 0, 0, 0, 0, 0, 0};

static AML_CHAR acMask1[25] = {1, 2, 3, 2, 1, 2, 5, 6, 5, 2, 3, 6, 8,
                               6, 3, 2, 5, 6, 5, 2, 1, 2, 3, 2, 1};

void SAMPLE_CVE_FilterCaseList(void)
{
    printf("\t\t  0) SrcType:U8C1 DstType:U8C1 FilterType:1.\n");
    printf("\t\t  1) SrcType:YUV420SP DstType:YUV420SP FilterType:0.\n");
    printf("\t\t  2) SrcType:YUV420SP DstType:YUV420SP FilterType:1.\n");
    printf("\t\t  3) SrcType:YUV422SP DstType:YUV422SP FilterType:0.\n");
    printf("\t\t  4) SrcType:YUV422SP DstType:YUV422SP FilterType:1.\n");
    return;
}

static AML_VOID SAMPLE_CVE_FilterUninit(SAMPLE_CVE_FILTER_INFO_T *pstFilterInfo)
{
    AML_S32 s32Ret = AML_SUCCESS;
    s32Ret = SAMPLE_COMM_CVE_WriteFile(&pstFilterInfo->stDst, pstFilterInfo->pFpDst);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Write Dst Image failed!\n", s32Ret);
    }

    CVE_PMZ_FREE(pstFilterInfo->stSrc.au64PhyAddr[0], pstFilterInfo->stSrc.au64VirAddr[0]);
    CVE_PMZ_FREE(pstFilterInfo->stDst.au64PhyAddr[0], pstFilterInfo->stDst.au64VirAddr[0]);
    CVE_CLOSE_FILE(pstFilterInfo->pFpSrc);
    CVE_CLOSE_FILE(pstFilterInfo->pFpDst);
}

static AML_S32 SAMPLE_CVE_FilterInit(SAMPLE_CVE_FILTER_INFO_T *pstFilterInfo,
                                     CVE_IMAGE_TYPE_E enSrcType, CVE_IMAGE_TYPE_E enDstType,
                                     AML_CHAR *pchSrcFileName, AML_CHAR *pchDstFileName,
                                     AML_U32 u32FilterType, AML_U32 u32Width, AML_U32 u32Height)
{
    AML_S32 s32Ret = AML_SUCCESS;

    memset(&pstFilterInfo->stSrc, 0, sizeof(CVE_SRC_IMAGE_T));
    memset(&pstFilterInfo->stDst, 0, sizeof(CVE_DST_IMAGE_T));

    CVE_OPEN_FILE(pstFilterInfo->pFpSrc, pchSrcFileName, "r+");
    CVE_OPEN_FILE(pstFilterInfo->pFpDst, pchDstFileName, "w");

    if (u32FilterType == 0) {
        memcpy(&pstFilterInfo->stFltCtrl.as8Mask, acMask0, 25);
        pstFilterInfo->stFltCtrl.u8Norm = 4;
    } else {
        memcpy(&pstFilterInfo->stFltCtrl.as8Mask, acMask1, 25);
        pstFilterInfo->stFltCtrl.u8Norm = 7;
    }

    s32Ret = SAMPLE_COMM_CVE_CreateImage(&pstFilterInfo->stSrc, enSrcType, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Src Image failed!\n", s32Ret);
        goto CVE_FILTER_EXIT1;
    }

    s32Ret = SAMPLE_COMM_CVE_CreateImage(&pstFilterInfo->stDst, enDstType, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Dst Image failed!\n", s32Ret);
        goto CVE_FILTER_EXIT2;
    }

    s32Ret = SAMPLE_COMM_CVE_ReadFile(&pstFilterInfo->stSrc, pstFilterInfo->pFpSrc);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Open Src Image failed!\n", s32Ret);
        goto CVE_FILTER_EXIT3;
    }
    goto CVE_FILTER_EXIT0;

CVE_FILTER_EXIT3:
    CVE_PMZ_FREE(pstFilterInfo->stDst.au64PhyAddr[0], pstFilterInfo->stDst.au64VirAddr[0]);
CVE_FILTER_EXIT2:
    CVE_PMZ_FREE(pstFilterInfo->stSrc.au64PhyAddr[0], pstFilterInfo->stSrc.au64VirAddr[0]);
CVE_FILTER_EXIT1:
    CVE_CLOSE_FILE(pstFilterInfo->pFpSrc);
    CVE_CLOSE_FILE(pstFilterInfo->pFpDst);
CVE_FILTER_EXIT0:

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_FilterProc(SAMPLE_CVE_FILTER_INFO_T *pstFilterInfo)
{
    AML_BOOL_E bInstant = AML_FALSE;
    AML_BOOL_E bFinish = AML_FALSE;
    CVE_HANDLE cveHandle;
    AML_S32 s32Ret = AML_SUCCESS;

    s32Ret = CVE_Filter(&cveHandle, &pstFilterInfo->stSrc, &pstFilterInfo->stDst,
                        &pstFilterInfo->stFltCtrl, bInstant);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Filter Proc error\n");
        return s32Ret;
    }
    s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Filter Query error\n");
        return s32Ret;
    }

    return s32Ret;
}

AML_S32 SAMPLE_CVE_FilterCase(AML_U32 u32CaseIndex)
{
    CVE_IMAGE_TYPE_E enSrcType;
    CVE_IMAGE_TYPE_E enDstType;
    AML_U32 u32FilterType;
    AML_U32 u32Width = 352;
    AML_U32 u32Height = 288;
    AML_S32 s32Ret;
    AML_CHAR *SrcName = "/data/cve/00_352x288_SP420.yuv";
    AML_CHAR pFpSrcName[64] = {0};
    AML_CHAR *pFpDstName = "/data/cve/filter.out";
    stFilterInfo.stFltCtrl.stCrop.enCrop = AML_FALSE;
    stFilterInfo.stFltCtrl.stCrop.u16X = 0;
    stFilterInfo.stFltCtrl.stCrop.u16Y = 0;
    stFilterInfo.stFltCtrl.stCrop.u16Width = 200;
    stFilterInfo.stFltCtrl.stCrop.u16Height = 200;

    switch (u32CaseIndex) {
    case 0:
        TEST_PRT("[INFO] FILTER CASE 0: SrcType:U8C1 DstType:U8C1 FilterType:1\n");
        enSrcType = CVE_IMAGE_TYPE_U8C1;
        enDstType = CVE_IMAGE_TYPE_U8C1;
        u32FilterType = 1;
        break;
    case 1:
        TEST_PRT("[INFO] FILTER CASE 1: SrcType:YUV420SP DstType:YUV420SP FilterType:0\n");
        enSrcType = CVE_IMAGE_TYPE_YUV420SP;
        enDstType = CVE_IMAGE_TYPE_YUV420SP;
        u32FilterType = 0;
        break;
    case 2:
        TEST_PRT("[INFO] FILTER CASE 2: SrcType:YUV420SP DstType:YUV420SP FilterType:1\n");
        enSrcType = CVE_IMAGE_TYPE_YUV420SP;
        enDstType = CVE_IMAGE_TYPE_YUV420SP;
        u32FilterType = 1;
        break;
    case 3:
        TEST_PRT("[INFO] FILTER CASE 3: SrcType:YUV422SP DstType:YUV422SP FilterType:0\n");
        enSrcType = CVE_IMAGE_TYPE_YUV422SP;
        enDstType = CVE_IMAGE_TYPE_YUV422SP;
        u32FilterType = 0;
        break;
    case 4:
        TEST_PRT("[INFO] FILTER CASE 4: SrcType:YUV422SP DstType:YUV422SP FilterType:1\n");
        enSrcType = CVE_IMAGE_TYPE_YUV422SP;
        enDstType = CVE_IMAGE_TYPE_YUV422SP;
        u32FilterType = 1;
        break;
    default:
        printf("the filter case index is invalid! valid index:\n");
        SAMPLE_CVE_FilterCaseList();
        return AML_FAILURE;
    }

    printf("please input src file name:");
    SAMPLE_COMM_CVE_Input_FileName(pFpSrcName, sizeof(pFpSrcName), SrcName);
    printf("src file name:%s\n", pFpSrcName);
    printf("dst file name:%s\n", pFpDstName);

    s32Ret = SAMPLE_CVE_FilterInit(&stFilterInfo, enSrcType, enDstType, pFpSrcName, pFpDstName,
                                   u32FilterType, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_FilterInit failed!\n", s32Ret);
        return s32Ret;
    }

    s32Ret = SAMPLE_CVE_FilterProc(&stFilterInfo);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_FilterProc failed!\n", s32Ret);
    }

    SAMPLE_CVE_FilterUninit(&stFilterInfo);
    memset(&stFilterInfo, 0, sizeof(SAMPLE_CVE_FILTER_INFO_T));

    return s32Ret;
}
