/*
 * Copyright (C) 2019-2024 Amlogic, Inc. All rights reserved.
 *
 * All information contained herein is Amlogic confidential.
 *
 */
#include "lib_cve.h"
#include "sample_comm_cve.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

typedef enum {
    CVE_OP_SUB = 0x0,
    CVE_OP_OR = 0x1,
    CVE_OP_XOR = 0x2,
    CVE_OP_AND = 0x3,
    CVE_OP_ADD = 0x4,
} CVE_ALU_OP_E;

typedef struct {
    CVE_SRC_IMAGE_T stSrc1;
    CVE_SRC_IMAGE_T stSrc2;
    CVE_DST_IMAGE_T stDst;
    CVE_DST_MEM_INFO_T stMemDst;
    CVE_ALU_OP_E enOp;
    CVE_SUB_CTRL_T stSubCtrl;
    CVE_ADD_CTRL_T stAddCtrl;
    CVE_AND_CTRL_T stAndCtrl;
    CVE_OR_CTRL_T stOrCtrl;
    CVE_XOR_CTRL_T stXorCtrl;
    FILE *pFpSrc1;
    FILE *pFpSrc2;
    FILE *pFpDst;
} SAMPLE_CVE_ALU_INFO_T;

SAMPLE_CVE_ALU_INFO_T stAluInfo;

void SAMPLE_CVE_AluCaseList(void)
{
    printf("\t\t  0) OP:Sub OM: Abs.\n");
    printf("\t\t  1) OP:Sub OM: Shift.\n");
    printf("\t\t  2) OP:Sub OM: Thresh.\n");
    printf("\t\t  3) OP:Or.\n");
    printf("\t\t  4) OP:Xor.\n");
    printf("\t\t  5) OP:And.\n");
    printf("\t\t  6) OP:Add.\n");

    return;
}

static AML_VOID SAMPLE_CVE_AluUninit(SAMPLE_CVE_ALU_INFO_T *pstAluInfo)
{
    AML_S32 s32Ret = AML_SUCCESS;
    s32Ret = SAMPLE_COMM_CVE_WriteFile(&pstAluInfo->stDst, pstAluInfo->pFpDst);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Write Dst Image failed!\n", s32Ret);
    }
    CVE_PMZ_FREE(pstAluInfo->stSrc1.au64PhyAddr[0], pstAluInfo->stSrc1.au64VirAddr[0]);
    CVE_PMZ_FREE(pstAluInfo->stSrc2.au64PhyAddr[0], pstAluInfo->stSrc2.au64VirAddr[0]);
    CVE_PMZ_FREE(pstAluInfo->stDst.au64PhyAddr[0], pstAluInfo->stDst.au64VirAddr[0]);
    CVE_PMZ_FREE(pstAluInfo->stMemDst.u64PhyAddr, pstAluInfo->stMemDst.u64VirAddr);
    CVE_CLOSE_FILE(pstAluInfo->pFpSrc1);
    CVE_CLOSE_FILE(pstAluInfo->pFpSrc2);
    CVE_CLOSE_FILE(pstAluInfo->pFpDst);
}

static AML_S32 SAMPLE_CVE_AluInit(SAMPLE_CVE_ALU_INFO_T *pstAluInfo, AML_U32 u32Width,
                                  AML_U32 u32Height, AML_CHAR *pchSrc1FileName,
                                  AML_CHAR *pchSrc2FileName, AML_CHAR *pchDstFileName)
{
    AML_S32 s32Ret = AML_SUCCESS;
    CVE_IMAGE_TYPE_E enDstImageType = CVE_IMAGE_TYPE_U8C1;

    memset(&pstAluInfo->stSrc1, 0, sizeof(CVE_SRC_IMAGE_T));
    memset(&pstAluInfo->stSrc2, 0, sizeof(CVE_SRC_IMAGE_T));
    memset(&pstAluInfo->stDst, 0, sizeof(CVE_DST_IMAGE_T));

    CVE_OPEN_FILE(pstAluInfo->pFpSrc1, pchSrc1FileName, "r+");
    CVE_OPEN_FILE(pstAluInfo->pFpSrc2, pchSrc2FileName, "r+");
    CVE_OPEN_FILE(pstAluInfo->pFpDst, pchDstFileName, "w");

    if (pstAluInfo->enOp == CVE_OP_SUB && pstAluInfo->stSubCtrl.enMode == CVE_SUB_MODE_SHIFT) {
        enDstImageType = CVE_IMAGE_TYPE_S8C1;
    }
    s32Ret =
        SAMPLE_COMM_CVE_CreateImage(&pstAluInfo->stSrc1, CVE_IMAGE_TYPE_U8C1, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Src Image failed!\n", s32Ret);
        goto CVE_ALU_EXIT1;
    }

    s32Ret =
        SAMPLE_COMM_CVE_CreateImage(&pstAluInfo->stSrc2, CVE_IMAGE_TYPE_U8C1, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Src Image failed!\n", s32Ret);
        goto CVE_ALU_EXIT2;
    }

    s32Ret = SAMPLE_COMM_CVE_CreateImage(&pstAluInfo->stDst, enDstImageType, u32Width, u32Height);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Dst Image failed!\n", s32Ret);
        goto CVE_ALU_EXIT3;
    }

    s32Ret = SAMPLE_COMM_CVE_CreateMemInfo(&pstAluInfo->stMemDst, sizeof(AML_U32) * 2);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Create Dst Mem failed!\n", s32Ret);
        goto CVE_ALU_EXIT5;
    }

    s32Ret = SAMPLE_COMM_CVE_ReadFile(&pstAluInfo->stSrc1, pstAluInfo->pFpSrc1);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Read Src1 Image failed!\n", s32Ret);
        goto CVE_ALU_EXIT4;
    }
    s32Ret = SAMPLE_COMM_CVE_ReadFile(&pstAluInfo->stSrc2, pstAluInfo->pFpSrc2);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] Read Src2 Image failed!\n", s32Ret);
        goto CVE_ALU_EXIT4;
    }
    goto CVE_ALU_EXIT0;

CVE_ALU_EXIT5:
    CVE_PMZ_FREE(pstAluInfo->stMemDst.u64PhyAddr, pstAluInfo->stMemDst.u64VirAddr);
CVE_ALU_EXIT4:
    CVE_PMZ_FREE(pstAluInfo->stDst.au64PhyAddr[0], pstAluInfo->stDst.au64VirAddr[0]);
CVE_ALU_EXIT3:
    CVE_PMZ_FREE(pstAluInfo->stSrc2.au64PhyAddr[0], pstAluInfo->stSrc2.au64VirAddr[0]);
CVE_ALU_EXIT2:
    CVE_PMZ_FREE(pstAluInfo->stSrc1.au64PhyAddr[0], pstAluInfo->stSrc1.au64VirAddr[0]);
CVE_ALU_EXIT1:
    CVE_CLOSE_FILE(pstAluInfo->pFpSrc1);
    CVE_CLOSE_FILE(pstAluInfo->pFpSrc2);
    CVE_CLOSE_FILE(pstAluInfo->pFpDst);
CVE_ALU_EXIT0:

    return s32Ret;
}

static AML_S32 SAMPLE_CVE_AluProc(SAMPLE_CVE_ALU_INFO_T *pstAluInfo)
{
    AML_BOOL_E bInstant = AML_FALSE;
    AML_BOOL_E bFinish = AML_FALSE;
    CVE_HANDLE cveHandle;
    AML_S32 s32Ret = AML_SUCCESS;

    if (pstAluInfo->enOp == CVE_OP_SUB) {
        s32Ret = CVE_Sub(&cveHandle, &pstAluInfo->stSrc1, &pstAluInfo->stSrc2, &pstAluInfo->stDst,
                         &pstAluInfo->stMemDst, &pstAluInfo->stSubCtrl, bInstant);
    } else if (pstAluInfo->enOp == CVE_OP_OR) {
        s32Ret = CVE_Or(&cveHandle, &pstAluInfo->stSrc1, &pstAluInfo->stSrc2, &pstAluInfo->stDst,
                        &pstAluInfo->stOrCtrl, bInstant);
    } else if (pstAluInfo->enOp == CVE_OP_XOR) {
        s32Ret = CVE_Xor(&cveHandle, &pstAluInfo->stSrc1, &pstAluInfo->stSrc2, &pstAluInfo->stDst,
                         &pstAluInfo->stXorCtrl, bInstant);
    } else if (pstAluInfo->enOp == CVE_OP_AND) {
        s32Ret = CVE_And(&cveHandle, &pstAluInfo->stSrc1, &pstAluInfo->stSrc2, &pstAluInfo->stDst,
                         &pstAluInfo->stAndCtrl, bInstant);
    } else if (pstAluInfo->enOp == CVE_OP_ADD) {
        s32Ret = CVE_Add(&cveHandle, &pstAluInfo->stSrc1, &pstAluInfo->stSrc2, &pstAluInfo->stDst,
                         &pstAluInfo->stMemDst, &pstAluInfo->stAddCtrl, bInstant);
    } else {
        TEST_PRT("[ERROR]Invalid mode!\n");
        s32Ret = AML_FAILURE;
    }
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Alu Proc error\n");
        return s32Ret;
    }
    s32Ret = CVE_Query(cveHandle, &bFinish, AML_TRUE);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("CVE Alu Query error\n");
        return s32Ret;
    }

    return s32Ret;
}

AML_S32 SAMPLE_CVE_AluCase(AML_U32 u32CaseIndex)
{
    AML_U32 u32Width = 352;
    AML_U32 u32Height = 288;
    AML_S32 s32Ret;
    AML_CHAR *Src1Name = "/data/cve/00_352x288_y.yuv";
    AML_CHAR *Src2Name = "/data/cve/02_352x288_y.yuv";
    AML_CHAR pFpSrc1Name[64] = {0};
    AML_CHAR pFpSrc2Name[64] = {0};
    AML_CHAR *pFpDstName = "/data/cve/Alu.out";
    stAluInfo.stAddCtrl.stCrop.enCrop = AML_FALSE;
    stAluInfo.stAddCtrl.stCrop.u16X = 0;
    stAluInfo.stAddCtrl.stCrop.u16Y = 0;
    stAluInfo.stAddCtrl.stCrop.u16Width = 160;
    stAluInfo.stAddCtrl.stCrop.u16Height = 160;

    stAluInfo.stSubCtrl.stCrop.enCrop = AML_FALSE;
    stAluInfo.stSubCtrl.stCrop.u16X = 0;
    stAluInfo.stSubCtrl.stCrop.u16Y = 0;
    stAluInfo.stSubCtrl.stCrop.u16Width = 160;
    stAluInfo.stSubCtrl.stCrop.u16Height = 160;

    stAluInfo.stAndCtrl.stCrop.enCrop = AML_FALSE;
    stAluInfo.stAndCtrl.stCrop.u16X = 0;
    stAluInfo.stAndCtrl.stCrop.u16Y = 0;
    stAluInfo.stAndCtrl.stCrop.u16Width = 160;
    stAluInfo.stAndCtrl.stCrop.u16Height = 160;

    stAluInfo.stOrCtrl.stCrop.enCrop = AML_FALSE;
    stAluInfo.stOrCtrl.stCrop.u16X = 0;
    stAluInfo.stOrCtrl.stCrop.u16Y = 0;
    stAluInfo.stOrCtrl.stCrop.u16Width = 160;
    stAluInfo.stOrCtrl.stCrop.u16Height = 160;

    stAluInfo.stXorCtrl.stCrop.enCrop = AML_FALSE;
    stAluInfo.stXorCtrl.stCrop.u16X = 0;
    stAluInfo.stXorCtrl.stCrop.u16Y = 0;
    stAluInfo.stXorCtrl.stCrop.u16Width = 160;
    stAluInfo.stXorCtrl.stCrop.u16Height = 160;

    switch (u32CaseIndex) {
    case 0:
        TEST_PRT("[INFO] Alu CASE 0: OP:Sub OM: Abs\n");
        stAluInfo.enOp = CVE_OP_SUB;
        stAluInfo.stSubCtrl.enMode = CVE_SUB_MODE_ABS;
        break;
    case 1:
        TEST_PRT("[INFO] Alu CASE 1: OP:Sub OM: Shift\n");
        stAluInfo.enOp = CVE_OP_SUB;
        stAluInfo.stSubCtrl.enMode = CVE_SUB_MODE_SHIFT;
        break;
    case 2:
        TEST_PRT("[INFO] Alu CASE 2: OP:Sub OM: Thresh\n");
        stAluInfo.enOp = CVE_OP_SUB;
        stAluInfo.stSubCtrl.enMode = CVE_SUB_MODE_THRESH;
        stAluInfo.stSubCtrl.u16ThreshRatio = 13107;
        break;
    case 3:
        TEST_PRT("[INFO] Alu CASE 2: OP:Or\n");
        stAluInfo.enOp = CVE_OP_OR;
        break;
    case 4:
        TEST_PRT("[INFO] Alu CASE 3: OP:Xor\n");
        stAluInfo.enOp = CVE_OP_XOR;
        break;
    case 5:
        TEST_PRT("[INFO] Alu CASE 4: OP:And\n");
        stAluInfo.enOp = CVE_OP_AND;
        break;
    case 6:
        TEST_PRT("[INFO] Alu CASE 5: OP:Add\n");
        stAluInfo.enOp = CVE_OP_ADD;
        stAluInfo.stAddCtrl.u0q16X = 32768;
        stAluInfo.stAddCtrl.u0q16Y = 65536 - stAluInfo.stAddCtrl.u0q16X;
        break;
    default:
        printf("the alu case index is invalid! valid index:\n");
        SAMPLE_CVE_AluCaseList();
        return AML_FAILURE;
    }

    printf("please input src1 file name:");
    SAMPLE_COMM_CVE_Input_FileName(pFpSrc1Name, sizeof(pFpSrc1Name), Src1Name);
    printf("please input src2 file name:");
    SAMPLE_COMM_CVE_Input_FileName(pFpSrc2Name, sizeof(pFpSrc2Name), Src2Name);
    printf("src1 file name:%s\n", pFpSrc1Name);
    printf("src2 file name:%s\n", pFpSrc2Name);
    printf("dst file name:%s\n", pFpDstName);

    s32Ret =
        SAMPLE_CVE_AluInit(&stAluInfo, u32Width, u32Height, pFpSrc1Name, pFpSrc2Name, pFpDstName);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_AluInit failed!\n", s32Ret);
        return s32Ret;
    }

    s32Ret = SAMPLE_CVE_AluProc(&stAluInfo);
    if (s32Ret != AML_SUCCESS) {
        TEST_PRT("[ERROR][%x] SAMPLE_CVE_AluProc failed!\n", s32Ret);
    }

    SAMPLE_CVE_AluUninit(&stAluInfo);
    memset(&stAluInfo, 0, sizeof(SAMPLE_CVE_ALU_INFO_T));

    return s32Ret;
}
