/*
 * Copyright (c) 2019 Amlogic, Inc. All rights reserved.
 *
 * This source code is subject to the terms and conditions defined in below
 * which is part of this source code package.
 *
 * Memory Allocator functions for ion
 *
 */
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "IONmem.h"
#include "aml_type.h"
#include "cve.h"
#include "cve_alloc.h"

#define ION_BUFF_NUM 32

struct cve_ion_buffer ion_buffer[ION_BUFF_NUM];
struct usr_ctx_s *ctx = NULL;

int create_ctx(struct usr_ctx_s *ctx)
{
    memset(ctx, 0, sizeof(struct usr_ctx_s));
    ctx->fd = -1;
    ctx->ion_dev = -1;

    ctx->ion_dev = ion_mem_init();
    if (ctx->ion_dev < 0) {
        printf("ion open failed error=%d, %s", errno, strerror(errno));
        return -1;
    }
    return 0;
}

int destroy_ctx(struct usr_ctx_s *ctx)
{
    if (ctx->cmemParm != NULL) {
        free((IONMEM_AllocParams *)ctx->cmemParm);
        ctx->cmemParm = NULL;
    }
    if (ctx->i_buff != NULL) {
        munmap(ctx->i_buff, ctx->i_len);
        ctx->i_buff = NULL;
        ctx->i_len = 0;
    }
    if (ctx->ion_dev >= 0) {
        ion_mem_exit(ctx->ion_dev);
    }

    return 0;
}

int ion_alloc_dma_buffer(struct usr_ctx_s *ctx, uint32_t type, size_t len)
{
    int ret = -1;
    int dma_fd = -1;

    switch (type) {
    case INPUT_BUFF_TYPE:
        ctx->cmemParm = malloc(sizeof(IONMEM_AllocParams));
        if (ctx->cmemParm == NULL)
            return -1;
        ret = ion_mem_alloc(ctx->ion_dev, len, ctx->cmemParm, false);
        if (ret < 0) {
            printf("%s,%d,Not enough memory\n", __func__, __LINE__);
            return -1;
        }
        dma_fd = ((IONMEM_AllocParams *)ctx->cmemParm)->mImageFd;
        ctx->i_len = len;
        ctx->i_buff = (char *)(mmap(NULL, len, PROT_READ | PROT_WRITE, MAP_SHARED, dma_fd, 0));
        if (!ctx->i_buff) {
            ctx->i_buff = NULL;
            printf("Failed to alloc i_buff:%s\n", strerror(errno));
            return -1;
        }
        ctx->fd = dma_fd;
        break;
    default:
        printf("Error no such buff type\n");
        break;
    }
    return dma_fd;
}

int alloc_dma_buffer(struct usr_ctx_s *ctx, uint32_t type, size_t len)
{
    return ion_alloc_dma_buffer(ctx, type, len);
}

int init_ion_buffer(void)
{
    memset(ion_buffer, 0, sizeof(ion_buffer));
    ctx = (struct usr_ctx_s *)malloc(sizeof(struct usr_ctx_s) * ION_BUFF_NUM);
    return 0;
}

static int find_empty_buffer(void)
{
    int i;

    for (i = 0; i < ION_BUFF_NUM; i++) {
        if (ion_buffer[i].used == AML_FALSE) {
            ion_buffer[i].used = AML_TRUE;
            break;
        }
    }
    if (i == ION_BUFF_NUM) {
        i = -1;
    }
    ion_buffer[i].ctx = &ctx[i];
    if (create_ctx(ion_buffer[i].ctx)) {
        return -1;
    }
    return i;
}

static int find_used_buffer(AML_U64 *u64VirAddr)
{
    int i;
    for (i = 0; i < ION_BUFF_NUM; i++) {
        if (ion_buffer[i].viraddr == *u64VirAddr) {
            break;
        }
    }
    if (i == ION_BUFF_NUM) {
        i = -1;
    }
    return i;
}

int cve_alloc_ion(int *dmafd, AML_VOID **u64VirAddr, AML_U32 len)
{
    int ret;
    int num;

    num = find_empty_buffer();
    if (num == -1) {
        return AML_FAILURE;
    }
    ion_buffer[num].size = len;

    ret = alloc_dma_buffer(ion_buffer[num].ctx, INPUT_BUFF_TYPE, len);
    if (ret < 0)
        return AML_FAILURE;
    ion_buffer[num].dmafd = ret;
    *u64VirAddr = (AML_VOID *)ion_buffer[num].ctx->i_buff;
    *dmafd = ret;
    return AML_SUCCESS;
}

int cve_free_ion(int dmafd, AML_VOID *u64VirAddr)
{
    int num;

    num = find_used_buffer((AML_U64 *)u64VirAddr);
    if (num == -1) {
        return AML_FAILURE;
    }
    destroy_ctx(ion_buffer[num].ctx);
    dmafd = 0;

    memset(&ion_buffer[num], 0, sizeof(struct cve_ion_buffer));
    return 0;
}
