// SPDX-License-Identifier: (GPL-2.0+ OR MIT)
/*
 * Copyright (C) 2019-2024 Amlogic, Inc. All rights reserved.
 */
#include "cve_op.h"

#define CVE_GET_HI64(x) (((x)&0xFFFFFFFF00000000ULL) >> 32)
#define CVE_GET_LO64(x) ((x)&0x00000000FFFFFFFFULL)
#define CVE_MIN(a, b) ((a) < (b) ? (a) : (b))
#define CVE_MAX(a, b) ((a) > (b) ? (a) : (b))

#define CMD_QUEUE_ADDR(op, reg, rbase, hmask, lmask)                                               \
    (((op) << OP_SHIFT) | (((reg + rbase) & 0x3ff) << ADDR_SHIFT) | (hmask << HMASK_SHIFT) |       \
     (lmask << LMASK_SHIFT))

#define CMD_QUEUE_DATE(val, hmask, lmask)                                                          \
    (val >> lmask & ((~(unsigned int)0 >> (31 - hmask + lmask))))

#define CMD_QUEUE_CREATE_PREPARE(op, buf)                                                          \
    unsigned int cmd_buf_offset = 0;                                                               \
    unsigned int cmd_op = op;

#define CMD_QUEUE_ADD_PROCESS(reg, reg_, val, n)                                                   \
    cmd_buf[cmd_buf_offset++] = CMD_QUEUE_DATE(val, reg_##HMASK##n, reg_##LMASK##n);               \
    cmd_buf[cmd_buf_offset++] =                                                                    \
        CMD_QUEUE_ADDR(cmd_op, reg, reg_##BASE, reg_##HMASK##n, reg_##LMASK##n);

#define CMD_QUEUE_ADD_PROCESS_1(reg, val) CMD_QUEUE_ADD_PROCESS(reg, reg##_, val, 0)

#ifdef CVE_DEBUG
#define CMD_QUEUE_DEBUG_DUMP(name)                                                                 \
    dump_cmd_queue("/tmp/cve_cmd_queue_" #name ".txt", cmd_buf, cmd_buf_offset / 2);               \
    parse_cmd_queue("/tmp/cve_cmd_queue_" #name "_p.txt", cmd_buf, cmd_buf_offset / 2);
#else
#define CMD_QUEUE_DEBUG_DUMP(name)
#endif

#define CMD_QUEUE_ADD_COMMON(comm)                                                                 \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG0, comm.reg_02.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_RDMIF_PACK_MODE, comm.reg_d8.reg);                                 \
    CMD_QUEUE_ADD_PROCESS_1(CVE_RDMIF1_PACK_MODE, comm.reg_e0.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_RDMIF2_PACK_MODE, comm.reg_e8.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_WRMIF_PACK_MODE, comm.reg_c0.reg);                                 \
    CMD_QUEUE_ADD_PROCESS_1(CVE_WRMIF1_PACK_MODE, comm.reg_c8.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_WRMIF2_PACK_MODE, comm.reg_d0.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG1_0, comm.reg_03.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG1_1, comm.reg_04.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG1_2, comm.reg_05.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG1_3, comm.reg_06.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG1_4, comm.reg_07.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG1_5, comm.reg_08.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG1_6, comm.reg_09.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG1_7, comm.reg_0a.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG2_0, comm.reg_0b.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG2_1, comm.reg_0c.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG2_2, comm.reg_0d.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG2_3, comm.reg_0e.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG2_4, comm.reg_0f.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG3_0, comm.reg_10.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG3_1, comm.reg_11.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG3_2, comm.reg_12.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG3_3, comm.reg_13.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG4_0, comm.reg_14.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG4_1, comm.reg_15.reg);                              \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG5, comm.reg_16.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG6, comm.reg_17.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG7, comm.reg_18.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG8, comm.reg_19.reg);                                \
    CMD_QUEUE_ADD_PROCESS_1(CVE_COMMON_CTRL_REG9, comm.reg_1a.reg);

#define CMD_QUEUE_END(reg)                                                                         \
    cmd_buf[cmd_buf_offset++] = CMD_QUEUE_DATE(0x0, reg##_HMASK0, reg##_LMASK0);                   \
    cmd_buf[cmd_buf_offset++] =                                                                    \
        CMD_QUEUE_ADDR(cmd_op, reg, reg##_BASE, reg##_HMASK0, reg##_LMASK0);                       \
    cmd_buf[cmd_buf_offset++] = CMD_QUEUE_DATE(0x1, reg##_HMASK0, reg##_LMASK0);                   \
    cmd_buf[cmd_buf_offset++] =                                                                    \
        CMD_QUEUE_ADDR(cmd_op, reg, reg##_BASE, reg##_HMASK0, reg##_LMASK0);                       \
    cmd_buf[cmd_buf_offset++] = CMD_QUEUE_DATE(0x0, reg##_HMASK0, reg##_LMASK0);                   \
    cmd_buf[cmd_buf_offset++] = CMD_QUEUE_ADDR(0x3ff, reg, reg##_BASE, reg##_HMASK0, reg##_LMASK0);

#define CMD_QUEUE_CREATE_END CMD_QUEUE_END(CVE_TOP_CTRL_REG0)

#define CMD_QUEUE_RETURN return cmd_buf_offset >> 1;

typedef enum {
    CVE_DMA = 0,
    CVE_LUMA,
    CVE_FILTER,
    CVE_CSC,
    CVE_FILTER_AND_CSC,
    CVE_SOBEL,
    CVE_MAG_AND_ANG,
    CVE_MATCH_BG_MODEL,
    CVE_DILATE,
    CVE_ERODE,
    CVE_THRESH,
    CVE_ALU_SUB,
    CVE_ALU_OR,
    CVE_ALU_AND,
    CVE_ALU_XOR,
    CVE_ALU_ADD,
    CVE_INTEG,
    CVE_HIST,
    CVE_THRESH_S16,
    CVE_THRESH_U16,
    CVE_16BIT_TO_8BIT,
    CVE_ORD_STAT_FILTER,
    CVE_MAP,
    CVE_EQUALIZE_HIST,
    CVE_NCC,
    CVE_CCL,
    CVE_GMM,
    CVE_CANNY_EDGE,
    CVE_LBP,
    CVE_NORM_GRAD,
    CVE_BUILD_LK_OPTICAL_FLOW_PYR,
    CVE_LK_OPTICAL_FLOW_PYR,
    CVE_ST_CANDI_CORNER,
    CVE_SAD,
    CVE_GRAD_FG,
    CVE_UPDATE_BG_MODEL,
    CVE_TOF,
    CVE_MODULE_NUMS,
} CVE_MODULE_OPS;

typedef struct {
    CVE_MODULE_OPS ops;
    AML_U32 min_width;
    AML_U32 min_height;
    AML_U32 max_width;
    AML_U32 max_height;
} CVE_Module_Image_Size;

CVE_Module_Image_Size module_img_size[] = {
    {CVE_DMA, 64, 64, 7680, 4320},
    {CVE_LUMA, 64, 64, 1920, 1080},
    {CVE_FILTER, 64, 64, 1920, 1080},
    {CVE_CSC, 64, 64, 1920, 1080},
    {CVE_FILTER_AND_CSC, 64, 64, 1920, 1080},
    {CVE_SOBEL, 64, 64, 1920, 1080},
    {CVE_MAG_AND_ANG, 64, 64, 1920, 1080},
    {CVE_MATCH_BG_MODEL, 64, 64, 1280, 720},
    {CVE_DILATE, 64, 64, 1920, 1080},
    {CVE_ERODE, 64, 64, 1920, 1080},
    {CVE_THRESH, 64, 64, 1920, 1080},
    {CVE_ALU_SUB, 64, 64, 1920, 1080},
    {CVE_ALU_OR, 64, 64, 1920, 1080},
    {CVE_ALU_AND, 64, 64, 1920, 1080},
    {CVE_ALU_XOR, 64, 64, 1920, 1080},
    {CVE_ALU_ADD, 64, 64, 1920, 1080},
    {CVE_INTEG, 64, 64, 1920, 1080},
    {CVE_HIST, 64, 64, 1920, 1080},
    {CVE_THRESH_S16, 64, 64, 1920, 1080},
    {CVE_THRESH_U16, 64, 64, 1920, 1080},
    {CVE_16BIT_TO_8BIT, 64, 64, 1920, 1080},
    {CVE_ORD_STAT_FILTER, 64, 64, 1920, 1080},
    {CVE_MAP, 64, 64, 1920, 1080},
    {CVE_EQUALIZE_HIST, 64, 64, 1920, 1080},
    {CVE_NCC, 64, 64, 1920, 1080},
    {CVE_CCL, 64, 64, 1920, 1080},
    {CVE_GMM, 64, 64, 1920, 1080},
    {CVE_CANNY_EDGE, 64, 64, 1920, 1080},
    {CVE_LBP, 64, 64, 1920, 1080},
    {CVE_NORM_GRAD, 64, 64, 1920, 1080},
    {CVE_BUILD_LK_OPTICAL_FLOW_PYR, 64, 64, 1280, 720},
    {CVE_LK_OPTICAL_FLOW_PYR, 64, 64, 1280, 720},
    {CVE_ST_CANDI_CORNER, 64, 64, 1920, 1080},
    {CVE_SAD, 64, 64, 1920, 1080},
    {CVE_GRAD_FG, 64, 64, 1920, 1080},
    {CVE_UPDATE_BG_MODEL, 64, 64, 1280, 720},
    {CVE_TOF, 64, 64, 640, 480},
};

#define MODULE_MIN_WIDTH image_sz->min_width
#define MODULE_MIN_HEIGHT image_sz->min_height
#define MODULE_MAX_WIDTH image_sz->max_width
#define MODULE_MAX_HEIGHT image_sz->max_height

#define CVE_GET_IMAGE_SIZE(module_name)                                                            \
    CVE_Module_Image_Size *image_sz;                                                               \
    image_sz = get_image_size(module_name);                                                        \
    if (image_sz == NULL) {                                                                        \
        return AML_ERR_CVE_ILLEGAL_PARAM;                                                          \
    }

#define CVE_DATA_CHECK(module_name, cve_data)                                                      \
    do {                                                                                           \
        if (cve_data->dmafd == 0) {                                                                \
            CVE_ERR_TRACE("[%s] %s->dmafd=%d\n", #module_name, #cve_data, cve_data->dmafd);        \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if ((cve_data->u32Width < MODULE_MIN_WIDTH) || (cve_data->u32Width > MODULE_MAX_WIDTH) ||  \
            (cve_data->u32Height < MODULE_MIN_HEIGHT) ||                                           \
            (cve_data->u32Height > MODULE_MAX_HEIGHT)) {                                           \
            CVE_ERR_TRACE("[%s] %s->u32Width=%d %s->u32Height=%d error\n", #module_name,           \
                          #cve_data, cve_data->u32Width, #cve_data, cve_data->u32Height);          \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_data->u32Stride != CVE_ALIGN_UP(cve_data->u32Width, CVE_ALIGN)) {                  \
            CVE_ERR_TRACE("[%s] %s->u32Stride=%d error\n", #module_name, #cve_data,                \
                          cve_data->u32Stride);                                                    \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_check_mem(cve_data->u64PhyAddr, cve_data->u32Stride * cve_data->u32Height, 1)) {   \
            CVE_ERR_TRACE("[%s] Memory area is invalid\n", #module_name);                          \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_DATA_CROP_CHECK(module_name, cve_crop, dst, src)                                       \
    do {                                                                                           \
        if ((cve_crop.u16Width > MODULE_MAX_WIDTH) || (cve_crop.u16Height > MODULE_MAX_HEIGHT)) {  \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d %s->u16Height=%d error\n", #module_name,           \
                          #cve_crop, cve_crop.u16Width, #cve_crop, cve_crop.u16Height);            \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width + cve_crop.u16X > src->u32Width) {                                   \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Width=%d error\n", #module_name,         \
                          #cve_crop, cve_crop.u16Width, #src, src->u32Width);                      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Height + cve_crop.u16Y > src->u32Height) {                                 \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Height=%d error\n", #module_name,        \
                          #cve_crop, cve_crop.u16Width, #src, src->u32Height);                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width % 2 != 0) {                                                          \
            CVE_ERR_TRACE("[%s] %s->u16Width must be even number", #module_name, #cve_crop);       \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width > dst->u32Width) {                                                   \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Width=%d error\n", #module_name,         \
                          #cve_crop, cve_crop.u16Width, #dst, dst->u32Width);                      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Height > dst->u32Height) {                                                 \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Height=%d error\n", #module_name,        \
                          #cve_crop, cve_crop.u16Width, #dst, dst->u32Height);                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_MEM_CHECK(module_name, cve_mem)                                                        \
    do {                                                                                           \
        if (cve_mem->dmafd == 0) {                                                                 \
            CVE_ERR_TRACE("[%s] %s->dmafd=%d\n", #module_name, #cve_mem, cve_mem->dmafd);          \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_mem->u32Size == 0) {                                                               \
            CVE_ERR_TRACE("[%s] %s->u32Size=%d error\n", #module_name, #cve_mem,                   \
                          cve_mem->u32Size);                                                       \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_check_mem(cve_mem->u64PhyAddr, cve_mem->u32Size, CVE_ALIGN)) {                     \
            CVE_ERR_TRACE("[%s] Memory area is invalid\n", #module_name);                          \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_IMAGE_CHECK(module_name, cve_image)                                                    \
    do {                                                                                           \
        if (cve_image->dmafd == 0) {                                                               \
            CVE_ERR_TRACE("[%s] %s->dmafd=%d\n", #module_name, #cve_image, cve_image->dmafd);      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if ((cve_image->u32Width < MODULE_MIN_WIDTH) ||                                            \
            (cve_image->u32Width > MODULE_MAX_WIDTH) ||                                            \
            (cve_image->u32Height < MODULE_MIN_HEIGHT) ||                                          \
            (cve_image->u32Height > MODULE_MAX_HEIGHT)) {                                          \
            CVE_ERR_TRACE("[%s] %s->u32Width=%d %s->u32Height=%d error\n", #module_name,           \
                          #cve_image, cve_image->u32Width, #cve_image, cve_image->u32Height);      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_check_image_stride(cve_image)) {                                                   \
            CVE_ERR_TRACE("[%s] Stride is invalid\n", #module_name);                               \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_check_image_mem(cve_image, CVE_ALIGN)) {                                           \
            CVE_ERR_TRACE("[%s] Image memory area is invalid\n", #module_name);                    \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_IMAGE_CSC_CHECK(module_name, cve_image)                                                \
    do {                                                                                           \
        if (cve_image->dmafd == 0) {                                                               \
            CVE_ERR_TRACE("[%s] %s->dmafd=%d\n", #module_name, #cve_image, cve_image->dmafd);      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if ((cve_image->u32Width > MODULE_MAX_WIDTH) ||                                            \
            (cve_image->u32Height > MODULE_MAX_HEIGHT)) {                                          \
            CVE_ERR_TRACE("[%s] %s->u32Width=%d %s->u32Height=%d error\n", #module_name,           \
                          #cve_image, cve_image->u32Width, #cve_image, cve_image->u32Height);      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_check_image_stride(cve_image)) {                                                   \
            CVE_ERR_TRACE("[%s] Stride is invalid\n", #module_name);                               \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_check_image_mem(cve_image, CVE_ALIGN)) {                                           \
            CVE_ERR_TRACE("[%s] Image memory area is invalid\n", #module_name);                    \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_IMAGE_CROP_CHECK(module_name, cve_crop, dst, src)                                      \
    do {                                                                                           \
        if ((cve_crop.u16Width > MODULE_MAX_WIDTH) || (cve_crop.u16Height > MODULE_MAX_HEIGHT)) {  \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d %s->u16Height=%d error\n", #module_name,           \
                          #cve_crop, cve_crop.u16Width, #cve_crop, cve_crop.u16Height);            \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width + cve_crop.u16X > src->u32Width) {                                   \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Width=%d error\n", #module_name,         \
                          #cve_crop, cve_crop.u16Width, #src, src->u32Width);                      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Height + cve_crop.u16Y > src->u32Height) {                                 \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Height=%d error\n", #module_name,        \
                          #cve_crop, cve_crop.u16Width, #src, src->u32Height);                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width % 2 != 0) {                                                          \
            CVE_ERR_TRACE("[%s] %s->u16Width must be even number", #module_name, #cve_crop);       \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width > dst->u32Width) {                                                   \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Width=%d error\n", #module_name,         \
                          #cve_crop, cve_crop.u16Width, #dst, dst->u32Width);                      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Height > dst->u32Height) {                                                 \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Height=%d error\n", #module_name,        \
                          #cve_crop, cve_crop.u16Width, #dst, dst->u32Height);                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_ALU_CROP_CHECK(module_name, cve_crop, dst, src)                                        \
    do {                                                                                           \
        if ((cve_crop.u16Width > MODULE_MAX_WIDTH) || (cve_crop.u16Height > MODULE_MAX_HEIGHT)) {  \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d %s->u16Height=%d error\n", #module_name,           \
                          #cve_crop, cve_crop.u16Width, #cve_crop, cve_crop.u16Height);            \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width + cve_crop.u16X > src->u32Width) {                                   \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Width=%d error\n", #module_name,         \
                          #cve_crop, cve_crop.u16Width, #src, src->u32Width);                      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Height + cve_crop.u16Y > src->u32Height) {                                 \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Height=%d error\n", #module_name,        \
                          #cve_crop, cve_crop.u16Width, #src, src->u32Height);                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width % 2 != 0) {                                                          \
            CVE_ERR_TRACE("[%s] %s->u16Width must be even number", #module_name, #cve_crop);       \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Height % 2 != 0) {                                                         \
            CVE_ERR_TRACE("[%s] %s->u16Height must be even number", #module_name, #cve_crop);      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width > dst->u32Width) {                                                   \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Width=%d error\n", #module_name,         \
                          #cve_crop, cve_crop.u16Width, #dst, dst->u32Width);                      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Height > dst->u32Height) {                                                 \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Height=%d error\n", #module_name,        \
                          #cve_crop, cve_crop.u16Width, #dst, dst->u32Height);                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_SRC_CROP_CHECK(module_name, cve_crop, src)                                             \
    do {                                                                                           \
        if ((cve_crop.u16Width > MODULE_MAX_WIDTH) || (cve_crop.u16Height > MODULE_MAX_HEIGHT)) {  \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d %s->u16Height=%d error\n", #module_name,           \
                          #cve_crop, cve_crop.u16Width, #cve_crop, cve_crop.u16Height);            \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width + cve_crop.u16X > src->u32Width) {                                   \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Width=%d error\n", #module_name,         \
                          #cve_crop, cve_crop.u16Width, #src, src->u32Width);                      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Height + cve_crop.u16Y > src->u32Height) {                                 \
            CVE_ERR_TRACE("[%s] %s->u16Width=%d != %s->u32Height=%d error\n", #module_name,        \
                          #cve_crop, cve_crop.u16Width, #src, src->u32Height);                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_crop.u16Width % 2 != 0) {                                                          \
            CVE_ERR_TRACE("[%s] %s->u16Width must be even number", #module_name, #cve_crop);       \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_RAW_CHECK(module_name, cve_raw)                                                        \
    do {                                                                                           \
        if (cve_raw->dmafd == 0) {                                                                 \
            CVE_ERR_TRACE("[%s] %s->dmafd=%d\n", #module_name, #cve_raw, cve_raw->dmafd);          \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if ((cve_raw->u32Width < MODULE_MIN_WIDTH) || (cve_raw->u32Width > MODULE_MAX_WIDTH) ||    \
            (cve_raw->u32Height < MODULE_MIN_HEIGHT) ||                                            \
            (cve_raw->u32Height > MODULE_MAX_HEIGHT)) {                                            \
            CVE_ERR_TRACE("[%s] %s->u32Width=%d %s->u32Height=%d error\n", #module_name, #cve_raw, \
                          cve_raw->u32Width, #cve_raw, cve_raw->u32Height);                        \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_raw->u32Stride != CVE_ALIGN_UP(cve_raw->u32Width, CVE_ALIGN)) {                    \
            CVE_ERR_TRACE("[%s] Stride is invalid\n", #module_name);                               \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
        if (cve_check_mem(cve_raw->u64PhyAddr, cve_raw->u32Stride * cve_raw->u32Height,            \
                          CVE_ALIGN)) {                                                            \
            CVE_ERR_TRACE("[%s] Raw memory area is invalid\n", #module_name);                      \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_IMAGE_COMPARE(module_name, srcimage, dstimage)                                         \
    do {                                                                                           \
        if ((srcimage->u32Width < dstimage->u32Width) ||                                           \
            (srcimage->u32Height < dstimage->u32Height)) {                                         \
            CVE_ERR_TRACE("[%s] %s->u32Width(%d) %s->u32Width(%d) set error\n", #module_name,      \
                          #srcimage, srcimage->u32Width, #dstimage, dstimage->u32Width);           \
            CVE_ERR_TRACE("[%s] %s->u32Height(%d) %s->u32Height(%d) set error\n", #module_name,    \
                          #srcimage, srcimage->u32Height, #dstimage, dstimage->u32Height);         \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define RESOLUTION_DS_NONE 0
#define RESOLUTION_DS_MB_2X2 1
#define RESOLUTION_DS_MB_4X4 2
#define RESOLUTION_DS_MB_8X8 3
#define RESOLUTION_DS_MB_16X16 4
#define CVE_RESOLUTION_EQUAL(module_name, src, dst, ds)                                            \
    do {                                                                                           \
        if ((src->u32Width >> ds != dst->u32Width) || (src->u32Height >> ds != dst->u32Height)) {  \
            CVE_ERR_TRACE("[%s] %s->u32Width(%d) %s->u32Width(%d) set error\n", #module_name,      \
                          #src, src->u32Width, #dst, dst->u32Width);                               \
            CVE_ERR_TRACE("[%s] %s->u32Height(%d) %s->u32Height(%d) set error\n", #module_name,    \
                          #src, src->u32Height, #dst, dst->u32Height);                             \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_RANGE_CHECK(module_name, param, min, max)                                              \
    do {                                                                                           \
        if ((param > max) || (param < min)) {                                                      \
            CVE_ERR_TRACE("[%s] [%s](%d) set value error, value "                                  \
                          "must in[%d, %d]\n",                                                     \
                          #module_name, #param, param, min, max);                                  \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_UNSIGNED_RANGE_CHECK(module_name, param, max)                                          \
    do {                                                                                           \
        if (param > max) {                                                                         \
            CVE_ERR_TRACE("[%s] [%s](%d) set value error, value "                                  \
                          "must in[0, %d]\n",                                                      \
                          #module_name, #param, param, max);                                       \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_MODE_CHECK(module_name, mode, mode_butt)                                               \
    do {                                                                                           \
        if (mode >= mode_butt) {                                                                   \
            CVE_ERR_TRACE("[%s][%s] not support this mode [%d], valid range [0, %u]\n",            \
                          #module_name, #mode, mode, mode_butt - 1);                               \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_BOOL_CHECK(module_name, bool)                                                          \
    do {                                                                                           \
        if (bool != AML_TRUE && bool != AML_FALSE) {                                               \
            CVE_ERR_TRACE("[%s][%s] (%d) must be AML_TRUE or AML_FALSE!\n", #module_name, #bool,   \
                          bool);                                                                   \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_IMAGE_TYPE_CHECK(module_name, src, type)                                               \
    do {                                                                                           \
        if (src->enType != type) {                                                                 \
            CVE_ERR_TRACE("[%s] %s->enType(%d) set error, need set %s.\n", #module_name, #src,     \
                          src->enType, #type);                                                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#define CVE_IMAGE_TYPE_EQUAL(module_name, src, dst)                                                \
    do {                                                                                           \
        if (src->enType != dst->enType) {                                                          \
            CVE_ERR_TRACE("[%s] %s->enType(%d) not equal %s->enType(%d), need equal.\n",           \
                          #module_name, #src, src->enType, #dst, dst->enType);                     \
            return AML_ERR_CVE_ILLEGAL_PARAM;                                                      \
        }                                                                                          \
    } while (0)

#ifdef CVE_DEBUG
static void dump_cmd_queue(const char *file_name, unsigned int *cmd_buf, int cmd_num)
{
    int i = 0;

    for (i = 0; i < cmd_num; i++) {
        printk("0x%08x%08x\n", cmd_buf[2 * i + 1], cmd_buf[2 * i + 0]);
    }
    return;
}
static void parse_cmd_queue(char *file_name, unsigned int *cmd_buf, int cmd_num)
{
    int i = 0;
    unsigned int comq = 0;
    unsigned int value = 0;

    for (i = 0; i < cmd_num; i++) {
        value = cmd_buf[2 * i];
        comq = cmd_buf[2 * i + 1];
        printk("10'h%03x, 12'h%03x, 5'h%02x, 5'h%02x, 32'h%08x\n", (comq >> 22),
               CMD_QUEUE_DATE(comq, 21, 10), CMD_QUEUE_DATE(comq, 9, 5), CMD_QUEUE_DATE(comq, 4, 0),
               value);
    }

    return;
}
#endif

static CVE_Module_Image_Size *get_image_size(CVE_MODULE_OPS ops)
{
    int i;
    for (i = 0; i < CVE_MODULE_NUMS; i++) {
        if (module_img_size[i].ops == ops) {
            return &module_img_size[i];
        }
    }
    if (i == CVE_MODULE_NUMS) {
        CVE_ERR_TRACE("not find cve module image size set.");
    }
    return NULL;
}

static int cve_check_mem(unsigned long long phy_addr, unsigned long addr_len, unsigned int aligned)
{
    int ret = 0;

    if (aligned != 1 && (unsigned int)phy_addr & (aligned - 1)) {
        CVE_ERR_TRACE("phy_addr(0x%llx) must be %d byte align!\n", phy_addr, aligned);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }

    return ret;
}

static int cve_check_image_mem(CVE_IMAGE_T *pstImg, unsigned int aligned)
{
    unsigned int mem_size = 0;

    if (NULL == pstImg) {
        CVE_ERR_TRACE("pstImg is null\n");
        return AML_FAILURE;
    }

    switch (pstImg->enType) {
    case CVE_IMAGE_TYPE_U8C1:
    case CVE_IMAGE_TYPE_S8C1:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height;
        break;
    case CVE_IMAGE_TYPE_YVU420SP:
    case CVE_IMAGE_TYPE_YUV420SP:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * 3 / 2;
        break;
    case CVE_IMAGE_TYPE_YVU422SP:
    case CVE_IMAGE_TYPE_YUV422SP:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * 2;
        break;
    case CVE_IMAGE_TYPE_YUV420P:
        mem_size =
            pstImg->au32Stride[0] * pstImg->u32Height + pstImg->au32Stride[1] * pstImg->u32Height;
        break;
    case CVE_IMAGE_TYPE_YUV422P:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height +
                   pstImg->au32Stride[1] * pstImg->u32Height * 2;

        break;
    case CVE_IMAGE_TYPE_S8C2_PACKAGE:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * 2;
        break;
    case CVE_IMAGE_TYPE_S8C2_PLANAR:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * 2;
        break;
    case CVE_IMAGE_TYPE_S16C1:
    case CVE_IMAGE_TYPE_U16C1:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * sizeof(AML_U16);
        break;
    case CVE_IMAGE_TYPE_U8C3_PACKAGE:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * 3;
        break;
    case CVE_IMAGE_TYPE_U8C3_PLANAR:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * 3;
        break;
    case CVE_IMAGE_TYPE_S32C1:
    case CVE_IMAGE_TYPE_U32C1:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * sizeof(AML_U32);
        break;
    case CVE_IMAGE_TYPE_S64C1:
    case CVE_IMAGE_TYPE_U64C1:
        mem_size = pstImg->au32Stride[0] * pstImg->u32Height * sizeof(AML_U64);
        break;
    default:
        CVE_ERR_TRACE("Unknown image type[%d]\n", pstImg->enType);
        return AML_ERR_CVE_ILLEGAL_PARAM;
        break;
    }

    if (cve_check_mem(pstImg->au64PhyAddr[0], mem_size, aligned)) {
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }

    return 0;
}

static int cve_check_image_stride(CVE_IMAGE_T *pstImg)
{
    unsigned int stride = 0;

    if (NULL == pstImg) {
        CVE_ERR_TRACE("pstImg is null\n");
        return AML_FAILURE;
    }

    stride = CVE_ALIGN_UP(pstImg->u32Width, CVE_ALIGN);

    switch (pstImg->enType) {
    case CVE_IMAGE_TYPE_U8C1:
    case CVE_IMAGE_TYPE_S8C1:
    case CVE_IMAGE_TYPE_S16C1:
    case CVE_IMAGE_TYPE_U16C1:
    case CVE_IMAGE_TYPE_S32C1:
    case CVE_IMAGE_TYPE_U32C1:
    case CVE_IMAGE_TYPE_S64C1:
    case CVE_IMAGE_TYPE_U64C1:
        if (pstImg->au32Stride[0] != stride) {
            CVE_ERR_TRACE("Invalid stride0 [%d], expected stride0 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[0], stride, pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    case CVE_IMAGE_TYPE_YVU420SP:
    case CVE_IMAGE_TYPE_YVU422SP:
    case CVE_IMAGE_TYPE_YUV420SP:
    case CVE_IMAGE_TYPE_YUV422SP:
    case CVE_IMAGE_TYPE_S8C2_PACKAGE:
    case CVE_IMAGE_TYPE_S8C2_PLANAR:
        if (pstImg->au32Stride[0] != stride) {
            CVE_ERR_TRACE("Invalid stride0 [%d], expected stride0 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[0], stride, pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        if (pstImg->au32Stride[1] != stride) {
            CVE_ERR_TRACE("Invalid stride1 [%d], expected stride1 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[1], stride, pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    case CVE_IMAGE_TYPE_YUV420P:
    case CVE_IMAGE_TYPE_YUV422P:
        if (pstImg->au32Stride[0] != stride) {
            CVE_ERR_TRACE("Invalid stride0 [%d], expected stride0 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[0], stride, pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        if (pstImg->au32Stride[1] != CVE_ALIGN_UP(pstImg->u32Width, CVE_ALIGN)) {
            CVE_ERR_TRACE("Invalid stride1 [%d], expected stride1 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[1], CVE_ALIGN_UP(pstImg->u32Width, CVE_ALIGN),
                          pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        if (pstImg->au32Stride[2] != pstImg->au32Stride[1]) {
            CVE_ERR_TRACE("Invalid stride2 [%d], expected stride2 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[2], pstImg->au32Stride[1], pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    case CVE_IMAGE_TYPE_U8C3_PACKAGE:
    case CVE_IMAGE_TYPE_U8C3_PLANAR:
        if (pstImg->au32Stride[0] != stride) {
            CVE_ERR_TRACE("Invalid stride0 [%d], expected stride0 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[0], stride, pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        if (pstImg->au32Stride[1] != stride) {
            CVE_ERR_TRACE("Invalid stride1 [%d], expected stride1 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[1], stride, pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        if (pstImg->au32Stride[2] != stride) {
            CVE_ERR_TRACE("Invalid stride2 [%d], expected stride2 is equal to %d, width [%d]\n",
                          pstImg->au32Stride[2], stride, pstImg->u32Width);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    default:
        CVE_ERR_TRACE("Unknown image type[%d]\n", pstImg->enType);
        return AML_ERR_CVE_ILLEGAL_PARAM;
        break;
    }

    return 0;
}

static unsigned int fill_src_image(cve_comm_init_params_t *init_params,
                                   CVE_SRC_IMAGE_T *pstSrcImage, unsigned int *src_off)
{
    unsigned int offset = *src_off;

    if (offset + 1 > CVE_SRC_MAX) {
        return AML_ERR_CVE_BUF_FULL;
    }

    switch (pstSrcImage->enType) {
    case CVE_IMAGE_TYPE_U8C1:
    case CVE_IMAGE_TYPE_S8C1: {
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[0];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[0];
    } break;
    case CVE_IMAGE_TYPE_S8C2_PLANAR:
    case CVE_IMAGE_TYPE_YVU420SP:
    case CVE_IMAGE_TYPE_YVU422SP:
    case CVE_IMAGE_TYPE_YUV420SP:
    case CVE_IMAGE_TYPE_YUV422SP: {
        if (offset + 2 > CVE_SRC_MAX) {
            return AML_ERR_CVE_BUF_FULL;
        }
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[0];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[0];
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[1];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[1];
    } break;
    case CVE_IMAGE_TYPE_S8C2_PACKAGE:
    case CVE_IMAGE_TYPE_S16C1:
    case CVE_IMAGE_TYPE_U16C1: {
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[0];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[0] * 2;
    } break;
    case CVE_IMAGE_TYPE_U8C3_PACKAGE: {
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[0];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[0] * 3;
    } break;
    case CVE_IMAGE_TYPE_YUV420P:
    case CVE_IMAGE_TYPE_YUV422P:
    case CVE_IMAGE_TYPE_U8C3_PLANAR:
        if (offset + 3 > CVE_SRC_MAX) {
            return AML_ERR_CVE_BUF_FULL;
        }
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[0];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[0];
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[1];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[1];
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[2];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[2];
        break;
    case CVE_IMAGE_TYPE_S32C1:
    case CVE_IMAGE_TYPE_U32C1: {
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[0];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[0] * 4;
    } break;
    case CVE_IMAGE_TYPE_S64C1:
    case CVE_IMAGE_TYPE_U64C1: {
        init_params->src_addr[offset] = pstSrcImage->au64PhyAddr[0];
        init_params->src_stride[offset++] = pstSrcImage->au32Stride[0] * 8;
    } break;
    default:
        break;
    }

    *src_off = offset;

    return AML_SUCCESS;
}

static unsigned int fill_dst_image(cve_comm_init_params_t *init_params,
                                   CVE_DST_IMAGE_T *pstDstImage, unsigned int *dst_off)
{
    unsigned int offset = *dst_off;

    if (offset + 1 > CVE_DST_MAX) {
        return AML_ERR_CVE_BUF_FULL;
    }

    switch (pstDstImage->enType) {
    case CVE_IMAGE_TYPE_U8C1:
    case CVE_IMAGE_TYPE_S8C1: {
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[0];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[0];
    } break;
    case CVE_IMAGE_TYPE_S8C2_PLANAR:
    case CVE_IMAGE_TYPE_YVU420SP:
    case CVE_IMAGE_TYPE_YVU422SP:
    case CVE_IMAGE_TYPE_YUV420SP:
    case CVE_IMAGE_TYPE_YUV422SP: {
        if (offset + 2 > CVE_DST_MAX) {
            return AML_ERR_CVE_BUF_FULL;
        }
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[0];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[0];
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[1];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[1];
    } break;
    case CVE_IMAGE_TYPE_S8C2_PACKAGE:
    case CVE_IMAGE_TYPE_S16C1:
    case CVE_IMAGE_TYPE_U16C1: {
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[0];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[0] * 2;
    } break;
    case CVE_IMAGE_TYPE_U8C3_PACKAGE: {
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[0];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[0] * 3;
    } break;
    case CVE_IMAGE_TYPE_YUV420P:
    case CVE_IMAGE_TYPE_YUV422P:
    case CVE_IMAGE_TYPE_U8C3_PLANAR:
        if (offset + 3 > CVE_DST_MAX) {
            return AML_ERR_CVE_BUF_FULL;
        }
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[0];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[0];
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[1];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[1];
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[2];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[2];
        break;
    case CVE_IMAGE_TYPE_S32C1:
    case CVE_IMAGE_TYPE_U32C1: {
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[0];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[0] * 4;
    } break;
    case CVE_IMAGE_TYPE_S64C1:
    case CVE_IMAGE_TYPE_U64C1: {
        init_params->dst_addr[offset] = pstDstImage->au64PhyAddr[0];
        init_params->dst_stride[offset++] = pstDstImage->au32Stride[0] * 8;
    } break;
    default:
        break;
    }

    *dst_off = offset;

    return AML_SUCCESS;
}

static unsigned int fill_src_raw(cve_comm_init_params_t *init_params, CVE_SRC_RAW_T *pstSrcRaw,
                                 unsigned int *src_off)
{
    unsigned int offset = *src_off;

    if (offset + 1 > CVE_SRC_MAX) {
        return AML_ERR_CVE_BUF_FULL;
    }

    switch (pstSrcRaw->enMode) {
    case CVE_RAW_MODE_RAW6:
    case CVE_RAW_MODE_RAW7:
    case CVE_RAW_MODE_RAW8: {
        init_params->src_addr[offset] = pstSrcRaw->u64PhyAddr;
        init_params->src_stride[offset++] = pstSrcRaw->u32Stride;
    } break;
    case CVE_RAW_MODE_RAW10: {
        init_params->src_addr[offset] = pstSrcRaw->u64PhyAddr;
        init_params->src_stride[offset++] = CVE_ALIGN_UP((pstSrcRaw->u32Stride * 5 + 3) / 4, 16);

    } break;
    case CVE_RAW_MODE_RAW12: {
        init_params->src_addr[offset] = pstSrcRaw->u64PhyAddr;
        init_params->src_stride[offset++] = CVE_ALIGN_UP((pstSrcRaw->u32Stride * 3 + 1) / 2, 16);

    } break;
    case CVE_RAW_MODE_RAW14: {
        init_params->src_addr[offset] = pstSrcRaw->u64PhyAddr;
        init_params->src_stride[offset++] = init_params->src_stride[0] * 2;

    } break;
    default:
        break;
    }
    *src_off = offset;

    return AML_SUCCESS;
}

static unsigned int fill_dst_mem(cve_comm_init_params_t *init_params, CVE_DST_MEM_INFO_T *pstDstMem,
                                 unsigned int stride, unsigned int *dst_off)
{
    unsigned int offset = *dst_off;

    if (offset + 1 > CVE_DST_MAX) {
        return AML_ERR_CVE_BUF_FULL;
    }

    init_params->dst_addr[offset] = pstDstMem->u64PhyAddr;
    init_params->dst_stride[offset++] = stride;
    *dst_off = offset;

    return AML_SUCCESS;
}

static unsigned int fill_src_mem(cve_comm_init_params_t *init_params, CVE_DST_MEM_INFO_T *pstSrcMem,
                                 unsigned int stride, unsigned int *src_off)
{
    unsigned int offset = *src_off;

    if (offset + 1 > CVE_SRC_MAX) {
        return AML_ERR_CVE_BUF_FULL;
    }

    init_params->src_addr[offset] = pstSrcMem->u64PhyAddr;
    init_params->src_stride[offset++] = stride;
    *src_off = offset;

    return AML_SUCCESS;
}

static unsigned int fill_src_data(cve_comm_init_params_t *init_params, CVE_SRC_DATA_T *pstSrcData,
                                  unsigned int *src_off)
{
    unsigned int offset = *src_off;

    if (offset + 1 > CVE_SRC_MAX) {
        return AML_ERR_CVE_BUF_FULL;
    }

    init_params->src_addr[offset] = pstSrcData->u64PhyAddr;
    init_params->src_stride[offset++] = pstSrcData->u32Stride;
    *src_off = offset;

    return AML_SUCCESS;
}

static unsigned int fill_dst_data(cve_comm_init_params_t *init_params, CVE_DST_DATA_T *pstDstData,
                                  unsigned int *dst_off)
{
    unsigned int offset = *dst_off;

    if (offset + 1 > CVE_DST_MAX) {
        return AML_ERR_CVE_BUF_FULL;
    }

    init_params->dst_addr[offset] = pstDstData->u64PhyAddr;
    init_params->dst_stride[offset++] = pstDstData->u32Stride;
    *dst_off = offset;

    return AML_SUCCESS;
}

static void cve_common_params_init(cve_comm_params_t *comm_params,
                                   cve_comm_init_params_t *init_params)
{
    comm_params->reg_03.bits.src_addr0 = init_params->src_addr[0] >> 4;
    comm_params->reg_04.bits.src_addr1 = init_params->src_addr[1] >> 4;
    comm_params->reg_05.bits.src_addr2 = init_params->src_addr[2] >> 4;
    comm_params->reg_06.bits.src_addr3 = init_params->src_addr[3] >> 4;
    comm_params->reg_07.bits.src_addr4 = init_params->src_addr[4] >> 4;
    comm_params->reg_08.bits.src_addr5 = init_params->src_addr[5] >> 4;
    comm_params->reg_09.bits.src_addr6 = init_params->src_addr[6] >> 4;
    comm_params->reg_0a.bits.src_addr7 = init_params->src_addr[7] >> 4;
    comm_params->reg_0b.bits.dst_addr0 = init_params->dst_addr[0] >> 4;
    comm_params->reg_0c.bits.dst_addr1 = init_params->dst_addr[1] >> 4;
    comm_params->reg_0d.bits.dst_addr2 = init_params->dst_addr[2] >> 4;
    comm_params->reg_0e.bits.dst_addr3 = init_params->dst_addr[3] >> 4;
    comm_params->reg_0f.bits.dst_addr4 = init_params->dst_addr[4] >> 4;

    comm_params->reg_10.bits.src_stride_0 = init_params->src_stride[0] >> 4;
    comm_params->reg_10.bits.src_stride_1 = init_params->src_stride[1] >> 4;
    comm_params->reg_11.bits.src_stride_3 = init_params->src_stride[3] >> 4;
    comm_params->reg_11.bits.src_stride_2 = init_params->src_stride[2] >> 4;
    comm_params->reg_12.bits.src_stride_5 = init_params->src_stride[5] >> 4;
    comm_params->reg_12.bits.src_stride_4 = init_params->src_stride[4] >> 4;
    comm_params->reg_13.bits.src_stride_7 = init_params->src_stride[7] >> 4;
    comm_params->reg_13.bits.src_stride_6 = init_params->src_stride[6] >> 4;
    comm_params->reg_14.bits.dst_stride_1 = init_params->dst_stride[1] >> 4;
    comm_params->reg_14.bits.dst_stride_0 = init_params->dst_stride[0] >> 4;
    comm_params->reg_15.bits.dst_stride_3 = init_params->dst_stride[3] >> 4;
    comm_params->reg_15.bits.dst_stride_2 = init_params->dst_stride[2] >> 4;
    comm_params->reg_16.bits.dst_stride_4 = init_params->dst_stride[4] >> 4;

    comm_params->reg_17.bits.src_image_width = init_params->src_width;
    comm_params->reg_17.bits.src_image_height = init_params->src_height;
    comm_params->reg_18.bits.dst_image_width = init_params->dst_width;
    comm_params->reg_18.bits.dst_image_height = init_params->dst_height;
    comm_params->reg_19.bits.cve_crop_xstart = init_params->xstart;
    comm_params->reg_19.bits.cve_crop_ystart = init_params->ystart;
    comm_params->reg_1a.bits.cve_crop_xsize = init_params->xSize;
    comm_params->reg_1a.bits.cve_crop_ysize = init_params->ySize;
}

static unsigned int dma_task_cmd_queue(cve_op_dma_params_t *dma_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(dma_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(dma_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_DMA_REG0, dma_params->reg_1b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_DMA_REG1, dma_params->reg_1c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_DMA_REG2, dma_params->reg_1d.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_DMA_REG3, dma_params->reg_1e.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(dma)

    CMD_QUEUE_RETURN
}

static unsigned int filter_task_cmd_queue(cve_op_filter_params_t *filter_params,
                                          unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(filter_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(filter_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG2, filter_params->filter_1f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG1_0, filter_params->filter_20.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG1_1, filter_params->filter_21.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG2_3, filter_params->filter_22.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_0, filter_params->filter_67.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_1, filter_params->filter_68.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_2, filter_params->filter_69.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_3, filter_params->filter_6a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_4, filter_params->filter_6b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_5, filter_params->filter_6c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_6, filter_params->filter_6d.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(filter)

    CMD_QUEUE_RETURN
}

static unsigned int csc_task_cmd_queue(cve_op_csc_params_t *csc_params, unsigned int *cmd_buf)
{

    CMD_QUEUE_CREATE_PREPARE(csc_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(csc_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG0, csc_params->csc_23.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG, csc_params->csc_24.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG1, csc_params->csc_6e.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG1_1, csc_params->csc_6f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG2_0, csc_params->csc_70.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG2_1, csc_params->csc_71.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG2_2, csc_params->csc_72.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG3, csc_params->csc_73.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG3_1, csc_params->csc_74.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(csc)

    CMD_QUEUE_RETURN
}

static unsigned int
filter_and_csc_task_cmd_queue(cve_op_filter_and_csc_params_t *filter_and_csc_params,
                              unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(filter_and_csc_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(filter_and_csc_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG0, filter_and_csc_params->csc_23.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG, filter_and_csc_params->csc_24.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG1, filter_and_csc_params->csc_6e.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG1_1, filter_and_csc_params->csc_6f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG2_0, filter_and_csc_params->csc_70.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG2_1, filter_and_csc_params->csc_71.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG2_2, filter_and_csc_params->csc_72.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG3, filter_and_csc_params->csc_73.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CSC_REG3_1, filter_and_csc_params->csc_74.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_0, filter_and_csc_params->filter_67.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_1, filter_and_csc_params->filter_68.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_2, filter_and_csc_params->filter_69.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_3, filter_and_csc_params->filter_6a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_4, filter_and_csc_params->filter_6b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_5, filter_and_csc_params->filter_6c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_6, filter_and_csc_params->filter_6d.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(filter_and_csc)

    CMD_QUEUE_RETURN
}

static unsigned int sobel_task_cmd_queue(cve_op_sobel_params_t *sobel_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(sobel_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(sobel_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG0, sobel_params->sobel_25.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_0, sobel_params->sobel_75.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_1, sobel_params->sobel_76.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_2, sobel_params->sobel_77.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_3, sobel_params->sobel_78.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_4, sobel_params->sobel_79.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_5, sobel_params->sobel_7a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_6, sobel_params->sobel_7b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_0, sobel_params->sobel_7c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_1, sobel_params->sobel_7d.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_2, sobel_params->sobel_7e.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_3, sobel_params->sobel_7f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_4, sobel_params->sobel_80.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_5, sobel_params->sobel_81.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_6, sobel_params->sobel_82.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(sobel)

    CMD_QUEUE_RETURN
}

static unsigned int mag_and_ang_task_cmd_queue(cve_op_mag_and_ang_params_t *mag_and_ang_params,
                                               unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(mag_and_ang_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(mag_and_ang_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_MAGANDANG_REG0, mag_and_ang_params->mag_and_ang_26.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_0, mag_and_ang_params->sobel_75.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_1, mag_and_ang_params->sobel_76.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_2, mag_and_ang_params->sobel_77.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_3, mag_and_ang_params->sobel_78.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_4, mag_and_ang_params->sobel_79.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_5, mag_and_ang_params->sobel_7a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_6, mag_and_ang_params->sobel_7b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_0, mag_and_ang_params->sobel_7c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_1, mag_and_ang_params->sobel_7d.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_2, mag_and_ang_params->sobel_7e.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_3, mag_and_ang_params->sobel_7f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_4, mag_and_ang_params->sobel_80.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_5, mag_and_ang_params->sobel_81.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_6, mag_and_ang_params->sobel_82.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(mag_and_ang)

    CMD_QUEUE_RETURN
}

static unsigned int
match_bg_model_task_cmd_queue(cve_op_match_bg_model_params_t *match_bg_model_params,
                              unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(match_bg_model_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(match_bg_model_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_BGMODE_REG0, match_bg_model_params->bg_mode_27.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(match_bg_model)

    CMD_QUEUE_RETURN
}

static unsigned int erode_dilate_task_cmd_queue(cve_op_erode_dilate_params_t *erode_dilate_params,
                                                unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(erode_dilate_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(erode_dilate_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_ERODEDILATE_REG0, erode_dilate_params->erode_dilate_2E.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_0, erode_dilate_params->filter_67.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_1, erode_dilate_params->filter_68.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_2, erode_dilate_params->filter_69.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_3, erode_dilate_params->filter_6a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_4, erode_dilate_params->filter_6b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_5, erode_dilate_params->filter_6c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_FILTER_REG0_6, erode_dilate_params->filter_6d.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(erode_dilate)

    CMD_QUEUE_RETURN
}

static unsigned int thresh_task_cmd_queue(cve_op_thresh_params_t *thresh_params,
                                          unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(thresh_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(thresh_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_THRESH_REG0, thresh_params->thresh_2f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_THRESH_REG1, thresh_params->thresh_30.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(thresh)

    CMD_QUEUE_RETURN
}

static unsigned int alu_task_cmd_queue(cve_op_alu_params_t *alu_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(alu_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(alu_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_ALU_REG0, alu_params->alu_31.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_ALU_REG1, alu_params->alu_32.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SUB_THRESH_RATIO, alu_params->sub_99.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(alu)

    CMD_QUEUE_RETURN
}

static unsigned int integ_task_cmd_queue(cve_op_integ_params_t *integ_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(integ_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(integ_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_INTEG_REG0, integ_params->integ_33.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(integ)

    CMD_QUEUE_RETURN
}

static unsigned int hist_task_cmd_queue(cve_op_hist_params_t *hist_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(hist_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(hist_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_INTEG_REG0, hist_params->integ_33.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_EQHIST_REG0, hist_params->eqhist_34.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(hist)

    CMD_QUEUE_RETURN
}

static unsigned int thresh_s16_task_cmd_queue(cve_op_thresh_s16_params_t *thresh_s16_params,
                                              unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(thresh_s16_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(thresh_s16_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_THRESHS16_REG0, thresh_s16_params->thresh_s16_35.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_THRESHS16_REG1, thresh_s16_params->thresh_s16_36.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(thresh_s16)

    CMD_QUEUE_RETURN
}

static unsigned int thresh_u16_task_cmd_queue(cve_op_thresh_u16_params_t *thresh_u16_params,
                                              unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(thresh_u16_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(thresh_u16_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_THRESHU16_REG0, thresh_u16_params->thresh_u16_37.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_THRESHU16_REG1, thresh_u16_params->thresh_u16_38.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(thresh_u16)

    CMD_QUEUE_RETURN
}

static unsigned int
_16bit_to_8bit_task_cmd_queue(cve_op_16bit_to_8bit_params_t *_16bit_to_8bit_params,
                              unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(_16bit_to_8bit_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(_16bit_to_8bit_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_16BITTO8BIT_REG0, _16bit_to_8bit_params->_16bit_to_8bit_39.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_16BITTO8BIT_REG1, _16bit_to_8bit_params->_16bit_to_8bit_3a.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(_16bit_to_8bit)

    CMD_QUEUE_RETURN
}

static unsigned int
ord_stat_filter_task_cmd_queue(cve_op_ord_stat_filter_params_t *stat_filter_params,
                               unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(stat_filter_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(stat_filter_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_STATFILTER_REG0, stat_filter_params->stat_filter_3b.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(ord_stat_filter)

    CMD_QUEUE_RETURN
}

static unsigned int map_task_cmd_queue(cve_op_map_params_t *map_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(map_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(map_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_MAP_REG0, map_params->map_3c.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(map)

    CMD_QUEUE_RETURN
}

static unsigned int ncc_task_cmd_queue(cve_op_ncc_params_t *ncc_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(ncc_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(ncc_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_NCC_REG0, ncc_params->ncc_3c.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(ncc)

    CMD_QUEUE_RETURN
}

static unsigned int ccl_task_cmd_queue(cve_op_ccl_params_t *ccl_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(ccl_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(ccl_params->comm_params)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(ccl)

    CMD_QUEUE_RETURN
}

static unsigned int gmm_task_cmd_queue(cve_op_gmm_params_t *gmm_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(gmm_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(gmm_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_GMM_REG0, gmm_params->gmm_44.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_GMM_REG1, gmm_params->gmm_45.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_GMM_REG2, gmm_params->gmm_46.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_GMM_REG3, gmm_params->gmm_47.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_GMM_REG4, gmm_params->gmm_48.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_GMM_REG5, gmm_params->gmm_49.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(gmm)

    CMD_QUEUE_RETURN
}

static unsigned int canny_edge_task_cmd_queue(cve_op_canny_edge_params_t *canny_params,
                                              unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(canny_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(canny_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CANNY_REG0, canny_params->canny_4a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_CANNY_REG2, canny_params->canny_4c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_0, canny_params->sobel_75.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_1, canny_params->sobel_76.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_2, canny_params->sobel_77.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_3, canny_params->sobel_78.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_4, canny_params->sobel_79.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_5, canny_params->sobel_7a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_6, canny_params->sobel_7b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_0, canny_params->sobel_7c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_1, canny_params->sobel_7d.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_2, canny_params->sobel_7e.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_3, canny_params->sobel_7f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_4, canny_params->sobel_80.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_5, canny_params->sobel_81.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_6, canny_params->sobel_82.reg)

    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(canny_edge)

    CMD_QUEUE_RETURN
}

static unsigned int lbp_task_cmd_queue(cve_op_lbp_params_t *lbp_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(lbp_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(lbp_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_LBP_REG0, lbp_params->lbp_4d.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(lbp)

    CMD_QUEUE_RETURN
}

static unsigned int norm_grad_task_cmd_queue(cve_op_norm_grad_params_t *norm_grad_params,
                                             unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(norm_grad_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(norm_grad_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_NORMGRAD_RETG0, norm_grad_params->norm_grad_4e.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_0, norm_grad_params->sobel_75.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_1, norm_grad_params->sobel_76.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_2, norm_grad_params->sobel_77.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_3, norm_grad_params->sobel_78.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_4, norm_grad_params->sobel_79.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_5, norm_grad_params->sobel_7a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG1_6, norm_grad_params->sobel_7b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_0, norm_grad_params->sobel_7c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_1, norm_grad_params->sobel_7d.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_2, norm_grad_params->sobel_7e.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_3, norm_grad_params->sobel_7f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_4, norm_grad_params->sobel_80.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_5, norm_grad_params->sobel_81.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SOBEL_REG2_6, norm_grad_params->sobel_82.reg)

    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(norm_grad)

    CMD_QUEUE_RETURN
}

static unsigned int build_lk_optical_flow_pyr_task_cmd_queue(
    cve_op_build_lk_optical_flow_pyr_params_t *build_lk_optical_flow_pyr_params,
    unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(build_lk_optical_flow_pyr_params->comm_params.reg_02.bits.cve_op_type,
                             cmd_buf)
    CMD_QUEUE_ADD_COMMON(build_lk_optical_flow_pyr_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_BDLK_REG0, build_lk_optical_flow_pyr_params->bdlk_4f.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(build_lk_optical_flow_pyr)

    CMD_QUEUE_RETURN
}

static unsigned int
lk_optical_flow_pyr_task_cmd_queue(cve_op_lk_optical_flow_pyr_params_t *lk_optical_flow_pyr_params,
                                   unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(lk_optical_flow_pyr_params->comm_params.reg_02.bits.cve_op_type,
                             cmd_buf)
    CMD_QUEUE_ADD_COMMON(lk_optical_flow_pyr_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_LK_REG0, lk_optical_flow_pyr_params->lk_50.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_LK_REG1, lk_optical_flow_pyr_params->lk_51.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(lk_optical_flow_pyr)

    CMD_QUEUE_RETURN
}

static unsigned int
st_candi_corner_task_cmd_queue(cve_op_st_candi_corner_params_t *st_corner_params,
                               unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(st_corner_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(st_corner_params->comm_params)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(st_corner)

    CMD_QUEUE_RETURN
}

static unsigned int sad_task_cmd_queue(cve_op_sad_params_t *sad_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(sad_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(sad_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SAD_REG0, sad_params->sad_64.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_SAD_REG1, sad_params->sad_65.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(sad)

    CMD_QUEUE_RETURN
}

static unsigned int grad_fg_task_cmd_queue(cve_op_grad_fg_params_t *grad_fg_params,
                                           unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(grad_fg_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(grad_fg_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_GRADFG_RETG0, grad_fg_params->grad_fg_66.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(grad_fg)

    CMD_QUEUE_RETURN
}

static unsigned int
update_bg_model_task_cmd_queue(cve_op_update_bg_model_params_t *update_bg_model_params,
                               unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(update_bg_model_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(update_bg_model_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_BGMODE_REG0, update_bg_model_params->bg_mode_27.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_UPDATEBGMODE_REG0, update_bg_model_params->update_bg_mode_2a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_UPDATEBGMODE_REG1, update_bg_model_params->update_bg_mode_2b.reg)
    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(update_bg_model)

    CMD_QUEUE_RETURN
}

static unsigned int tof_task_cmd_queue(cve_op_tof_params_t *tof_params, unsigned int *cmd_buf)
{
    CMD_QUEUE_CREATE_PREPARE(tof_params->comm_params.reg_02.bits.cve_op_type, cmd_buf)
    CMD_QUEUE_ADD_COMMON(tof_params->comm_params)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG0, tof_params->tof_83.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG1, tof_params->tof_84.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG3_0, tof_params->tof_85.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG3_1, tof_params->tof_86.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG3_2, tof_params->tof_87.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG3_3, tof_params->tof_88.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG3_4, tof_params->tof_89.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG4, tof_params->tof_8a.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG5, tof_params->tof_8b.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG9_0, tof_params->tof_8c.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG9_1, tof_params->tof_8d.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG9_2, tof_params->tof_8e.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG9_3, tof_params->tof_8f.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG9_4, tof_params->tof_90.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG10, tof_params->tof_91.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG11, tof_params->tof_92.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG12, tof_params->tof_93.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG13, tof_params->tof_94.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG14, tof_params->tof_95.reg)
    CMD_QUEUE_ADD_PROCESS_1(CVE_TOF_REG15, tof_params->tof_96.reg)

    CMD_QUEUE_CREATE_END

    CMD_QUEUE_DEBUG_DUMP(tof)

    CMD_QUEUE_RETURN
}

AML_U32 cve_check_dma_param(CVE_DATA_T *pstSrcDATA, CVE_DST_DATA_T *pstDstDATA,
                            CVE_DMA_CTRL_T *pstDmaCtrl)
{
    AML_U8 hsegsize[5] = {2, 3, 4, 8, 16};
    int i;

    CVE_GET_IMAGE_SIZE(CVE_DMA);
    CVE_DATA_CHECK(CVE_DMA, pstDstDATA);

    if (pstDmaCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_DATA_CROP_CHECK(CVE_DMA, pstDmaCtrl->stCrop, pstDstDATA, pstSrcDATA);
    }
    if (pstDmaCtrl->enMode <= CVE_DMA_MODE_INTERVAL_COPY) {
        CVE_DATA_CHECK(CVE_DMA, pstSrcDATA);
        CVE_IMAGE_COMPARE(CVE_DMA, pstSrcDATA, pstDstDATA);
    }
    CVE_MODE_CHECK(CVE_DMA, pstDmaCtrl->enMode, CVE_DMA_MODE_BUTT);
    if (pstDmaCtrl->enMode == CVE_DMA_MODE_INTERVAL_COPY) {
        for (i = 0; i < 5; i++) {
            if (pstDmaCtrl->u8HorSegSize == hsegsize[i]) {
                break;
            }
        }
        if (i == 5) {
            CVE_ERR_TRACE("[CVE_DMA] pstDmaCtrl->u8HorSegSize(%d) set value error, value must in "
                          "{2, 3, 4, 8, 16}\n",
                          pstDmaCtrl->u8HorSegSize);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        if ((pstDmaCtrl->u8VerSegRows == 0) || (pstDmaCtrl->u8VerSegRows > 0xFF) ||
            (pstDmaCtrl->u8VerSegRows > ((pstDstDATA->u32Height > (0xFFFF / pstSrcDATA->u32Stride))
                                             ? (0xFFFF / pstSrcDATA->u32Stride)
                                             : pstDstDATA->u32Height))) {
            CVE_ERR_TRACE("[CVE_DMA] pstDmaCtrl->u8VerSegRows(%d), value must in [1, min(ysize, "
                          "255, 65535/srcStride)]\n",
                          pstDmaCtrl->u8HorSegSize);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        CVE_UNSIGNED_RANGE_CHECK(CVE_DMA, pstDmaCtrl->u8ElemSize, pstDmaCtrl->u8HorSegSize);
    }
    return AML_SUCCESS;
}

AML_S32 cve_check_luamStat_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_MEM_INFO_T *pstDstMem,
                                 CVE_RECT_U16_T *astCveLumaRect,
                                 CVE_LUMA_STAT_ARRAY_CTRL_T *pstLumaStatArrayCtrl)
{
    int i;

    CVE_GET_IMAGE_SIZE(CVE_LUMA);
    if (pstLumaStatArrayCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_SRC_CROP_CHECK(CVE_LUMA, pstLumaStatArrayCtrl->stCrop, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_LUMA, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_LUMA, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_MEM_CHECK(CVE_LUMA, pstDstMem);
    if (pstLumaStatArrayCtrl->u8MaxLumaRect > CVE_LUMA_RECT_MAX) {
        CVE_ERR_TRACE("[CVE_LUMA] pstLumaStatArrayCtrl->u8MaxLumaRect(%d) set value error, value "
                      "must in[1, %d]\n",
                      pstLumaStatArrayCtrl->u8MaxLumaRect, CVE_LUMA_RECT_MAX);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    } else {
        for (i = 0; i < pstLumaStatArrayCtrl->u8MaxLumaRect; i++) {
            if ((astCveLumaRect[i].u16Width < 8) || (astCveLumaRect[i].u16Height < 8)) {
                CVE_ERR_TRACE("[CVE_LUMA] astCveLumaRect[%d].u16Width(%d), "
                              "astCveLumaRect[%d].u16Height(%d) value must >8\n",
                              i, astCveLumaRect[i].u16Width, i, astCveLumaRect[i].u16Height);
                return AML_ERR_CVE_ILLEGAL_PARAM;
            }
            if (((astCveLumaRect[i].u16X + astCveLumaRect[i].u16Width) > pstSrcImage->u32Width) ||
                ((astCveLumaRect[i].u16Y + astCveLumaRect[i].u16Height) > pstSrcImage->u32Height)) {
                CVE_ERR_TRACE("[CVE_LUMA] "
                              "xstart+xsize<=src_width,ystart+ysize<=src_height,xsize=dst_width,"
                              "ysize=dst_height.\n");
                CVE_ERR_TRACE(
                    "[CVE_LUMA] astCveLumaRect[%d].u16X(%d),astCveLumaRect[%d].u16Width(%d), "
                    "srcWidth(%d)\n",
                    i, astCveLumaRect[i].u16X, i, astCveLumaRect[i].u16Width,
                    pstSrcImage->u32Width);
                CVE_ERR_TRACE(
                    "[CVE_LUMA] astCveLumaRect[%d].u16Y(%d),astCveLumaRect[%d].u16Height(%d), "
                    "srcHeight(%d)\n",
                    i, astCveLumaRect[i].u16X, i, astCveLumaRect[i].u16Width,
                    pstSrcImage->u32Height);
                return AML_ERR_CVE_ILLEGAL_PARAM;
            }
        }
    }
    return AML_SUCCESS;
}

AML_S32 cve_check_filter_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                               CVE_FILTER_CTRL_T *pstFilterCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_FILTER);
    CVE_IMAGE_CHECK(CVE_FILTER, pstSrcImage);
    if (pstFilterCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_FILTER, pstFilterCtrl->stCrop, pstDstImage, pstSrcImage);
    }
    if ((pstSrcImage->enType != CVE_IMAGE_TYPE_U8C1) &&
        (pstSrcImage->enType != CVE_IMAGE_TYPE_YUV420SP) &&
        (pstSrcImage->enType != CVE_IMAGE_TYPE_YUV422SP)) {
        CVE_ERR_TRACE("[CVE_FILTER] pstSrcImage->enType(%d) set error, not support this type\n",
                      pstSrcImage->enType);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }
    CVE_IMAGE_CHECK(CVE_FILTER, pstDstImage);
    CVE_RESOLUTION_EQUAL(CVE_FILTER, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_IMAGE_TYPE_EQUAL(CVE_FILTER, pstSrcImage, pstDstImage);
    if (pstFilterCtrl->u8Norm > 13) {
        CVE_ERR_TRACE("[CVE_FILTER] pstFilterCtrl->u8Norm(%d) set error, need set on [0, 13]\n",
                      pstFilterCtrl->u8Norm);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }
    return AML_SUCCESS;
}

AML_S32 cve_check_csc_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                            CVE_CSC_CTRL_T *pstCscCtrl)
{
    AML_U8 csc_image_type_support[] = {CVE_IMAGE_TYPE_YUV420SP,     CVE_IMAGE_TYPE_YUV422SP,
                                       CVE_IMAGE_TYPE_U8C3_PACKAGE, CVE_IMAGE_TYPE_U8C3_PLANAR,
                                       CVE_IMAGE_TYPE_YVU420SP,     CVE_IMAGE_TYPE_YVU422SP};
    int i;

    CVE_GET_IMAGE_SIZE(CVE_CSC);
    CVE_IMAGE_CSC_CHECK(CVE_CSC, pstSrcImage);
    CVE_IMAGE_CSC_CHECK(CVE_CSC, pstDstImage);
    if (pstCscCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_CSC, pstCscCtrl->stCrop, pstDstImage, pstSrcImage);
    } else {
        CVE_RESOLUTION_EQUAL(CVE_CSC, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    }

    for (i = 0; i < sizeof(csc_image_type_support) / sizeof(csc_image_type_support[0]); i++) {
        if (pstSrcImage->enType == csc_image_type_support[i]) {
            break;
        }
    }
    if (i == sizeof(csc_image_type_support) / sizeof(csc_image_type_support[0])) {
        CVE_ERR_TRACE("[CVE_CSC] pstSrcImage->enType(%d) not support.\n", pstSrcImage->enType);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }
    for (i = 0; i < sizeof(csc_image_type_support) / sizeof(csc_image_type_support[0]); i++) {
        if (pstDstImage->enType == csc_image_type_support[i]) {
            break;
        }
    }
    if (i == sizeof(csc_image_type_support) / sizeof(csc_image_type_support[0])) {
        CVE_ERR_TRACE("[CVE_CSC] stDstImage->enType(%d) not support.\n", pstDstImage->enType);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }
    CVE_MODE_CHECK(CVE_CSC, pstCscCtrl->enMode, CVE_CSC_MODE_BUTT);

    return AML_SUCCESS;
}

AML_S32 cve_check_filter_csc_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                   CVE_FILTER_AND_CSC_CTRL_T *pstFilterCscCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_FILTER_AND_CSC);
    if (pstFilterCscCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_FILTER_AND_CSC, pstFilterCscCtrl->stCrop, pstDstImage,
                             pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_FILTER_AND_CSC, pstSrcImage);
    CVE_IMAGE_CHECK(CVE_FILTER_AND_CSC, pstDstImage);
    CVE_RESOLUTION_EQUAL(CVE_FILTER_AND_CSC, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_FILTER_AND_CSC, pstFilterCscCtrl->enMode, CVE_CSC_MODE_BUTT);
    if ((pstSrcImage->enType != CVE_IMAGE_TYPE_YUV420SP) &&
        (pstSrcImage->enType != CVE_IMAGE_TYPE_YUV422SP)) {
        CVE_ERR_TRACE("[CVE_FILTER_AND_CSC] pstSrcImage->enType(%d) not support, only yuv420sp and "
                      "yuv422sp.\n",
                      pstDstImage->enType);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }
    if ((pstDstImage->enType != CVE_IMAGE_TYPE_U8C3_PACKAGE) &&
        (pstDstImage->enType != CVE_IMAGE_TYPE_U8C3_PLANAR)) {
        CVE_ERR_TRACE("[CVE_FILTER_AND_CSC] pstDstImage->enType(%d) not support, only U8C3_PACKAGE "
                      "and U8C3_PLANAR.\n",
                      pstDstImage->enType);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }

    return AML_SUCCESS;
}

AML_S32 cve_check_sobel_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstH,
                              CVE_DST_IMAGE_T *pstDstV, CVE_SOBEL_CTRL_T *pstSobelCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_SOBEL);
    if (pstSobelCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_SOBEL, pstSobelCtrl->stCrop, pstDstH, pstSrcImage);
        CVE_IMAGE_CROP_CHECK(CVE_SOBEL, pstSobelCtrl->stCrop, pstDstV, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_SOBEL, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_SOBEL, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_MODE_CHECK(CVE_SOBEL, pstSobelCtrl->enOutCtrl, CVE_SOBEL_OUT_CTRL_BUTT);
    if ((pstSobelCtrl->enOutCtrl == CVE_SOBEL_OUT_CTRL_BOTH) ||
        (pstSobelCtrl->enOutCtrl == CVE_SOBEL_OUT_CTRL_HOR)) {
        CVE_IMAGE_CHECK(CVE_SOBEL, pstDstH);
        CVE_IMAGE_TYPE_CHECK(CVE_SOBEL, pstDstH, CVE_IMAGE_TYPE_S16C1);
        CVE_RESOLUTION_EQUAL(CVE_SOBEL, pstSrcImage, pstDstH, RESOLUTION_DS_NONE);
    }
    if ((pstSobelCtrl->enOutCtrl == CVE_SOBEL_OUT_CTRL_BOTH) ||
        (pstSobelCtrl->enOutCtrl == CVE_SOBEL_OUT_CTRL_VER)) {
        CVE_IMAGE_CHECK(CVE_SOBEL, pstDstV);
        CVE_IMAGE_TYPE_CHECK(CVE_SOBEL, pstDstV, CVE_IMAGE_TYPE_S16C1);
        CVE_RESOLUTION_EQUAL(CVE_SOBEL, pstSrcImage, pstDstV, RESOLUTION_DS_NONE);
    }

    return AML_SUCCESS;
}

AML_S32 cve_check_mag_and_ang_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstMag,
                                    CVE_DST_IMAGE_T *pstDstAng,
                                    CVE_MAG_AND_ANG_CTRL_T *pstMagAndAngCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_MAG_AND_ANG);
    if (pstMagAndAngCtrl->stCrop.enCrop == AML_TRUE &&
        pstMagAndAngCtrl->enOutCtrl != CVE_MAG_AND_ANG_OUT_CTRL_MAG) {
        CVE_IMAGE_CROP_CHECK(CVE_MAG_AND_ANG, pstMagAndAngCtrl->stCrop, pstDstMag, pstSrcImage);
        CVE_IMAGE_CROP_CHECK(CVE_MAG_AND_ANG, pstMagAndAngCtrl->stCrop, pstDstAng, pstSrcImage);
    } else {
        CVE_IMAGE_CROP_CHECK(CVE_MAG_AND_ANG, pstMagAndAngCtrl->stCrop, pstDstMag, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_MAG_AND_ANG, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_MAG_AND_ANG, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_MAG_AND_ANG, pstDstMag);
    CVE_RESOLUTION_EQUAL(CVE_MAG_AND_ANG, pstSrcImage, pstDstMag, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_MAG_AND_ANG, pstMagAndAngCtrl->enOutCtrl, CVE_MAG_AND_ANG_OUT_CTRL_BUTT);
    CVE_IMAGE_TYPE_CHECK(CVE_MAG_AND_ANG, pstDstMag, CVE_IMAGE_TYPE_U16C1);
    if ((pstMagAndAngCtrl->enOutCtrl == CVE_MAG_AND_ANG_OUT_CTRL_MAG_AND_ANG) ||
        (pstMagAndAngCtrl->enOutCtrl == CVE_MAG_AND_ANG_OUT_CTRL_HOG)) {
        CVE_IMAGE_CHECK(CVE_MAG_AND_ANG, pstDstAng);
        CVE_RESOLUTION_EQUAL(CVE_MAG_AND_ANG, pstSrcImage, pstDstAng, RESOLUTION_DS_NONE);
        if (pstMagAndAngCtrl->enOutCtrl == CVE_MAG_AND_ANG_OUT_CTRL_MAG) {
            CVE_IMAGE_TYPE_CHECK(CVE_MAG_AND_ANG, pstDstAng, CVE_IMAGE_TYPE_U8C1);
        } else if (pstMagAndAngCtrl->enOutCtrl == CVE_MAG_AND_ANG_OUT_CTRL_HOG) {
            CVE_IMAGE_TYPE_CHECK(CVE_MAG_AND_ANG, pstDstAng, CVE_IMAGE_TYPE_U16C1);
        }
    }

    return AML_SUCCESS;
}

AML_S32 cve_check_match_bg_model_param(CVE_SRC_IMAGE_T *pstCurImg, CVE_SRC_IMAGE_T *pstPreImg,
                                       CVE_MEM_INFO_T *pstBgModel, CVE_DST_IMAGE_T *pstFg,
                                       CVE_DST_IMAGE_T *pstBg, CVE_DST_IMAGE_T *pstCurDiffBg,
                                       CVE_DST_IMAGE_T *pstFrmDiff, CVE_DST_MEM_INFO_T *pstStatData,
                                       CVE_MATCH_BG_MODEL_CTRL_T *pstMatchBgModelCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_MATCH_BG_MODEL);
    if (pstMatchBgModelCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->stCrop, pstFg, pstCurImg);
        CVE_IMAGE_CROP_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->stCrop, pstBg, pstCurImg);
        CVE_IMAGE_CROP_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->stCrop, pstCurDiffBg,
                             pstCurImg);
        CVE_IMAGE_CROP_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->stCrop, pstFrmDiff,
                             pstCurImg);
        CVE_IMAGE_CROP_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->stCrop, pstFg, pstPreImg);
        CVE_IMAGE_CROP_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->stCrop, pstBg, pstPreImg);
        CVE_IMAGE_CROP_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->stCrop, pstCurDiffBg,
                             pstPreImg);
        CVE_IMAGE_CROP_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->stCrop, pstFrmDiff,
                             pstPreImg);
    }
    CVE_IMAGE_CHECK(CVE_MATCH_BG_MODEL, pstCurImg);
    CVE_IMAGE_TYPE_CHECK(CVE_MATCH_BG_MODEL, pstCurImg, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_MATCH_BG_MODEL, pstPreImg);
    CVE_MEM_CHECK(CVE_MATCH_BG_MODEL, pstBgModel);
    CVE_MEM_CHECK(CVE_MATCH_BG_MODEL, pstStatData);
    CVE_RESOLUTION_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstPreImg, RESOLUTION_DS_NONE);
    CVE_RESOLUTION_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstFg,
                         pstMatchBgModelCtrl->enDownScaleMode);
    CVE_RESOLUTION_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstBg,
                         pstMatchBgModelCtrl->enDownScaleMode);
    CVE_RESOLUTION_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstCurDiffBg,
                         pstMatchBgModelCtrl->enDownScaleMode);
    CVE_RESOLUTION_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstFrmDiff,
                         pstMatchBgModelCtrl->enDownScaleMode);
    CVE_IMAGE_TYPE_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstPreImg);
    CVE_IMAGE_TYPE_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstFg);
    CVE_IMAGE_TYPE_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstBg);
    CVE_IMAGE_TYPE_EQUAL(CVE_MATCH_BG_MODEL, pstCurImg, pstCurDiffBg);
    CVE_UNSIGNED_RANGE_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->u8q4DistThr, 4095);
    CVE_MODE_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->enOutputMode,
                   CVE_MATCH_BG_MODEL_OUTPUT_MODE_BUTT);
    CVE_MODE_CHECK(CVE_MATCH_BG_MODEL, pstMatchBgModelCtrl->enDownScaleMode,
                   CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_BUTT);
    return AML_SUCCESS;
}

AML_S32 cve_check_dilate_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                               CVE_DILATE_CTRL_T *pstDilateCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_DILATE);
    if (pstDilateCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_SOBEL, pstDilateCtrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_DILATE, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_DILATE, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_DILATE, pstDstImage);
    CVE_IMAGE_TYPE_EQUAL(CVE_DILATE, pstSrcImage, pstDstImage);
    CVE_RESOLUTION_EQUAL(CVE_DILATE, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    return AML_SUCCESS;
}

AML_S32 cve_check_erode_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                              CVE_ERODE_CTRL_T *pstErodeCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_ERODE);
    if (pstErodeCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_SOBEL, pstErodeCtrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_ERODE, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_ERODE, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ERODE, pstDstImage);
    CVE_IMAGE_TYPE_EQUAL(CVE_ERODE, pstSrcImage, pstDstImage);
    CVE_RESOLUTION_EQUAL(CVE_ERODE, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);

    return AML_SUCCESS;
}

AML_S32 cve_check_thresh(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                         CVE_THRESH_CTRL_T *pstThreshCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_THRESH);
    if (pstThreshCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_THRESH, pstThreshCtrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_THRESH, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_THRESH, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_THRESH, pstDstImage);
    CVE_IMAGE_TYPE_EQUAL(CVE_THRESH, pstSrcImage, pstDstImage);
    CVE_RESOLUTION_EQUAL(CVE_THRESH, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_THRESH, pstThreshCtrl->enMode, CVE_THRESH_MODE_BUTT);
    switch (pstThreshCtrl->enMode) {
    case CVE_THRESH_MODE_BINARY:
    case CVE_THRESH_MODE_TRUNC:
    case CVE_THRESH_MODE_TO_MINVAL:
        break;
    case CVE_THRESH_MODE_MIN_MID_MAX:
    case CVE_THRESH_MODE_ORI_MID_MAX:
    case CVE_THRESH_MODE_MIN_MID_ORI:
    case CVE_THRESH_MODE_MIN_ORI_MAX:
    case CVE_THRESH_MODE_ORI_MID_ORI:
        if (pstThreshCtrl->u8LowThr > pstThreshCtrl->u8HighThr) {
            CVE_ERR_TRACE("[CVE_THRESH] pstThreshCtrl->u8LowThr(%d) pstThreshCtrl->u8HighThr(%d)\n",
                          pstThreshCtrl->u8LowThr, pstThreshCtrl->u8HighThr);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    default:
        CVE_ERR_TRACE("[CVE_THRESH] pstThreshCtrl->enMode(%d) set error\n", pstThreshCtrl->enMode);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }

    return AML_SUCCESS;
}

AML_S32 cve_check_sub_param(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                            CVE_DST_IMAGE_T *pstDst, CVE_SUB_CTRL_T *pstSubCtrl,
                            CVE_DST_MEM_INFO_T *pstDstmem)
{
    CVE_GET_IMAGE_SIZE(CVE_ALU_SUB);
    if (pstSubCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_ALU_CROP_CHECK(CVE_ALU_SUB, pstSubCtrl->stCrop, pstDst, pstSrcImage1);
        CVE_ALU_CROP_CHECK(CVE_ALU_SUB, pstSubCtrl->stCrop, pstDst, pstSrcImage2);
    }
    CVE_IMAGE_CHECK(CVE_ALU_SUB, pstSrcImage1);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_SUB, pstSrcImage1, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_SUB, pstSrcImage2);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_SUB, pstSrcImage2, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_SUB, pstDst);
    CVE_MODE_CHECK(CVE_ALU_SUB, pstSubCtrl->enMode, CVE_SUB_MODE_BUTT);
    CVE_MEM_CHECK(CVE_ALU_SUB, pstDstmem);
    if (pstSubCtrl->enMode == CVE_SUB_MODE_SHIFT) {
        CVE_IMAGE_TYPE_CHECK(CVE_ALU_SUB, pstDst, CVE_IMAGE_TYPE_S8C1);
    } else {
        CVE_IMAGE_TYPE_CHECK(CVE_ALU_SUB, pstDst, CVE_IMAGE_TYPE_U8C1);
    }
    CVE_RESOLUTION_EQUAL(CVE_ALU_SUB, pstSrcImage1, pstSrcImage2, RESOLUTION_DS_NONE);
    CVE_RESOLUTION_EQUAL(CVE_ALU_SUB, pstSrcImage1, pstDst, RESOLUTION_DS_NONE);
    return AML_SUCCESS;
}

AML_S32 cve_check_or_param(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                           CVE_DST_IMAGE_T *pstDst, CVE_OR_CTRL_T *pstOrCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_ALU_OR);
    if (pstOrCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_ALU_CROP_CHECK(CVE_ALU_OR, pstOrCtrl->stCrop, pstDst, pstSrcImage1);
        CVE_ALU_CROP_CHECK(CVE_ALU_OR, pstOrCtrl->stCrop, pstDst, pstSrcImage2);
    }
    CVE_IMAGE_CHECK(CVE_ALU_OR, pstSrcImage1);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_OR, pstSrcImage1, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_OR, pstSrcImage2);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_OR, pstSrcImage2, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_OR, pstDst);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_OR, pstDst, CVE_IMAGE_TYPE_U8C1);
    CVE_RESOLUTION_EQUAL(CVE_ALU_OR, pstSrcImage1, pstSrcImage2, RESOLUTION_DS_NONE);
    CVE_RESOLUTION_EQUAL(CVE_ALU_OR, pstSrcImage1, pstDst, RESOLUTION_DS_NONE);
    return AML_SUCCESS;
}

AML_S32 cve_check_and_param(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                            CVE_DST_IMAGE_T *pstDst, CVE_AND_CTRL_T *pstAndCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_ALU_AND);
    if (pstAndCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_ALU_CROP_CHECK(CVE_ALU_AND, pstAndCtrl->stCrop, pstDst, pstSrcImage1);
        CVE_ALU_CROP_CHECK(CVE_ALU_AND, pstAndCtrl->stCrop, pstDst, pstSrcImage2);
    }
    CVE_IMAGE_CHECK(CVE_ALU_AND, pstSrcImage1);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_AND, pstSrcImage1, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_AND, pstSrcImage2);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_AND, pstSrcImage2, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_AND, pstDst);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_AND, pstDst, CVE_IMAGE_TYPE_U8C1);
    CVE_RESOLUTION_EQUAL(CVE_ALU_AND, pstSrcImage1, pstSrcImage2, RESOLUTION_DS_NONE);
    CVE_RESOLUTION_EQUAL(CVE_ALU_AND, pstSrcImage1, pstDst, RESOLUTION_DS_NONE);
    return AML_SUCCESS;
}

AML_S32 cve_check_xor_param(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                            CVE_DST_IMAGE_T *pstDst, CVE_XOR_CTRL_T *pstXorCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_ALU_XOR);
    if (pstXorCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_ALU_CROP_CHECK(CVE_ALU_XOR, pstXorCtrl->stCrop, pstDst, pstSrcImage1);
        CVE_ALU_CROP_CHECK(CVE_ALU_XOR, pstXorCtrl->stCrop, pstDst, pstSrcImage2);
    }
    CVE_IMAGE_CHECK(CVE_ALU_XOR, pstSrcImage1);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_XOR, pstSrcImage1, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_XOR, pstSrcImage2);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_XOR, pstSrcImage2, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_XOR, pstDst);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_XOR, pstDst, CVE_IMAGE_TYPE_U8C1);
    CVE_RESOLUTION_EQUAL(CVE_ALU_XOR, pstSrcImage1, pstSrcImage2, RESOLUTION_DS_NONE);
    CVE_RESOLUTION_EQUAL(CVE_ALU_XOR, pstSrcImage1, pstDst, RESOLUTION_DS_NONE);
    return AML_SUCCESS;
}

AML_S32 cve_check_add_param(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                            CVE_DST_IMAGE_T *pstDst, CVE_ADD_CTRL_T *pstAddCtrl,
                            CVE_DST_MEM_INFO_T *pstDstmem)
{
    CVE_GET_IMAGE_SIZE(CVE_ALU_ADD);
    if (pstAddCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_ALU_CROP_CHECK(CVE_ALU_ADD, pstAddCtrl->stCrop, pstDst, pstSrcImage1);
        CVE_ALU_CROP_CHECK(CVE_ALU_ADD, pstAddCtrl->stCrop, pstDst, pstSrcImage2);
    }
    CVE_IMAGE_CHECK(CVE_ALU_ADD, pstSrcImage1);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_ADD, pstSrcImage1, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_ADD, pstSrcImage2);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_ADD, pstSrcImage2, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ALU_ADD, pstDst);
    CVE_IMAGE_TYPE_CHECK(CVE_ALU_ADD, pstDst, CVE_IMAGE_TYPE_U8C1);
    CVE_MEM_CHECK(CVE_ALU_SUB, pstDstmem);
    CVE_RESOLUTION_EQUAL(CVE_ALU_ADD, pstSrcImage1, pstSrcImage2, RESOLUTION_DS_NONE);
    CVE_RESOLUTION_EQUAL(CVE_ALU_ADD, pstSrcImage1, pstDst, RESOLUTION_DS_NONE);
    if (pstAddCtrl->u0q16X + pstAddCtrl->u0q16Y != 65536) {
        CVE_ERR_TRACE("[CVE_ALU_ADD] pstAddCtrl->u0q16X(%d) pstAddCtrl->u0q16Y(%d) set error. need "
                      "x+y=65536\n",
                      pstAddCtrl->u0q16X, pstAddCtrl->u0q16Y);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }
    return AML_SUCCESS;
}

AML_S32 cve_check_integ_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                              CVE_INTEG_CTRL_T *pstIntegCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_INTEG);
    if (pstIntegCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_ALU_ADD, pstIntegCtrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_INTEG, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_INTEG, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_INTEG, pstDstImage);
    CVE_RESOLUTION_EQUAL(CVE_INTEG, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_INTEG, pstIntegCtrl->enOutCtrl, CVE_INTEG_OUT_CTRL_BUTT);
    if ((pstIntegCtrl->enOutCtrl == CVE_INTEG_OUT_CTRL_COMBINE) ||
        (pstIntegCtrl->enOutCtrl == CVE_INTEG_OUT_CTRL_TQSUM)) {
        CVE_IMAGE_TYPE_CHECK(CVE_INTEG, pstDstImage, CVE_IMAGE_TYPE_U64C1);
    } else if (pstIntegCtrl->enOutCtrl == CVE_INTEG_OUT_CTRL_TUM) {
        CVE_IMAGE_TYPE_CHECK(CVE_INTEG, pstDstImage, CVE_IMAGE_TYPE_U32C1);
    }

    return AML_SUCCESS;
}

AML_S32 cve_check_hist_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_MEM_INFO_T *pstDstMem,
                             CVE_HIST_CTRL_T *pstHistCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_HIST);
    if (pstHistCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_SRC_CROP_CHECK(CVE_HIST, pstHistCtrl->stCrop, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_HIST, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_HIST, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_MEM_CHECK(CVE_HIST, pstDstMem);

    return AML_SUCCESS;
}

AML_S32 cve_check_thresh_s16_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                   CVE_THRESH_S16_CTRL_T *pstThreshS16Ctrl)
{
    CVE_GET_IMAGE_SIZE(CVE_THRESH_S16);
    if (pstThreshS16Ctrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_THRESH_S16, pstThreshS16Ctrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_THRESH_S16, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_THRESH_S16, pstSrcImage, CVE_IMAGE_TYPE_S16C1);
    CVE_IMAGE_CHECK(CVE_THRESH_S16, pstDstImage);
    CVE_MODE_CHECK(CVE_THRESH_S16, pstThreshS16Ctrl->enMode, CVE_THRESH_S16_MODE_BUTT);
    if ((pstThreshS16Ctrl->enMode == CVE_THRESH_S16_MODE_S16_TO_S8_MIN_MID_MAX) ||
        (pstThreshS16Ctrl->enMode == CVE_THRESH_S16_MODE_S16_TO_S8_MIN_ORI_MAX)) {
        CVE_IMAGE_TYPE_CHECK(CVE_THRESH_S16, pstDstImage, CVE_IMAGE_TYPE_S8C1);
    } else if ((pstThreshS16Ctrl->enMode == CVE_THRESH_S16_MODE_S16_TO_U8_MIN_MID_MAX) ||
               (pstThreshS16Ctrl->enMode == CVE_THRESH_S16_MODE_S16_TO_U8_MIN_ORI_MAX)) {
        CVE_IMAGE_TYPE_CHECK(CVE_THRESH_S16, pstDstImage, CVE_IMAGE_TYPE_U8C1);
    }
    CVE_RESOLUTION_EQUAL(CVE_THRESH_S16, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);

    switch (pstThreshS16Ctrl->enMode) {
    case CVE_THRESH_S16_MODE_S16_TO_U8_MIN_MID_MAX:
    case CVE_THRESH_S16_MODE_S16_TO_S8_MIN_MID_MAX:
        if (pstThreshS16Ctrl->s16LowThr > pstThreshS16Ctrl->s16HighThr) {
            CVE_ERR_TRACE("[CVE_THRESH_S16] pstThreshS16Ctrl->s16LowThr(%d) "
                          "pstThreshS16Ctrl->s16HighThr(%d) set error.\n",
                          pstThreshS16Ctrl->s16LowThr, pstThreshS16Ctrl->s16HighThr);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    case CVE_THRESH_S16_MODE_S16_TO_S8_MIN_ORI_MAX:
        if ((pstThreshS16Ctrl->s16LowThr > pstThreshS16Ctrl->s16HighThr) ||
            (pstThreshS16Ctrl->s16LowThr < -129) || (pstThreshS16Ctrl->s16HighThr > 127)) {
            CVE_ERR_TRACE("[CVE_THRESH_S16] pstThreshS16Ctrl->s16LowThr(%d) "
                          "pstThreshS16Ctrl->s16HighThr(%d) set error.\n",
                          pstThreshS16Ctrl->s16LowThr, pstThreshS16Ctrl->s16HighThr);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    case CVE_THRESH_S16_MODE_S16_TO_U8_MIN_ORI_MAX:
        if ((pstThreshS16Ctrl->s16LowThr > pstThreshS16Ctrl->s16HighThr) ||
            (pstThreshS16Ctrl->s16LowThr < -1) || (pstThreshS16Ctrl->s16HighThr > 255)) {
            CVE_ERR_TRACE("[CVE_THRESH_S16] pstThreshS16Ctrl->s16LowThr(%d) "
                          "pstThreshS16Ctrl->s16HighThr(%d) set error.\n",
                          pstThreshS16Ctrl->s16LowThr, pstThreshS16Ctrl->s16HighThr);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    default:
        CVE_ERR_TRACE("[CVE_THRESH_S16] pstThreshS16Ctrl->enMode(%d) set error.\n",
                      pstThreshS16Ctrl->enMode);
        return AML_ERR_CVE_ILLEGAL_PARAM;
        break;
    }
    return AML_SUCCESS;
}

AML_S32 cve_check_thresh_u16_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                   CVE_THRESH_U16_CTRL_T *pstThreshU16Ctrl)
{
    CVE_GET_IMAGE_SIZE(CVE_THRESH_U16);
    if (pstThreshU16Ctrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_THRESH_U16, pstThreshU16Ctrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_THRESH_U16, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_THRESH_U16, pstSrcImage, CVE_IMAGE_TYPE_U16C1);
    CVE_IMAGE_CHECK(CVE_THRESH_U16, pstDstImage);
    CVE_IMAGE_TYPE_CHECK(CVE_THRESH_U16, pstDstImage, CVE_IMAGE_TYPE_U8C1);
    CVE_RESOLUTION_EQUAL(CVE_THRESH_U16, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_THRESH_U16, pstThreshU16Ctrl->enMode, CVE_THRESH_U16_MODE_BUTT);
    switch (pstThreshU16Ctrl->enMode) {
    case CVE_THRESH_U16_MODE_U16_TO_U8_MIN_MID_MAX:
        if (pstThreshU16Ctrl->u16LowThr > pstThreshU16Ctrl->u16HighThr) {
            CVE_ERR_TRACE("[CVE_THRESH_U16] pstThreshU16Ctrl->u16LowThr(%d) "
                          "pstThreshU16Ctrl->u16HighThr(%d) set error.\n",
                          pstThreshU16Ctrl->u16LowThr, pstThreshU16Ctrl->u16HighThr);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    case CVE_THRESH_U16_MODE_U16_TO_U8_MIN_ORI_MAX:
        if ((pstThreshU16Ctrl->u16LowThr > pstThreshU16Ctrl->u16HighThr) ||
            (pstThreshU16Ctrl->u16HighThr > 255) || (pstThreshU16Ctrl->u16LowThr < 0)) {
            CVE_ERR_TRACE("[CVE_THRESH_U16] pstThreshU16Ctrl->u16LowThr(%d) "
                          "pstThreshU16Ctrl->u16HighThr(%d) set error.\n",
                          pstThreshU16Ctrl->u16LowThr, pstThreshU16Ctrl->u16HighThr);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;

    default:
        CVE_ERR_TRACE("[CVE_THRESH_U16] pstThreshU16Ctrl->enMode(%d) set error.\n",
                      pstThreshU16Ctrl->enMode);
        return AML_ERR_CVE_ILLEGAL_PARAM;
        break;
    }

    return AML_SUCCESS;
}

AML_S32 cve_check_16bit_to_8bit_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                      CVE_16BIT_TO_8BIT_CTRL_T *pst16BitTo8BitCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_16BIT_TO_8BIT);
    if (pst16BitTo8BitCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_16BIT_TO_8BIT, pst16BitTo8BitCtrl->stCrop, pstDstImage,
                             pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_16BIT_TO_8BIT, pstSrcImage);
    CVE_IMAGE_CHECK(CVE_16BIT_TO_8BIT, pstDstImage);
    CVE_RESOLUTION_EQUAL(CVE_16BIT_TO_8BIT, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_16BIT_TO_8BIT, pst16BitTo8BitCtrl->enMode, CVE_16BIT_TO_8BIT_MODE_BUTT);
    switch (pst16BitTo8BitCtrl->enMode) {
    case CVE_16BIT_TO_8BIT_MODE_S16_TO_S8:
        if ((pstSrcImage->enType != CVE_IMAGE_TYPE_S16C1) ||
            (pstDstImage->enType != CVE_IMAGE_TYPE_S8C1)) {
            CVE_ERR_TRACE("[CVE_16BIT_TO_8BIT] pstSrcImage->enType(%d) pstDstImage->enType(%d), "
                          "not support this type\n",
                          pstSrcImage->enType, pstDstImage->enType);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    case CVE_16BIT_TO_8BIT_MODE_S16_TO_U8_ABS:
    case CVE_16BIT_TO_8BIT_MODE_S16_TO_U8_BIAS:
        if ((pstSrcImage->enType != CVE_IMAGE_TYPE_S16C1) ||
            (pstDstImage->enType != CVE_IMAGE_TYPE_U8C1)) {
            CVE_ERR_TRACE("[CVE_16BIT_TO_8BIT] pstSrcImage->enType(%d) pstDstImage->enType(%d), "
                          "not support this type\n",
                          pstSrcImage->enType, pstDstImage->enType);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    case CVE_16BIT_TO_8BIT_MODE_U16_TO_U8:
        if ((pstSrcImage->enType != CVE_IMAGE_TYPE_U16C1) ||
            (pstDstImage->enType != CVE_IMAGE_TYPE_U8C1)) {
            CVE_ERR_TRACE("[CVE_16BIT_TO_8BIT] pstSrcImage->enType(%d) pstDstImage->enType(%d), "
                          "not support this type\n",
                          pstSrcImage->enType, pstDstImage->enType);
            return AML_ERR_CVE_ILLEGAL_PARAM;
        }
        break;
    default:
        CVE_ERR_TRACE("[CVE_16BIT_TO_8BIT] pstSrcImage->enType(%d) pstDstImage->enType(%d), not "
                      "support this type\n",
                      pstSrcImage->enType, pstDstImage->enType);
        break;
    }

    return AML_SUCCESS;
}
AML_S32 cve_check_ord_stat_filter_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                        CVE_ORD_STAT_FILTER_CTRL_T *pstOrdStatFltCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_ORD_STAT_FILTER);
    if (pstOrdStatFltCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_ORD_STAT_FILTER, pstOrdStatFltCtrl->stCrop, pstDstImage,
                             pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_ORD_STAT_FILTER, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_ORD_STAT_FILTER, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ORD_STAT_FILTER, pstDstImage);
    CVE_IMAGE_TYPE_CHECK(CVE_ORD_STAT_FILTER, pstDstImage, CVE_IMAGE_TYPE_U8C1);
    CVE_RESOLUTION_EQUAL(CVE_ORD_STAT_FILTER, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_ORD_STAT_FILTER, pstOrdStatFltCtrl->enMode, CVE_ORD_STAT_FILTER_MODE_BUTT);

    return AML_SUCCESS;
}

AML_S32 cve_check_map_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_SRC_MEM_INFO_T *pstMap,
                            CVE_DST_IMAGE_T *pstDstImage, CVE_MAP_CTRL_T *pstMapCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_MAP);
    if (pstMapCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_MAP, pstMapCtrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_MAP, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_MAP, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_MEM_CHECK(CVE_MAP, pstMap);
    CVE_IMAGE_CHECK(CVE_MAP, pstDstImage);
    CVE_RESOLUTION_EQUAL(CVE_MAP, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_MAP, pstMapCtrl->enMode, CVE_MAP_MODE_BUTT);
    if (pstMapCtrl->enMode == CVE_MAP_MODE_U8) {
        CVE_IMAGE_TYPE_CHECK(CVE_MAP, pstDstImage, CVE_IMAGE_TYPE_U8C1);
    } else if (pstMapCtrl->enMode == CVE_MAP_MODE_U16) {
        CVE_IMAGE_TYPE_CHECK(CVE_MAP, pstDstImage, CVE_IMAGE_TYPE_U16C1);
    } else if (pstMapCtrl->enMode == CVE_MAP_MODE_S16) {
        CVE_IMAGE_TYPE_CHECK(CVE_MAP, pstDstImage, CVE_IMAGE_TYPE_S16C1);
    }
    return AML_SUCCESS;
}

AML_S32 cve_check_equalize_hist_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                      CVE_EQUALIZE_HIST_CTRL_T *pstEqualizeHistCtrl)
{
    CVE_MEM_INFO_T *pstMem = &pstEqualizeHistCtrl->stMem;

    CVE_GET_IMAGE_SIZE(CVE_EQUALIZE_HIST);
    if (pstEqualizeHistCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_EQUALIZE_HIST, pstEqualizeHistCtrl->stCrop, pstDstImage,
                             pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_EQUALIZE_HIST, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_EQUALIZE_HIST, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_EQUALIZE_HIST, pstDstImage);
    CVE_IMAGE_TYPE_CHECK(CVE_EQUALIZE_HIST, pstDstImage, CVE_IMAGE_TYPE_U8C1);
    CVE_RESOLUTION_EQUAL(CVE_EQUALIZE_HIST, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MEM_CHECK(CVE_EQUALIZE_HIST, pstMem);
    return AML_SUCCESS;
}

AML_S32 cve_check_ncc_param(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                            CVE_DST_MEM_INFO_T *pstDstmem, CVE_NCC_CTRL_T *pstNccCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_NCC);
    if (pstNccCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_SRC_CROP_CHECK(CVE_NCC, pstNccCtrl->stCrop, pstSrcImage1);
        CVE_SRC_CROP_CHECK(CVE_NCC, pstNccCtrl->stCrop, pstSrcImage2);
    }
    CVE_IMAGE_CHECK(CVE_NCC, pstSrcImage1);
    CVE_IMAGE_TYPE_CHECK(CVE_NCC, pstSrcImage1, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_NCC, pstSrcImage2);
    CVE_IMAGE_TYPE_CHECK(CVE_NCC, pstSrcImage2, CVE_IMAGE_TYPE_U8C1);
    CVE_RESOLUTION_EQUAL(CVE_NCC, pstSrcImage1, pstSrcImage2, RESOLUTION_DS_NONE);
    CVE_MEM_CHECK(CVE_NCC, pstDstmem);
    CVE_MODE_CHECK(CVE_NCC, pstNccCtrl->enMode, CVE_NCC_MODE_BUTT);
    return AML_SUCCESS;
}

AML_S32 cve_check_ccl_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                            CVE_DST_MEM_INFO_T *pstBlob, CVE_CCL_CTRL_T *pstCclCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_CCL);
    if (pstCclCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_CCL, pstCclCtrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_CHECK(CVE_CCL, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_CCL, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_CCL, pstDstImage);
    CVE_IMAGE_TYPE_CHECK(CVE_CCL, pstDstImage, CVE_IMAGE_TYPE_U16C1);
    CVE_RESOLUTION_EQUAL(CVE_CCL, pstSrcImage, pstDstImage, RESOLUTION_DS_NONE);
    CVE_MEM_CHECK(CVE_CCL, pstBlob);
    CVE_MODE_CHECK(CVE_CCL, pstCclCtrl->enInputDataMode, CVE_CCL_INPUT_DATA_MODE_BUTT);
    return AML_SUCCESS;
}

AML_S32 cve_check_gmm_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_SRC_IMAGE_T *pstFactor,
                            CVE_DST_IMAGE_T *pstFg, CVE_DST_IMAGE_T *pstBg,
                            CVE_MEM_INFO_T *pstModel, CVE_GMM_CTRL_T *pstGmmCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_GMM);
    CVE_IMAGE_CHECK(CVE_GMM, pstSrcImage);
    if ((pstSrcImage->enType != CVE_IMAGE_TYPE_U8C1) &&
        (pstSrcImage->enType != CVE_IMAGE_TYPE_U8C3_PACKAGE)) {
        CVE_ERR_TRACE("[CVE_GMM] pstSrcImage->enType(%d) set error.\n", pstSrcImage->enType);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }
    if (pstGmmCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_GMM, pstGmmCtrl->stCrop, pstFg, pstSrcImage);
        CVE_IMAGE_CROP_CHECK(CVE_GMM, pstGmmCtrl->stCrop, pstBg, pstSrcImage);
        CVE_IMAGE_CROP_CHECK(CVE_GMM, pstGmmCtrl->stCrop, pstFg, pstFactor);
        CVE_IMAGE_CROP_CHECK(CVE_GMM, pstGmmCtrl->stCrop, pstBg, pstFactor);
    }
    CVE_IMAGE_CHECK(CVE_GMM, pstFactor);
    CVE_IMAGE_TYPE_CHECK(CVE_GMM, pstFactor, CVE_IMAGE_TYPE_U16C1);
    CVE_IMAGE_CHECK(CVE_GMM, pstFg);
    CVE_IMAGE_TYPE_CHECK(CVE_GMM, pstFg, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_GMM, pstBg);
    CVE_IMAGE_TYPE_EQUAL(CVE_GMM, pstSrcImage, pstBg);
    CVE_MEM_CHECK(CVE_GMM, pstModel);
    CVE_MODE_CHECK(CVE_GMM, pstGmmCtrl->enOutputMode, CVE_GMM_OUTPUT_MODE_BUTT);
    CVE_MODE_CHECK(CVE_GMM, pstGmmCtrl->enSnsFactorMode, CVE_GMM_SNS_FACTOR_MODE_BUTT);
    CVE_MODE_CHECK(CVE_GMM, pstGmmCtrl->enDurationUpdateFactorMode,
                   CVE_GMM_DURATION_UPDATE_FACTOR_MODE_BUTT);
    CVE_MODE_CHECK(CVE_GMM, pstGmmCtrl->enDownScaleMode, CVE_GMM_DOWN_SCALE_MODE_BUTT);
    CVE_MODE_CHECK(CVE_GMM, pstGmmCtrl->enOutputDataMode, CVE_GMM_OUTPUT_DATA_MODE_BUTT);
    CVE_RANGE_CHECK(CVE_GMM, pstGmmCtrl->u8ModelNum, 1, 5);
    CVE_UNSIGNED_RANGE_CHECK(CVE_GMM, pstGmmCtrl->u3q7SigmaScale, 1023);
    CVE_BOOL_CHECK(CVE_GMM, pstGmmCtrl->enFastLearn);
    return AML_SUCCESS;
}

AML_S32 cve_check_canny_hys_edge_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstEdge,
                                       CVE_DST_MEM_INFO_T *pstStack,
                                       CVE_CANNY_HYS_EDGE_CTRL_T *pstCannyHysEdgeCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_CANNY_EDGE);
    if (pstCannyHysEdgeCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_CANNY_EDGE, pstCannyHysEdgeCtrl->stCrop, pstEdge, pstSrcImage);
    }
    CVE_IMAGE_TYPE_CHECK(CVE_CANNY_EDGE, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_CANNY_EDGE, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_CANNY_EDGE, pstEdge, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_CANNY_EDGE, pstEdge);
    CVE_MEM_CHECK(CVE_CANNY_EDGE, pstStack);
    CVE_MEM_CHECK(CVE_CANNY_EDGE, (&pstCannyHysEdgeCtrl->stMem));
    CVE_RESOLUTION_EQUAL(CVE_CANNY_EDGE, pstSrcImage, pstEdge, RESOLUTION_DS_NONE);
    CVE_BOOL_CHECK(CVE_CANNY_EDGE, pstCannyHysEdgeCtrl->bGauss);

    return AML_SUCCESS;
}

AML_S32 cve_check_lbp_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                            CVE_LBP_CTRL_T *pstLbpCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_LBP);
    if (pstLbpCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_LBP, pstLbpCtrl->stCrop, pstDstImage, pstSrcImage);
    }
    CVE_IMAGE_TYPE_CHECK(CVE_LBP, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_LBP, pstSrcImage);
    CVE_IMAGE_TYPE_CHECK(CVE_LBP, pstDstImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_LBP, pstDstImage);

    CVE_MODE_CHECK(CVE_LBP, pstLbpCtrl->enMode, CVE_LBP_CMP_MODE_BUTT);

    return AML_SUCCESS;
}

AML_S32 cve_check_norm_grad_param(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstH,
                                  CVE_DST_IMAGE_T *pstDstV, CVE_DST_IMAGE_T *pstDstHV,
                                  CVE_NORM_GRAD_CTRL_T *pstNormGradCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_NORM_GRAD);
    if (pstNormGradCtrl->stCrop.enCrop == AML_TRUE &&
        pstNormGradCtrl->enOutCtrl == CVE_NORM_GRAD_OUT_CTRL_HOR_AND_VER) {
        CVE_IMAGE_CROP_CHECK(CVE_NORM_GRAD, pstNormGradCtrl->stCrop, pstDstH, pstSrcImage);
        CVE_IMAGE_CROP_CHECK(CVE_NORM_GRAD, pstNormGradCtrl->stCrop, pstDstV, pstSrcImage);
    } else if (pstNormGradCtrl->stCrop.enCrop == AML_TRUE &&
               pstNormGradCtrl->enOutCtrl == CVE_NORM_GRAD_OUT_CTRL_HOR) {
        CVE_IMAGE_CROP_CHECK(CVE_NORM_GRAD, pstNormGradCtrl->stCrop, pstDstH, pstSrcImage);
    } else if (pstNormGradCtrl->stCrop.enCrop == AML_TRUE &&
               pstNormGradCtrl->enOutCtrl == CVE_NORM_GRAD_OUT_CTRL_VER) {
        CVE_IMAGE_CROP_CHECK(CVE_NORM_GRAD, pstNormGradCtrl->stCrop, pstDstV, pstSrcImage);
    } else {
        CVE_IMAGE_CROP_CHECK(CVE_NORM_GRAD, pstNormGradCtrl->stCrop, pstDstHV, pstSrcImage);
    }
    CVE_IMAGE_TYPE_CHECK(CVE_NORM_GRAD, pstSrcImage, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_NORM_GRAD, pstSrcImage);

    CVE_MODE_CHECK(CVE_NORM_GRAD, pstNormGradCtrl->enOutCtrl, CVE_NORM_GRAD_OUT_CTRL_BUTT);
    switch (pstNormGradCtrl->enOutCtrl) {
    case CVE_NORM_GRAD_OUT_CTRL_HOR_AND_VER:
        CVE_IMAGE_TYPE_CHECK(CVE_NORM_GRAD, pstDstH, CVE_IMAGE_TYPE_S8C1);
        CVE_IMAGE_CHECK(CVE_NORM_GRAD, pstDstH);
        CVE_IMAGE_TYPE_CHECK(CVE_NORM_GRAD, pstDstV, CVE_IMAGE_TYPE_S8C1);
        CVE_IMAGE_CHECK(CVE_NORM_GRAD, pstDstV);
        CVE_RESOLUTION_EQUAL(CVE_NORM_GRAD, pstSrcImage, pstDstH, RESOLUTION_DS_NONE);
        CVE_RESOLUTION_EQUAL(CVE_NORM_GRAD, pstSrcImage, pstDstV, RESOLUTION_DS_NONE);
        break;
    case CVE_NORM_GRAD_OUT_CTRL_HOR:
        CVE_IMAGE_TYPE_CHECK(CVE_NORM_GRAD, pstDstH, CVE_IMAGE_TYPE_S8C1);
        CVE_IMAGE_CHECK(CVE_NORM_GRAD, pstDstH);
        CVE_RESOLUTION_EQUAL(CVE_NORM_GRAD, pstSrcImage, pstDstH, RESOLUTION_DS_NONE);
        break;
    case CVE_NORM_GRAD_OUT_CTRL_VER:
        CVE_IMAGE_TYPE_CHECK(CVE_NORM_GRAD, pstDstV, CVE_IMAGE_TYPE_S8C1);
        CVE_IMAGE_CHECK(CVE_NORM_GRAD, pstDstV);
        CVE_RESOLUTION_EQUAL(CVE_NORM_GRAD, pstSrcImage, pstDstV, RESOLUTION_DS_NONE);
        break;
    case CVE_NORM_GRAD_OUT_CTRL_COMBINE:
        CVE_IMAGE_TYPE_CHECK(CVE_NORM_GRAD, pstDstHV, CVE_IMAGE_TYPE_S8C2_PACKAGE);
        CVE_IMAGE_CHECK(CVE_NORM_GRAD, pstDstHV);
        CVE_RESOLUTION_EQUAL(CVE_NORM_GRAD, pstSrcImage, pstDstHV, RESOLUTION_DS_NONE);
        break;
    default:
        break;
    }

    CVE_UNSIGNED_RANGE_CHECK(CVE_NORM_GRAD, pstNormGradCtrl->u8Norm, 13);

    return AML_SUCCESS;
}

AML_S32 cve_check_build_lk_optical_flow_pyr_param(
    CVE_SRC_IMAGE_T *pstSrcPyr, CVE_SRC_IMAGE_T astDstPyr[],
    CVE_BUILD_LK_OPTICAL_FLOW_PYR_CTRL_T *pstBuildLkOptiFlowPyrCtrl)
{
    int i;

    CVE_GET_IMAGE_SIZE(CVE_BUILD_LK_OPTICAL_FLOW_PYR);

    CVE_IMAGE_TYPE_CHECK(CVE_BUILD_LK_OPTICAL_FLOW_PYR, pstSrcPyr, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_BUILD_LK_OPTICAL_FLOW_PYR, pstSrcPyr);
    CVE_UNSIGNED_RANGE_CHECK(CVE_BUILD_LK_OPTICAL_FLOW_PYR, pstBuildLkOptiFlowPyrCtrl->u8MaxLevel,
                             3);
    for (i = 0; i <= pstBuildLkOptiFlowPyrCtrl->u8MaxLevel; i++) {
        if (pstBuildLkOptiFlowPyrCtrl->stCrop.enCrop == AML_TRUE) {
            CVE_IMAGE_CROP_CHECK(CVE_BUILD_LK_OPTICAL_FLOW_PYR, pstBuildLkOptiFlowPyrCtrl->stCrop,
                                 (&astDstPyr[i]), pstSrcPyr);
        }
        CVE_IMAGE_TYPE_CHECK(CVE_BUILD_LK_OPTICAL_FLOW_PYR, (&astDstPyr[i]), CVE_IMAGE_TYPE_U8C1);
        CVE_RESOLUTION_EQUAL(CVE_BUILD_LK_OPTICAL_FLOW_PYR, pstSrcPyr, (&astDstPyr[i]), i);
    }

    return AML_SUCCESS;
}

AML_S32
cve_check_lk_optical_flow_pyr_param(CVE_SRC_IMAGE_T astSrcPrevPyr[],
                                    CVE_SRC_IMAGE_T astSrcNextPyr[], CVE_SRC_MEM_INFO_T *pstPrevPts,
                                    CVE_MEM_INFO_T *pstNextPts, CVE_DST_MEM_INFO_T *pstStatus,
                                    CVE_LK_OPTICAL_FLOW_PYR_CTRL_T *pstLkOptiFlowPyrCtrl)
{
    int i;

    CVE_GET_IMAGE_SIZE(CVE_LK_OPTICAL_FLOW_PYR);
    CVE_UNSIGNED_RANGE_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstLkOptiFlowPyrCtrl->u8MaxLevel, 3);
    for (i = 0; i <= pstLkOptiFlowPyrCtrl->u8MaxLevel; i++) {
        if (pstLkOptiFlowPyrCtrl->stCrop.enCrop == AML_TRUE) {
            CVE_SRC_CROP_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstLkOptiFlowPyrCtrl->stCrop,
                               (&astSrcPrevPyr[i]));
            CVE_SRC_CROP_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstLkOptiFlowPyrCtrl->stCrop,
                               (&astSrcNextPyr[i]));
        }
        CVE_IMAGE_TYPE_CHECK(CVE_LK_OPTICAL_FLOW_PYR, (&astSrcPrevPyr[i]), CVE_IMAGE_TYPE_U8C1);
        CVE_IMAGE_TYPE_CHECK(CVE_LK_OPTICAL_FLOW_PYR, (&astSrcNextPyr[i]), CVE_IMAGE_TYPE_U8C1);
        CVE_RESOLUTION_EQUAL(CVE_LK_OPTICAL_FLOW_PYR, (&astSrcPrevPyr[0]), (&astSrcPrevPyr[i]), i);
        CVE_RESOLUTION_EQUAL(CVE_LK_OPTICAL_FLOW_PYR, (&astSrcPrevPyr[0]), (&astSrcNextPyr[i]), i);
    }

    CVE_MODE_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstLkOptiFlowPyrCtrl->enOutMode,
                   CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_BUTT);

    CVE_MEM_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstPrevPts);
    CVE_MEM_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstNextPts);
    CVE_MEM_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstStatus);

    CVE_BOOL_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstLkOptiFlowPyrCtrl->bUseInitFlow);
    CVE_RANGE_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstLkOptiFlowPyrCtrl->u16PtsNum, 1, 500);
    CVE_RANGE_CHECK(CVE_LK_OPTICAL_FLOW_PYR, pstLkOptiFlowPyrCtrl->u8IterCnt, 1, 20);

    return AML_SUCCESS;
}

AML_S32 cve_check_st_candi_corner_param(CVE_SRC_IMAGE_T *pstSrc, CVE_DST_IMAGE_T *pstLabel,
                                        CVE_DST_IMAGE_T *pstCandiCorner,
                                        CVE_DST_MEM_INFO_T *pstCandiCornerPoint,
                                        CVE_ST_CANDI_CORNER_CTRL_T *pstStCandiCornerCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_ST_CANDI_CORNER);
    CVE_IMAGE_TYPE_CHECK(CVE_ST_CANDI_CORNER, pstSrc, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ST_CANDI_CORNER, pstSrc);
    CVE_IMAGE_TYPE_CHECK(CVE_ST_CANDI_CORNER, pstLabel, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_ST_CANDI_CORNER, pstLabel);
    CVE_IMAGE_TYPE_CHECK(CVE_ST_CANDI_CORNER, pstCandiCorner, CVE_IMAGE_TYPE_U16C1);
    CVE_IMAGE_CHECK(CVE_ST_CANDI_CORNER, pstCandiCorner);
    CVE_MEM_CHECK(CVE_ST_CANDI_CORNER, pstCandiCornerPoint);
    CVE_MEM_CHECK(CVE_ST_CANDI_CORNER, (&pstStCandiCornerCtrl->stMem));
    if (pstStCandiCornerCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_ST_CANDI_CORNER, pstStCandiCornerCtrl->stCrop, pstLabel, pstSrc);
        CVE_IMAGE_CROP_CHECK(CVE_ST_CANDI_CORNER, pstStCandiCornerCtrl->stCrop, pstCandiCorner,
                             pstSrc);
    } else {
        CVE_RESOLUTION_EQUAL(CVE_ST_CANDI_CORNER, pstSrc, pstLabel, RESOLUTION_DS_NONE);
        CVE_RESOLUTION_EQUAL(CVE_ST_CANDI_CORNER, pstSrc, pstCandiCorner, RESOLUTION_DS_NONE);
    }

    CVE_MODE_CHECK(CVE_ST_CANDI_CORNER, pstStCandiCornerCtrl->enOutputDataMode,
                   CVE_ST_CANDI_CORNER_OUTPUT_DATA_MODE_BUTT);

    return AML_SUCCESS;
}

AML_S32 cve_check_sad_param(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                            CVE_DST_IMAGE_T *pstSad, CVE_DST_IMAGE_T *pstThr,
                            CVE_SAD_CTRL_T *pstSadCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_SAD);
    CVE_IMAGE_TYPE_CHECK(CVE_SAD, pstSrcImage1, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_SAD, pstSrcImage1);
    CVE_IMAGE_TYPE_CHECK(CVE_SAD, pstSrcImage2, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_SAD, pstSrcImage2);
    CVE_IMAGE_TYPE_CHECK(CVE_SAD, pstSad, CVE_IMAGE_TYPE_U16C1);
    CVE_IMAGE_TYPE_CHECK(CVE_SAD, pstThr, CVE_IMAGE_TYPE_U8C1);
    if (pstSadCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_SRC_CROP_CHECK(CVE_SAD, pstSadCtrl->stCrop, pstSrcImage1);
        CVE_SRC_CROP_CHECK(CVE_SAD, pstSadCtrl->stCrop, pstSrcImage2);
    }

    CVE_RESOLUTION_EQUAL(CVE_SAD, pstSrcImage1, pstSrcImage2, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_SAD, pstSadCtrl->enMode, CVE_SAD_MODE_BUTT);
    if (pstSadCtrl->enMode == CVE_SAD_MODE_MB_4X4) {
        CVE_RESOLUTION_EQUAL(CVE_SAD, pstSrcImage1, pstSad, RESOLUTION_DS_MB_4X4);
        CVE_RESOLUTION_EQUAL(CVE_SAD, pstSrcImage1, pstThr, RESOLUTION_DS_MB_4X4);
    } else if (pstSadCtrl->enMode == CVE_SAD_MODE_MB_8X8) {
        CVE_RESOLUTION_EQUAL(CVE_SAD, pstSrcImage1, pstSad, RESOLUTION_DS_MB_8X8);
        CVE_RESOLUTION_EQUAL(CVE_SAD, pstSrcImage1, pstThr, RESOLUTION_DS_MB_8X8);
    } else {
        CVE_RESOLUTION_EQUAL(CVE_SAD, pstSrcImage1, pstSad, RESOLUTION_DS_MB_16X16);
        CVE_RESOLUTION_EQUAL(CVE_SAD, pstSrcImage1, pstThr, RESOLUTION_DS_MB_16X16);
    }

    return AML_SUCCESS;
}

AML_S32 cve_check_grad_fg_param(CVE_SRC_IMAGE_T *pstBgDiffFg, CVE_SRC_IMAGE_T *pstCurGrad,
                                CVE_SRC_IMAGE_T *pstBgGrad, CVE_DST_IMAGE_T *pstGradFg,
                                CVE_GRAD_FG_CTRL_T *pstGradFgCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_GRAD_FG);
    if (pstGradFgCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_IMAGE_CROP_CHECK(CVE_GRAD_FG, pstGradFgCtrl->stCrop, pstGradFg, pstBgDiffFg);
        CVE_IMAGE_CROP_CHECK(CVE_GRAD_FG, pstGradFgCtrl->stCrop, pstGradFg, pstCurGrad);
        CVE_IMAGE_CROP_CHECK(CVE_GRAD_FG, pstGradFgCtrl->stCrop, pstGradFg, pstBgGrad);
    }
    CVE_IMAGE_TYPE_CHECK(CVE_GRAD_FG, pstBgDiffFg, CVE_IMAGE_TYPE_U8C1);
    CVE_IMAGE_CHECK(CVE_GRAD_FG, pstBgDiffFg);
    CVE_IMAGE_TYPE_CHECK(CVE_GRAD_FG, pstCurGrad, CVE_IMAGE_TYPE_S8C2_PACKAGE);
    CVE_IMAGE_CHECK(CVE_GRAD_FG, pstCurGrad);
    CVE_IMAGE_TYPE_CHECK(CVE_GRAD_FG, pstBgGrad, CVE_IMAGE_TYPE_S8C2_PACKAGE);
    CVE_IMAGE_CHECK(CVE_GRAD_FG, pstBgGrad);
    CVE_IMAGE_TYPE_CHECK(CVE_GRAD_FG, pstGradFg, CVE_IMAGE_TYPE_S8C2_PACKAGE);
    CVE_IMAGE_CHECK(CVE_GRAD_FG, pstGradFg);

    CVE_RESOLUTION_EQUAL(CVE_GRAD_FG, pstBgDiffFg, pstCurGrad, RESOLUTION_DS_NONE);
    CVE_RESOLUTION_EQUAL(CVE_GRAD_FG, pstBgDiffFg, pstBgGrad, RESOLUTION_DS_NONE);
    CVE_RESOLUTION_EQUAL(CVE_GRAD_FG, pstBgDiffFg, pstGradFg, RESOLUTION_DS_NONE);

    CVE_MODE_CHECK(CVE_GRAD_FG, pstGradFgCtrl->enMode, CVE_GRAD_FG_MODE_BUTT);

    return AML_SUCCESS;
}

AML_S32 cve_check_update_bg_model_param(CVE_SRC_IMAGE_T *pstCurImg, CVE_MEM_INFO_T *pstBgModel1,
                                        CVE_MEM_INFO_T *pstBgModel2,
                                        CVE_DST_MEM_INFO_T *pstStatData,
                                        CVE_UPDATE_BG_MODEL_CTRL_T *pstUpdateBgModelCtrl)
{
    CVE_GET_IMAGE_SIZE(CVE_UPDATE_BG_MODEL);
    if (pstUpdateBgModelCtrl->stCrop.enCrop == AML_TRUE) {
        CVE_SRC_CROP_CHECK(CVE_UPDATE_BG_MODEL, pstUpdateBgModelCtrl->stCrop, pstCurImg);
    }
    CVE_IMAGE_CHECK(CVE_UPDATE_BG_MODEL, pstCurImg);
    CVE_IMAGE_TYPE_CHECK(CVE_UPDATE_BG_MODEL, pstCurImg, CVE_IMAGE_TYPE_U8C1);
    CVE_MEM_CHECK(CVE_UPDATE_BG_MODEL, pstBgModel1);
    CVE_MEM_CHECK(CVE_UPDATE_BG_MODEL, pstBgModel2);
    CVE_MEM_CHECK(CVE_UPDATE_BG_MODEL, pstStatData);
    CVE_MODE_CHECK(CVE_UPDATE_BG_MODEL, pstUpdateBgModelCtrl->enDownScaleMode,
                   CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_BUTT);
    return AML_SUCCESS;
}
AML_S32 cve_check_tof_param(CVE_SRC_RAW_T *pstSrcRaw, CVE_SRC_RAW_T *pstSrcFpn,
                            CVE_SRC_MEM_INFO_T *pstSrcCoef, CVE_SRC_MEM_INFO_T *pstBpc,
                            CVE_DST_MEM_INFO_T *pstDtsStatus, CVE_DST_MEM_INFO_T *pstDtsIR,
                            CVE_DST_MEM_INFO_T *pstDtsData, CVE_DST_MEM_INFO_T *pstDstHist,
                            CVE_TOF_CTRL_T *pstTofCtrl)
{
    int i;

    CVE_GET_IMAGE_SIZE(CVE_TOF);
    CVE_RAW_CHECK(CVE_TOF, pstSrcRaw);
    CVE_RAW_CHECK(CVE_TOF, pstSrcFpn);
    CVE_MEM_CHECK(CVE_TOF, pstSrcCoef);
    CVE_MEM_CHECK(CVE_TOF, pstBpc);
    CVE_MEM_CHECK(CVE_TOF, pstDtsStatus);
    CVE_MEM_CHECK(CVE_TOF, pstDtsIR);
    CVE_MEM_CHECK(CVE_TOF, pstDtsData);
    CVE_MEM_CHECK(CVE_TOF, pstDstHist);
    CVE_RESOLUTION_EQUAL(CVE_TOF, pstSrcRaw, pstSrcFpn, RESOLUTION_DS_NONE);
    CVE_MODE_CHECK(CVE_TOF, pstTofCtrl->enRawMode, CVE_RAW_MODE_BUTT);
    CVE_MODE_CHECK(CVE_TOF, pstTofCtrl->enFpnMode, CVE_TOF_FPN_MODE_BUTT);
    CVE_BOOL_CHECK(CVE_TOF, pstTofCtrl->bRawShift);
    CVE_BOOL_CHECK(CVE_TOF, pstTofCtrl->bBypass);
    CVE_BOOL_CHECK(CVE_TOF, pstTofCtrl->bSpa1En);
    CVE_BOOL_CHECK(CVE_TOF, pstTofCtrl->bSpa2En);
    for (i = 0; i < 5; i++) {
        CVE_RANGE_CHECK(CVE_TOF, pstTofCtrl->as32PCoef[i], -(1 << 23), (1 << 23) - 1);
    }
    CVE_RANGE_CHECK(CVE_TOF, pstTofCtrl->as16TCoef1[0], -(1 << 11), (1 << 11) - 1);
    CVE_RANGE_CHECK(CVE_TOF, pstTofCtrl->as16TCoef1[1], -(1 << 8), (1 << 8) - 1);
    CVE_RANGE_CHECK(CVE_TOF, pstTofCtrl->as16TCoef1[2], -(1 << 11), (1 << 11) - 1);
    CVE_RANGE_CHECK(CVE_TOF, pstTofCtrl->as16TCoef1[3], -(1 << 8), (1 << 8) - 1);
    CVE_UNSIGNED_RANGE_CHECK(CVE_TOF, pstTofCtrl->u16HistXstart,
                             CVE_MIN(pstSrcRaw->u32Width, 1024));
    CVE_UNSIGNED_RANGE_CHECK(CVE_TOF, pstTofCtrl->u16HistYstart,
                             CVE_MIN(pstSrcRaw->u32Height, 1024));
    CVE_RANGE_CHECK(CVE_TOF, pstTofCtrl->u16HistXend, pstTofCtrl->u16HistXstart,
                    CVE_MIN(pstSrcRaw->u32Width, 1024));
    CVE_RANGE_CHECK(CVE_TOF, pstTofCtrl->u16HistYend, pstTofCtrl->u16HistYstart,
                    CVE_MIN(pstSrcRaw->u32Height, 1024));
    CVE_UNSIGNED_RANGE_CHECK(CVE_TOF, pstTofCtrl->u16Q1HighThr, (1 << 12) - 1);
    CVE_UNSIGNED_RANGE_CHECK(CVE_TOF, pstTofCtrl->u16Q23HighThr, (1 << 12) - 1);
    CVE_UNSIGNED_RANGE_CHECK(CVE_TOF, pstTofCtrl->u16IRLowThr, (1 << 12) - 1);
    CVE_RANGE_CHECK(CVE_TOF, pstTofCtrl->u16IRHighThr, pstTofCtrl->u16IRLowThr, (1 << 12) - 1);
    CVE_UNSIGNED_RANGE_CHECK(CVE_TOF, pstTofCtrl->u8SpaNorm, 12);

    return AML_SUCCESS;
}

int cve_fill_dma_task(CVE_DATA_T *pstSrcDATA, CVE_DST_DATA_T *pstDstDATA,
                      CVE_DMA_CTRL_T *pstDmaCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_dma_params_t dma_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&dma_params, 0, sizeof(cve_op_dma_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_data(&init_params, pstSrcDATA, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_data(&init_params, pstDstDATA, &dst_off);
    if (ret != 0) {
        return ret;
    }
    init_params.src_addr[0] = pstSrcDATA->u64PhyAddr;
    init_params.dst_addr[0] = pstDstDATA->u64PhyAddr;
    init_params.src_stride[0] = pstSrcDATA->u32Stride;
    init_params.dst_stride[0] = pstDstDATA->u32Stride;
    init_params.src_width = pstSrcDATA->u32Width;
    init_params.src_height = pstSrcDATA->u32Height;
    if (pstDmaCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.dst_width = pstDmaCtrl->stCrop.u16Width;
        init_params.dst_height = pstDmaCtrl->stCrop.u16Height;
        init_params.xstart = pstDmaCtrl->stCrop.u16X;
        init_params.ystart = pstDmaCtrl->stCrop.u16Y;
        init_params.xSize = pstDmaCtrl->stCrop.u16Width;
        init_params.ySize = pstDmaCtrl->stCrop.u16Height;
    } else {
        init_params.dst_width = pstDstDATA->u32Width;
        init_params.dst_height = pstDstDATA->u32Height;
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&dma_params.comm_params, &init_params);

    /* dma op params initialization */
    if (pstDmaCtrl->enMode == CVE_DMA_MODE_DIRECT_COPY ||
        pstDmaCtrl->enMode == CVE_DMA_MODE_SET_3BYTE ||
        pstDmaCtrl->enMode == CVE_DMA_MODE_SET_8BYTE || pstDmaCtrl->enMode == CVE_DMA_MODE_NOT) {
        dma_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0x4;
        dma_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0x4;
    } else if (pstDmaCtrl->enMode == CVE_DMA_MODE_INTERVAL_COPY) {
        dma_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
        dma_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    }
    dma_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_DMA;
    dma_params.reg_1b.bits.dma_mode_ctrl = pstDmaCtrl->enMode;

    if (dma_params.reg_1b.bits.dma_mode_ctrl == CVE_DMA_MODE_SET_3BYTE ||
        dma_params.reg_1b.bits.dma_mode_ctrl == CVE_DMA_MODE_SET_8BYTE) {
        dma_params.reg_1c.bits.dma_u64val_lsb = CVE_GET_LO64(pstDmaCtrl->u64Val);
        dma_params.reg_1d.bits.dma_u64val_msb = CVE_GET_HI64(pstDmaCtrl->u64Val);
        /*FIX ME: FILL high or low bit?*/
        dma_params.reg_1b.bits.dma_endmode = 0;
    }

    if (dma_params.reg_1b.bits.dma_mode_ctrl == CVE_DMA_MODE_INTERVAL_COPY) {
        dma_params.reg_1b.bits.dma_interval_vsegsize = pstDmaCtrl->u8VerSegRows;
        dma_params.reg_1b.bits.dma_interval_hsegsize = pstDmaCtrl->u8HorSegSize;
        dma_params.reg_1b.bits.dma_interval_elemsize = pstDmaCtrl->u8ElemSize;

        dma_params.comm_params.reg_18.bits.dst_image_width =
            dma_params.comm_params.reg_1a.bits.cve_crop_xsize / pstDmaCtrl->u8HorSegSize *
            pstDmaCtrl->u8ElemSize;

        dma_params.comm_params.reg_14.bits.dst_stride_0 =
            CVE_ALIGN_UP(dma_params.comm_params.reg_18.bits.dst_image_width, CVE_ALIGN) >> 4;

        dma_params.comm_params.reg_18.bits.dst_image_height =
            CVE_ALIGN_UP(dma_params.comm_params.reg_1a.bits.cve_crop_ysize,
                         pstDmaCtrl->u8VerSegRows) /
            pstDmaCtrl->u8VerSegRows;

        dma_params.reg_1e.bits.dma_interval_xlength =
            dma_params.comm_params.reg_1a.bits.cve_crop_xsize / pstDmaCtrl->u8HorSegSize *
            pstDmaCtrl->u8HorSegSize;
    }

    *cmd_line_num = dma_task_cmd_queue(&dma_params, (unsigned int *)cmd_buf);

    return AML_SUCCESS;
}

int cve_fill_luma_stat_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_MEM_INFO_T *pstDstMem,
                            CVE_RECT_U16_T *pstCveLumaRect,
                            CVE_LUMA_STAT_ARRAY_CTRL_T *pstLumaStatArrayCtrl,
                            unsigned char idx_windows, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_dma_params_t luma_stat_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&luma_stat_params, 0, sizeof(cve_op_dma_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_mem(&init_params, pstDstMem,
                       CVE_ALIGN_UP(pstLumaStatArrayCtrl->u8MaxLumaRect * 4, CVE_ALIGN), &dst_off);
    if (ret != 0) {
        return ret;
    }
    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstLumaStatArrayCtrl->u8MaxLumaRect * 4;
    init_params.dst_height = 1;
    if (pstLumaStatArrayCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstLumaStatArrayCtrl->stCrop.u16X;
        init_params.ystart = pstLumaStatArrayCtrl->stCrop.u16Y;
        init_params.xSize = pstLumaStatArrayCtrl->stCrop.u16Width;
        init_params.ySize = pstLumaStatArrayCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = pstCveLumaRect->u16X;
        init_params.ystart = pstCveLumaRect->u16Y;
        init_params.xSize = pstCveLumaRect->u16Width;
        init_params.ySize = pstCveLumaRect->u16Height;
    }
    cve_common_params_init(&luma_stat_params.comm_params, &init_params);

    /* luma stat op params initialization */

    luma_stat_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    luma_stat_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    luma_stat_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_DMA;
    luma_stat_params.reg_1b.bits.dma_mode_ctrl = 0x5;
    luma_stat_params.reg_1e.bits.cve_caption_stat_step = pstLumaStatArrayCtrl->enMode;
    luma_stat_params.reg_1e.bits.cve_caption_stat_wincnt = pstLumaStatArrayCtrl->u8MaxLumaRect;
    luma_stat_params.reg_1e.bits.cve_caption_stat_winidx = idx_windows;

    *cmd_line_num = dma_task_cmd_queue(&luma_stat_params, (unsigned int *)cmd_buf);

    return AML_SUCCESS;
}

int cve_fill_filter_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                         CVE_FILTER_CTRL_T *pstFilterCtrl, char *cmd_buf,
                         unsigned int *cmd_line_num)
{
    cve_op_filter_params_t filter_params;
    cve_comm_init_params_t init_params;
    char uv_mask[9] = {-1, 2, 1, 2, -4, 2, 1, 2, 1};
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&filter_params, 0, sizeof(filter_params));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }
    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstFilterCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstFilterCtrl->stCrop.u16X;
        init_params.ystart = pstFilterCtrl->stCrop.u16Y;
        init_params.xSize = pstFilterCtrl->stCrop.u16Width;
        init_params.ySize = pstFilterCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&filter_params.comm_params, &init_params);

    /* filter op params initialization */
    filter_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_FILTER;
    filter_params.comm_params.reg_02.bits.src_image_type = pstSrcImage->enType;
    filter_params.comm_params.reg_02.bits.dst_image_type = pstSrcImage->enType;
    if (pstSrcImage->enType == CVE_IMAGE_TYPE_U8C1) {
        filter_params.filter_1f.bits.filter_uv_en = 0;
        filter_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 2;
        filter_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 2;
    } else {
        filter_params.filter_1f.bits.filter_uv_en = 1;
    }

    /*FIX ME:filter_norm_uv?*/
    filter_params.filter_1f.bits.filter_norm_uv = 4;

    memcpy(&filter_params.filter_67.reg, pstFilterCtrl->as8Mask, 24);
    filter_params.filter_6d.bits.filter_coef24 = pstFilterCtrl->as8Mask[24];
    filter_params.filter_6d.bits.filter_norm = pstFilterCtrl->u8Norm;

    memcpy(&filter_params.filter_20.reg, uv_mask, 8);
    filter_params.filter_22.bits.filter_coef_uv8 = uv_mask[8];

    *cmd_line_num = filter_task_cmd_queue(&filter_params, (unsigned int *)cmd_buf);

    return AML_SUCCESS;
}

int cve_fill_csc_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                      CVE_CSC_CTRL_T *pstCscCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_csc_params_t csc_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;
    /* YUV2RGB conversion matrix  */
    int csc_offset_inp0[3] = {0, -128, -128};
    int csc_matrix0[3][3] = {{256, 0, 359}, {256, -88, -183}, {256, 454, 0}};
    int csc_offset_oup0[3] = {0, 0, 0};

    /* RGB2YUV conversion matrix */
    int csc_offset_inp1[3] = {0, 0, 0};
    int csc_matrix1[3][3] = {{76, 150, 29}, {-43, -84, 128}, {128, -107, -21}};
    int csc_offset_oup1[3] = {0, 128, 128};

    /* RGB2BGR or BGR2RGB RGB2HSV RGB2LAB conversion matrix: */
    int csc_offset_inp2[3] = {0, 0, 0};
    int csc_matrix2[3][3] = {{256, 0, 0}, {0, 256, 0}, {0, 0, 256}};
    int csc_offset_oup2[3] = {0, 0, 0};

    int *inp = NULL;
    int(*matric)[3] = NULL;
    int *oup = NULL;

    memset(&csc_params, 0, sizeof(cve_op_csc_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }
    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstCscCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstCscCtrl->stCrop.u16X;
        init_params.ystart = pstCscCtrl->stCrop.u16Y;
        init_params.xSize = pstCscCtrl->stCrop.u16Width;
        init_params.ySize = pstCscCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&csc_params.comm_params, &init_params);

    /* csc op params initialization */
    if (pstSrcImage->enType == CVE_IMAGE_TYPE_YVU420SP) {
        pstSrcImage->enType = CVE_IMAGE_TYPE_YUV420SP;
    }
    if (pstSrcImage->enType == CVE_IMAGE_TYPE_YVU422SP) {
        pstSrcImage->enType = CVE_IMAGE_TYPE_YUV422SP;
    }
    csc_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_CSC;
    csc_params.comm_params.reg_02.bits.src_image_type = pstSrcImage->enType;
    csc_params.comm_params.reg_02.bits.dst_image_type = pstDstImage->enType;
    /*FIX ME: yuv_image_type nv12 or nv21*/
    if ((pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2RGB) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2BGR) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2HSV) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2LAB) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_RGB2YUV) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_BGR2YUV)) {
        csc_params.comm_params.reg_02.bits.yuv_image_type = 1;
    } else if ((pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YVU2RGB) ||
               (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YVU2HSV) ||
               (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YVU2LAB) ||
               (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YVU2BGR) ||
               (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_RGB2YVU) ||
               (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_BGR2YVU)) {
        csc_params.comm_params.reg_02.bits.yuv_image_type = 0;
    }

    if ((pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2RGB) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2BGR) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2HSV) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2LAB) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YVU2RGB) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YVU2HSV) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YVU2LAB) ||
        (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YVU2BGR)) {
        inp = csc_offset_inp0;
        matric = csc_matrix0;
        oup = csc_offset_oup0;
    } else if ((pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_RGB2YUV) ||
               (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_BGR2YUV) ||
               (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_RGB2YVU) ||
               (pstCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_BGR2YVU)) {
        inp = csc_offset_inp1;
        matric = csc_matrix1;
        oup = csc_offset_oup1;
    } else {
        inp = csc_offset_inp2;
        matric = csc_matrix2;
        oup = csc_offset_oup2;
    }

    if ((pstSrcImage->enType == CVE_IMAGE_TYPE_YUV420SP) ||
        (pstSrcImage->enType == CVE_IMAGE_TYPE_YUV422SP)) {
        csc_params.comm_params.reg_d8.bits.rdmif_pack_mode = 1;
        csc_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 1;
    } else if (pstSrcImage->enType == CVE_IMAGE_TYPE_U8C3_PACKAGE) {
        csc_params.comm_params.reg_d8.bits.rdmif_pack_mode = 2;
    } else if (pstSrcImage->enType == CVE_IMAGE_TYPE_U8C3_PLANAR) {
        csc_params.comm_params.reg_d8.bits.rdmif_pack_mode = 1;
        csc_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 1;
        csc_params.comm_params.reg_e8.bits.rdmif2_pack_mode = 1;
    }

    if ((pstDstImage->enType == CVE_IMAGE_TYPE_YUV420SP) ||
        (pstDstImage->enType == CVE_IMAGE_TYPE_YUV422SP)) {
        csc_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
        csc_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
    } else if (pstDstImage->enType == CVE_IMAGE_TYPE_U8C3_PACKAGE) {
        csc_params.comm_params.reg_c0.bits.wdmif_pack_mode = 2;
    } else if (pstDstImage->enType == CVE_IMAGE_TYPE_U8C3_PLANAR) {
        csc_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
        csc_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
        csc_params.comm_params.reg_d0.bits.wdmif2_pack_mode = 0;
    }

    csc_params.csc_24.bits.src_u6order = 0x24;
    csc_params.csc_24.bits.dst_u6order = 0x24;
    switch (pstCscCtrl->enMode) {
    case CVE_CSC_MODE_PIC_BT601_YUV2RGB: {
        csc_params.csc_23.bits.csc_mode = 0;
        csc_params.csc_24.bits.dst_u6order = 0x24;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_YUV2BGR: {
        csc_params.csc_23.bits.csc_mode = 0;
        csc_params.csc_24.bits.dst_u6order = 0x6;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_YUV2HSV: {
        csc_params.csc_23.bits.csc_mode = 1;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_YUV2LAB: {
        csc_params.csc_23.bits.csc_mode = 2;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_RGB2YUV: {
        csc_params.csc_23.bits.csc_mode = 3;
        csc_params.csc_24.bits.src_u6order = 0x24;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_BGR2YUV: {
        csc_params.csc_23.bits.csc_mode = 3;
        csc_params.csc_24.bits.src_u6order = 0x6;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_RGB2HSV: {
        csc_params.csc_23.bits.csc_mode = 4;
        csc_params.csc_24.bits.src_u6order = 0x24;
        csc_params.csc_24.bits.dst_u6order = 0x24;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_RGB2LAB: {
        csc_params.csc_23.bits.csc_mode = 5;
        csc_params.csc_24.bits.src_u6order = 0x24;
        csc_params.csc_24.bits.dst_u6order = 0x24;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_RGB2BGR: {
        csc_params.csc_23.bits.csc_mode = 0;
        csc_params.csc_24.bits.src_u6order = 0x24;
        csc_params.csc_24.bits.dst_u6order = 0x6;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_BGR2RGB: {
        csc_params.csc_23.bits.csc_mode = 0;
        csc_params.csc_24.bits.src_u6order = 0x6;
        csc_params.csc_24.bits.dst_u6order = 0x24;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_YVU2RGB: {
        csc_params.csc_23.bits.csc_mode = 0;
        csc_params.csc_24.bits.dst_u6order = 0x24;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_YVU2HSV: {
        csc_params.csc_23.bits.csc_mode = 1;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_YVU2LAB: {
        csc_params.csc_23.bits.csc_mode = 2;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_YVU2BGR: {
        csc_params.csc_23.bits.csc_mode = 0;
        csc_params.csc_24.bits.dst_u6order = 0x6;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_RGB2YVU: {
        csc_params.csc_23.bits.csc_mode = 3;
        csc_params.csc_24.bits.src_u6order = 0x24;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_BGR2YVU: {
        csc_params.csc_23.bits.csc_mode = 3;
        csc_params.csc_24.bits.src_u6order = 0x6;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_BGR2HSV: {
        csc_params.csc_23.bits.csc_mode = 4;
        csc_params.csc_24.bits.src_u6order = 0x6;
        csc_params.csc_24.bits.dst_u6order = 0x24;
        break;
    }
    case CVE_CSC_MODE_PIC_BT601_BGR2LAB: {
        csc_params.csc_23.bits.csc_mode = 5;
        csc_params.csc_24.bits.src_u6order = 0x6;
        csc_params.csc_24.bits.dst_u6order = 0x24;
        break;
    }

    default:
        CVE_ERR_TRACE("Invalid CSC mode\n");
        break;
    }

    csc_params.csc_23.bits.csc_gamma = 1;
    csc_params.csc_23.bits.csc_yuv422toyuv420_mode = 0;
    csc_params.csc_23.bits.csc_yuv444toyuv422_mode = 0;
    csc_params.csc_23.bits.csc_yuv420toyuv422_mode = 0;
    csc_params.csc_23.bits.csc_yuv422toyuv444_mode = 0;

    csc_params.csc_6e.bits.csc_offset_inp_0 = inp[0];
    csc_params.csc_6e.bits.csc_offset_inp_1 = inp[1];
    csc_params.csc_6f.bits.csc_offset_inp_2 = inp[2];

    csc_params.csc_70.bits.csc_3x3matrix_0_0 = matric[0][0];
    csc_params.csc_70.bits.csc_3x3matrix_1_0 = matric[1][0];
    csc_params.csc_70.bits.csc_3x3matrix_2_0 = matric[2][0];
    csc_params.csc_71.bits.csc_3x3matrix_0_1 = matric[0][1];
    csc_params.csc_71.bits.csc_3x3matrix_1_1 = matric[1][1];
    csc_params.csc_71.bits.csc_3x3matrix_2_1 = matric[2][1];
    csc_params.csc_72.bits.csc_3x3matrix_0_2 = matric[0][2];
    csc_params.csc_72.bits.csc_3x3matrix_1_2 = matric[1][2];
    csc_params.csc_72.bits.csc_3x3matrix_2_2 = matric[2][2];

    csc_params.csc_73.bits.csc_offset_oup_0 = oup[0];
    csc_params.csc_73.bits.csc_offset_oup_1 = oup[1];
    csc_params.csc_74.bits.csc_offset_oup_2 = oup[2];

    *cmd_line_num = csc_task_cmd_queue(&csc_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_filter_and_csc_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                 CVE_FILTER_AND_CSC_CTRL_T *pstFilterCscCtrl, char *cmd_buf,
                                 unsigned int *cmd_line_num)
{
    cve_op_filter_and_csc_params_t filter_and_csc_params;
    cve_comm_init_params_t init_params;
    // Color space conversion_2 yuv2rgb input 8_bit
    int offser_inp[3] = {0, -128, -128};
    int matric[3][3] = {{256, 0, 358}, {256, -87, -182}, {256, 453, 0}};
    int offser_oup[3] = {0, 0, 0};
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&filter_and_csc_params, 0, sizeof(cve_op_filter_and_csc_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }
    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstFilterCscCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstFilterCscCtrl->stCrop.u16X;
        init_params.ystart = pstFilterCscCtrl->stCrop.u16Y;
        init_params.xSize = pstFilterCscCtrl->stCrop.u16Width;
        init_params.ySize = pstFilterCscCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&filter_and_csc_params.comm_params, &init_params);

    /* csc op params initialization */
    filter_and_csc_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    filter_and_csc_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 1;
    filter_and_csc_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_FILTER_AND_CSC;
    filter_and_csc_params.comm_params.reg_02.bits.src_image_type = pstSrcImage->enType;
    filter_and_csc_params.comm_params.reg_02.bits.dst_image_type = pstDstImage->enType;
    /*FIX ME: yuv_image_type nv12 or nv21*/
    filter_and_csc_params.comm_params.reg_02.bits.yuv_image_type = 1;

    if (pstDstImage->enType == CVE_IMAGE_TYPE_U8C3_PACKAGE) {
        filter_and_csc_params.comm_params.reg_c0.bits.wdmif_pack_mode = 2;
    }

    if (pstDstImage->enType == CVE_IMAGE_TYPE_U8C3_PLANAR) {
        filter_and_csc_params.comm_params.reg_d0.bits.wdmif2_pack_mode = 0;
        filter_and_csc_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
        filter_and_csc_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    }

    filter_and_csc_params.csc_23.bits.csc_gamma = 1;
    filter_and_csc_params.csc_23.bits.csc_yuv422toyuv420_mode = 0;
    filter_and_csc_params.csc_23.bits.csc_yuv444toyuv422_mode = 0;
    filter_and_csc_params.csc_23.bits.csc_yuv420toyuv422_mode = 0;
    filter_and_csc_params.csc_23.bits.csc_yuv422toyuv444_mode = 0;

    if (pstFilterCscCtrl->enMode == CVE_CSC_MODE_PIC_BT601_YUV2RGB) {
        filter_and_csc_params.csc_24.bits.dst_u6order = 0x24;
    } else {
        filter_and_csc_params.csc_24.bits.dst_u6order = 0x6;
    }
    filter_and_csc_params.csc_24.bits.src_u6order = 0x24;

    filter_and_csc_params.csc_6e.bits.csc_offset_inp_0 = offser_inp[0];
    filter_and_csc_params.csc_6e.bits.csc_offset_inp_1 = offser_inp[1];
    filter_and_csc_params.csc_6f.bits.csc_offset_inp_2 = offser_inp[2];

    filter_and_csc_params.csc_70.bits.csc_3x3matrix_0_0 = matric[0][0];
    filter_and_csc_params.csc_70.bits.csc_3x3matrix_1_0 = matric[1][0];
    filter_and_csc_params.csc_70.bits.csc_3x3matrix_2_0 = matric[2][0];
    filter_and_csc_params.csc_71.bits.csc_3x3matrix_0_1 = matric[0][1];
    filter_and_csc_params.csc_71.bits.csc_3x3matrix_1_1 = matric[1][1];
    filter_and_csc_params.csc_71.bits.csc_3x3matrix_2_1 = matric[2][1];
    filter_and_csc_params.csc_72.bits.csc_3x3matrix_0_2 = matric[0][2];
    filter_and_csc_params.csc_72.bits.csc_3x3matrix_1_2 = matric[1][2];
    filter_and_csc_params.csc_72.bits.csc_3x3matrix_2_2 = matric[2][2];

    filter_and_csc_params.csc_73.bits.csc_offset_oup_0 = offser_oup[0];
    filter_and_csc_params.csc_73.bits.csc_offset_oup_1 = offser_oup[1];
    filter_and_csc_params.csc_74.bits.csc_offset_oup_2 = offser_oup[2];

    memcpy(&filter_and_csc_params.filter_67.reg, pstFilterCscCtrl->as8Mask, 24);
    filter_and_csc_params.filter_6d.bits.filter_coef24 = pstFilterCscCtrl->as8Mask[24];
    filter_and_csc_params.filter_6d.bits.filter_norm = pstFilterCscCtrl->u8Norm;

    *cmd_line_num = filter_and_csc_task_cmd_queue(&filter_and_csc_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_sobel_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstH,
                        CVE_DST_IMAGE_T *pstDstV, CVE_SOBEL_CTRL_T *pstSobelCtrl, char *cmd_buf,
                        unsigned int *cmd_line_num)
{
    cve_op_sobel_params_t sobel_params;
    cve_comm_init_params_t init_params;
    char mask_v[25];
    int i = 0;
    int j = 0;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&sobel_params, 0, sizeof(cve_op_sobel_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstDstH, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstDstV, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstSrcImage->u32Width;
    init_params.dst_height = pstSrcImage->u32Height;
    if (pstSobelCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstSobelCtrl->stCrop.u16X;
        init_params.ystart = pstSobelCtrl->stCrop.u16Y;
        init_params.xSize = pstSobelCtrl->stCrop.u16Width;
        init_params.ySize = pstSobelCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&sobel_params.comm_params, &init_params);

    /* sobel op params initialization */
    sobel_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    sobel_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    sobel_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_SOBEL;
    sobel_params.sobel_25.bits.sobel_output_mode = pstSobelCtrl->enOutCtrl;

    memcpy(&sobel_params.sobel_75.reg, pstSobelCtrl->as8Mask, 24);
    sobel_params.sobel_7b.bits.sobel_coef_h_24 = pstSobelCtrl->as8Mask[24];

    for (j = 0; j < 5; j++) {
        for (i = 0; i < 5; i++) {
            mask_v[j * 5 + i] = pstSobelCtrl->as8Mask[i * 5 + j];
        }
    }

    memcpy(&sobel_params.sobel_7c.reg, mask_v, 24);
    sobel_params.sobel_82.bits.sobel_coef_v_24 = mask_v[24];

    *cmd_line_num = sobel_task_cmd_queue(&sobel_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_mag_and_ang_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstMag,
                              CVE_DST_IMAGE_T *pstDstAng, CVE_MAG_AND_ANG_CTRL_T *pstMagAndAngCtrl,
                              char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_mag_and_ang_params_t mag_and_ang_params;
    cve_comm_init_params_t init_params;
    char mask_v[25];
    int i = 0;
    int j = 0;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&mag_and_ang_params, 0, sizeof(cve_op_mag_and_ang_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstMag, &dst_off);
    if (ret != 0) {
        return ret;
    }
    if ((pstMagAndAngCtrl->enOutCtrl == CVE_MAG_AND_ANG_OUT_CTRL_MAG_AND_ANG) ||
        (pstMagAndAngCtrl->enOutCtrl == CVE_MAG_AND_ANG_OUT_CTRL_HOG)) {
        ret = fill_dst_image(&init_params, pstDstAng, &dst_off);
        if (ret != 0) {
            return ret;
        }
    }
    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstSrcImage->u32Width;
    init_params.dst_height = pstSrcImage->u32Height;
    if (pstMagAndAngCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstMagAndAngCtrl->stCrop.u16X;
        init_params.ystart = pstMagAndAngCtrl->stCrop.u16Y;
        init_params.xSize = pstMagAndAngCtrl->stCrop.u16Width;
        init_params.ySize = pstMagAndAngCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&mag_and_ang_params.comm_params, &init_params);
    if (pstMagAndAngCtrl->enOutCtrl == CVE_MAG_AND_ANG_OUT_CTRL_MAG_AND_ANG) {
        mag_and_ang_params.comm_params.reg_14.bits.dst_stride_1 =
            mag_and_ang_params.comm_params.reg_14.bits.dst_stride_1 >> 1;
    }

    /*mag_and_ang op params initialization */
    mag_and_ang_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    mag_and_ang_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    mag_and_ang_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
    mag_and_ang_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_MAG_AND_ANG;
    mag_and_ang_params.mag_and_ang_26.bits.magandang_output_ctrl = pstMagAndAngCtrl->enOutCtrl;
    mag_and_ang_params.mag_and_ang_26.bits.magandang_u16thr = pstMagAndAngCtrl->u16Thr;
    mag_and_ang_params.mag_and_ang_26.bits.magandang_mode = 1;

    if (pstMagAndAngCtrl->enOutCtrl == CVE_MAG_AND_ANG_OUT_CTRL_MAG_AND_ANG) {
        mag_and_ang_params.comm_params.reg_14.bits.dst_stride_1 =
            ((init_params.dst_stride[1] >> 4) + 1) >> 1;
    }
    memcpy(&mag_and_ang_params.sobel_75.reg, pstMagAndAngCtrl->as8Mask, 24);
    mag_and_ang_params.sobel_7b.bits.sobel_coef_h_24 = pstMagAndAngCtrl->as8Mask[24];

    for (j = 0; j < 5; j++) {
        for (i = 0; i < 5; i++) {
            mask_v[j * 5 + i] = pstMagAndAngCtrl->as8Mask[i * 5 + j];
        }
    }

    memcpy(&mag_and_ang_params.sobel_7c.reg, mask_v, 24);
    mag_and_ang_params.sobel_82.bits.sobel_coef_v_24 = mask_v[24];

    *cmd_line_num = mag_and_ang_task_cmd_queue(&mag_and_ang_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_match_bg_model_task(CVE_SRC_IMAGE_T *pstCurImg, CVE_SRC_IMAGE_T *pstPreImg,
                                 CVE_MEM_INFO_T *pstBgModel, CVE_DST_IMAGE_T *pstFg,
                                 CVE_DST_IMAGE_T *pstBg, CVE_DST_IMAGE_T *pstCurDiffBg,
                                 CVE_DST_IMAGE_T *pstFrmDiff,
                                 CVE_MATCH_BG_MODEL_CTRL_T *pstMatchBgModelCtrl, char *cmd_buf,
                                 unsigned int *cmd_line_num)
{
    cve_op_match_bg_model_params_t match_bg_model_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    unsigned int dst4_stride = 0;
    int ret = 0;

    memset(&match_bg_model_params, 0, sizeof(cve_op_match_bg_model_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstCurImg, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_src_image(&init_params, pstPreImg, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstFg, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstBg, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstCurDiffBg, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstFrmDiff, &dst_off);
    if (ret != 0) {
        return ret;
    }
    dst4_stride = pstCurImg->au32Stride[0] >> pstMatchBgModelCtrl->enDownScaleMode;
    ret = fill_dst_mem(&init_params, pstBgModel, dst4_stride * 16, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstCurImg->u32Width;
    init_params.src_height = pstCurImg->u32Height;
    init_params.dst_width = pstFg->u32Width;
    init_params.dst_height = pstFg->u32Height;
    if (pstMatchBgModelCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstMatchBgModelCtrl->stCrop.u16X;
        init_params.ystart = pstMatchBgModelCtrl->stCrop.u16Y;
        init_params.xSize = pstMatchBgModelCtrl->stCrop.u16Width;
        init_params.ySize = pstMatchBgModelCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&match_bg_model_params.comm_params, &init_params);

    /*match_bg_model op params initialization */
    match_bg_model_params.comm_params.reg_02.bits.output_mode = 0;

    match_bg_model_params.comm_params.reg_e8.bits.rdmif2_pack_mode = 0;
    match_bg_model_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    match_bg_model_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_MATCH_BG_MODEL;

    if (pstMatchBgModelCtrl->enDownScaleMode == CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_2X2) {
        match_bg_model_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 1;
        match_bg_model_params.comm_params.reg_d8.bits.rdmif_pack_mode = 1;
    } else if (pstMatchBgModelCtrl->enDownScaleMode == CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_4X4) {
        match_bg_model_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 2;
        match_bg_model_params.comm_params.reg_d8.bits.rdmif_pack_mode = 2;
    } else {
        match_bg_model_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;
        match_bg_model_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    }

    match_bg_model_params.bg_mode_27.bits.bgmodel_ds_mode = pstMatchBgModelCtrl->enDownScaleMode;
    match_bg_model_params.bg_mode_27.bits.bgmodel_output_ctrl = pstMatchBgModelCtrl->enOutputMode;
    match_bg_model_params.bg_mode_27.bits.bgmodel_u8gray_thr = pstMatchBgModelCtrl->u8GrayThr;
    match_bg_model_params.bg_mode_27.bits.bgmodel_u8q4dist_thr = pstMatchBgModelCtrl->u8q4DistThr;

    *cmd_line_num = match_bg_model_task_cmd_queue(&match_bg_model_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_dilate_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                         CVE_DILATE_CTRL_T *pstDilateCtrl, char *cmd_buf,
                         unsigned int *cmd_line_num)
{
    cve_op_erode_dilate_params_t erode_dilate_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&erode_dilate_params, 0, sizeof(cve_op_erode_dilate_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstDilateCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstDilateCtrl->stCrop.u16X;
        init_params.ystart = pstDilateCtrl->stCrop.u16Y;
        init_params.xSize = pstDilateCtrl->stCrop.u16Width;
        init_params.ySize = pstDilateCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&erode_dilate_params.comm_params, &init_params);
    /*dilate op params initialization */

    erode_dilate_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    erode_dilate_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    erode_dilate_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_DILATE_AND_ERODE;
    erode_dilate_params.comm_params.reg_02.bits.output_mode = 0;
    erode_dilate_params.erode_dilate_2E.bits.erodedilate_sel = 1;

    memcpy(&erode_dilate_params.filter_67.reg, pstDilateCtrl->au8Mask, 24);
    erode_dilate_params.filter_6d.bits.filter_coef24 = pstDilateCtrl->au8Mask[24];

    *cmd_line_num = erode_dilate_task_cmd_queue(&erode_dilate_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_erode_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                        CVE_ERODE_CTRL_T *pstErodeCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_erode_dilate_params_t erode_dilate_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&erode_dilate_params, 0, sizeof(cve_op_erode_dilate_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstErodeCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstErodeCtrl->stCrop.u16X;
        init_params.ystart = pstErodeCtrl->stCrop.u16Y;
        init_params.xSize = pstErodeCtrl->stCrop.u16Width;
        init_params.ySize = pstErodeCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&erode_dilate_params.comm_params, &init_params);
    /*erode op params initialization */

    erode_dilate_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    erode_dilate_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    erode_dilate_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_DILATE_AND_ERODE;
    erode_dilate_params.comm_params.reg_02.bits.output_mode = 0;
    erode_dilate_params.erode_dilate_2E.bits.erodedilate_sel = 0;

    memcpy(&erode_dilate_params.filter_67.reg, pstErodeCtrl->au8Mask, 24);
    erode_dilate_params.filter_6d.bits.filter_coef24 = pstErodeCtrl->au8Mask[24];

    *cmd_line_num = erode_dilate_task_cmd_queue(&erode_dilate_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_thresh_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                         CVE_THRESH_CTRL_T *pstThreshCtrl, char *cmd_buf,
                         unsigned int *cmd_line_num)
{
    cve_op_thresh_params_t thresh_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&thresh_params, 0, sizeof(cve_op_thresh_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstThreshCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstThreshCtrl->stCrop.u16X;
        init_params.ystart = pstThreshCtrl->stCrop.u16Y;
        init_params.xSize = pstThreshCtrl->stCrop.u16Width;
        init_params.ySize = pstThreshCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&thresh_params.comm_params, &init_params);

    /*thresh op params initialization */
    thresh_params.comm_params.reg_d8.bits.rdmif_pack_mode = 2;
    thresh_params.comm_params.reg_c0.bits.wdmif_pack_mode = 2;
    thresh_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_THRESH;
    thresh_params.thresh_2f.bits.thresh_u8max_val = pstThreshCtrl->u8MaxVal;
    thresh_params.thresh_2f.bits.thresh_mode = pstThreshCtrl->enMode;
    thresh_params.thresh_30.bits.thresh_u8mid_val = pstThreshCtrl->u8MidVal;
    thresh_params.thresh_30.bits.thresh_u8min_val = pstThreshCtrl->u8MinVal;
    thresh_params.thresh_30.bits.thresh_u8high_thr = pstThreshCtrl->u8HighThr;
    thresh_params.thresh_30.bits.thresh_u8low_thr = pstThreshCtrl->u8LowThr;

    *cmd_line_num = thresh_task_cmd_queue(&thresh_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_alu1_task(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                       CVE_DST_IMAGE_T *pstDst, cve_alu_sel_e alu_sel, CVE_SUB_CTRL_T *pstSubCtrl,
                       CVE_ADD_CTRL_T *pstAddCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_alu_params_t alu_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&alu_params, 0, sizeof(cve_op_alu_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage1, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_src_image(&init_params, pstSrcImage2, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDst, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage1->u32Width;
    init_params.src_height = pstSrcImage1->u32Height;
    init_params.dst_width = pstDst->u32Width;
    init_params.dst_height = pstDst->u32Height;
    if (alu_sel == CVE_ALU_SEL_SUB && pstSubCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstSubCtrl->stCrop.u16X;
        init_params.ystart = pstSubCtrl->stCrop.u16Y;
        init_params.xSize = pstSubCtrl->stCrop.u16Width;
        init_params.ySize = pstSubCtrl->stCrop.u16Height;
    } else if (alu_sel == CVE_ALU_SEL_ADD && pstAddCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstAddCtrl->stCrop.u16X;
        init_params.ystart = pstAddCtrl->stCrop.u16Y;
        init_params.xSize = pstAddCtrl->stCrop.u16Width;
        init_params.ySize = pstAddCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&alu_params.comm_params, &init_params);

    /* alu op params initialization */
    alu_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0x2;
    alu_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0x2;
    alu_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0x2;
    alu_params.comm_params.reg_02.bits.src_image_type = pstSrcImage1->enType;
    alu_params.comm_params.reg_02.bits.dst_image_type = pstDst->enType;
    alu_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_ALU;
    alu_params.comm_params.reg_02.bits.intput_mode = 0;
    alu_params.comm_params.reg_02.bits.output_mode = 0;
    alu_params.alu_31.bits.cve_alu_sel = alu_sel;
    if (alu_sel == CVE_ALU_SEL_SUB) {
        alu_params.alu_31.bits.sub_output_mode = pstSubCtrl->enMode;
        if (pstSubCtrl->enMode == CVE_SUB_MODE_THRESH) {
            alu_params.sub_99.bits.sub_thresh_ratio = pstSubCtrl->u16ThreshRatio;
        }
    } else if (alu_sel == CVE_ALU_SEL_ADD) {
        alu_params.alu_32.bits.add_u0q16x = pstAddCtrl->u0q16X;
        alu_params.alu_32.bits.add_u0q16y = pstAddCtrl->u0q16Y;
    }

    *cmd_line_num = alu_task_cmd_queue(&alu_params, (unsigned int *)cmd_buf);

    return AML_SUCCESS;
}

int cve_fill_alu2_task(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                       CVE_DST_IMAGE_T *pstDst, cve_alu_sel_e alu_sel, CVE_AND_CTRL_T *pstAndCtrl,
                       CVE_OR_CTRL_T *pstOrCtrl, CVE_XOR_CTRL_T *pstXorCtrl, char *cmd_buf,
                       unsigned int *cmd_line_num)
{
    cve_op_alu_params_t alu_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&alu_params, 0, sizeof(cve_op_alu_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage1, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_src_image(&init_params, pstSrcImage2, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDst, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage1->u32Width;
    init_params.src_height = pstSrcImage1->u32Height;
    init_params.dst_width = pstDst->u32Width;
    init_params.dst_height = pstDst->u32Height;
    if (alu_sel == CVE_ALU_SEL_AND && pstAndCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstAndCtrl->stCrop.u16X;
        init_params.ystart = pstAndCtrl->stCrop.u16Y;
        init_params.xSize = pstAndCtrl->stCrop.u16Width;
        init_params.ySize = pstAndCtrl->stCrop.u16Height;
    } else if (alu_sel == CVE_ALU_SEL_OR && pstOrCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstOrCtrl->stCrop.u16X;
        init_params.ystart = pstOrCtrl->stCrop.u16Y;
        init_params.xSize = pstOrCtrl->stCrop.u16Width;
        init_params.ySize = pstOrCtrl->stCrop.u16Height;
    } else if (alu_sel == CVE_ALU_SEL_XOR && pstXorCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstXorCtrl->stCrop.u16X;
        init_params.ystart = pstXorCtrl->stCrop.u16Y;
        init_params.xSize = pstXorCtrl->stCrop.u16Width;
        init_params.ySize = pstXorCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&alu_params.comm_params, &init_params);

    /* alu op params initialization */
    alu_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0x2;
    alu_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0x2;
    alu_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0x2;
    alu_params.comm_params.reg_02.bits.src_image_type = pstSrcImage1->enType;
    alu_params.comm_params.reg_02.bits.dst_image_type = pstDst->enType;
    alu_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_ALU;
    alu_params.comm_params.reg_02.bits.intput_mode = 0;
    alu_params.comm_params.reg_02.bits.output_mode = 0;
    alu_params.alu_31.bits.cve_alu_sel = alu_sel;

    *cmd_line_num = alu_task_cmd_queue(&alu_params, (unsigned int *)cmd_buf);

    return AML_SUCCESS;
}

int cve_fill_integ_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                        CVE_INTEG_CTRL_T *pstIntegCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_integ_params_t integ_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&integ_params, 0, sizeof(cve_op_integ_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }
    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    init_params.xstart = 0;
    init_params.ystart = 0;
    init_params.xSize = init_params.src_width;
    init_params.ySize = init_params.src_height;
    if ((pstIntegCtrl->enOutCtrl == CVE_INTEG_OUT_CTRL_COMBINE) ||
        (pstIntegCtrl->enOutCtrl == CVE_INTEG_OUT_CTRL_TQSUM)) {
        init_params.dst_stride[1] = init_params.dst_stride[0];
    }
    cve_common_params_init(&integ_params.comm_params, &init_params);

    /*thresh op params initialization */
    integ_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    integ_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    integ_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
    integ_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_INTEG;
    integ_params.integ_33.bits.integ_out_ctrl = pstIntegCtrl->enOutCtrl;

    *cmd_line_num = integ_task_cmd_queue(&integ_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_hist_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_MEM_INFO_T *pstDstMem,
                       CVE_HIST_CTRL_T *pstHistCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_hist_params_t hist_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    int ret = 0;

    memset(&hist_params, 0, sizeof(cve_op_hist_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    if (pstHistCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstHistCtrl->stCrop.u16X;
        init_params.ystart = pstHistCtrl->stCrop.u16Y;
        init_params.xSize = pstHistCtrl->stCrop.u16Width;
        init_params.ySize = pstHistCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&hist_params.comm_params, &init_params);
    /*hist op params initialization */
    hist_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    hist_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_HIST;
    hist_params.integ_33.bits.eqhist_curv_mode = 1;
    /*FIX ME: (1<<32) / height /width*/
    hist_params.eqhist_34.bits.eqhist_norm = (0xffffffff / init_params.xSize) / init_params.ySize;

    *cmd_line_num = hist_task_cmd_queue(&hist_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_thresh_s16_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                             CVE_THRESH_S16_CTRL_T *pstThreshS16Ctrl, char *cmd_buf,
                             unsigned int *cmd_line_num)
{
    cve_op_thresh_s16_params_t thresh_s16_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&thresh_s16_params, 0, sizeof(cve_op_thresh_s16_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstThreshS16Ctrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstThreshS16Ctrl->stCrop.u16X;
        init_params.ystart = pstThreshS16Ctrl->stCrop.u16Y;
        init_params.xSize = pstThreshS16Ctrl->stCrop.u16Width;
        init_params.ySize = pstThreshS16Ctrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&thresh_s16_params.comm_params, &init_params);

    /*thresh_s16 op params initialization */
    thresh_s16_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    thresh_s16_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    thresh_s16_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_THRESH_S16;
    thresh_s16_params.comm_params.reg_02.bits.dst_image_type = pstDstImage->enType;
    thresh_s16_params.comm_params.reg_02.bits.src_image_type = pstSrcImage->enType;
    thresh_s16_params.thresh_s16_35.bits.thresh_max_val = pstThreshS16Ctrl->un8MaxVal.u8Val;
    thresh_s16_params.thresh_s16_35.bits.thresh_s16tos8oru8_mode = pstThreshS16Ctrl->enMode;
    thresh_s16_params.thresh_s16_35.bits.thresh_mid_val = pstThreshS16Ctrl->un8MidVal.u8Val;
    thresh_s16_params.thresh_s16_35.bits.thresh_min_val = pstThreshS16Ctrl->un8MinVal.u8Val;
    thresh_s16_params.thresh_s16_36.bits.thresh_s16highthr = pstThreshS16Ctrl->s16HighThr;
    thresh_s16_params.thresh_s16_36.bits.thresh_s16lowthr = pstThreshS16Ctrl->s16LowThr;

    *cmd_line_num = thresh_s16_task_cmd_queue(&thresh_s16_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_thresh_u16_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                             CVE_THRESH_U16_CTRL_T *pstThreshU16Ctrl, char *cmd_buf,
                             unsigned int *cmd_line_num)
{
    cve_op_thresh_u16_params_t thresh_u16_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&thresh_u16_params, 0, sizeof(cve_op_thresh_u16_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstThreshU16Ctrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstThreshU16Ctrl->stCrop.u16X;
        init_params.ystart = pstThreshU16Ctrl->stCrop.u16Y;
        init_params.xSize = pstThreshU16Ctrl->stCrop.u16Width;
        init_params.ySize = pstThreshU16Ctrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&thresh_u16_params.comm_params, &init_params);

    /*thresh_u16 op params initialization */
    thresh_u16_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    thresh_u16_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    thresh_u16_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_THRESH_U16;
    thresh_u16_params.comm_params.reg_02.bits.dst_image_type = pstDstImage->enType;
    thresh_u16_params.comm_params.reg_02.bits.src_image_type = pstSrcImage->enType;
    thresh_u16_params.thresh_u16_37.bits.thresh_max_val = pstThreshU16Ctrl->u8MaxVal;
    thresh_u16_params.thresh_u16_37.bits.thresh_u16tou8_mode = pstThreshU16Ctrl->enMode;
    thresh_u16_params.thresh_u16_37.bits.thresh_mid_val = pstThreshU16Ctrl->u8MidVal;
    thresh_u16_params.thresh_u16_37.bits.thresh_min_val = pstThreshU16Ctrl->u8MinVal;
    thresh_u16_params.thresh_u16_38.bits.thresh_u16_u16highthr = pstThreshU16Ctrl->u16HighThr;
    thresh_u16_params.thresh_u16_38.bits.thresh_u16_u16lowthr = pstThreshU16Ctrl->u16LowThr;

    *cmd_line_num = thresh_u16_task_cmd_queue(&thresh_u16_params, (unsigned int *)cmd_buf);

    return 0;
}

unsigned int cve_fill_16bit_to_8bit_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                         CVE_16BIT_TO_8BIT_CTRL_T *pst16BitTo8BitCtrl,
                                         char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_16bit_to_8bit_params_t _16bit_to_8bit_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&_16bit_to_8bit_params, 0, sizeof(cve_op_16bit_to_8bit_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pst16BitTo8BitCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pst16BitTo8BitCtrl->stCrop.u16X;
        init_params.ystart = pst16BitTo8BitCtrl->stCrop.u16Y;
        init_params.xSize = pst16BitTo8BitCtrl->stCrop.u16Width;
        init_params.ySize = pst16BitTo8BitCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&_16bit_to_8bit_params.comm_params, &init_params);

    /*_16bit_to_8bit op params initialization */
    _16bit_to_8bit_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    _16bit_to_8bit_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    _16bit_to_8bit_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_16BIT_TO_8BIT;
    _16bit_to_8bit_params.comm_params.reg_02.bits.dst_image_type = pstDstImage->enType;
    _16bit_to_8bit_params.comm_params.reg_02.bits.src_image_type = pstSrcImage->enType;
    _16bit_to_8bit_params._16bit_to_8bit_39.bits._16bitto8bit_mode = pst16BitTo8BitCtrl->enMode;
    _16bit_to_8bit_params._16bit_to_8bit_3a.bits._16bitto8bit_s8bias = pst16BitTo8BitCtrl->s8Bias;
    /*FIX ME: _16bitto8bit_u0q16norm how to calculate ?*/
    _16bit_to_8bit_params._16bit_to_8bit_3a.bits._16bitto8bit_u0q16norm =
        pst16BitTo8BitCtrl->u8LineCoefThr;

    *cmd_line_num = _16bit_to_8bit_task_cmd_queue(&_16bit_to_8bit_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_ord_stat_filter_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                  CVE_ORD_STAT_FILTER_CTRL_T *pstOrdStatFltCtrl, char *cmd_buf,
                                  unsigned int *cmd_line_num)
{
    cve_op_ord_stat_filter_params_t stat_filter_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&stat_filter_params, 0, sizeof(cve_op_ord_stat_filter_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstOrdStatFltCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstOrdStatFltCtrl->stCrop.u16X;
        init_params.ystart = pstOrdStatFltCtrl->stCrop.u16Y;
        init_params.xSize = pstOrdStatFltCtrl->stCrop.u16Width;
        init_params.ySize = pstOrdStatFltCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&stat_filter_params.comm_params, &init_params);

    /*stat_filter op params initialization */
    stat_filter_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    stat_filter_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    stat_filter_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_ORD_STAT_FILTER;
    stat_filter_params.stat_filter_3b.bits.stat_filter_outnum = 1;
    if (pstOrdStatFltCtrl->enMode == CVE_ORD_STAT_FILTER_MODE_MEDIAN) {
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx0 = 4;
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx1 = 4;
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx2 = 4;
    } else if (pstOrdStatFltCtrl->enMode == CVE_ORD_STAT_FILTER_MODE_MAX) {
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx0 = 8;
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx1 = 8;
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx2 = 8;
    } else {
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx0 = 0;
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx1 = 0;
        stat_filter_params.stat_filter_3b.bits.stat_filter_outidx2 = 0;
    }

    *cmd_line_num = ord_stat_filter_task_cmd_queue(&stat_filter_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_map_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_SRC_MEM_INFO_T *pstMap,
                      CVE_DST_IMAGE_T *pstDstImage, CVE_MAP_CTRL_T *pstMapCtrl, char *cmd_buf,
                      unsigned int *cmd_line_num)
{
    cve_op_map_params_t map_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&map_params, 0, sizeof(cve_op_map_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstMapCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstMapCtrl->stCrop.u16X;
        init_params.ystart = pstMapCtrl->stCrop.u16Y;
        init_params.xSize = pstMapCtrl->stCrop.u16Width;
        init_params.ySize = pstMapCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.dst_width;
        init_params.ySize = init_params.dst_height;
    }

    cve_common_params_init(&map_params.comm_params, &init_params);

    /*map op params initialization */
    map_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    map_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    map_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_MAP;
    map_params.comm_params.reg_02.bits.src_image_type = pstSrcImage->enType;
    map_params.comm_params.reg_02.bits.dst_image_type = pstDstImage->enType;
    map_params.map_3c.bits.map_mode = pstMapCtrl->enMode;

    *cmd_line_num = map_task_cmd_queue(&map_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_equalize_hist_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                                CVE_EQUALIZE_HIST_CTRL_T *pstEqualizeHistCtrl, char *cmd_buf,
                                unsigned int *cmd_line_num)
{
    cve_op_hist_params_t equalize_hist_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&equalize_hist_params, 0, sizeof(cve_op_hist_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstEqualizeHistCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstEqualizeHistCtrl->stCrop.u16X;
        init_params.ystart = pstEqualizeHistCtrl->stCrop.u16Y;
        init_params.xSize = pstEqualizeHistCtrl->stCrop.u16Width;
        init_params.ySize = pstEqualizeHistCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&equalize_hist_params.comm_params, &init_params);
    /*equalize hist op params initialization */
    equalize_hist_params.comm_params.reg_d8.bits.rdmif_pack_mode = 2;
    equalize_hist_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    equalize_hist_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_EQUALIZE_HIST;
    equalize_hist_params.integ_33.bits.eqhist_curv_mode = 1;
    /*FIX ME: (1<<32) / height /width*/
    equalize_hist_params.eqhist_34.bits.eqhist_norm =
        (0xffffffff / init_params.xSize) / init_params.ySize;

    *cmd_line_num = hist_task_cmd_queue(&equalize_hist_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_ncc_task(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                      CVE_NCC_CTRL_T *pstNccCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_ncc_params_t ncc_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    int ret = 0;

    memset(&ncc_params, 0, sizeof(cve_op_ncc_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage1, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_src_image(&init_params, pstSrcImage2, &src_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage1->u32Width;
    init_params.src_height = pstSrcImage1->u32Height;
    if (pstNccCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstNccCtrl->stCrop.u16X;
        init_params.ystart = pstNccCtrl->stCrop.u16Y;
        init_params.xSize = pstNccCtrl->stCrop.u16Width;
        init_params.ySize = pstNccCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }
    cve_common_params_init(&ncc_params.comm_params, &init_params);

    /*ncc op params initialization */
    ncc_params.comm_params.reg_d8.bits.rdmif_pack_mode = 2;
    ncc_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 2;
    ncc_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_NCC;

    ncc_params.ncc_3c.bits.ncc_mode = pstNccCtrl->enMode;
    if (pstNccCtrl->enMode == CVE_NCC_MODE_SIMILAR) {
        ncc_params.ncc_3c.bits.ncc_offset1 = pstNccCtrl->u8Src1Offset;
        ncc_params.ncc_3c.bits.ncc_offset0 = pstNccCtrl->u8Src0Offset;
    }

    /*FIX ME: How to get ncc data?*/

    *cmd_line_num = ncc_task_cmd_queue(&ncc_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_ccl_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                      CVE_DST_MEM_INFO_T *pstBlob, CVE_CCL_CTRL_T *pstCclCtrl, char *cmd_buf,
                      unsigned int *cmd_line_num)
{
    cve_op_ccl_params_t ccl_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&ccl_params, 0, sizeof(cve_op_ccl_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_src_image(&init_params, pstDstImage, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstBlob, sizeof(CVE_REGION_T) * CVE_MAX_REGION_NUM, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstCclCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstCclCtrl->stCrop.u16X;
        init_params.ystart = pstCclCtrl->stCrop.u16Y;
        init_params.xSize = pstCclCtrl->stCrop.u16Width;
        init_params.ySize = pstCclCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&ccl_params.comm_params, &init_params);

    /*ccl op params initialization */
    ccl_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    ccl_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    ccl_params.comm_params.reg_d0.bits.wdmif2_pack_mode = 0;
    ccl_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;
    ccl_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_CCL;
    ccl_params.comm_params.reg_02.bits.intput_mode = pstCclCtrl->enInputDataMode;

    *cmd_line_num = ccl_task_cmd_queue(&ccl_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_gmm_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_SRC_IMAGE_T *pstFactor,
                      CVE_DST_IMAGE_T *pstFg, CVE_DST_IMAGE_T *pstBg, CVE_MEM_INFO_T *pstModel,
                      CVE_GMM_CTRL_T *pstGmmCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_gmm_params_t gmm_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    unsigned int channel_num = 0;
    unsigned int model_len = 0;
    int ret = 0;

    if (pstSrcImage->enType == CVE_IMAGE_TYPE_U8C3_PACKAGE) {
        channel_num = 3;
    } else {
        channel_num = 1;
    }

    model_len = CVE_ALIGN_UP(
                    pstFg->u32Width * (8 + pstGmmCtrl->u8ModelNum * (32 + 16 * channel_num)), 128) /
                8;

    memset(&gmm_params, 0, sizeof(cve_op_ccl_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_src_image(&init_params, pstFactor, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_src_mem(&init_params, pstModel, model_len, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstFg, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstBg, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstModel, model_len, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstFg->u32Width;
    init_params.dst_height = pstFg->u32Height;
    if (pstGmmCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstGmmCtrl->stCrop.u16X;
        init_params.ystart = pstGmmCtrl->stCrop.u16Y;
        init_params.xSize = pstGmmCtrl->stCrop.u16Width;
        init_params.ySize = pstGmmCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&gmm_params.comm_params, &init_params);

    /*ccl op params initialization */
    gmm_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    gmm_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    gmm_params.comm_params.reg_e8.bits.rdmif2_pack_mode = 0;
    gmm_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;
    gmm_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_GMM;
    gmm_params.comm_params.reg_02.bits.output_mode = pstGmmCtrl->enOutputDataMode;

    gmm_params.gmm_44.bits.gmm_u0q16initweight = pstGmmCtrl->u0q16InitWeight;
    gmm_params.gmm_44.bits.gmm_u0q16learnrate = pstGmmCtrl->u0q16LearnRate;
    gmm_params.gmm_45.bits.gmm_u10q6sigma_init = pstGmmCtrl->u10q6NoiseVar;
    gmm_params.gmm_45.bits.gmm_ds_mode = pstGmmCtrl->enDownScaleMode;
    gmm_params.gmm_45.bits.gmm_output_bg_en = pstGmmCtrl->enOutputMode;
    gmm_params.gmm_46.bits.gmm_u10q6sigma_max = pstGmmCtrl->u10q6MaxVar;
    gmm_params.gmm_46.bits.gmm_u2nchannels = channel_num;

    gmm_params.gmm_46.bits.gmm_u3modelnum = pstGmmCtrl->u8ModelNum;

    gmm_params.gmm_47.bits.gmm_modellen_in128b =
        CVE_ALIGN_UP(pstFg->u32Width * (8 + pstGmmCtrl->u8ModelNum *
                                                (32 + 16 * gmm_params.gmm_46.bits.gmm_u2nchannels)),
                     128) /
        128;

    gmm_params.gmm_47.bits.gmm_u10q6sigma_min = pstGmmCtrl->u10q6MinVar;
    gmm_params.gmm_48.bits.gmm_u3q7sigma_scale = pstGmmCtrl->u3q7SigmaScale;
    gmm_params.gmm_48.bits.gmm_u0q16weight_sum_thr = pstGmmCtrl->u0q16WeightThr;

    gmm_params.gmm_49.bits.gmm_piclen_in128b = pstSrcImage->u32Width * channel_num / 16;
    gmm_params.gmm_49.bits.gmm_update_factor_mode_en = pstGmmCtrl->enDurationUpdateFactorMode;
    gmm_params.gmm_49.bits.gmm_acc_lr_en = pstGmmCtrl->enFastLearn;
    gmm_params.gmm_49.bits.gmm_sns_factor_mode_en = pstGmmCtrl->enSnsFactorMode;

    *cmd_line_num = gmm_task_cmd_queue(&gmm_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_canny_hys_edge_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstEdge,
                                 CVE_DST_MEM_INFO_T *pstStack,
                                 CVE_CANNY_HYS_EDGE_CTRL_T *pstCannyHysEdgeCtrl, char *cmd_buf,
                                 unsigned int *cmd_line_num)
{
    cve_op_canny_edge_params_t canny_params;
    cve_comm_init_params_t init_params;
    char mask_v[25];
    int i = 0;
    int j = 0;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&canny_params, 0, sizeof(cve_op_canny_edge_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_src_mem(&init_params, &pstCannyHysEdgeCtrl->stMem, pstSrcImage->au32Stride[0] * 4,
                       &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstEdge, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstStack, pstEdge->au32Stride[0] * sizeof(CVE_POINT_U16_T),
                       &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, &pstCannyHysEdgeCtrl->stMem, pstEdge->au32Stride[0] * 4,
                       &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstEdge->u32Width;
    init_params.dst_height = pstEdge->u32Height;
    if (pstCannyHysEdgeCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstCannyHysEdgeCtrl->stCrop.u16X;
        init_params.ystart = pstCannyHysEdgeCtrl->stCrop.u16Y;
        init_params.xSize = pstCannyHysEdgeCtrl->stCrop.u16Width;
        init_params.ySize = pstCannyHysEdgeCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&canny_params.comm_params, &init_params);

    /*canny op params initialization */
    canny_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    canny_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    canny_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 2;
    canny_params.comm_params.reg_d0.bits.wdmif2_pack_mode = 0;
    canny_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;
    canny_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_CANNY_HYS_EDGE;

    memcpy(&canny_params.sobel_75.reg, pstCannyHysEdgeCtrl->as8Mask, 24);
    canny_params.sobel_7b.bits.sobel_coef_h_24 = pstCannyHysEdgeCtrl->as8Mask[24];
    for (j = 0; j < 5; j++) {
        for (i = 0; i < 5; i++) {
            mask_v[j * 5 + i] = pstCannyHysEdgeCtrl->as8Mask[i * 5 + j];
        }
    }

    memcpy(&canny_params.sobel_7c.reg, mask_v, 24);
    canny_params.sobel_82.bits.sobel_coef_v_24 = mask_v[24];

    canny_params.comm_params.reg_14.bits.dst_stride_0 =
        (canny_params.comm_params.reg_14.bits.dst_stride_0 + 3) >> 2;
    canny_params.canny_4a.bits.canny_u16highthr = pstCannyHysEdgeCtrl->u16HighThr;
    canny_params.canny_4a.bits.canny_u16lowthr = pstCannyHysEdgeCtrl->u16LowThr;
    canny_params.canny_4c.bits.canny_gauss_en = pstCannyHysEdgeCtrl->bGauss;

    *cmd_line_num = canny_edge_task_cmd_queue(&canny_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_lbp_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstImage,
                      CVE_LBP_CTRL_T *pstLbpCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_lbp_params_t lbp_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&lbp_params, 0, sizeof(cve_op_lbp_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstDstImage, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstDstImage->u32Width;
    init_params.dst_height = pstDstImage->u32Height;
    if (pstLbpCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstLbpCtrl->stCrop.u16X;
        init_params.ystart = pstLbpCtrl->stCrop.u16Y;
        init_params.xSize = pstLbpCtrl->stCrop.u16Width;
        init_params.ySize = pstLbpCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&lbp_params.comm_params, &init_params);

    /*lbp op params initialization */
    lbp_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    lbp_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    lbp_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_LBP;
    lbp_params.lbp_4d.bits.u8bitthr = pstLbpCtrl->un8BitThr.u8Val;
    lbp_params.lbp_4d.bits.cmp_mode = pstLbpCtrl->enMode;

    *cmd_line_num = lbp_task_cmd_queue(&lbp_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_norm_grad_task(CVE_SRC_IMAGE_T *pstSrcImage, CVE_DST_IMAGE_T *pstDstH,
                            CVE_DST_IMAGE_T *pstDstV, CVE_DST_IMAGE_T *pstDstHV,
                            CVE_NORM_GRAD_CTRL_T *pstNormGradCtrl, char *cmd_buf,
                            unsigned int *cmd_line_num)
{
    cve_op_norm_grad_params_t norm_grad_params;
    cve_comm_init_params_t init_params;
    char mask_v[25];
    int i = 0;
    int j = 0;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&norm_grad_params, 0, sizeof(cve_op_norm_grad_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstDstH, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstDstV, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstDstHV, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcImage->u32Width;
    init_params.src_height = pstSrcImage->u32Height;
    init_params.dst_width = pstSrcImage->u32Width;
    init_params.dst_height = pstSrcImage->u32Height;
    if (pstNormGradCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstNormGradCtrl->stCrop.u16X;
        init_params.ystart = pstNormGradCtrl->stCrop.u16Y;
        init_params.xSize = pstNormGradCtrl->stCrop.u16Width;
        init_params.ySize = pstNormGradCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&norm_grad_params.comm_params, &init_params);

    /*lbp op params initialization */
    norm_grad_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    norm_grad_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    norm_grad_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
    norm_grad_params.comm_params.reg_d0.bits.wdmif2_pack_mode = 0;

    norm_grad_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_NROM_GRAD;
    norm_grad_params.norm_grad_4e.bits.normgrad_u8norm = pstNormGradCtrl->u8Norm;
    norm_grad_params.norm_grad_4e.bits.normgrad_outmode = pstNormGradCtrl->enOutCtrl;

    memcpy(&norm_grad_params.sobel_75.reg, pstNormGradCtrl->as8Mask, 24);
    norm_grad_params.sobel_7b.bits.sobel_coef_h_24 = pstNormGradCtrl->as8Mask[24];
    for (j = 0; j < 5; j++) {
        for (i = 0; i < 5; i++) {
            mask_v[j * 5 + i] = pstNormGradCtrl->as8Mask[i * 5 + j];
        }
    }

    memcpy(&norm_grad_params.sobel_7c.reg, mask_v, 24);
    norm_grad_params.sobel_82.bits.sobel_coef_v_24 = mask_v[24];

    *cmd_line_num = norm_grad_task_cmd_queue(&norm_grad_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_build_lk_optical_flow_pyr_task(
    CVE_SRC_IMAGE_T *pstSrcPyr, CVE_SRC_IMAGE_T astDstPyr[],
    CVE_BUILD_LK_OPTICAL_FLOW_PYR_CTRL_T *pstBuildLkOptiFlowPyrCtrl, char *cmd_buf,
    unsigned int *cmd_line_num)
{
    cve_op_build_lk_optical_flow_pyr_params_t build_lk_optical_flow_pyr_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;
    int i = 0;

    memset(&build_lk_optical_flow_pyr_params, 0, sizeof(cve_op_build_lk_optical_flow_pyr_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcPyr, &src_off);
    if (ret != 0) {
        return ret;
    }

    for (i = 0; i <= pstBuildLkOptiFlowPyrCtrl->u8MaxLevel; i++) {
        ret = fill_dst_image(&init_params, &astDstPyr[i], &dst_off);
        if (ret != 0) {
            return ret;
        }
    }

    init_params.src_width = pstSrcPyr->u32Width;
    init_params.src_height = pstSrcPyr->u32Height;
    init_params.dst_width = pstSrcPyr->u32Width;
    init_params.dst_height = pstSrcPyr->u32Height;
    if (pstBuildLkOptiFlowPyrCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstBuildLkOptiFlowPyrCtrl->stCrop.u16X;
        init_params.ystart = pstBuildLkOptiFlowPyrCtrl->stCrop.u16Y;
        init_params.xSize = pstBuildLkOptiFlowPyrCtrl->stCrop.u16Width;
        init_params.ySize = pstBuildLkOptiFlowPyrCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&build_lk_optical_flow_pyr_params.comm_params, &init_params);

    /*lk_optical_flow_pyr op params initialization */
    build_lk_optical_flow_pyr_params.comm_params.reg_d8.bits.rdmif_pack_mode = 1;
    build_lk_optical_flow_pyr_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    build_lk_optical_flow_pyr_params.comm_params.reg_02.bits.cve_op_type =
        CVE_OP_TYPE_BULID_LK_OPTICAL_FLOW_PYR;

    if (pstBuildLkOptiFlowPyrCtrl->u8MaxLevel == 2) {
        build_lk_optical_flow_pyr_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
    }

    if (pstBuildLkOptiFlowPyrCtrl->u8MaxLevel == 3) {
        build_lk_optical_flow_pyr_params.comm_params.reg_d0.bits.wdmif2_pack_mode = 0;
    }

    build_lk_optical_flow_pyr_params.bdlk_4f.bits.bdlk_maxLevel =
        pstBuildLkOptiFlowPyrCtrl->u8MaxLevel;

    *cmd_line_num = build_lk_optical_flow_pyr_task_cmd_queue(&build_lk_optical_flow_pyr_params,
                                                             (unsigned int *)cmd_buf);
    return 0;
}

int cve_fill_lk_optical_flow_pyr_task(CVE_SRC_IMAGE_T astSrcPrevPyr[],
                                      CVE_SRC_IMAGE_T astSrcNextPyr[],
                                      CVE_SRC_MEM_INFO_T *pstPrevPts, CVE_MEM_INFO_T *pstNextPts,
                                      CVE_LK_OPTICAL_FLOW_PYR_CTRL_T *pstLkOptiFlowPyrCtrl,
                                      char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_lk_optical_flow_pyr_params_t lk_optical_flow_pyr_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;
    int i = 0;

    memset(&lk_optical_flow_pyr_params, 0, sizeof(cve_op_lk_optical_flow_pyr_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    for (i = 0; i <= 3; i++) {
        ret = fill_src_image(&init_params, &astSrcPrevPyr[i], &src_off);
        if (ret != 0) {
            return ret;
        }
    }
    for (i = 0; i <= 3; i++) {
        ret = fill_src_image(&init_params, &astSrcNextPyr[i], &src_off);
        if (ret != 0) {
            return ret;
        }
    }

    ret = fill_dst_mem(&init_params, pstPrevPts, sizeof(CVE_POINT_S12Q7_T) * CVE_ST_MAX_CORNER_NUM,
                       &dst_off);

    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstNextPts, sizeof(CVE_POINT_S12Q7_T) * CVE_ST_MAX_CORNER_NUM,
                       &dst_off);
    if (ret != 0) {
        return ret;
    }

    /*    if ((pstLkOptiFlowPyrCtrl->enOutMode == CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_ERR) ||
            (pstLkOptiFlowPyrCtrl->enOutMode == CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_BOTH)) {
            ret = fill_dst_mem(&init_params, pstErr, CVE_ST_MAX_CORNER_NUM, &dst_off);
            if (ret != 0) {
                return ret;
            }
        }*/

    init_params.src_width = astSrcPrevPyr[0].u32Width;
    init_params.src_height = astSrcPrevPyr[0].u32Height;
    init_params.dst_width = astSrcPrevPyr[0].u32Width;
    init_params.dst_height = astSrcPrevPyr[0].u32Height;
    if (pstLkOptiFlowPyrCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstLkOptiFlowPyrCtrl->stCrop.u16X;
        init_params.ystart = pstLkOptiFlowPyrCtrl->stCrop.u16Y;
        init_params.xSize = pstLkOptiFlowPyrCtrl->stCrop.u16Width;
        init_params.ySize = pstLkOptiFlowPyrCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    init_params.dst_stride[3] = init_params.dst_stride[2];
    cve_common_params_init(&lk_optical_flow_pyr_params.comm_params, &init_params);

    /*lk_optical_flow_pyr op params initialization */
    lk_optical_flow_pyr_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    lk_optical_flow_pyr_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;
    lk_optical_flow_pyr_params.comm_params.reg_e8.bits.rdmif2_pack_mode = 0;
    lk_optical_flow_pyr_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    lk_optical_flow_pyr_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
    lk_optical_flow_pyr_params.comm_params.reg_d0.bits.wdmif2_pack_mode = 0;
    lk_optical_flow_pyr_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_LK_OPTIAL_FLOW_PRY;

    lk_optical_flow_pyr_params.lk_50.bits.lk_u10pstnum = pstLkOptiFlowPyrCtrl->u16PtsNum;
    lk_optical_flow_pyr_params.lk_50.bits.lk_buseinitflow = pstLkOptiFlowPyrCtrl->bUseInitFlow;
    lk_optical_flow_pyr_params.lk_50.bits.lk_mode = pstLkOptiFlowPyrCtrl->enOutMode;
    lk_optical_flow_pyr_params.lk_51.bits.u0q8eps = pstLkOptiFlowPyrCtrl->u0q8Eps;
    lk_optical_flow_pyr_params.lk_51.bits.lk_u8itercnt = pstLkOptiFlowPyrCtrl->u8IterCnt;
    lk_optical_flow_pyr_params.lk_51.bits.lk_u2maxlevel = pstLkOptiFlowPyrCtrl->u8MaxLevel;

    *cmd_line_num =
        lk_optical_flow_pyr_task_cmd_queue(&lk_optical_flow_pyr_params, (unsigned int *)cmd_buf);
    return 0;
}

int cve_fill_st_candi_corner_task(CVE_SRC_IMAGE_T *pstSrc, CVE_DST_IMAGE_T *pstLabel,
                                  CVE_DST_IMAGE_T *pstCandiCorner,
                                  CVE_DST_MEM_INFO_T *pstCandiCornerPoint,
                                  CVE_ST_CANDI_CORNER_CTRL_T *pstStCandiCornerCtrl, char *cmd_buf,
                                  unsigned int *cmd_line_num)
{
    cve_op_st_candi_corner_params_t st_candi_corner_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&st_candi_corner_params, 0, sizeof(cve_op_st_candi_corner_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrc, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_src_mem(&init_params, &pstStCandiCornerCtrl->stMem,
                       pstCandiCorner->au32Stride[0] * 2, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_image(&init_params, pstLabel, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, &pstStCandiCornerCtrl->stMem,
                       pstCandiCorner->au32Stride[0] * 2, &dst_off);
    if (ret != 0) {
        return ret;
    }

    dst_off++;

    ret = fill_dst_image(&init_params, pstCandiCorner, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstCandiCornerPoint,
                       pstCandiCorner->au32Stride[0] * sizeof(CVE_POINT_U16_T), &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrc->u32Width;
    init_params.src_height = pstSrc->u32Height;
    init_params.dst_width = pstCandiCorner->u32Width;
    init_params.dst_height = pstCandiCorner->u32Height;

    if (pstStCandiCornerCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstStCandiCornerCtrl->stCrop.u16X;
        init_params.ystart = pstStCandiCornerCtrl->stCrop.u16Y;
        init_params.xSize = pstStCandiCornerCtrl->stCrop.u16Width;
        init_params.ySize = pstStCandiCornerCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&st_candi_corner_params.comm_params, &init_params);

    /*st_corner op params initialization */
    st_candi_corner_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    st_candi_corner_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;
    st_candi_corner_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;

    st_candi_corner_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_ST_CANDI_CORNER;
    st_candi_corner_params.comm_params.reg_02.bits.output_mode =
        pstStCandiCornerCtrl->enOutputDataMode;

    *cmd_line_num =
        st_candi_corner_task_cmd_queue(&st_candi_corner_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_sad_task(CVE_SRC_IMAGE_T *pstSrcImage1, CVE_SRC_IMAGE_T *pstSrcImage2,
                      CVE_DST_IMAGE_T *pstSad, CVE_DST_IMAGE_T *pstThr, CVE_SAD_CTRL_T *pstSadCtrl,
                      char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_sad_params_t sad_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&sad_params, 0, sizeof(cve_op_sad_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstSrcImage1, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_src_image(&init_params, pstSrcImage2, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstSad, &dst_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstThr, &dst_off);
    if (ret != 0) {
        return ret;
    }
    init_params.src_width = pstSrcImage1->u32Width;
    init_params.src_height = pstSrcImage1->u32Height;
    init_params.dst_width = pstSrcImage1->u32Width;
    init_params.dst_height = pstSrcImage1->u32Height;
    if (pstSadCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstSadCtrl->stCrop.u16X;
        init_params.ystart = pstSadCtrl->stCrop.u16Y;
        init_params.xSize = pstSadCtrl->stCrop.u16Width;
        init_params.ySize = pstSadCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&sad_params.comm_params, &init_params);

    /*sad op params initialization */
    sad_params.comm_params.reg_d8.bits.rdmif_pack_mode = 2;
    sad_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    sad_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
    sad_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 2;

    sad_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_SAD;
    sad_params.sad_64.bits.sad_u8minval = pstSadCtrl->u8MinVal;
    sad_params.sad_64.bits.sad_mode = pstSadCtrl->enMode;
    sad_params.sad_65.bits.sad_u4rightshift = 0;
    sad_params.sad_65.bits.sad_u8maxval = pstSadCtrl->u8MaxVal;
    sad_params.sad_65.bits.sad_u16thr = pstSadCtrl->u16Thr;

    *cmd_line_num = sad_task_cmd_queue(&sad_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_grad_fg_task(CVE_SRC_IMAGE_T *pstBgDiffFg, CVE_SRC_IMAGE_T *pstCurGrad,
                          CVE_SRC_IMAGE_T *pstBgGrad, CVE_DST_IMAGE_T *pstGradFg,
                          CVE_GRAD_FG_CTRL_T *pstGradFgCtrl, char *cmd_buf,
                          unsigned int *cmd_line_num)
{
    cve_op_grad_fg_params_t grad_fg_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    int ret = 0;

    memset(&grad_fg_params, 0, sizeof(cve_op_grad_fg_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    /* common params initialization */
    ret = fill_src_image(&init_params, pstBgDiffFg, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_src_image(&init_params, pstCurGrad, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_src_image(&init_params, pstBgGrad, &src_off);
    if (ret != 0) {
        return ret;
    }
    ret = fill_dst_image(&init_params, pstGradFg, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstBgDiffFg->u32Width;
    init_params.src_height = pstBgDiffFg->u32Height;
    init_params.dst_width = pstBgDiffFg->u32Width;
    init_params.dst_height = pstBgDiffFg->u32Height;
    if (pstGradFgCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstGradFgCtrl->stCrop.u16X;
        init_params.ystart = pstGradFgCtrl->stCrop.u16Y;
        init_params.xSize = pstGradFgCtrl->stCrop.u16Width;
        init_params.ySize = pstGradFgCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    if ((pstGradFgCtrl->enMode == CVE_GRAD_FG_MODE_THR) ||
        (pstGradFgCtrl->enMode == CVE_GRAD_FG_MODE_THR_FG)) {
        init_params.src_stride[3] = init_params.src_stride[2];
        init_params.dst_stride[1] = init_params.dst_stride[0] * 2;
    }
    cve_common_params_init(&grad_fg_params.comm_params, &init_params);

    /*grad_fg op params initialization */
    grad_fg_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    grad_fg_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    grad_fg_params.comm_params.reg_e8.bits.rdmif2_pack_mode = 0;
    grad_fg_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;

    grad_fg_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_GRAD_FG;
    grad_fg_params.grad_fg_66.bits.gradfg_mode = pstGradFgCtrl->enMode;
    grad_fg_params.grad_fg_66.bits.gradfg_u8minmagdiff = pstGradFgCtrl->u8MinMagDiff;

    *cmd_line_num = grad_fg_task_cmd_queue(&grad_fg_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_update_bg_model_task(CVE_SRC_IMAGE_T *pstCurImg, CVE_MEM_INFO_T *pstBgModel1,
                                  CVE_MEM_INFO_T *pstBgModel2,
                                  CVE_UPDATE_BG_MODEL_CTRL_T *pstUpdateBgModelCtrl, char *cmd_buf,
                                  unsigned int *cmd_line_num)
{
    cve_op_update_bg_model_params_t update_bg_model_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    unsigned int dst_stride = 0;
    int ret = 0;

    /* common params initialization */
    memset(&update_bg_model_params, 0, sizeof(cve_op_update_bg_model_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    ret = fill_src_image(&init_params, pstCurImg, &src_off);
    if (ret != 0) {
        return ret;
    }

    dst_stride = pstCurImg->au32Stride[0] >> pstUpdateBgModelCtrl->enDownScaleMode;
    ret = fill_dst_mem(&init_params, pstBgModel1, dst_stride * 16, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstBgModel2, dst_stride * 16, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstCurImg->u32Width;
    init_params.src_height = pstCurImg->u32Height;
    init_params.dst_width = pstCurImg->u32Width >> pstUpdateBgModelCtrl->enDownScaleMode;
    init_params.dst_height = pstCurImg->u32Height >> pstUpdateBgModelCtrl->enDownScaleMode;
    if (pstUpdateBgModelCtrl->stCrop.enCrop == AML_TRUE) {
        init_params.xstart = pstUpdateBgModelCtrl->stCrop.u16X;
        init_params.ystart = pstUpdateBgModelCtrl->stCrop.u16Y;
        init_params.xSize = pstUpdateBgModelCtrl->stCrop.u16Width;
        init_params.ySize = pstUpdateBgModelCtrl->stCrop.u16Height;
    } else {
        init_params.xstart = 0;
        init_params.ystart = 0;
        init_params.xSize = init_params.src_width;
        init_params.ySize = init_params.src_height;
    }

    cve_common_params_init(&update_bg_model_params.comm_params, &init_params);

    /*update_bg_model op params initialization */
    update_bg_model_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    update_bg_model_params.comm_params.reg_e8.bits.rdmif2_pack_mode = 0;
    update_bg_model_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;
    update_bg_model_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_UPDATE_BG_MODEL;

    if (pstUpdateBgModelCtrl->enDownScaleMode == CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_2X2) {
        update_bg_model_params.comm_params.reg_d8.bits.rdmif_pack_mode = 1;
    } else if (pstUpdateBgModelCtrl->enDownScaleMode == CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_4X4) {
        update_bg_model_params.comm_params.reg_d8.bits.rdmif_pack_mode = 2;
    } else {
        update_bg_model_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    }
    update_bg_model_params.update_bg_mode_2a.bits.updatebgmodel_u16del_thr =
        pstUpdateBgModelCtrl->u16DelThr;
    update_bg_model_params.update_bg_mode_2a.bits.updatebgmodel_u16frq_thr =
        pstUpdateBgModelCtrl->u16FrqThr;
    update_bg_model_params.update_bg_mode_2b.bits.updatebgmodel_u0q16lr =
        pstUpdateBgModelCtrl->u0q16LearnRate;
    update_bg_model_params.update_bg_mode_2b.bits.updatebgmodel_u16interval_thr =
        pstUpdateBgModelCtrl->u16IntervalThr;

    update_bg_model_params.bg_mode_27.bits.bgmodel_ds_mode = pstUpdateBgModelCtrl->enDownScaleMode;
    *cmd_line_num =
        update_bg_model_task_cmd_queue(&update_bg_model_params, (unsigned int *)cmd_buf);

    return 0;
}

int cve_fill_tof_task(CVE_SRC_RAW_T *pstSrcRaw, CVE_SRC_RAW_T *pstSrcFpn,
                      CVE_SRC_MEM_INFO_T *pstSrcCoef, CVE_DST_MEM_INFO_T *pstDstStatus,
                      CVE_DST_MEM_INFO_T *pstDstIR, CVE_DST_MEM_INFO_T *pstDstData,
                      CVE_TOF_CTRL_T *pstTofCtrl, char *cmd_buf, unsigned int *cmd_line_num)
{
    cve_op_tof_params_t tof_params;
    cve_comm_init_params_t init_params;
    unsigned int src_off = 0;
    unsigned int dst_off = 0;
    unsigned char spa_mask[15] = {0, 1, 2, 1, 0, 0, 2, 4, 2, 0, 0, 1, 2, 1, 0};
    int ret = 0;

    /* common params initialization */
    memset(&tof_params, 0, sizeof(cve_op_update_bg_model_params_t));
    memset(&init_params, 0, sizeof(cve_comm_init_params_t));

    ret = fill_src_raw(&init_params, pstSrcRaw, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_src_raw(&init_params, pstSrcFpn, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_src_mem(&init_params, pstSrcCoef, pstSrcRaw->u32Stride * 8, &src_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstDstStatus, pstSrcRaw->u32Stride * 2, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstDstIR, pstSrcRaw->u32Stride * 2, &dst_off);
    if (ret != 0) {
        return ret;
    }

    ret = fill_dst_mem(&init_params, pstDstData, pstSrcRaw->u32Stride * 2, &dst_off);
    if (ret != 0) {
        return ret;
    }

    init_params.src_width = pstSrcRaw->u32Width;
    init_params.src_height = pstSrcRaw->u32Height;
    init_params.dst_width = pstSrcRaw->u32Width;
    init_params.dst_height = pstSrcRaw->u32Height;
    init_params.xstart = 0;
    init_params.ystart = 0;
    init_params.xSize = init_params.src_width;
    init_params.ySize = init_params.src_height;

    init_params.src_stride[3] = init_params.src_stride[2];
    init_params.dst_stride[3] = init_params.dst_stride[2];
    cve_common_params_init(&tof_params.comm_params, &init_params);

    /*update_bg_model op params initialization */
    tof_params.comm_params.reg_c0.bits.wdmif_pack_mode = 0;
    tof_params.comm_params.reg_c8.bits.wdmif1_pack_mode = 0;
    tof_params.comm_params.reg_d0.bits.wdmif2_pack_mode = 0;
    tof_params.comm_params.reg_d8.bits.rdmif_pack_mode = 0;
    tof_params.comm_params.reg_e8.bits.rdmif2_pack_mode = 0;
    tof_params.comm_params.reg_e0.bits.rdmif1_pack_mode = 0;
    tof_params.comm_params.reg_02.bits.cve_op_type = CVE_OP_TYPE_TOF;

    tof_params.tof_83.bits.cve_tof_hist_x_start = pstTofCtrl->u16HistXstart;
    tof_params.tof_83.bits.cve_tof_hist_y_start = pstTofCtrl->u16HistYstart;
    tof_params.tof_84.bits.cve_tof_hist_x_end = pstTofCtrl->u16HistXend;
    tof_params.tof_84.bits.cve_tof_hist_y_end = pstTofCtrl->u16HistYend;

    tof_params.tof_85.bits.cve_tof_p_coef_0 = pstTofCtrl->as32PCoef[0];
    tof_params.tof_86.bits.cve_tof_p_coef_1 = pstTofCtrl->as32PCoef[1];
    tof_params.tof_87.bits.cve_tof_p_coef_2 = pstTofCtrl->as32PCoef[2];
    tof_params.tof_88.bits.cve_tof_p_coef_3 = pstTofCtrl->as32PCoef[3];
    tof_params.tof_89.bits.cve_tof_p_coef_4 = pstTofCtrl->as32PCoef[4];

    tof_params.tof_8a.bits.cve_tof_t_coef1_0 = pstTofCtrl->as16TCoef1[0];
    tof_params.tof_8a.bits.cve_tof_t_coef1_1 = pstTofCtrl->as16TCoef1[1];
    tof_params.tof_8b.bits.cve_tof_t_coef1_2 = pstTofCtrl->as16TCoef1[2];
    tof_params.tof_8b.bits.cve_tof_t_coef1_3 = pstTofCtrl->as16TCoef1[3];

    tof_params.tof_8c.bits.cve_tof_spa_mask_0 = spa_mask[0];
    tof_params.tof_8c.bits.cve_tof_spa_mask_1 = spa_mask[1];
    tof_params.tof_8c.bits.cve_tof_spa_mask_2 = spa_mask[2];
    tof_params.tof_8d.bits.cve_tof_spa_mask_3 = spa_mask[3];
    tof_params.tof_8d.bits.cve_tof_spa_mask_4 = spa_mask[4];
    tof_params.tof_8d.bits.cve_tof_spa_mask_5 = spa_mask[5];
    tof_params.tof_8e.bits.cve_tof_spa_mask_6 = spa_mask[6];
    tof_params.tof_8e.bits.cve_tof_spa_mask_7 = spa_mask[7];
    tof_params.tof_8e.bits.cve_tof_spa_mask_8 = spa_mask[8];
    tof_params.tof_8f.bits.cve_tof_spa_mask_9 = spa_mask[9];
    tof_params.tof_8f.bits.cve_tof_spa_mask_10 = spa_mask[10];
    tof_params.tof_8f.bits.cve_tof_spa_mask_11 = spa_mask[11];
    tof_params.tof_90.bits.cve_tof_spa_mask_12 = spa_mask[12];
    tof_params.tof_90.bits.cve_tof_spa_mask_13 = spa_mask[13];
    tof_params.tof_90.bits.cve_tof_spa_mask_14 = spa_mask[14];

    tof_params.tof_91.bits.cve_tof_raw_mode = pstTofCtrl->enRawMode;
    tof_params.tof_91.bits.cve_tof_bp_num = pstTofCtrl->u8BadPointNum;
    tof_params.tof_91.bits.cve_tof_spa_norm = pstTofCtrl->u8SpaNorm;
    tof_params.tof_91.bits.cve_tof_raw_shift_12bit_en = pstTofCtrl->bRawShift;
    tof_params.tof_92.bits.cve_tof_temperature_int = pstTofCtrl->s8IntTemp;
    tof_params.tof_92.bits.cve_tof_temperature_ext = pstTofCtrl->s8ExtTemp;
    tof_params.tof_93.bits.cve_tof_q1_high_thr = pstTofCtrl->u16Q1HighThr;
    tof_params.tof_93.bits.cve_tof_q23_high_thr = pstTofCtrl->u16Q23HighThr;
    tof_params.tof_94.bits.cve_tof_ir_high_thr = pstTofCtrl->u16IRHighThr;
    tof_params.tof_94.bits.cve_tof_ir_low_thr = pstTofCtrl->u16IRLowThr;
    tof_params.tof_95.bits.cve_tof_dis_max = pstTofCtrl->u16DepthMax;
    tof_params.tof_95.bits.cve_tof_dis_min = pstTofCtrl->u16DepthMin;
    tof_params.tof_96.bits.cve_tof_bypass_en = pstTofCtrl->bBypass;
    tof_params.tof_96.bits.cve_tof_fpn_cali_mode = pstTofCtrl->enFpnMode;
    tof_params.tof_96.bits.cve_tof_spa1_en = pstTofCtrl->bSpa1En;
    tof_params.tof_96.bits.cve_tof_spa2_en = pstTofCtrl->bSpa2En;

    *cmd_line_num = tof_task_cmd_queue(&tof_params, (unsigned int *)cmd_buf);

    return 0;
}
