// SPDX-License-Identifier: (GPL-2.0+ OR MIT)
/*
 * Copyright (C) 2019-2024 Amlogic, Inc. All rights reserved.
 */
#include "cve.h"
#include "aml_common.h"
#include "aml_cve.h"
#include "cve_op.h"
#include <linux/arm-smccc.h>
#include <linux/clk.h>
#include <linux/dma-buf.h>
#include <linux/err.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/printk.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/seq_file.h>
#include <linux/time.h>
#include <linux/uaccess.h>
#include <linux/version.h>
#include <linux/wait.h>
#include <linux/workqueue.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0)
#include <linux/amlogic/ion.h>
#else
#include <linux/ion.h>
#endif

typedef enum {
    CVE_SYS_STATUS_READY = 0,
    CVE_SYS_STATUS_STOP = 1,
    CVE_SYS_STATUS_IDLE = 2,
} cve_sys_status_e;

#define CVE_OP_NODE_CMD_SIZE_MAX 512 // byte addr size 512/8=64
#define CMD_HANDLE_MAX 0x10000000
#define TASK_CMD_MAX 64
#define TASK_CMD_LINE_SIZE 8
#define CVE_QUERY_TIMEOUT 500
#define CVE_SYS_TIMEOUT 6000000


struct cve_device_s {
    dev_t dev_num;
    char name[10];
    struct device *cve_pdev;
    struct class *cve_class;
    struct resource *mem;
    int cve_irq;
    int major;
    struct proc_dir_entry *proc_node_entry;
    int cve_rate;
};

struct cve_device_s cve_device;

unsigned short cve_node_num = 512;
cve_sys_status_e cve_state = CVE_SYS_STATUS_IDLE;
cve_context_t cve_context;
static spinlock_t cve_spinlock;
bool cve_timeout_flag;
atomic_t cve_user_ref;
int cve_rate;
unsigned long long g_cve_mmz_base_addr;
int cve_irq;
void __iomem *cve_regs_map;
struct device *cve_pdev;
char *multi_cmd_buf[CVE_LUMA_MULTI_CMD_MAX];
unsigned int multi_cmd_line_num[CVE_LUMA_MULTI_CMD_MAX];
unsigned long long tasklet_start_time, tasklet_end_time;

static bool check_cve_reg_map(void)
{

    if (cve_regs_map) {
        return true;
    }

    CVE_ERR_TRACE("cve reg not map\n");

    return false;
}

static int cve_reg_init(void)
{
    unsigned int base_addr;

    base_addr = CVE_REG_BASE << 2;

    if (!cve_regs_map) {
        cve_regs_map = ioremap(base_addr, CVE_REG_SIZE);
    }

    return 0;
}

static unsigned int cve_reg_read(unsigned int reg)
{
    unsigned int addr = 0;
    unsigned int val = 0;

    addr = CVE_REG_ADDR_OFFSET(reg);
    if (check_cve_reg_map()) {
        val = ioread32(cve_regs_map + addr);
    }

    return val;
}

static void cve_reg_write(unsigned int reg, unsigned int val)
{
    unsigned int addr = 0;

    addr = CVE_REG_ADDR_OFFSET(reg);
    if (check_cve_reg_map()) {
        iowrite32(val, cve_regs_map + addr);
    }

    return;
}

static inline unsigned int cve_reg_get_bits(unsigned int reg, const unsigned int start,
                                            const unsigned int len)
{
    unsigned int val;

    val = (cve_reg_read(reg) >> (start)) & ((1L << (len)) - 1);
    return val;
}

static inline void cve_reg_bits_set(unsigned int reg, const unsigned int value,
                                    const unsigned int start, const unsigned int len)
{
    cve_reg_write(reg, ((cve_reg_read(reg) & ~(((1L << (len)) - 1) << (start))) |
                        (((value) & ((1L << (len)) - 1)) << (start))));
}
#if 0
static unsigned int cve_task_id_get()
{
    cve_reg_get_bits(CVE_ID_STA_REG1, 0, 16);

    return 0;
}
#endif
static void cve_cq0_int_enable(void)
{
    /* enable cq0 interrupt*/
    cve_reg_bits_set(CVE_INTR_MASKN_REG0, 1, 0, 1);

    return;
}

static void cve_cq0_int_disable(void)
{
    /* enable cq0 interrupt*/
    cve_reg_bits_set(CVE_INTR_MASKN_REG0, 0, 0, 1);

    return;
}

static void cve_cq0_int_clear(void)
{
    /*  clear cq0 interrupt */
    cve_reg_bits_set(CVE_INTR_STAT_WRITE_REG0, 0, 0, 1);
    /* disable cq0 */
    cve_reg_bits_set(CVE_CQ_REG2, 0, 24, 1);

    return;
}

static void cve_cq0_function_enable(void)
{
    /* enable cmd queue cq0 func */
    cve_reg_bits_set(CVE_CQ_REG2, 1, 26, 1);

    return;
}

static void cve_cq0_function_disable(void)
{
    /*need the clock*/
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG1, 2, 0, 2);
    /* disable cmd queue cq0 func */
    cve_reg_bits_set(CVE_CQ_REG2, 0, 26, 1);
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG1, 0, 0, 2);

    return;
}
#if 0
static void cve_sys_timeout_enable(unsigned int timeout)
{
    /* enable timeout func */
    cve_reg_bits_set(CVE_TIMEOUT_REG0, 1, 0, 1);
    /* enable timeout func */
    cve_reg_bits_set(CVE_TIMEOUT_REG1, timeout, 0, 32);

    return;
}

static void cve_sys_timeout_disable(void)
{
    /* enable disable func */
    cve_reg_bits_set(CVE_TIMEOUT_REG0, 0, 0, 1);

    return;
}
#endif
static unsigned int cve_int_status_get(void)
{
    unsigned int value;

    value = cve_reg_read(CVE_STATUS_REG);

    return value;
}

static bool cve_is_timeout_int(unsigned int status)
{
    if (status) {
        return true;
    }

    return false;
}

static void cve_set_reset(void)
{
    /* soft reset for all cve ops */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 1, 0, 1);
    /* release soft reset function */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 0, 1);

    return;
}

static void cve_op_run_time_cycle_enable(void)
{
    /*enable once op cycles calculate*/
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 1, 2, 1);
}

static unsigned int cve_get_op_run_time_cycle(void)
{
    /*get once op cycles*/
    return cve_reg_get_bits(CVE_OP_ONCE_RUN_CYCLES_REG, 0, 32);
}

static void cve_ute_stat_enable(unsigned int set_total_cycle)
{
    cve_reg_bits_set(CVE_UTE_REG0, set_total_cycle, 2, 30);
    cve_reg_bits_set(CVE_UTE_REG0, 1, 0, 2);
}

static unsigned int cve_get_ute_cycle_in_total(void)
{
    return cve_reg_get_bits(CVE_UTE_REG1, 0, 30);
}

#if 0
static void cve_clk_gate_enable(void)
{
    /* Clk gate control for cmd queue enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 4, 2);
    /* Clk gate control for reg_top enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 6, 2);
    /* Clk gate control for rdmif enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 8, 2);
    /* Clk gate control for rdmif1 enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 10, 2);
    /* Clk gate control for rdmif2 enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 12, 2);
    /* Clk gate control for wrmif enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 14, 2);
    /* Clk gate control for wrmif1 enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 16, 2);
    /* Clk gate control for wrmif2 enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 18, 2);
    /* Clk gate control for reg_top_hw enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 20, 2);
    /* Clk gate control for lut enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 22, 2);
    /* Clk gate control for interrupt enable */
    cve_reg_bits_set(CVE_TOP_HW_CTRL_REG0, 0, 24, 2);

    return;
}
#endif

extern unsigned int meson_ion_cma_heap_id_get(void);
extern unsigned int meson_ion_fb_heap_id_get(void);
extern unsigned int meson_ion_codecmm_heap_id_get(void);
extern void meson_ion_buffer_to_phys(struct ion_buffer *buffer, phys_addr_t *addr, size_t *len);
#define ION_FLAG_EXTEND_MESON_HEAP BIT(30)

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0)
int cve_ion_alloc_buffer(cve_ion_buffer_t *cve_buffer)
{
    int id;
    size_t len;
    struct dma_buf *dma_buf = NULL;
    struct ion_buffer *ionbuffer = NULL;
    enum dma_data_direction dir;
    int ret = -1;
    struct dma_buf_attachment *d_att = NULL;

    dir = DMA_TO_DEVICE;
    len = cve_buffer->size;
    id = meson_ion_fb_heap_id_get();
    if (id) {
        dma_buf = ion_alloc(len, (1 << id), ION_FLAG_EXTEND_MESON_HEAP);
    }
    if (IS_ERR_OR_NULL(dma_buf)) {
        id = meson_ion_cma_heap_id_get();
        dma_buf = ion_alloc(len, (1 << id), ION_FLAG_EXTEND_MESON_HEAP);
    }
    if (IS_ERR_OR_NULL(dma_buf)) {
        id = meson_ion_codecmm_heap_id_get();
        dma_buf = ion_alloc(len, (1 << id), ION_FLAG_EXTEND_MESON_HEAP);
    }
    if (IS_ERR(dma_buf)) {
        return -ENOMEM;
    }

    if (IS_ERR_OR_NULL(dma_buf)) {
        return PTR_ERR(dma_buf);
    }
    cve_buffer->dma_buf = dma_buf;
    ionbuffer = (struct ion_buffer *)(dma_buf->priv);
    cve_buffer->ionbuffer = ionbuffer;
    // sg_dma_address(ionbuffer->sg_table->sgl) = sg_phys(ionbuffer->sg_table->sgl);
    /*dma_sync_sg_for_device(cve_device.cve_pdev,
                           ionbuffer->sg_table->sgl,
                           ionbuffer->sg_table->nents,
                           dir);*/
    meson_ion_buffer_to_phys(ionbuffer, (phys_addr_t *)&cve_buffer->phys_start, &len);
    d_att = dma_buf_attach(dma_buf, cve_device.cve_pdev);
    if (!d_att) {
        CVE_ERR_TRACE("Failed to set dma attach");
        goto attach_err;
    }
    ret = dma_buf_begin_cpu_access(dma_buf, dir);
    if (ret != 0) {
        CVE_ERR_TRACE("Failed to access dma buff");
        goto access_err;
    }

    cve_buffer->virt_start = (unsigned long)ionbuffer->vaddr;
    cve_buffer->d_att = d_att;
    return 0;

access_err:
    dma_buf_detach(dma_buf, d_att);

attach_err:
    dma_buf_put(dma_buf); // free ion

    return ret;
}

void cve_ion_free_buffer(cve_ion_buffer_t *cve_buffer)
{
    if (cve_buffer->ionbuffer) {
        dma_buf_end_cpu_access(cve_buffer->dma_buf, DMA_TO_DEVICE);
        dma_buf_detach(cve_buffer->dma_buf, cve_buffer->d_att);

        dma_buf_put(cve_buffer->dma_buf); // ion_free
        cve_buffer->ionbuffer = NULL;
        cve_buffer->dma_buf = NULL;
    } else {
        CVE_ERR_TRACE("cve_ion_free_buffer cve_bufferis null\n");
    }
}
#else
int cve_ion_alloc_buffer(cve_ion_buffer_t *cve_buffer)
{
    int id;
    size_t len;
    struct dma_buf *dma_buf = NULL;
    struct ion_buffer *ionbuffer = NULL;

    len = cve_buffer->size;
    id = meson_ion_fb_heap_id_get();
    if (id) {
        dma_buf = ion_alloc(len, (1 << id), ION_FLAG_EXTEND_MESON_HEAP);
    }
    if (IS_ERR_OR_NULL(dma_buf)) {
        id = meson_ion_cma_heap_id_get();
        dma_buf = ion_alloc(len, (1 << id), ION_FLAG_EXTEND_MESON_HEAP);
    }
    if (IS_ERR_OR_NULL(dma_buf)) {
        id = meson_ion_codecmm_heap_id_get();
        dma_buf = ion_alloc(len, (1 << id), ION_FLAG_EXTEND_MESON_HEAP);
    }
    if (IS_ERR(dma_buf)) {
        return -ENOMEM;
    }

    if (IS_ERR_OR_NULL(dma_buf)) {
        return PTR_ERR(dma_buf);
    }
    cve_buffer->dma_buf = dma_buf;
    ionbuffer = (struct ion_buffer *)(dma_buf->priv);
    cve_buffer->ionbuffer = ionbuffer;
    meson_ion_buffer_to_phys(ionbuffer, (phys_addr_t *)&cve_buffer->phys_start, &len);
    dma_buf_vmap(dma_buf);
    cve_buffer->virt_start = (unsigned long)dma_buf->vmap_ptr;
    return 0;
}

void cve_ion_free_buffer(cve_ion_buffer_t *cve_buffer)
{
    if (cve_buffer->ionbuffer) {
        dma_buf_vunmap(cve_buffer->dma_buf, (void *)cve_buffer->virt_start);
        ion_free(cve_buffer->ionbuffer);
        cve_buffer->ionbuffer = NULL;
        cve_buffer->dma_buf = NULL;
    } else {
        CVE_ERR_TRACE("cve_ion_free_buffer cve_bufferis null\n");
    }
}
#endif

static long cve_dmabuf_get_phyaddr(int fd)
{
    unsigned long phy_addr = 0;
    struct dma_buf *dbuf = NULL;
    struct sg_table *table = NULL;
    struct page *page = NULL;
    struct dma_buf_attachment *attach = NULL;

    dbuf = dma_buf_get(fd);
    attach = dma_buf_attach(dbuf, cve_pdev);
    if (IS_ERR(attach))
        return 0;

    table = dma_buf_map_attachment(attach, DMA_BIDIRECTIONAL);
    page = sg_page(table->sgl);
    phy_addr = PFN_PHYS(page_to_pfn(page));
    dma_buf_unmap_attachment(attach, table, DMA_BIDIRECTIONAL);
    dma_buf_detach(dbuf, attach);
    dma_buf_put(dbuf);

    return phy_addr;
}

static void set_image_phy_addr(CVE_IMAGE_T *pstImg)
{
    unsigned long phyaddr;

    if (pstImg->dmafd == 0)
        return;
    phyaddr = cve_dmabuf_get_phyaddr(pstImg->dmafd);

    pstImg->au64PhyAddr[0] = phyaddr;
    switch (pstImg->enType) {
    case CVE_IMAGE_TYPE_S8C2_PLANAR:
    case CVE_IMAGE_TYPE_YUV420SP:
    case CVE_IMAGE_TYPE_YUV422SP: {
        pstImg->au64PhyAddr[1] =
            pstImg->au64PhyAddr[0] + (AML_U64)pstImg->au32Stride[0] * (AML_U64)pstImg->u32Height;
    } break;
    case CVE_IMAGE_TYPE_S8C2_PACKAGE: {
        pstImg->au64PhyAddr[1] = pstImg->au64PhyAddr[0] + 1;
    } break;
    case CVE_IMAGE_TYPE_U8C3_PACKAGE: {
        pstImg->au64PhyAddr[1] = pstImg->au64PhyAddr[0] + 1;
        pstImg->au64PhyAddr[2] = pstImg->au64PhyAddr[1] + 1;
    } break;
    case CVE_IMAGE_TYPE_YUV420P: {
        pstImg->au64PhyAddr[1] = pstImg->au64PhyAddr[0] + pstImg->au32Stride[0] * pstImg->u32Height;
        pstImg->au64PhyAddr[2] =
            pstImg->au64PhyAddr[1] + pstImg->au32Stride[1] * pstImg->u32Height / 2;
    } break;
    case CVE_IMAGE_TYPE_YUV422P:
    case CVE_IMAGE_TYPE_U8C3_PLANAR: {
        pstImg->au64PhyAddr[1] = pstImg->au64PhyAddr[0] + pstImg->au32Stride[0] * pstImg->u32Height;
        pstImg->au64PhyAddr[2] = pstImg->au64PhyAddr[1] + pstImg->au32Stride[1] * pstImg->u32Height;
    } break;
    default:
        break;
    }
}

static void set_raw_phy_addr(CVE_RAW_T *pstRaw)
{
    unsigned long phyaddr;

    phyaddr = cve_dmabuf_get_phyaddr(pstRaw->dmafd);
    pstRaw->u64PhyAddr = phyaddr;
}

static void set_mem_phy_addr(CVE_MEM_INFO_T *pstMemInfo)
{
    unsigned long phyaddr;

    phyaddr = cve_dmabuf_get_phyaddr(pstMemInfo->dmafd);
    pstMemInfo->u64PhyAddr = phyaddr;
}

static void set_data_phy_addr(CVE_DATA_T *pstData)
{
    unsigned long phyaddr;

    phyaddr = cve_dmabuf_get_phyaddr(pstData->dmafd);
    pstData->u64PhyAddr = phyaddr;
}

static void *cve_vmap(unsigned long phys, unsigned int size)
{
    pgprot_t pgprot;
    unsigned int npages, offset;
    struct page **pages = NULL;
    static unsigned char *vaddr = 0;
    int i;

    npages = PAGE_ALIGN(size) / PAGE_SIZE;
    offset = phys & ~PAGE_MASK;
    if ((offset + size) > PAGE_SIZE)
        npages++;
    pages = vmalloc(sizeof(struct page *) * npages);
    if (!pages)
        return NULL;
    for (i = 0; i < npages; i++) {
        pages[i] = phys_to_page(phys);
        phys += PAGE_SIZE;
    }
    /*nocache*/
    pgprot = pgprot_writecombine(PAGE_KERNEL);
    vaddr = vmap(pages, npages, VM_MAP, pgprot);
    if (!vaddr) {
        CVE_ERR_TRACE("the phy(%lx) vmaped fail, size: %d\n", phys, npages << PAGE_SHIFT);
        vfree(pages);
        return NULL;
    }
    vfree(pages);
    return (void *)vaddr;
}

static void cve_vunmap(void *virt)
{
    static void *vaddr;
    vaddr = (void *)(PAGE_MASK & (ulong)virt);
    vunmap(vaddr);
}

static char *cmd_buf_get(void)
{
    cve_cmd_buf_t *cmd_bufs = &cve_context.cmd_bufs;
    char *cmd_buf;

    if (cmd_bufs == NULL) {
        CVE_ERR_TRACE("cmd_bufs is NULL\n");
        return NULL;
    }

    if (down_interruptible(&cmd_bufs->cmd_buf_sema)) {
        return NULL;
    }
    if ((cmd_bufs->cur_index + 1 > cve_node_num)) {
        cmd_bufs->cur_index = 0;
    }
    cmd_buf = (char *)cmd_bufs->virt_start + cmd_bufs->cmd_size * cmd_bufs->cur_index;
    cmd_bufs->cur_index++;

    up(&cmd_bufs->cmd_buf_sema);

    return cmd_buf;
}

static bool cve_task_io_process_record(cve_task_desc_t *task_desc, cve_op_io_info_t *info,
                                       bool check_only)
{
    if (info->inp_flags == 0 && info->outp_flags == 0) {
        return true;
    }

    if (task_desc->bOutput || task_desc->bInput) {
        if (task_desc->output_process_flags & info->outp_flags) {
            return false;
        }
        if (task_desc->input_process_flags & info->inp_flags) {
            return false;
        }
        if ((task_desc->output_process_flags & HIST_OUTPUT_PROCESS_LUT) &&
            (info->inp_flags == EQUALIZE_HIST_INPUT_PROCESS_LUT)) {
            return false;
        }
        if ((task_desc->input_process_flags & EQUALIZE_HIST_INPUT_PROCESS_LUT) &&
            (info->outp_flags == HIST_OUTPUT_PROCESS_LUT)) {
            return false;
        }
    }

    if (check_only) {
        return true;
    }

    if (info->inp_flags) {
        task_desc->input_process_flags |= info->inp_flags;
        task_desc->bInput = AML_TRUE;
    }
    if (info->outp_flags) {
        task_desc->output_process_flags |= info->outp_flags;
        task_desc->bOutput = AML_TRUE;
    }

    return true;
}

static int cve_manage_handle(CVE_HANDLE *cveHandle, unsigned int cmd_num)
{
    cve_task_info_t *task_info = &cve_context.task_info;

    if (task_info->cmd_handle + cmd_num > CMD_HANDLE_MAX) {
        task_info->cmd_handle = task_info->cmd_handle + cmd_num - CMD_HANDLE_MAX;
        task_info->cmd_handle_wrap++;
    } else {
        task_info->cmd_handle = task_info->cmd_handle + cmd_num;
    }

    *cveHandle = task_info->cmd_handle - 1;

    return 0;
}

static void task_descs_init(cve_task_desc_t *task_desc)
{
    task_desc->total_cmd_num = 0;
    task_desc->total_cmd_line_num = 0;
    task_desc->bOutput = AML_FALSE;
    task_desc->bInput = AML_FALSE;
    task_desc->bInvalid = AML_FALSE;
    task_desc->input_process_flags = 0;
    task_desc->output_process_flags = 0;
    INIT_LIST_HEAD(&task_desc->cmd_list);

    return;
}

static int cve_create_task(cve_cmd_desc_t *cmd_desc, char *cmd_buf)
{
    cve_cq_desc_t *cq_desc_wait;
    cve_task_desc_t *task_desc;

    AML_ASSERT(cmd_desc != NULL);
    AML_ASSERT(cmd_buf != NULL);
    AML_ASSERT((cmd_desc->instant == AML_TRUE) || (cmd_desc->instant == AML_FALSE));

    cq_desc_wait = &cve_context.cq_desc[cve_context.queue_wait];
    if (cq_desc_wait->end_cmd_id >= cve_node_num) {
        CVE_ERR_TRACE("the waiting queue is full of cmd nodes! end_cmd_id = %d,max cmd nodes = %d, "
                      "need 1 node\n",
                      cq_desc_wait->end_cmd_id, cve_node_num);
        return AML_ERR_CVE_BUF_FULL;
    }

    task_desc = &cq_desc_wait->task_descs[cq_desc_wait->task_descs_create_index];
    if ((task_desc->bInvalid == AML_FALSE) && (task_desc->total_cmd_num != 0)) {
        if (cq_desc_wait->task_instant == 1 ||
            task_desc->total_cmd_num >= TASK_CMD_MAX ||
            !cve_task_io_process_record(task_desc, &cmd_desc->io_info, true)) {
            cq_desc_wait->task_descs_create_index++;
            cq_desc_wait->task_virt_offset += task_desc->total_cmd_num * CVE_OP_NODE_CMD_SIZE_MAX;
            cq_desc_wait->task_phys_offset += task_desc->total_cmd_num * CVE_OP_NODE_CMD_SIZE_MAX;
            task_desc = &cq_desc_wait->task_descs[cq_desc_wait->task_descs_create_index];
            task_descs_init(task_desc);
        }
    } else {
        task_descs_init(task_desc);
    }

    task_desc->virt_addr = cq_desc_wait->virt_start + cq_desc_wait->task_virt_offset;
    task_desc->phys_addr = cq_desc_wait->phys_start + cq_desc_wait->task_phys_offset;

    memcpy(((char *)(uintptr_t)task_desc->virt_addr +
            task_desc->total_cmd_line_num * TASK_CMD_LINE_SIZE),
           cmd_buf, cmd_desc->cmd_line_num * TASK_CMD_LINE_SIZE);
    cmd_desc->virt_addr = task_desc->virt_addr + task_desc->total_cmd_line_num * TASK_CMD_LINE_SIZE;
    cmd_desc->phys_addr = task_desc->phys_addr + task_desc->total_cmd_line_num * TASK_CMD_LINE_SIZE;
    cmd_desc->task_desc = task_desc;
    list_add_tail(&cmd_desc->list, &task_desc->cmd_list);
    task_desc->total_cmd_line_num += cmd_desc->cmd_line_num;
    task_desc->total_cmd_num++;

    cve_task_io_process_record(task_desc, &cmd_desc->io_info, false);
    cq_desc_wait->task_instant = cmd_desc->instant;

    cq_desc_wait->end_cmd_id++;

    return 0;
}

static int cve_create_task_multi_cmd(cve_cmd_desc_t *cmd_desc, char *cmd_buf_arr[],
                                     unsigned int cmd_line_num_arr[], unsigned int cmd_num)
{
    cve_cq_desc_t *cq_desc_wait;
    cve_task_desc_t *task_desc;
    int i = 0;
    int ofs = 0;

    AML_ASSERT(cmd_desc != NULL);
    AML_ASSERT(cmd_buf_arr != NULL);
    AML_ASSERT((cmd_desc->instant == AML_TRUE) || (cmd_desc->instant == AML_FALSE));

    cq_desc_wait = &cve_context.cq_desc[cve_context.queue_wait];
    if (cq_desc_wait->end_cmd_id + cmd_num >= cve_node_num) {
        CVE_ERR_TRACE("the waiting queue is full of cmd nodes! end_cmd_id = %d,max cmd nodes = %d "
                      "need %d nodes\n",
                      cq_desc_wait->end_cmd_id, cve_node_num, cmd_num);
        return AML_ERR_CVE_BUF_FULL;
    }

    task_desc = &cq_desc_wait->task_descs[cq_desc_wait->task_descs_create_index];
    if (cq_desc_wait->task_instant == 1 ||
        task_desc->total_cmd_num >= TASK_CMD_MAX ||
        !cve_task_io_process_record(task_desc, &cmd_desc->io_info, true)) {
        cq_desc_wait->task_descs_create_index++;
        cq_desc_wait->task_virt_offset += task_desc->total_cmd_line_num * TASK_CMD_LINE_SIZE;
        cq_desc_wait->task_phys_offset += task_desc->total_cmd_line_num * TASK_CMD_LINE_SIZE;
        task_desc = &cq_desc_wait->task_descs[cq_desc_wait->task_descs_create_index];
    }
    if (task_desc->bInvalid == AML_TRUE) {
        task_desc->virt_addr = cq_desc_wait->virt_start + cq_desc_wait->task_virt_offset;
        task_desc->phys_addr = cq_desc_wait->phys_start + cq_desc_wait->task_phys_offset;
        task_desc->total_cmd_num = 0;
        task_desc->total_cmd_line_num = 0;
        task_desc->bOutput = AML_FALSE;
        task_desc->bInput = AML_FALSE;
        task_desc->bInvalid = AML_FALSE;
        task_desc->input_process_flags = 0;
        task_desc->output_process_flags = 0;
        INIT_LIST_HEAD(&task_desc->cmd_list);
    }
    task_desc->virt_addr = cq_desc_wait->virt_start + cq_desc_wait->task_virt_offset;
    task_desc->phys_addr = cq_desc_wait->phys_start + cq_desc_wait->task_phys_offset;

    for (i = 0; i < cmd_num; i++) {
        memcpy((char *)(uintptr_t)task_desc->virt_addr + ofs, cmd_buf_arr[i],
               cmd_desc->cmd_line_num * 8);
        ofs += cmd_line_num_arr[i] * TASK_CMD_LINE_SIZE;
    }

    cmd_desc->virt_addr = task_desc->virt_addr + task_desc->total_cmd_line_num * TASK_CMD_LINE_SIZE;
    cmd_desc->phys_addr = task_desc->phys_addr + task_desc->total_cmd_line_num * TASK_CMD_LINE_SIZE;
    cmd_desc->task_desc = task_desc;
    list_add_tail(&cmd_desc->list, &task_desc->cmd_list);
    task_desc->total_cmd_line_num += cmd_desc->cmd_line_num;
    task_desc->total_cmd_num += cmd_num;
    cve_task_io_process_record(task_desc, &cmd_desc->io_info, false);
    cq_desc_wait->task_instant = cmd_desc->instant;
    cq_desc_wait->end_cmd_id += cmd_num;

    return 0;
}

static void cve_start_task(cve_task_desc_t *task_desc)
{
    cve_reg_write(CVE_CQ_REG0, task_desc->phys_addr >> 4);
    cve_reg_bits_set(CVE_CQ_REG2, task_desc->total_cmd_line_num, 0, 12);
    cve_reg_bits_set(CVE_CQ_REG3, task_desc->total_cmd_num, 0, 9);

    cve_cq0_int_enable();
    /* for dma luma stat */
    cve_cq0_function_enable();
    /* enable cq0 */
    cve_reg_bits_set(CVE_CQ_REG2, 1, 24, 1);
}

static void cve_ute_time_cycle(void)
{
    unsigned int tm = 0;
    cve_run_time_cycle_t *cve_cycle = &cve_context.run_time_info.cve_cycle;
    cve_cycle->cve_ute_cycles = cve_get_ute_cycle_in_total();
    tm = cve_cycle->cve_ute_cycles * 1 / 800;
    cve_context.run_time_info.total_cve_run_time = tm;
}

static void cve_op_time_cycle(cve_op_io_info_t *info)
{
    unsigned int tm = 0;
    unsigned int tmax = 0;
    cve_run_time_cycle_t *cve_cycle = &cve_context.run_time_info.cve_cycle;
    cve_cycle->cve_once_run_cycles = cve_get_op_run_time_cycle();
    tm = cve_cycle->cve_once_run_cycles * 1 / 800;
    if (tm > tmax)
        tmax = tm;
    switch (info->op_type) {
    case CVE_OP_TYPE_DMA: {
        if (info->op_mode == CVE_MODE_DMA) {
            cve_context.run_time_info.cve_op_cycle.dma = tmax;
        } else if (info->op_mode == CVE_MODE_LUMA_STAT) {
            cve_context.run_time_info.cve_op_cycle.luma_stat = tmax;
        }
        break;
    }
    case CVE_OP_TYPE_FILTER: {
        cve_context.run_time_info.cve_op_cycle.filter = tmax;
        break;
    }
    case CVE_OP_TYPE_CSC: {
        cve_context.run_time_info.cve_op_cycle.csc = tmax;
        break;
    }
    case CVE_OP_TYPE_FILTER_AND_CSC: {
        cve_context.run_time_info.cve_op_cycle.filter_and_csc = tmax;
        break;
    }
    case CVE_OP_TYPE_SOBEL: {
        cve_context.run_time_info.cve_op_cycle.sobel = tmax;
        break;
    }
    case CVE_OP_TYPE_MAG_AND_ANG: {
        cve_context.run_time_info.cve_op_cycle.mag_and_ang = tmax;
        break;
    }
    case CVE_OP_TYPE_MATCH_BG_MODEL: {
        cve_context.run_time_info.cve_op_cycle.match_bg_model = tmax;
        break;
    }
    case CVE_OP_TYPE_DILATE_AND_ERODE: {
        if (info->op_mode == CVE_MODE_DILATE) {
            cve_context.run_time_info.cve_op_cycle.dilate = tmax;
        } else if (info->op_mode == CVE_MODE_ERODE) {
            cve_context.run_time_info.cve_op_cycle.erode = tmax;
        }
        break;
    }
    case CVE_OP_TYPE_THRESH: {
        cve_context.run_time_info.cve_op_cycle.thresh = tmax;
        break;
    }
    case CVE_OP_TYPE_ALU: {
        switch (info->op_mode) {
        case CVE_ALU_MODE_AND: {
            cve_context.run_time_info.cve_op_cycle.and = tmax;
            break;
        }
        case CVE_ALU_MODE_OR: {
            cve_context.run_time_info.cve_op_cycle.or = tmax;
            break;
        }
        case CVE_ALU_MODE_XOR: {
            cve_context.run_time_info.cve_op_cycle.xor = tmax;
            break;
        }
        case CVE_ALU_MODE_ADD: {
            cve_context.run_time_info.cve_op_cycle.add = tmax;
            break;
        }
        case CVE_ALU_MODE_SUB: {
            cve_context.run_time_info.cve_op_cycle.sub = tmax;
            break;
        }
        }
        break;
    }
    case CVE_OP_TYPE_INTEG: {
        cve_context.run_time_info.cve_op_cycle.integ = tmax;
        break;
    }
    case CVE_OP_TYPE_HIST: {
        cve_context.run_time_info.cve_op_cycle.hist = tmax;
        break;
    }
    case CVE_OP_TYPE_THRESH_S16: {
        cve_context.run_time_info.cve_op_cycle.thresh_s16 = tmax;
        break;
    }
    case CVE_OP_TYPE_THRESH_U16: {
        cve_context.run_time_info.cve_op_cycle.thresh_u16 = tmax;
        break;
    }
    case CVE_OP_TYPE_16BIT_TO_8BIT: {
        cve_context.run_time_info.cve_op_cycle._16bit_to_8bit = tmax;
        break;
    }
    case CVE_OP_TYPE_ORD_STAT_FILTER: {
        cve_context.run_time_info.cve_op_cycle.ord_stat_filter = tmax;
        break;
    }
    case CVE_OP_TYPE_MAP: {
        cve_context.run_time_info.cve_op_cycle.map = tmax;
        break;
    }
    case CVE_OP_TYPE_EQUALIZE_HIST: {
        cve_context.run_time_info.cve_op_cycle.equalize_hist = tmax;
        break;
    }
    case CVE_OP_TYPE_NCC: {
        cve_context.run_time_info.cve_op_cycle.ncc = tmax;
        break;
    }
    case CVE_OP_TYPE_CCL: {
        cve_context.run_time_info.cve_op_cycle.ccl = tmax;
        break;
    }
    case CVE_OP_TYPE_GMM: {
        cve_context.run_time_info.cve_op_cycle.gmm = tmax;
        break;
    }
    case CVE_OP_TYPE_CANNY_HYS_EDGE: {
        cve_context.run_time_info.cve_op_cycle.canny_edge = tmax;
        break;
    }
    case CVE_OP_TYPE_LBP: {
        cve_context.run_time_info.cve_op_cycle.lbp = tmax;
        break;
    }
    case CVE_OP_TYPE_NROM_GRAD: {
        cve_context.run_time_info.cve_op_cycle.nrom_grad = tmax;
        break;
    }
    case CVE_OP_TYPE_BULID_LK_OPTICAL_FLOW_PYR: {
        cve_context.run_time_info.cve_op_cycle.bulid_lk_optical_flow_pyr = tmax;
        break;
    }
    case CVE_OP_TYPE_LK_OPTIAL_FLOW_PRY: {
        cve_context.run_time_info.cve_op_cycle.lk_optial_flow_pry = tmax;
        break;
    }
    case CVE_OP_TYPE_ST_CANDI_CORNER: {
        cve_context.run_time_info.cve_op_cycle.st_candi_corner = tmax;
        break;
    }
    case CVE_OP_TYPE_SAD: {
        cve_context.run_time_info.cve_op_cycle.sad = tmax;
        break;
    }
    case CVE_OP_TYPE_GRAD_FG: {
        cve_context.run_time_info.cve_op_cycle.grad_fg = tmax;
        break;
    }
    case CVE_OP_TYPE_UPDATE_BG_MODEL: {
        cve_context.run_time_info.cve_op_cycle.update_bg_model = tmax;
        break;
    }
    case CVE_OP_TYPE_TOF: {
        cve_context.run_time_info.cve_op_cycle.tof = tmax;
        break;
    }
    default: {
        CVE_ERR_TRACE(" not come op time \n");
        break;
    }
    }
}

extern void ktime_get_real_ts64(struct timespec64 *tv);
static unsigned long long cve_get_sys_time(void)
{
    unsigned long long cur_time;
    struct timespec64 ts;

    ktime_get_real_ts64(&ts);
    cur_time = ts.tv_nsec / 1000 + ts.tv_sec * 1000000;

    return cur_time;
}

static void cve_irq_time_cycle(void)
{
    unsigned long long costTm = 0;
    unsigned long long costTmax = 0;

    tasklet_end_time = cve_get_sys_time();
    costTm = tasklet_end_time - tasklet_start_time;
    if (costTm > costTmax) {
        costTmax = costTm;
        cve_context.run_time_info.int_cost_time_max = costTmax;
    }
    cve_context.run_time_info.last_int_cost_time = costTm;
}

static void cve_reset_driver(void)
{
    int i;

    cve_set_reset();
    cve_timeout_flag = true;
    cve_context.queue_wait = CVE_STATUS_CQ0;
    cve_context.queue_busy = CVE_STATUS_IDLE;
    cve_context.run_time_info.system_timeout_cnt++;
    cve_context.run_time_info.last_instant = false;
    for (i = 0; i < 2; i++) {
        cve_context.cq_desc[i].task_phys_offset = 0;
        cve_context.cq_desc[i].task_virt_offset = 0;
        cve_context.cq_desc[i].cur_cmd_id = 0;
        cve_context.cq_desc[i].end_cmd_id = 0;
        cve_context.cq_desc[i].task_descs_create_index = 0;
        cve_context.cq_desc[i].task_descs_invoke_index = 0;
        cve_context.cq_desc[i].task_instant = 0;
        cve_context.cq_desc[i].cmd_descs_index = 0;
    }
    memset(&cve_context.task_info, 0, sizeof(cve_task_info_t));
}

static void cve_input_process(cve_op_io_info_t *info)
{
    void *pvKvirt = cve_vmap(info->inp_phys_addr, info->inp_size);

    if (pvKvirt == NULL) {
        CVE_ERR_TRACE("rmmap failed\n");
        return;
    }

    switch (info->op_type) {

    case CVE_OP_TYPE_EQUALIZE_HIST: {
        unsigned char *pau8curv;
        int i = 0;

        pau8curv = (unsigned char *)pvKvirt;
        cve_reg_write(CVE_LUT_REG0, CVE_EQHIST_AU8CURV_LUT);
        for (i = 0; i < CVE_HIST_NUM; i++) {
            cve_reg_write(CVE_LUT_REG1, pau8curv[i]);
        }
        cve_reg_write(CVE_LUT_REG0, CVE_LUT_DONE);

        break;
    }
    case CVE_OP_TYPE_MAP: {
        unsigned char *pau8map;
        unsigned short *pau16map;
        short *pas16map;

        CVE_MAP_MODE_E enMode = (CVE_MAP_MODE_E)info->op_mode;
        int i = 0;

        cve_reg_write(CVE_LUT_REG0, CVE_MAP_AU16MAP_LUT);
        if (enMode == CVE_MAP_MODE_U8) {
            pau8map = (unsigned char *)pvKvirt;
            for (i = 0; i < CVE_MAP_NUM; i++) {
                cve_reg_write(CVE_LUT_REG1, pau8map[i]);
            }
        } else if (enMode == CVE_MAP_MODE_U16) {
            pau16map = (unsigned short *)pvKvirt;
            for (i = 0; i < CVE_MAP_NUM; i++) {
                cve_reg_write(CVE_LUT_REG1, pau16map[i]);
            }
        } else {
            pas16map = (short *)pvKvirt;
            for (i = 0; i < CVE_MAP_NUM; i++) {
                cve_reg_write(CVE_LUT_REG1, pas16map[i]);
            }
        }
        cve_reg_write(CVE_LUT_REG0, CVE_LUT_DONE);

        break;
    }
    case CVE_OP_TYPE_TOF: {
        unsigned int *phist;
        int i = 0;

        cve_reg_write(CVE_LUT_REG0, CVE_TOF_BPC_LUT);
        phist = (unsigned int *)pvKvirt;
        for (i = 0; i < CVE_MAP_NUM; i++) {
            cve_reg_write(CVE_LUT_REG1, phist[i]);
        }
        cve_reg_write(CVE_LUT_REG0, CVE_LUT_DONE);

        break;
    }
    default:
        break;
    }

    cve_vunmap(pvKvirt);

    return;
}

static void cve_output_process(cve_op_io_info_t *info)
{

    void *pvKvirt = cve_vmap(info->outp_phys_addr, info->outp_size);

    if (pvKvirt == NULL) {
        CVE_ERR_TRACE("rmmap failed\n");
        return;
    }
    switch (info->op_type) {

    case CVE_OP_TYPE_ALU: {
        unsigned int *psubadd;
        psubadd = (unsigned int *)pvKvirt;
        if (info->outp_flags == SUB_OUTPUT_PROCESS_REG) {
            psubadd[0] = cve_reg_read(CVE_SUB_AREA_REG);
        } else if (info->outp_flags == ADD_OUTPUT_PROCESS_REG) {
            psubadd[0] = cve_reg_read(CVE_ADD_DSTSUM_REG);
        }
        break;
    }
    case CVE_OP_TYPE_HIST: {
        unsigned int *pau21hist;
        unsigned char *pau8curv;
        int i = 0;

        pau21hist = (unsigned int *)pvKvirt;
        cve_reg_write(CVE_LUT_REG0, CVE_HIST_LUT);
        for (i = 0; i < CVE_HIST_NUM; i++) {
            pau21hist[i] = cve_reg_read(CVE_LUT_REG1);
        }
        cve_reg_write(CVE_LUT_REG0, CVE_LUT_DONE);

        pau8curv = (unsigned char *)(pau21hist + CVE_HIST_NUM);
        cve_reg_write(CVE_LUT_REG0, CVE_EQHIST_AU8CURV_LUT);
        for (i = 0; i < CVE_HIST_NUM; i++) {
            pau8curv[i] = (unsigned char)cve_reg_read(CVE_LUT_REG1);
        }
        cve_reg_write(CVE_LUT_REG0, CVE_LUT_DONE);
        break;
    }
    case CVE_OP_TYPE_NCC: {
        CVE_NCC_DST_MEM_T *pstNccMem = (CVE_NCC_DST_MEM_T *)pvKvirt;
        pstNccMem->u64Numerator = cve_reg_read(CVE_NCC_U64NUM_REG0);
        pstNccMem->u64Numerator += (AML_U64)cve_reg_read(CVE_NCC_U64NUM_REG1) << 32;
        pstNccMem->u64QuadSum1 = cve_reg_read(CVE_NCC_U64QUADSUM1_REG0);
        pstNccMem->u64QuadSum1 += (AML_U64)cve_reg_read(CVE_NCC_U64QUADSUM2_REG0) << 32;
        pstNccMem->u64QuadSum2 = cve_reg_read(CVE_NCC_U64QUADSUM2_REG0);
        pstNccMem->u64QuadSum2 += (AML_U64)cve_reg_read(CVE_NCC_U64QUADSUM2_REG1) << 32;
        break;
    }
    case CVE_OP_TYPE_CCL: {
        CVE_CCBLOB_T *pstBlob = (CVE_CCBLOB_T *)pvKvirt;

        pstBlob->u16RegionNum = (unsigned short)cve_reg_read(CVE_CCL_REG0);
        break;
    }
    case CVE_OP_TYPE_MATCH_BG_MODEL: {
        CVE_BG_STAT_DATA_T *pstBgStatData = (CVE_BG_STAT_DATA_T *)pvKvirt;

        pstBgStatData->u32PixNum = cve_reg_read(CVE_BGMODE_REG1);
        pstBgStatData->u32SumLum = cve_reg_read(CVE_BGMODE_REG2);
        break;
    }
    case CVE_OP_TYPE_UPDATE_BG_MODEL: {
        CVE_BG_STAT_DATA_T *pstBgStatData = (CVE_BG_STAT_DATA_T *)pvKvirt;

        pstBgStatData->u32PixNum = cve_reg_read(CVE_UPDATEBGMODE_REG2);
        pstBgStatData->u32SumLum = cve_reg_read(CVE_UPDATEBGMODE_REG3);
        break;
    }
    case CVE_OP_TYPE_LK_OPTIAL_FLOW_PRY: {
        unsigned int *lk_status;
        int i = 0;
        lk_status = (unsigned int *)pvKvirt;
        for (i = 0; i < 17; i++) {
            lk_status[i] = cve_reg_read(CVE_LK_REG2_0 + i);
        }
        break;
    }
    case CVE_OP_TYPE_ST_CANDI_CORNER: {
        unsigned int *cn = (unsigned int *)pvKvirt;
        *cn = cve_reg_read(CVE_STCORNER_REG0);
        break;
    }
    case CVE_OP_TYPE_CANNY_HYS_EDGE: {
        unsigned int *cc = (unsigned int *)pvKvirt;
        *cc = cve_reg_read(CVE_CANNY_REG1);
        break;
    }
    case CVE_OP_TYPE_TOF: {
        unsigned int *phist;
        unsigned int v32;
        int i = 0;

        phist = (unsigned int *)pvKvirt;
        cve_reg_write(CVE_LUT_REG0, CVE_TOF_IR_HIST_LUT);
        for (i = 0; i < CVE_HIST_NUM; i++) {
            v32 = cve_reg_read(CVE_LUT_REG1);
            phist[i] = v32;
        }
        cve_reg_write(CVE_LUT_REG0, CVE_LUT_DONE);
        phist = (unsigned int *)(phist + CVE_HIST_NUM);
        cve_reg_write(CVE_LUT_REG0, CVE_TOF_Q2_HIST_LUT);
        for (i = 0; i < CVE_HIST_NUM; i++) {
            v32 = cve_reg_read(CVE_LUT_REG1);
            phist[i] = v32;
        }
        cve_reg_write(CVE_LUT_REG0, CVE_LUT_DONE);
        phist = (unsigned int *)(phist + CVE_HIST_NUM * 2);
        cve_reg_write(CVE_LUT_REG0, CVE_TOF_Q3_HIST_LUT);
        for (i = 0; i < CVE_HIST_NUM; i++) {
            v32 = cve_reg_read(CVE_LUT_REG1);
            phist[i] = v32;
        }
        cve_reg_write(CVE_LUT_REG0, CVE_LUT_DONE);
    }
    default:
        break;
    }
    cve_vunmap(pvKvirt);

    return;
}
static int cve_start_process(cve_cq_desc_t *cq_desc_busy)
{
    cve_task_desc_t *task_desc;
    cve_cmd_desc_t *cmd_desc_tmp;

    task_desc = &cq_desc_busy->task_descs[cq_desc_busy->task_descs_invoke_index];
    if (task_desc->bInput) {
        list_for_each_entry(cmd_desc_tmp, &task_desc->cmd_list, list)
        {
            if (cmd_desc_tmp->io_info.inp_flags != 0 && cmd_desc_tmp->io_info.inp_phys_addr != 0) {
                cve_input_process(&cmd_desc_tmp->io_info);
            }
        }
    }

    tasklet_start_time = cve_get_sys_time();
    cve_start_task(task_desc);
    return 0;
}

static void cve_cq_updata(void)
{
    cve_task_info_t *task_info;
    cve_cq_desc_t *cq_desc_busy;
    cve_cq_desc_t *cq_desc_wait;
    unsigned long flags;

    cq_desc_busy = &cve_context.cq_desc[cve_context.queue_busy];
    cq_desc_wait = &cve_context.cq_desc[cve_context.queue_wait];
    task_info = &cve_context.task_info;

    if (cq_desc_busy->cur_cmd_id > cq_desc_busy->end_cmd_id) {
        panic("\nASSERT at:\n	>Function : %s\n  >Line No. : %d\n	>Condition: %s\n",
              __FUNCTION__, __LINE__, "busy_cur_cmd_id <= busy_end_cmd_id");
    }

    if (cq_desc_busy->cur_cmd_id == cq_desc_busy->end_cmd_id) {
        cq_desc_busy->task_phys_offset = 0;
        cq_desc_busy->task_virt_offset = 0;
        cq_desc_busy->cur_cmd_id = 0;
        cq_desc_busy->end_cmd_id = 0;
        cq_desc_busy->task_descs_create_index = 0;
        cq_desc_busy->task_descs_invoke_index = 0;
        cq_desc_busy->task_instant = 0;
        /*FIX ME: If you want to zero out, you need to synchronize with the task creation
         * process*/
        cq_desc_busy->cmd_descs_index = 0;
        task_info->cur_finish_cmd_id = 0;
        task_info->last_finish_cmd_id = 0;

        if (cq_desc_wait->cur_cmd_id > cq_desc_wait->end_cmd_id) {
            panic("\nASSERT at:\n	>Function : %s\n  >Line No. : %d\n	>Condition: %s\n",
                  __FUNCTION__, __LINE__, "wait_cur_cmd_id <= wait_end_cmd_id");
        }
        spin_lock_irqsave(&cve_spinlock, flags);
        if (cq_desc_wait->cur_cmd_id == cq_desc_wait->end_cmd_id) {
            cve_context.queue_busy = CVE_STATUS_IDLE;
        } else {
            cve_context.queue_busy = cve_context.queue_wait;
            cve_context.queue_wait = CVE_STATUS_CQ1 - cve_context.queue_wait;
        }
        spin_unlock_irqrestore(&cve_spinlock, flags);
    } else {
        task_info->last_finish_cmd_id = task_info->cur_finish_cmd_id + 1;
    }
}

static irqreturn_t cve_irq_handler_thread(int irq, void *work)
{
    cve_task_desc_t *task_desc;
    cve_cmd_desc_t *cmd_desc;
    task_desc = cve_context.task_desc_outp;

    if (task_desc) {
        list_for_each_entry(cmd_desc, &task_desc->cmd_list, list)
        {
            if (cmd_desc->io_info.outp_flags != 0 && cmd_desc->io_info.outp_phys_addr != 0) {
                cve_output_process(&cmd_desc->io_info);
            }
        }
        INIT_LIST_HEAD(&task_desc->cmd_list);
        cve_context.task_desc_outp = NULL;
        cve_context.task_info.cmd_finish_cnt += task_desc->total_cmd_num;
        cve_cq_updata();
        wake_up(&cve_context.cve_wait);
    }
    if (cve_context.queue_busy != CVE_STATUS_IDLE) {
        cve_start_process(&cve_context.cq_desc[cve_context.queue_busy]);
    }

    return IRQ_HANDLED;
}

static int cve_continue_task(void)
{
    cve_task_info_t *task_info;
    cve_cq_desc_t *cq_desc_busy;
    cve_task_desc_t *task_desc;
    cve_cmd_desc_t *cmd_desc;
    int ret = IRQ_HANDLED;

    task_info = &cve_context.task_info;

    if (cve_context.queue_busy != CVE_STATUS_IDLE) {
        cq_desc_busy = &cve_context.cq_desc[cve_context.queue_busy];
        if (cve_context.queue_wait != CVE_STATUS_CQ0 && cve_context.queue_wait != CVE_STATUS_CQ1) {
            panic("\nASSERT at:\n  >Function : %s\n  >Line No. : %d\n  >Condition: %s\n",
                  __FUNCTION__, __LINE__,
                  "(queue_wait == CVE_WAIT_STATUS_CQ0) || (queue_wait == CVE_WAIT_STATUS_CQ1)");
        }
        cve_context.run_time_info.total_int_cnt++;
        task_desc = &cq_desc_busy->task_descs[cq_desc_busy->task_descs_invoke_index++];
        task_desc->bInvalid = AML_TRUE;

        task_info->cur_finish_cmd_id += task_desc->total_cmd_num;

        if (task_info->cmd_finish_cnt > CMD_HANDLE_MAX) {
            task_info->cmd_finish_cnt -= CMD_HANDLE_MAX;
            task_info->finish_cmd_wrap++;
        }

        cq_desc_busy->cur_cmd_id += task_desc->total_cmd_num;

        if (cve_context.run_time_info.cve_cycle.cve_ute_stat_en == true) {
            cve_ute_time_cycle();
        }
        if (cve_context.run_time_info.cve_cycle.cve_dbg_run_cycles_en == true &&
            task_desc->total_cmd_num == 1) {
            if (!list_empty(&task_desc->cmd_list)) {
                cmd_desc = list_first_entry(&task_desc->cmd_list, cve_cmd_desc_t, list);
                cve_op_time_cycle(&cmd_desc->io_info);
            }
        }

        if (task_desc->bOutput) {
            cve_context.task_desc_outp = task_desc;
            ret = IRQ_WAKE_THREAD;
        } else {
            task_info->cmd_finish_cnt += task_desc->total_cmd_num;
            cve_cq_updata();
            wake_up(&cve_context.cve_wait);
            if (cve_context.queue_busy != CVE_STATUS_IDLE) {
                cq_desc_busy = &cve_context.cq_desc[cve_context.queue_busy];
                task_desc = &cq_desc_busy->task_descs[cq_desc_busy->task_descs_invoke_index];
                if (task_desc->bInput) {
                    ret = IRQ_WAKE_THREAD;
                } else {
                    tasklet_start_time = cve_get_sys_time();
                    cve_start_task(task_desc);
                    ret = IRQ_HANDLED;
                }
            } else {
                ret = IRQ_HANDLED;
            }
        }
    }

    return ret;
}

static irqreturn_t cve_irq_handler(int handle, void *data)
{
    unsigned int status;
    int ret = IRQ_HANDLED;
    status = cve_int_status_get();
    cve_cq0_int_clear();
    /*for dma luma stat*/
    cve_cq0_function_disable();
    cve_irq_time_cycle();
    if (!cve_is_timeout_int(status)) {
        ret = cve_continue_task();
    } else {
        cve_reset_driver();
        CVE_ERR_TRACE("wait task finish time out, reset cve driver\n");
    }
    return ret;
}

static int request_op_cmd(cve_cmd_desc_t **cmd_desc, unsigned int cmd_num)
{
    cve_cq_desc_t *cq_desc_wait;

    if (cmd_desc == NULL) {
        CVE_ERR_TRACE("cmd_desc is NULL\n");
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }

    cq_desc_wait = &cve_context.cq_desc[cve_context.queue_wait];

    if (cq_desc_wait->cmd_descs_index + cmd_num > cve_node_num) {
        CVE_ERR_TRACE("Waiting queue is full of task! end_cmd_id = %d,max task = %d\n",
                      cq_desc_wait->cmd_descs_index + cmd_num, cve_node_num);
        return AML_ERR_CVE_BUF_FULL;
    }

    *cmd_desc = &cq_desc_wait->cmd_descs[cq_desc_wait->cmd_descs_index];
    cq_desc_wait->cmd_descs_index += cmd_num;

    return 0;
}

static int cve_post_process(char *cmd_buf, unsigned int cmd_line_num, CVE_HANDLE *cveHandle,
                            AML_BOOL_E bInstant, cve_op_io_info_t *info)
{
    unsigned long flags;
    cve_cmd_desc_t *cmd_desc;
    int ret = 0;

    AML_ASSERT(cmd_buf != NULL);
    AML_ASSERT(cveHandle != NULL);
    AML_ASSERT((bInstant == AML_TRUE) || (bInstant == AML_FALSE));

    spin_lock_irqsave(&cve_spinlock, flags);
    ret = request_op_cmd(&cmd_desc, 1);
    if (ret) {
        CVE_ERR_TRACE("request op cmd failed!\n");
        spin_unlock_irqrestore(&cve_spinlock, flags);
        return ret;
    }

    cmd_desc->instant = bInstant;
    cmd_desc->cmd_line_num = cmd_line_num;
    INIT_LIST_HEAD(&cmd_desc->list);
    memcpy((void *)&cmd_desc->io_info, (void *)info, sizeof(cve_op_io_info_t));
    ret = cve_create_task(cmd_desc, cmd_buf);
    if (ret) {
        CVE_ERR_TRACE("creat task failed!\n");
        spin_unlock_irqrestore(&cve_spinlock, flags);
        return ret;
    }
    cve_manage_handle(cveHandle, 1);
    cmd_desc->cveHandle = *cveHandle;
    if (cve_context.queue_busy == CVE_STATUS_IDLE) {
        cve_context.queue_busy = cve_context.queue_wait;
        cve_context.queue_wait = CVE_STATUS_CQ1 - cve_context.queue_wait;
        spin_unlock_irqrestore(&cve_spinlock, flags);
        cve_start_process(&cve_context.cq_desc[cve_context.queue_busy]);
    } else {
        spin_unlock_irqrestore(&cve_spinlock, flags);
    }

    return ret;
}

static int cve_dma(CVE_OP_DMA_T *pstOpDMA)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }
    if (pstOpDMA == NULL) {
        CVE_ERR_TRACE("dma arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_data_phy_addr(&pstOpDMA->stSrcDATA);
    set_data_phy_addr(&pstOpDMA->stDstDATA);
    ret = cve_check_dma_param(&pstOpDMA->stSrcDATA, &pstOpDMA->stDstDATA, &pstOpDMA->stDmaCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check dma parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_dma_task(&pstOpDMA->stSrcDATA, &pstOpDMA->stDstDATA, &pstOpDMA->stDmaCtrl,
                            cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill dma task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_DMA;
    io_info.op_mode = CVE_MODE_DMA;
    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpDMA->cveHandle, pstOpDMA->bInstant, &io_info);
    cve_context.invoke_count.dma++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_luma_stat(CVE_OP_LUAM_STAT_ARRAY_T *pstOpLuamStat)
{
    unsigned int cmd_line_num_total = 0;
    unsigned long flags;
    cve_cmd_desc_t *cmd_descs;
    int ret = 0;
    int i = 0;

    if (pstOpLuamStat == NULL) {
        CVE_ERR_TRACE("luamStat arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }
    set_image_phy_addr(&pstOpLuamStat->stSrcImage);
    set_mem_phy_addr(&pstOpLuamStat->stDstMem);
    ret = cve_check_luamStat_param(&pstOpLuamStat->stSrcImage, &pstOpLuamStat->stDstMem,
                                   pstOpLuamStat->astCveLumaRect,
                                   &pstOpLuamStat->stLumaStatArrayCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check luamStat parameters failed!\n", ret);
        return ret;
    }

    for (i = 0; i < pstOpLuamStat->stLumaStatArrayCtrl.u8MaxLumaRect; i++) {
        multi_cmd_buf[i] = cmd_buf_get();
        if (multi_cmd_buf[i] == NULL) {
            return AML_ERR_CVE_NOBUF;
        }
        ret = cve_fill_luma_stat_task(
            &pstOpLuamStat->stSrcImage, &pstOpLuamStat->stDstMem, &pstOpLuamStat->astCveLumaRect[i],
            &pstOpLuamStat->stLumaStatArrayCtrl, i, multi_cmd_buf[i], &multi_cmd_line_num[i]);
        if (ret) {
            CVE_ERR_TRACE("ERRID(%x): fill luamStat task failed!\n", ret);
            return ret;
        }
        cmd_line_num_total += multi_cmd_line_num[i];
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    spin_lock_irqsave(&cve_spinlock, flags);
    ret = request_op_cmd(&cmd_descs, pstOpLuamStat->stLumaStatArrayCtrl.u8MaxLumaRect);
    if (ret) {
        spin_unlock_irqrestore(&cve_spinlock, flags);
        up(&cve_context.cve_sema);
        return ret;
    }

    cmd_descs->instant = pstOpLuamStat->bInstant;
    cmd_descs->cmd_line_num = cmd_line_num_total;
    INIT_LIST_HEAD(&cmd_descs->list);

    cmd_descs->io_info.op_type = CVE_OP_TYPE_DMA;
    cmd_descs->io_info.op_mode = CVE_MODE_LUMA_STAT;
    cmd_descs->io_info.inp_flags = 0;
    cmd_descs->io_info.inp_phys_addr = 0;
    cmd_descs->io_info.outp_flags = 0;
    cmd_descs->io_info.outp_phys_addr = 0;

    ret = cve_create_task_multi_cmd(cmd_descs, multi_cmd_buf, multi_cmd_line_num,
                                    pstOpLuamStat->stLumaStatArrayCtrl.u8MaxLumaRect);
    if (ret) {
        CVE_ERR_TRACE("creat task failed!\n");
        spin_unlock_irqrestore(&cve_spinlock, flags);
        up(&cve_context.cve_sema);
        return ret;
    }
    cve_manage_handle(&pstOpLuamStat->cveHandle, pstOpLuamStat->stLumaStatArrayCtrl.u8MaxLumaRect);
    cmd_descs->cveHandle = pstOpLuamStat->cveHandle;
    if (cve_context.queue_busy == CVE_STATUS_IDLE) {
        cve_context.queue_busy = cve_context.queue_wait;
        cve_context.queue_wait = CVE_STATUS_CQ1 - cve_context.queue_wait;
        spin_unlock_irqrestore(&cve_spinlock, flags);
        cve_start_process(&cve_context.cq_desc[cve_context.queue_busy]);
    } else {
        spin_unlock_irqrestore(&cve_spinlock, flags);
    }

    cve_context.invoke_count.luma_stat++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_and(CVE_OP_AND_T *pstOpAnd)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpAnd == NULL) {
        CVE_ERR_TRACE("and arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpAnd->stSrcImage1);
    set_image_phy_addr(&pstOpAnd->stSrcImage2);
    set_image_phy_addr(&pstOpAnd->stDst);
    ret = cve_check_and_param(&pstOpAnd->stSrcImage1, &pstOpAnd->stSrcImage2, &pstOpAnd->stDst,
                              &pstOpAnd->stAndCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check and parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_alu2_task(&pstOpAnd->stSrcImage1, &pstOpAnd->stSrcImage2, &pstOpAnd->stDst,
                             CVE_ALU_SEL_AND, &pstOpAnd->stAndCtrl, NULL, NULL, cmd_buf,
                             &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill and task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_ALU;
    io_info.op_mode = CVE_ALU_MODE_AND;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpAnd->cveHandle, pstOpAnd->bInstant, &io_info);

    cve_context.invoke_count.and ++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_or(CVE_OP_OR_T *pstOpOr)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpOr == NULL) {
        CVE_ERR_TRACE("or arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpOr->stSrcImage1);
    set_image_phy_addr(&pstOpOr->stSrcImage2);
    set_image_phy_addr(&pstOpOr->stDst);
    ret = cve_check_or_param(&pstOpOr->stSrcImage1, &pstOpOr->stSrcImage2, &pstOpOr->stDst,
                             &pstOpOr->stOrCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check or parameters failed!\n", ret);
        return ret;
    }

    ret =
        cve_fill_alu2_task(&pstOpOr->stSrcImage1, &pstOpOr->stSrcImage2, &pstOpOr->stDst,
                           CVE_ALU_SEL_OR, NULL, &pstOpOr->stOrCtrl, NULL, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill or task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_ALU;
    io_info.op_mode = CVE_ALU_MODE_OR;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpOr->cveHandle, pstOpOr->bInstant, &io_info);

    cve_context.invoke_count.or ++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_xor(CVE_OP_XOR_T *pstOpXor)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpXor == NULL) {
        CVE_ERR_TRACE("xor arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpXor->stSrcImage1);
    set_image_phy_addr(&pstOpXor->stSrcImage2);
    set_image_phy_addr(&pstOpXor->stDst);
    ret = cve_check_xor_param(&pstOpXor->stSrcImage1, &pstOpXor->stSrcImage2, &pstOpXor->stDst,
                              &pstOpXor->stXorCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check xor parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_alu2_task(&pstOpXor->stSrcImage1, &pstOpXor->stSrcImage2, &pstOpXor->stDst,
                             CVE_ALU_SEL_XOR, NULL, NULL, &pstOpXor->stXorCtrl, cmd_buf,
                             &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill xor task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_ALU;
    io_info.op_mode = CVE_ALU_MODE_XOR;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpXor->cveHandle, pstOpXor->bInstant, &io_info);

    cve_context.invoke_count.xor ++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_sub(CVE_OP_SUB_T *pstOpSub)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }
    if (pstOpSub == NULL) {
        CVE_ERR_TRACE("sub arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpSub->stSrcImage1);
    set_image_phy_addr(&pstOpSub->stSrcImage2);
    set_image_phy_addr(&pstOpSub->stDst);
    ret = cve_check_sub_param(&pstOpSub->stSrcImage1, &pstOpSub->stSrcImage2, &pstOpSub->stDst,
                              &pstOpSub->stSubCtrl, &pstOpSub->stDstMem);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check sub parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_alu1_task(&pstOpSub->stSrcImage1, &pstOpSub->stSrcImage2, &pstOpSub->stDst,
                             CVE_ALU_SEL_SUB, &pstOpSub->stSubCtrl, NULL, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill sub task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_ALU;
    io_info.op_mode = CVE_ALU_MODE_SUB;
    io_info.outp_flags = SUB_OUTPUT_PROCESS_REG;
    io_info.outp_size = pstOpSub->stDstMem.u32Size;
    io_info.outp_phys_addr = pstOpSub->stDstMem.u64PhyAddr;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpSub->cveHandle, pstOpSub->bInstant, &io_info);

    cve_context.invoke_count.sub++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_add(CVE_OP_ADD_T *pstOpAdd)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpAdd == NULL) {
        CVE_ERR_TRACE("add arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpAdd->stSrcImage1);
    set_image_phy_addr(&pstOpAdd->stSrcImage2);
    set_image_phy_addr(&pstOpAdd->stDst);
    ret = cve_check_add_param(&pstOpAdd->stSrcImage1, &pstOpAdd->stSrcImage2, &pstOpAdd->stDst,
                              &pstOpAdd->stAddCtrl, &pstOpAdd->stDstMem);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check add parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_alu1_task(&pstOpAdd->stSrcImage1, &pstOpAdd->stSrcImage2, &pstOpAdd->stDst,
                             CVE_ALU_SEL_ADD, NULL, &pstOpAdd->stAddCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill add task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_ALU;
    io_info.op_mode = CVE_ALU_MODE_ADD;
    io_info.outp_flags = ADD_OUTPUT_PROCESS_REG;
    io_info.outp_size = pstOpAdd->stDstMem.u32Size;
    io_info.outp_phys_addr = pstOpAdd->stDstMem.u64PhyAddr;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpAdd->cveHandle, pstOpAdd->bInstant, &io_info);

    cve_context.invoke_count.add++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_equalize_hist(CVE_OP_EQUALIZE_HIST_T *pstOpEqualizeHist)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpEqualizeHist == NULL) {
        CVE_ERR_TRACE("equalize hist arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpEqualizeHist->stSrcImage);
    set_image_phy_addr(&pstOpEqualizeHist->stDstImage);
    set_mem_phy_addr(&pstOpEqualizeHist->stEqualizeHistCtrl.stMem);
    ret = cve_check_equalize_hist_param(&pstOpEqualizeHist->stSrcImage,
                                        &pstOpEqualizeHist->stDstImage,
                                        &pstOpEqualizeHist->stEqualizeHistCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check equalize_hist parameters failed!\n", ret);
        return ret;
    }

    ret =
        cve_fill_equalize_hist_task(&pstOpEqualizeHist->stSrcImage, &pstOpEqualizeHist->stDstImage,
                                    &pstOpEqualizeHist->stEqualizeHistCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill equalize_hist task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_EQUALIZE_HIST;
    io_info.inp_flags = EQUALIZE_HIST_INPUT_PROCESS_LUT;
    io_info.inp_size = pstOpEqualizeHist->stEqualizeHistCtrl.stMem.u32Size;
    io_info.inp_phys_addr = pstOpEqualizeHist->stEqualizeHistCtrl.stMem.u64PhyAddr;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpEqualizeHist->cveHandle,
                           pstOpEqualizeHist->bInstant, &io_info);

    cve_context.invoke_count.equalize_hist++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_filter(CVE_OP_FILTER_T *pstOpFilter)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpFilter == NULL) {
        CVE_ERR_TRACE("filter arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpFilter->stSrcImage);
    set_image_phy_addr(&pstOpFilter->stDstImage);
    ret = cve_check_filter_param(&pstOpFilter->stSrcImage, &pstOpFilter->stDstImage,
                                 &pstOpFilter->stFilterCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check filter parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_filter_task(&pstOpFilter->stSrcImage, &pstOpFilter->stDstImage,
                               &pstOpFilter->stFilterCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill filter task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_FILTER;

    cve_context.invoke_count.filter++;
    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpFilter->cveHandle, pstOpFilter->bInstant,
                           &io_info);
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_csc(CVE_OP_CSC_T *pstOpCsc)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpCsc == NULL) {
        CVE_ERR_TRACE("csc arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpCsc->stSrcImage);
    set_image_phy_addr(&pstOpCsc->stDstImage);
    ret = cve_check_csc_param(&pstOpCsc->stSrcImage, &pstOpCsc->stDstImage, &pstOpCsc->stCscCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check csc parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_csc_task(&pstOpCsc->stSrcImage, &pstOpCsc->stDstImage, &pstOpCsc->stCscCtrl,
                            cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill csc task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_CSC;

    cve_context.invoke_count.csc++;
    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpCsc->cveHandle, pstOpCsc->bInstant, &io_info);
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_filter_and_csc(CVE_OP_FILTER_AND_CSC_T *pstOpFilterCsc)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpFilterCsc == NULL) {
        CVE_ERR_TRACE("filter_csc arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpFilterCsc->stSrcImage);
    set_image_phy_addr(&pstOpFilterCsc->stDstImage);
    ret = cve_check_filter_csc_param(&pstOpFilterCsc->stSrcImage, &pstOpFilterCsc->stDstImage,
                                     &pstOpFilterCsc->stFilterCscCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check filter_and_csc parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_filter_and_csc_task(&pstOpFilterCsc->stSrcImage, &pstOpFilterCsc->stDstImage,
                                       &pstOpFilterCsc->stFilterCscCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill filter_and_csc task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_FILTER_AND_CSC;

    cve_context.invoke_count.filter_and_csc++;
    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpFilterCsc->cveHandle,
                           pstOpFilterCsc->bInstant, &io_info);
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_sobel(CVE_OP_SOBEL_T *pstOpSobel)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpSobel == NULL) {
        CVE_ERR_TRACE("sobel arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpSobel->stSrcImage);
    set_image_phy_addr(&pstOpSobel->stDstH);
    set_image_phy_addr(&pstOpSobel->stDstV);
    ret = cve_check_sobel_param(&pstOpSobel->stSrcImage, &pstOpSobel->stDstH, &pstOpSobel->stDstV,
                                &pstOpSobel->stSobelCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check sobel parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_sobel_task(&pstOpSobel->stSrcImage, &pstOpSobel->stDstH, &pstOpSobel->stDstV,
                              &pstOpSobel->stSobelCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill sobel task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_SOBEL;

    cve_context.invoke_count.sobel++;
    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpSobel->cveHandle, pstOpSobel->bInstant,
                           &io_info);
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_dilate(CVE_OP_DILATE_T *stOpDilate)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (stOpDilate == NULL) {
        CVE_ERR_TRACE("dilate arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&stOpDilate->stSrcImage);
    set_image_phy_addr(&stOpDilate->stDstImage);
    ret = cve_check_dilate_param(&stOpDilate->stSrcImage, &stOpDilate->stDstImage,
                                 &stOpDilate->stDilateCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check dilate parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_dilate_task(&stOpDilate->stSrcImage, &stOpDilate->stDstImage,
                               &stOpDilate->stDilateCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill dilate task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_DILATE_AND_ERODE;
    io_info.op_mode = CVE_MODE_DILATE;

    cve_context.invoke_count.dilate++;
    ret = cve_post_process(cmd_buf, cmd_line_num, &stOpDilate->cveHandle, stOpDilate->bInstant,
                           &io_info);
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_erode(CVE_OP_ERODE_T *stOpErode)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (stOpErode == NULL) {
        CVE_ERR_TRACE("erode arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&stOpErode->stSrcImage);
    set_image_phy_addr(&stOpErode->stDstImage);
    ret = cve_check_erode_param(&stOpErode->stSrcImage, &stOpErode->stDstImage,
                                &stOpErode->stErodeCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check erode parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_erode_task(&stOpErode->stSrcImage, &stOpErode->stDstImage,
                              &stOpErode->stErodeCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill erode task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_DILATE_AND_ERODE;
    io_info.op_mode = CVE_MODE_ERODE;

    cve_context.invoke_count.erode++;
    ret = cve_post_process(cmd_buf, cmd_line_num, &stOpErode->cveHandle, stOpErode->bInstant,
                           &io_info);
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_thresh(CVE_OP_THRESH_T *pstOpThresh)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpThresh == NULL) {
        CVE_ERR_TRACE("thresh arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpThresh->stSrcImage);
    set_image_phy_addr(&pstOpThresh->stDstImage);
    ret = cve_check_thresh(&pstOpThresh->stSrcImage, &pstOpThresh->stDstImage,
                           &pstOpThresh->stThreshCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check thresh parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_thresh_task(&pstOpThresh->stSrcImage, &pstOpThresh->stDstImage,
                               &pstOpThresh->stThreshCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill thresh task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_THRESH;

    cve_context.invoke_count.thresh++;
    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpThresh->cveHandle, pstOpThresh->bInstant,
                           &io_info);
    up(&cve_context.cve_sema);

    return ret;
}
static int cve_integ(CVE_OP_INTEG_T *pstOpInteg)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpInteg == NULL) {
        CVE_ERR_TRACE("integ arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpInteg->stSrcImage);
    set_image_phy_addr(&pstOpInteg->stDstImage);
    ret = cve_check_integ_param(&pstOpInteg->stSrcImage, &pstOpInteg->stDstImage,
                                &pstOpInteg->stIntegCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check integ parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_integ_task(&pstOpInteg->stSrcImage, &pstOpInteg->stDstImage,
                              &pstOpInteg->stIntegCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill integ task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_INTEG;

    cve_context.invoke_count.integ++;
    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpInteg->cveHandle, pstOpInteg->bInstant,
                           &io_info);
    up(&cve_context.cve_sema);

    return ret;
}
static int cve_hist(CVE_OP_HIST_T *pstOpHist)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpHist == NULL) {
        CVE_ERR_TRACE("hist arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpHist->stSrcImage);
    set_mem_phy_addr(&pstOpHist->stDstMem);
    ret =
        cve_check_hist_param(&pstOpHist->stSrcImage, &pstOpHist->stDstMem, &pstOpHist->stHistCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check hist parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_hist_task(&pstOpHist->stSrcImage, &pstOpHist->stDstMem, &pstOpHist->stHistCtrl,
                             cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill hist task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_HIST;
    io_info.outp_flags = HIST_OUTPUT_PROCESS_LUT;
    io_info.outp_size = pstOpHist->stDstMem.u32Size;
    io_info.outp_phys_addr = pstOpHist->stDstMem.u64PhyAddr;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpHist->cveHandle, pstOpHist->bInstant,
                           &io_info);

    cve_context.invoke_count.hist++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_ncc(CVE_OP_NCC_T *pstOpNcc)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpNcc == NULL) {
        CVE_ERR_TRACE("ncc arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpNcc->stSrcImage1);
    set_image_phy_addr(&pstOpNcc->stSrcImage2);
    set_mem_phy_addr(&pstOpNcc->stDstmem);
    ret = cve_check_ncc_param(&pstOpNcc->stSrcImage1, &pstOpNcc->stSrcImage2, &pstOpNcc->stDstmem,
                              &pstOpNcc->stNccCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check ncc parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_ncc_task(&pstOpNcc->stSrcImage1, &pstOpNcc->stSrcImage2, &pstOpNcc->stNccCtrl,
                            cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill ncc task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_NCC;
    io_info.outp_flags = NCC_OUTPUT_PROCESS_REG;
    io_info.outp_size = pstOpNcc->stDstmem.u32Size;
    io_info.outp_phys_addr = pstOpNcc->stDstmem.u64PhyAddr;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpNcc->cveHandle, pstOpNcc->bInstant, &io_info);

    cve_context.invoke_count.ncc++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_norm_grad(CVE_OP_NORM_GRAD_T *pstOpNormGrad)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpNormGrad == NULL) {
        CVE_ERR_TRACE("norm grad arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpNormGrad->stSrcImage);
    set_image_phy_addr(&pstOpNormGrad->stDstH);
    set_image_phy_addr(&pstOpNormGrad->stDstV);
    set_image_phy_addr(&pstOpNormGrad->stDstHV);
    ret = cve_check_norm_grad_param(&pstOpNormGrad->stSrcImage, &pstOpNormGrad->stDstH,
                                    &pstOpNormGrad->stDstV, &pstOpNormGrad->stDstHV,
                                    &pstOpNormGrad->stNormGradCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check norm_grad parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_norm_grad_task(&pstOpNormGrad->stSrcImage, &pstOpNormGrad->stDstH,
                                  &pstOpNormGrad->stDstV, &pstOpNormGrad->stDstHV,
                                  &pstOpNormGrad->stNormGradCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill norm_grad task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_NROM_GRAD;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpNormGrad->cveHandle,
                           pstOpNormGrad->bInstant, &io_info);
    cve_context.invoke_count.nrom_grad++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_lbp(CVE_OP_LBP_T *pstOpLbp)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpLbp == NULL) {
        CVE_ERR_TRACE("lbp arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpLbp->stSrcImage);
    set_image_phy_addr(&pstOpLbp->stDstImage);
    ret = cve_check_lbp_param(&pstOpLbp->stSrcImage, &pstOpLbp->stDstImage, &pstOpLbp->stLbpCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check lbp parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_lbp_task(&pstOpLbp->stSrcImage, &pstOpLbp->stDstImage, &pstOpLbp->stLbpCtrl,
                            cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill LBP task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_LBP;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpLbp->cveHandle, pstOpLbp->bInstant, &io_info);

    cve_context.invoke_count.lbp++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_ccl(CVE_OP_CCL_T *pstOpCcl)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpCcl == NULL) {
        CVE_ERR_TRACE("ccl arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpCcl->stSrcImage);
    set_image_phy_addr(&pstOpCcl->stDstImage);
    set_mem_phy_addr(&pstOpCcl->stBlob);
    ret = cve_check_ccl_param(&pstOpCcl->stSrcImage, &pstOpCcl->stDstImage, &pstOpCcl->stBlob,
                              &pstOpCcl->stCclCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check ccl parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_ccl_task(&pstOpCcl->stSrcImage, &pstOpCcl->stDstImage, &pstOpCcl->stBlob,
                            &pstOpCcl->stCclCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill ccl task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_CCL;
    io_info.outp_flags = CCL_OUTPUT_PROCESS_REG;
    io_info.outp_size = pstOpCcl->stBlob.u32Size;
    io_info.outp_phys_addr = pstOpCcl->stBlob.u64PhyAddr;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpCcl->cveHandle, pstOpCcl->bInstant, &io_info);

    cve_context.invoke_count.ccl++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_gmm(CVE_OP_GMM_T *pstOpGmm)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpGmm == NULL) {
        CVE_ERR_TRACE("ccl arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpGmm->stSrcImage);
    set_image_phy_addr(&pstOpGmm->stFactor);
    set_image_phy_addr(&pstOpGmm->stFg);
    set_image_phy_addr(&pstOpGmm->stBg);
    set_mem_phy_addr(&pstOpGmm->stModel);
    ret = cve_check_gmm_param(&pstOpGmm->stSrcImage, &pstOpGmm->stFactor, &pstOpGmm->stFg,
                              &pstOpGmm->stBg, &pstOpGmm->stModel, &pstOpGmm->stGmmCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check gmm parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_gmm_task(&pstOpGmm->stSrcImage, &pstOpGmm->stFactor, &pstOpGmm->stFg,
                            &pstOpGmm->stBg, &pstOpGmm->stModel, &pstOpGmm->stGmmCtrl, cmd_buf,
                            &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill gmm task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_GMM;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpGmm->cveHandle, pstOpGmm->bInstant, &io_info);

    cve_context.invoke_count.gmm++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_map(CVE_OP_MAP_T *pstOpMap)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpMap == NULL) {
        CVE_ERR_TRACE("map arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpMap->stSrcImage);
    set_mem_phy_addr(&pstOpMap->stMap);
    set_image_phy_addr(&pstOpMap->stDstImage);
    ret = cve_check_map_param(&pstOpMap->stSrcImage, &pstOpMap->stMap, &pstOpMap->stDstImage,
                              &pstOpMap->stMapCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check map parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_map_task(&pstOpMap->stSrcImage, &pstOpMap->stMap, &pstOpMap->stDstImage,
                            &pstOpMap->stMapCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill map task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_MAP;
    io_info.op_mode = pstOpMap->stMapCtrl.enMode;
    io_info.inp_flags = MAP_INPUT_PROCESS_LUT;
    io_info.inp_size = pstOpMap->stMap.u32Size;
    io_info.inp_phys_addr = pstOpMap->stMap.u64PhyAddr;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpMap->cveHandle, pstOpMap->bInstant, &io_info);

    cve_context.invoke_count.map++;
    up(&cve_context.cve_sema);

    return ret;
}
static int cve_thresh_s16(CVE_OP_THRESH_S16_T *pstOpThreshS16)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpThreshS16 == NULL) {
        CVE_ERR_TRACE("thresh_s16 arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpThreshS16->stSrcImage);
    set_image_phy_addr(&pstOpThreshS16->stDstImage);
    ret = cve_check_thresh_s16_param(&pstOpThreshS16->stSrcImage, &pstOpThreshS16->stDstImage,
                                     &pstOpThreshS16->stThreshS16Ctrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check thresh_s16 parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_thresh_s16_task(&pstOpThreshS16->stSrcImage, &pstOpThreshS16->stDstImage,
                                   &pstOpThreshS16->stThreshS16Ctrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill thresh_s16 task failed!\n", ret);
        return ret;
    }
    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_THRESH_S16;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpThreshS16->cveHandle,
                           pstOpThreshS16->bInstant, &io_info);

    cve_context.invoke_count.thresh_s16++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_thresh_u16(CVE_OP_THRESH_U16_T *pstOpThreshU16)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpThreshU16 == NULL) {
        CVE_ERR_TRACE("thresh_u16 arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpThreshU16->stSrcImage);
    set_image_phy_addr(&pstOpThreshU16->stDstImage);
    ret = cve_check_thresh_u16_param(&pstOpThreshU16->stSrcImage, &pstOpThreshU16->stDstImage,
                                     &pstOpThreshU16->stThreshU16Ctrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check thresh_u16 parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_thresh_u16_task(&pstOpThreshU16->stSrcImage, &pstOpThreshU16->stDstImage,
                                   &pstOpThreshU16->stThreshU16Ctrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill thresh_u16 task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_THRESH_U16;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpThreshU16->cveHandle,
                           pstOpThreshU16->bInstant, &io_info);

    cve_context.invoke_count.thresh_u16++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_ord_stat_filter(CVE_OP_ORD_STAT_FILTER_T *pstOpOrdStatFilter)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpOrdStatFilter == NULL) {
        CVE_ERR_TRACE("ord_stat_filter arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpOrdStatFilter->stSrcImage);
    set_image_phy_addr(&pstOpOrdStatFilter->stDstImage);
    ret = cve_check_ord_stat_filter_param(&pstOpOrdStatFilter->stSrcImage,
                                          &pstOpOrdStatFilter->stDstImage,
                                          &pstOpOrdStatFilter->stOrdStatFltCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check ord_stat_filter parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_ord_stat_filter_task(
        &pstOpOrdStatFilter->stSrcImage, &pstOpOrdStatFilter->stDstImage,
        &pstOpOrdStatFilter->stOrdStatFltCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill ord_stat_filter task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_ORD_STAT_FILTER;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpOrdStatFilter->cveHandle,
                           pstOpOrdStatFilter->bInstant, &io_info);

    cve_context.invoke_count.ord_stat_filter++;
    up(&cve_context.cve_sema);
    return ret;
}

static int cve_16_to_8(CVE_OP_16BIT_TO_8BIT_T *pstOp16bitTo8bit)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOp16bitTo8bit == NULL) {
        CVE_ERR_TRACE("16bit_to_8bit arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOp16bitTo8bit->stSrcImage);
    set_image_phy_addr(&pstOp16bitTo8bit->stDstImage);
    ret =
        cve_check_16bit_to_8bit_param(&pstOp16bitTo8bit->stSrcImage, &pstOp16bitTo8bit->stDstImage,
                                      &pstOp16bitTo8bit->st16BitTo8BitCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check 16bit_to_8bit parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_16bit_to_8bit_task(&pstOp16bitTo8bit->stSrcImage, &pstOp16bitTo8bit->stDstImage,
                                      &pstOp16bitTo8bit->st16BitTo8BitCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill 16bit_to_8bit task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_16BIT_TO_8BIT;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOp16bitTo8bit->cveHandle,
                           pstOp16bitTo8bit->bInstant, &io_info);

    cve_context.invoke_count._16bit_to_8bit++;
    up(&cve_context.cve_sema);

    return ret;
}
static int cve_sad(CVE_OP_SAD_T *stOpSAD)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (stOpSAD == NULL) {
        CVE_ERR_TRACE("sad arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&stOpSAD->stSrcImage1);
    set_image_phy_addr(&stOpSAD->stSrcImage2);
    set_image_phy_addr(&stOpSAD->stSad);
    set_image_phy_addr(&stOpSAD->stThr);
    ret = cve_check_sad_param(&stOpSAD->stSrcImage1, &stOpSAD->stSrcImage2, &stOpSAD->stSad,
                              &stOpSAD->stThr, &stOpSAD->stSadCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check sad parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_sad_task(&stOpSAD->stSrcImage1, &stOpSAD->stSrcImage2, &stOpSAD->stSad,
                            &stOpSAD->stThr, &stOpSAD->stSadCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill sad task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_SAD;

    ret = cve_post_process(cmd_buf, cmd_line_num, &stOpSAD->cveHandle, stOpSAD->bInstant, &io_info);

    cve_context.invoke_count.sad++;
    up(&cve_context.cve_sema);
    return ret;
}

static int cve_grad_fg(CVE_OP_GRAD_FG_T *stOpGradFg)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (stOpGradFg == NULL) {
        CVE_ERR_TRACE("gradFg arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&stOpGradFg->stFg);
    set_image_phy_addr(&stOpGradFg->stCurGrad);
    set_image_phy_addr(&stOpGradFg->stBgGrad);
    set_image_phy_addr(&stOpGradFg->stGradFg);
    ret = cve_check_grad_fg_param(&stOpGradFg->stFg, &stOpGradFg->stCurGrad, &stOpGradFg->stBgGrad,
                                  &stOpGradFg->stGradFg, &stOpGradFg->stGradFgCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check grad_fg parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_grad_fg_task(&stOpGradFg->stFg, &stOpGradFg->stCurGrad, &stOpGradFg->stBgGrad,
                                &stOpGradFg->stGradFg, &stOpGradFg->stGradFgCtrl, cmd_buf,
                                &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill grad_fg task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_GRAD_FG;

    ret = cve_post_process(cmd_buf, cmd_line_num, &stOpGradFg->cveHandle, stOpGradFg->bInstant,
                           &io_info);

    cve_context.invoke_count.grad_fg++;
    up(&cve_context.cve_sema);
    return ret;
}

static int cve_mag_and_ang(CVE_OP_MAG_AND_ANG_T *pstOpMagAndAng)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpMagAndAng == NULL) {
        CVE_ERR_TRACE("mag_and_ang arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpMagAndAng->stSrcImage);
    set_image_phy_addr(&pstOpMagAndAng->stDstMag);
    set_image_phy_addr(&pstOpMagAndAng->stDstAng);
    ret = cve_check_mag_and_ang_param(&pstOpMagAndAng->stSrcImage, &pstOpMagAndAng->stDstMag,
                                      &pstOpMagAndAng->stDstAng, &pstOpMagAndAng->stMagAndAngCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check mag_and_ang parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_mag_and_ang_task(&pstOpMagAndAng->stSrcImage, &pstOpMagAndAng->stDstMag,
                                    &pstOpMagAndAng->stDstAng, &pstOpMagAndAng->stMagAndAngCtrl,
                                    cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill mag_and_ang task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_MAG_AND_ANG;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpMagAndAng->cveHandle,
                           pstOpMagAndAng->bInstant, &io_info);

    cve_context.invoke_count.mag_and_ang++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_match_bg_model(CVE_OP_MATCH_BG_MODEL_T *pstOpMatchBgModel)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpMatchBgModel == NULL) {
        CVE_ERR_TRACE("match_bg_model arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpMatchBgModel->stCurImg);
    set_image_phy_addr(&pstOpMatchBgModel->stPreImg);
    set_mem_phy_addr(&pstOpMatchBgModel->stBgModel);
    set_image_phy_addr(&pstOpMatchBgModel->stFg);
    set_image_phy_addr(&pstOpMatchBgModel->stBg);
    set_image_phy_addr(&pstOpMatchBgModel->stCurDiffBg);
    set_image_phy_addr(&pstOpMatchBgModel->stFrmDiff);
    set_mem_phy_addr(&pstOpMatchBgModel->stStatData);
    ret = cve_check_match_bg_model_param(
        &pstOpMatchBgModel->stCurImg, &pstOpMatchBgModel->stPreImg, &pstOpMatchBgModel->stBgModel,
        &pstOpMatchBgModel->stFg, &pstOpMatchBgModel->stBg, &pstOpMatchBgModel->stCurDiffBg,
        &pstOpMatchBgModel->stFrmDiff, &pstOpMatchBgModel->stStatData,
        &pstOpMatchBgModel->stMatchBgModelCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check match_bg_model parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_match_bg_model_task(
        &pstOpMatchBgModel->stCurImg, &pstOpMatchBgModel->stPreImg, &pstOpMatchBgModel->stBgModel,
        &pstOpMatchBgModel->stFg, &pstOpMatchBgModel->stBg, &pstOpMatchBgModel->stCurDiffBg,
        &pstOpMatchBgModel->stFrmDiff, &pstOpMatchBgModel->stMatchBgModelCtrl, cmd_buf,
        &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill match_bg_model task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_MATCH_BG_MODEL;
    io_info.outp_flags = MATCH_BG_MODEL_OUTPUT_PROCESS_REG;
    io_info.outp_size = pstOpMatchBgModel->stStatData.u32Size;
    io_info.outp_phys_addr = pstOpMatchBgModel->stStatData.u64PhyAddr;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpMatchBgModel->cveHandle,
                           pstOpMatchBgModel->bInstant, &io_info);

    cve_context.invoke_count.match_bg_model++;
    up(&cve_context.cve_sema);

    return ret;
}

static int cve_update_bg_model(CVE_OP_UPDATE_BG_MODEL_T *pstOpUpdateBgModel)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpUpdateBgModel == NULL) {
        CVE_ERR_TRACE("update_bg_model arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpUpdateBgModel->stCurImg);
    set_mem_phy_addr(&pstOpUpdateBgModel->stBgModel1);
    set_mem_phy_addr(&pstOpUpdateBgModel->stBgModel2);
    set_mem_phy_addr(&pstOpUpdateBgModel->stStatData);
    ret = cve_check_update_bg_model_param(
        &pstOpUpdateBgModel->stCurImg, &pstOpUpdateBgModel->stBgModel1,
        &pstOpUpdateBgModel->stBgModel2, &pstOpUpdateBgModel->stStatData,
        &pstOpUpdateBgModel->stUpdateBgModelCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check update_bg_model parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_update_bg_model_task(
        &pstOpUpdateBgModel->stCurImg, &pstOpUpdateBgModel->stBgModel1,
        &pstOpUpdateBgModel->stBgModel2, &pstOpUpdateBgModel->stUpdateBgModelCtrl, cmd_buf,
        &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill update_bg_model task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_UPDATE_BG_MODEL;
    io_info.outp_flags = UPDATE_BG_MODEL_OUTPUT_PROCESS_REG;
    io_info.outp_size = pstOpUpdateBgModel->stStatData.u32Size;
    io_info.outp_phys_addr = pstOpUpdateBgModel->stStatData.u64PhyAddr;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpUpdateBgModel->cveHandle,
                           pstOpUpdateBgModel->bInstant, &io_info);

    cve_context.invoke_count.update_bg_model++;
    up(&cve_context.cve_sema);

    return ret;
}

static int
cve_build_lk_optical_flow_pyr(CVE_OP_BUILD_LK_OPTICAL_FLOW_PYR_T *pstOpBuildLkOpticalFlowPyr)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;
    int i = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpBuildLkOpticalFlowPyr == NULL) {
        CVE_ERR_TRACE("build_lk_optical_flow_pyr arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpBuildLkOpticalFlowPyr->stSrcPyr);
    for (i = 0; i < pstOpBuildLkOpticalFlowPyr->stLkBuildOptiFlowPyrCtrl.u8MaxLevel; i++) {
        set_image_phy_addr(&pstOpBuildLkOpticalFlowPyr->astDstPyr[i]);
    };
    ret = cve_check_build_lk_optical_flow_pyr_param(
        &pstOpBuildLkOpticalFlowPyr->stSrcPyr, pstOpBuildLkOpticalFlowPyr->astDstPyr,
        &pstOpBuildLkOpticalFlowPyr->stLkBuildOptiFlowPyrCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check build_lk_optical_flow_pyr parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_build_lk_optical_flow_pyr_task(
        &pstOpBuildLkOpticalFlowPyr->stSrcPyr, pstOpBuildLkOpticalFlowPyr->astDstPyr,
        &pstOpBuildLkOpticalFlowPyr->stLkBuildOptiFlowPyrCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill build_lk_optical_flow_pyr task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_BULID_LK_OPTICAL_FLOW_PYR;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpBuildLkOpticalFlowPyr->cveHandle,
                           pstOpBuildLkOpticalFlowPyr->bInstant, &io_info);

    cve_context.invoke_count.bulid_lk_optical_flow_pyr++;
    up(&cve_context.cve_sema);
    return ret;
}

static int cve_lk_optical_flow_pyr(CVE_OP_LK_OPTICAL_FLOW_PYR_T *pstOpLkOpticalFlowPyr)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;
    int i = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpLkOpticalFlowPyr == NULL) {
        CVE_ERR_TRACE("lk_optical_flow_pyr arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    for (i = 0; i < pstOpLkOpticalFlowPyr->stLkOptiFlowPyrCtrl.u8MaxLevel; i++) {
        set_image_phy_addr(&pstOpLkOpticalFlowPyr->astSrcPrevPyr[i]);
        set_image_phy_addr(&pstOpLkOpticalFlowPyr->astSrcNextPyr[i]);
    }
    set_mem_phy_addr(&pstOpLkOpticalFlowPyr->stPrevPts);
    set_mem_phy_addr(&pstOpLkOpticalFlowPyr->stNextPts);
    set_mem_phy_addr(&pstOpLkOpticalFlowPyr->stStatus);
    set_mem_phy_addr(&pstOpLkOpticalFlowPyr->stErr);
    ret = cve_check_lk_optical_flow_pyr_param(
        pstOpLkOpticalFlowPyr->astSrcPrevPyr, pstOpLkOpticalFlowPyr->astSrcNextPyr,
        &pstOpLkOpticalFlowPyr->stPrevPts, &pstOpLkOpticalFlowPyr->stNextPts,
        &pstOpLkOpticalFlowPyr->stStatus, &pstOpLkOpticalFlowPyr->stLkOptiFlowPyrCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check lk_optical_flow_pyr parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_lk_optical_flow_pyr_task(
        pstOpLkOpticalFlowPyr->astSrcPrevPyr, pstOpLkOpticalFlowPyr->astSrcNextPyr,
        &pstOpLkOpticalFlowPyr->stPrevPts, &pstOpLkOpticalFlowPyr->stNextPts,
        &pstOpLkOpticalFlowPyr->stLkOptiFlowPyrCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill lk_optical_flow_pyr task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_LK_OPTIAL_FLOW_PRY;
    io_info.outp_flags = LK_OPTICAL_FLOWPYR_OUTPUT_PROCESS_REG;
    io_info.outp_size = pstOpLkOpticalFlowPyr->stStatus.u32Size;
    io_info.outp_phys_addr = pstOpLkOpticalFlowPyr->stStatus.u64PhyAddr;

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpLkOpticalFlowPyr->cveHandle,
                           pstOpLkOpticalFlowPyr->bInstant, &io_info);

    cve_context.invoke_count.lk_optial_flow_pry++;
    up(&cve_context.cve_sema);
    return ret;
}

static int cve_st_candi_corner(CVE_OP_ST_CANDI_CORNER_T *pstOpSTCandiCorner)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpSTCandiCorner == NULL) {
        CVE_ERR_TRACE("st_candi_corner arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpSTCandiCorner->stSrc);
    set_image_phy_addr(&pstOpSTCandiCorner->stLabel);
    set_image_phy_addr(&pstOpSTCandiCorner->stCandiCorner);
    set_mem_phy_addr(&pstOpSTCandiCorner->stCandiCornerPoint);
    ret = cve_check_st_candi_corner_param(&pstOpSTCandiCorner->stSrc, &pstOpSTCandiCorner->stLabel,
                                          &pstOpSTCandiCorner->stCandiCorner,
                                          &pstOpSTCandiCorner->stCandiCornerPoint,
                                          &pstOpSTCandiCorner->stStCandiCornerCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check st_candi_corner parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_st_candi_corner_task(
        &pstOpSTCandiCorner->stSrc, &pstOpSTCandiCorner->stLabel,
        &pstOpSTCandiCorner->stCandiCorner, &pstOpSTCandiCorner->stCandiCornerPoint,
        &pstOpSTCandiCorner->stStCandiCornerCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill st_candi_corner task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_ST_CANDI_CORNER;
    io_info.outp_flags = STCORNER_OUTPUT_PROCESS_REG;
    io_info.outp_size = sizeof(AML_U32);
    io_info.outp_phys_addr = pstOpSTCandiCorner->stCandiCornerPoint.u64PhyAddr +
                             pstOpSTCandiCorner->stCandiCornerPoint.u32Size - sizeof(AML_U32);

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpSTCandiCorner->cveHandle,
                           pstOpSTCandiCorner->bInstant, &io_info);

    cve_context.invoke_count.st_candi_corner++;
    up(&cve_context.cve_sema);
    return ret;
}

static int cve_canny_hys_edge(CVE_OP_CANNY_HYS_EDGE_T *pstOpCannyHysEdge)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpCannyHysEdge == NULL) {
        CVE_ERR_TRACE("canny_hys_edge arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_image_phy_addr(&pstOpCannyHysEdge->stSrcImage);
    set_image_phy_addr(&pstOpCannyHysEdge->stEdge);
    set_mem_phy_addr(&pstOpCannyHysEdge->stStack);
    ret = cve_check_canny_hys_edge_param(&pstOpCannyHysEdge->stSrcImage, &pstOpCannyHysEdge->stEdge,
                                         &pstOpCannyHysEdge->stStack,
                                         &pstOpCannyHysEdge->stCannyHysEdgeCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check canny_hys_edge parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_canny_hys_edge_task(
        &pstOpCannyHysEdge->stSrcImage, &pstOpCannyHysEdge->stEdge, &pstOpCannyHysEdge->stStack,
        &pstOpCannyHysEdge->stCannyHysEdgeCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill canny_hys_edge task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_CANNY_HYS_EDGE;
    io_info.outp_size = sizeof(AML_U32);
    io_info.outp_flags = CANNY_OUTPUT_PROCESS_REG;
    io_info.outp_phys_addr = pstOpCannyHysEdge->stStack.u64PhyAddr +
                             pstOpCannyHysEdge->stStack.u32Size - sizeof(AML_U32);

    ret = cve_post_process(cmd_buf, cmd_line_num, &pstOpCannyHysEdge->cveHandle,
                           pstOpCannyHysEdge->bInstant, &io_info);

    cve_context.invoke_count.canny_edge++;
    up(&cve_context.cve_sema);
    return ret;
}

static int cve_tof(CVE_OP_TOF_T *pstOpTof)
{
    char *cmd_buf = cmd_buf_get();
    unsigned int cmd_line_num = 0;
    cve_op_io_info_t io_info;
    int ret = 0;

    if (cmd_buf == NULL) {
        return AML_ERR_CVE_NOBUF;
    }

    if (pstOpTof == NULL) {
        CVE_ERR_TRACE("tof arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    set_raw_phy_addr(&pstOpTof->stSrcRaw);
    set_raw_phy_addr(&pstOpTof->stSrcFpn);
    set_mem_phy_addr(&pstOpTof->stSrcCoef);
    set_mem_phy_addr(&pstOpTof->stBpc);
    set_mem_phy_addr(&pstOpTof->stDtsStatus);
    set_mem_phy_addr(&pstOpTof->stDtsIR);
    set_mem_phy_addr(&pstOpTof->stDtsData);
    set_mem_phy_addr(&pstOpTof->stDstHist);
    ret = cve_check_tof_param(&pstOpTof->stSrcRaw, &pstOpTof->stSrcFpn, &pstOpTof->stSrcCoef,
                              &pstOpTof->stBpc, &pstOpTof->stDtsStatus, &pstOpTof->stDtsIR,
                              &pstOpTof->stDtsData, &pstOpTof->stDstHist, &pstOpTof->stTofCtrl);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): check tof parameters failed!\n", ret);
        return ret;
    }

    ret = cve_fill_tof_task(&pstOpTof->stSrcRaw, &pstOpTof->stSrcFpn, &pstOpTof->stSrcCoef,
                            &pstOpTof->stDtsStatus, &pstOpTof->stDtsIR, &pstOpTof->stDtsData,
                            &pstOpTof->stTofCtrl, cmd_buf, &cmd_line_num);
    if (ret) {
        CVE_ERR_TRACE("ERRID(%x): fill tof task failed!\n", ret);
        return ret;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    memset((void *)&io_info, 0, sizeof(cve_op_io_info_t));
    io_info.op_type = CVE_OP_TYPE_TOF;
    io_info.inp_flags = TOF_INPUT_PROCESS_LUT;
    io_info.inp_size = pstOpTof->stBpc.u32Size;
    io_info.inp_phys_addr = pstOpTof->stBpc.u64PhyAddr;
    io_info.outp_flags = TOF_OUTPUT_PROCESS_LUT;
    io_info.outp_size = pstOpTof->stDstHist.u32Size;
    io_info.outp_phys_addr = pstOpTof->stDstHist.u64PhyAddr;

    ret =
        cve_post_process(cmd_buf, cmd_line_num, &pstOpTof->cveHandle, pstOpTof->bInstant, &io_info);

    cve_context.invoke_count.tof++;
    up(&cve_context.cve_sema);

    return ret;
}
static int cve_check_op_proc_completed(const void *data)
{
    cve_task_info_t *info = &cve_context.task_info;
    unsigned long flags;
    CVE_HANDLE cveHandle;

    if (data == NULL) {
        panic("\nASSERT at:\n	>Function : %s\n  >Line No. : %d\n	>Condition: %s\n",
              __FUNCTION__, __LINE__, "data != NULL");
        return false;
    }
    cveHandle = *(CVE_HANDLE *)data;

    spin_lock_irqsave(&cve_spinlock, flags);

    if ((cveHandle < info->cmd_finish_cnt &&
         info->cmd_finish_cnt - cveHandle <= cve_node_num * 2) ||
        (cveHandle > info->cmd_finish_cnt && cveHandle - info->cmd_finish_cnt > cve_node_num * 2)) {
        spin_unlock_irqrestore(&cve_spinlock, flags);
        return true;
    }
    spin_unlock_irqrestore(&cve_spinlock, flags);

    return false;
}

static int cve_query_task(CVE_HANDLE cveHandle, AML_BOOL_E bBlock, AML_BOOL_E *pbFinish)
{
    int ret;

    if (bBlock != AML_TRUE && bBlock != AML_FALSE) {
        CVE_ERR_TRACE("bBlock (%d) must be AML_TRUE or AML_FALSE!\n", bBlock);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }
    if (cveHandle >= CMD_HANDLE_MAX) {
        CVE_ERR_TRACE("Error,CveHandle(%d) must be (%d,%d)\n", cveHandle, AML_INVALID_HANDLE,
                      CMD_HANDLE_MAX);
        return AML_ERR_CVE_ILLEGAL_PARAM;
    }

    if (down_interruptible(&cve_context.cve_sema)) {
        CVE_ERR_TRACE("osal_down_interruptible cve_sema error\n");
        return -ERESTARTSYS;
    }

    if (cve_check_op_proc_completed(&cveHandle)) {
        goto check_cve_timeout;
    }

    if (!bBlock) {
        *pbFinish = AML_FALSE;
        up(&cve_context.cve_sema);
        return 0;
    }
    ret = wait_event_interruptible_timeout(
        cve_context.cve_wait, cve_check_op_proc_completed(&cveHandle), CVE_QUERY_TIMEOUT);
    if (ret < 0 && ret != -ERESTARTSYS) {
        CVE_ERR_TRACE("CVE query parameter invalid!\n");
        ret = AML_ERR_CVE_ILLEGAL_PARAM;
        goto query_error;
    } else if (ret == -ERESTARTSYS) {
        CVE_ERR_TRACE("CVE query failed!\n");
        goto query_error;
    } else if (!ret) {
        CVE_ERR_TRACE("CVE query time out!\n");
        cve_context.run_time_info.query_timeout_cnt++;
        ret = AML_ERR_CVE_QUERY_TIMEOUT;
        goto query_error;
    }

    if (cve_state) {
        CVE_ERR_TRACE("CVE is unexist!\n");
        ret = AML_ERR_CVE_UNEXIST;
        goto query_error;
    }

check_cve_timeout:
    if (cve_timeout_flag) {
        cve_timeout_flag = false;
        CVE_ERR_TRACE("CVE process timeout!\n");
        ret = AML_ERR_CVE_SYS_TIMEOUT;
        goto query_error;
    }

    *pbFinish = AML_TRUE;
    up(&cve_context.cve_sema);

    return 0;

query_error:
    up(&cve_context.cve_sema);
    return ret;
}
static int cve_query(CVE_OP_QUERY_T *pstOpQuery)
{
    int ret = 0;

    if (pstOpQuery == NULL) {
        CVE_ERR_TRACE("query arg is NULL\n");
        return AML_ERR_CVE_NULL_PTR;
    }

    ret = cve_query_task(pstOpQuery->cveHandle, pstOpQuery->bBlock, &pstOpQuery->bFinish);

    return ret;
}

static int cve_proc_show(struct seq_file *proc_entry, void *arg)
{
    cve_task_info_t *ti;
    cve_cq_desc_t *cqb;
    cve_cq_desc_t *cqw;
    cve_run_time_info_t *rti;
    cve_op_info_t *ic;

    ti = &cve_context.task_info;
    cqw = &cve_context.cq_desc[cve_context.queue_wait];
    cqb = &cve_context.cq_desc[CVE_STATUS_CQ1 - cve_context.queue_wait];
    rti = &cve_context.run_time_info;
    ic = &cve_context.invoke_count;

    seq_printf(
        proc_entry,
        "\n-------------------------------MODULE PARAM--------------------------------------\n");
    seq_printf(proc_entry, "%16s\n", "max_node_num");
    seq_printf(proc_entry, "%16u\n", cve_node_num);
    seq_printf(
        proc_entry,
        "\n-------------------------------CVE QUEUE INFO------------------------------------\n");
    seq_printf(proc_entry, "%8s%8s%13s%13s%13s%13s\n", "Wait", "Busy", "WaitCurId", "WaitEndId",
               "BusyCurId", "BusyEndId");
    seq_printf(proc_entry, "%8d%8d%13d%13d%13d%13d\n", cve_context.queue_wait,
               cve_context.queue_busy, cqw->cur_cmd_id, cqw->end_cmd_id, cqb->cur_cmd_id,
               cqb->end_cmd_id);
    seq_printf(
        proc_entry,
        "\n--------------------------------CVE TASK INFO------------------------------------\n");
    seq_printf(proc_entry, "%7s%11s%10s%10s%11s%11s\n", "Hnd", "TaskFsh", "LastId", "TaskId",
               "HndWrap", "FshWrap");
    seq_printf(proc_entry, "%7d%11d%10d%10d%11d%11d\n", ti->cmd_handle, ti->cmd_finish_cnt,
               ti->last_finish_cmd_id, ti->cur_finish_cmd_id, ti->cmd_handle_wrap,
               ti->finish_cmd_wrap);
    seq_printf(
        proc_entry,
        "\n-----------------------------------CVE RUN-TIME INFO-----------------------------\n");
    seq_printf(proc_entry, "%12s%13s%16s%22s%15s%9s%9s\n", "LastInst", "CntPerSec", "MaxCntPerSec",
               "TotalIntCntLastSec", "TotalIntCnt", "QTCnt", "STCnt");
    seq_printf(proc_entry, "%12d%13d%16d%22d%15d%9d%9d\n", rti->last_instant,
               rti->last_int_cnt_per_sec, rti->max_int_cnt_per_sec, rti->total_int_cnt_last_sec,
               rti->total_int_cnt, rti->query_timeout_cnt, rti->system_timeout_cnt);
    seq_printf(proc_entry, "\n%10s%11s%16s%17s%18s%9s%9s\n", "CostTm", "MCostTm", "CostTmPerSec",
               "MCostTmPerSec", "TotalIntCostTm", "RunTm", "(us)");
    seq_printf(proc_entry, "%10d%11d%16d%17d%18d%9d\n", rti->last_int_cost_time,
               rti->int_cost_time_max, rti->last_persec_int_cost_time,
               rti->persec_int_cost_time_max, rti->total_int_cost_time, rti->total_cve_run_time);
    seq_printf(
        proc_entry,
        "\n----------------------------------CVE INVOKE INFO--------------------------------\n");
    seq_printf(proc_entry, "\n%13s%13s%13s%13s%13s%13s%13s%13s\n", "DMA", "Filter", "CSC", "FltCsc",
               "Sobel", "MagAng", "Dilate", "Erode");
    seq_printf(proc_entry, "%13d%13d%13d%13d%13d%13d%13d%13d\n", ic->dma, ic->filter, ic->csc,
               ic->filter_and_csc, ic->sobel, ic->mag_and_ang, ic->dilate, ic->erode);
    seq_printf(proc_entry, "\n%13s%13s%13s%13s%13s%13s%13s%13s\n", "Thresh", "Integ", "Hist",
               "ThreshS16", "ThreshU16", "And", "Sub", "Or");
    seq_printf(proc_entry, "%13d%13d%13d%13d%13d%13d%13d%13d\n", ic->thresh, ic->integ, ic->hist,
               ic->thresh_s16, ic->thresh_u16, ic->and, ic->sub, ic->or);
    seq_printf(proc_entry, "\n%13s%13s%13s%13s%13s%13s%13s%13s\n", "Add", "Xor", "16to8", "OrdStat",
               "Map", "EqualH", "NCC", "SAD");
    seq_printf(proc_entry, "%13d%13d%13d%13d%13d%13d%13d%13d\n", ic->add, ic->xor,
               ic->_16bit_to_8bit, ic->ord_stat_filter, ic->map, ic->equalize_hist, ic->ncc,
               ic->sad);
    seq_printf(proc_entry, "\n%13s%13s%13s%13s%13s%13s%13s%13s\n", "CCL", "GMM", "Canny", "LBP",
               "NormGrad", "LK", "STconcer", "GradFg");
    seq_printf(proc_entry, "%13d%13d%13d%13d%13d%13d%13d%13d\n", ic->ccl, ic->gmm, ic->canny_edge,
               ic->lbp, ic->nrom_grad, ic->lk_optial_flow_pry, ic->st_candi_corner, ic->grad_fg);
    seq_printf(proc_entry, "\n%13s%13s%13s\n", "MatchMod", "UpdateMod", "TOF");
    seq_printf(proc_entry, "%13d%13d%13d\n", ic->match_bg_model, ic->update_bg_model, ic->tof);
    seq_printf(
        proc_entry,
        "\n----------------------------------CVE OPRUNTIME INFO--------------------------------\n");
    seq_printf(proc_entry, "\n%13s%13s%13s%13s%13s%13s%13s%13s\n", "DMA", "lumaStat", "Filter",
               "CSC", "FltCsc", "Sobel", "MagAng", "Dilate");
    seq_printf(proc_entry, "%13d%13d%13d%13d%13d%13d%13d%13d\n", rti->cve_op_cycle.dma,
               rti->cve_op_cycle.luma_stat, rti->cve_op_cycle.filter, rti->cve_op_cycle.csc,
               rti->cve_op_cycle.filter_and_csc, rti->cve_op_cycle.sobel,
               rti->cve_op_cycle.mag_and_ang, rti->cve_op_cycle.dilate);
    seq_printf(proc_entry, "\n%13s%13s%13s%13s%13s%13s%13s%13s\n", "Erode", "Thresh", "Integ",
               "Hist", "ThreshS16", "ThreshU16", "And", "Sub");
    seq_printf(proc_entry, "%13d%13d%13d%13d%13d%13d%13d%13d\n", rti->cve_op_cycle.erode,
               rti->cve_op_cycle.thresh, rti->cve_op_cycle.integ, rti->cve_op_cycle.hist,
               rti->cve_op_cycle.thresh_s16, rti->cve_op_cycle.thresh_u16, rti->cve_op_cycle.and,
               rti->cve_op_cycle.sub);
    seq_printf(proc_entry, "\n%13s%13s%13s%13s%13s%13s%13s%13s\n", "Or", "Add", "Xor", "16to8",
               "OrdStat", "Map", "EqualH", "NCC");
    seq_printf(proc_entry, "%13d%13d%13d%13d%13d%13d%13d%13d\n", rti->cve_op_cycle.or,
               rti->cve_op_cycle.add, rti->cve_op_cycle.xor, rti->cve_op_cycle._16bit_to_8bit,
               rti->cve_op_cycle.ord_stat_filter, rti->cve_op_cycle.map,
               rti->cve_op_cycle.equalize_hist, rti->cve_op_cycle.ncc);
    seq_printf(proc_entry, "\n%13s%13s%13s%13s%13s%13s%13s%13s\n", "SAD", "CCL", "GMM", "Canny",
               "LBP", "NormGrad", "LK", "STconcer");
    seq_printf(proc_entry, "%13d%13d%13d%13d%13d%13d%13d%13d\n", rti->cve_op_cycle.sad,
               rti->cve_op_cycle.ccl, rti->cve_op_cycle.gmm, rti->cve_op_cycle.canny_edge,
               rti->cve_op_cycle.lbp, rti->cve_op_cycle.nrom_grad,
               rti->cve_op_cycle.lk_optial_flow_pry, rti->cve_op_cycle.st_candi_corner);
    seq_printf(proc_entry, "\n%13s%13s%13s%13s\n", "GradFg", "MatchMod", "UpdateMod", "TOF");
    seq_printf(proc_entry, "%13d%13d%13d%13d\n", rti->cve_op_cycle.grad_fg,
               rti->cve_op_cycle.match_bg_model, rti->cve_op_cycle.update_bg_model,
               rti->cve_op_cycle.tof);

    return 0;
}

static ssize_t cve_debug_write(struct file *file, const char __user *buffer, size_t count,
                               loff_t *ppos)
{
    int ok = 0;
    char *buf;

    buf = kmalloc(count + 1, GFP_KERNEL);
    if (!buf)
        return -ENOMEM;

    memset(buf, 0, count + 1);
    if (copy_from_user(buf, buffer, count))
        goto exit;

exit:
    kfree(buf);
    if (ok)
        return count;
    else
        return -EINVAL;
}

static int cve_debug_open(struct inode *inode, struct file *file)
{
    return single_open(file, cve_proc_show, NULL);
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0)
static const struct proc_ops cve_proc_file_ops = {
    /// kernel 5.15
    .proc_open = cve_debug_open,   .proc_read = seq_read,          .proc_lseek = seq_lseek,
    .proc_write = cve_debug_write, .proc_release = single_release,
};
#else
static const struct file_operations cve_proc_file_ops = {
    .open = cve_debug_open,
    .read = seq_read,
    .llseek = seq_lseek,
    .write = cve_debug_write,
    .release = single_release,
};
#endif

static int cve_close(struct inode *inode, struct file *file)
{
    return 0;
}

static int cve_open(struct inode *inode, struct file *file)
{
    return 0;
}

static long cve_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
    int ret = 0;

    if (cve_state) {
        CVE_ERR_TRACE("cve is not ready\n");
        return AML_ERR_CVE_NOTREADY;
    }

    atomic_inc_return(&cve_user_ref);

    switch (cmd) {
    case CVE_OP_DILATE: {
        CVE_OP_DILATE_T stOpDilate;
        ret = copy_from_user(&stOpDilate, (void *)arg, sizeof(CVE_OP_DILATE_T));
        if (ret) {
            goto exit;
        }
        ret = cve_dilate(&stOpDilate);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpDilate, sizeof(CVE_OP_DILATE_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_SAD: {
        CVE_OP_SAD_T stOpSAD;

        ret = copy_from_user(&stOpSAD, (void *)arg, sizeof(CVE_OP_SAD_T));
        if (ret) {
            goto exit;
        }
        ret = cve_sad(&stOpSAD);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpSAD, sizeof(CVE_OP_SAD_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_GRAD_FG: {
        CVE_OP_GRAD_FG_T stOpGradFg;

        ret = copy_from_user(&stOpGradFg, (void *)arg, sizeof(CVE_OP_GRAD_FG_T));
        if (ret) {
            goto exit;
        }
        ret = cve_grad_fg(&stOpGradFg);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpGradFg, sizeof(CVE_OP_GRAD_FG_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_CANNY_HYS_EDGE: {
        CVE_OP_CANNY_HYS_EDGE_T stOpCannyHysEdge;

        ret = copy_from_user(&stOpCannyHysEdge, (void *)arg, sizeof(CVE_OP_CANNY_HYS_EDGE_T));
        if (ret) {
            goto exit;
        }
        ret = cve_canny_hys_edge(&stOpCannyHysEdge);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpCannyHysEdge, sizeof(CVE_OP_CANNY_HYS_EDGE_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_SOBEL: {
        CVE_OP_SOBEL_T stOpSobel;

        ret = copy_from_user(&stOpSobel, (void *)arg, sizeof(CVE_OP_SOBEL_T));
        if (ret) {
            goto exit;
        }
        ret = cve_sobel(&stOpSobel);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpSobel, sizeof(CVE_OP_SOBEL_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_LK_OPTICAL_FLOW_PYR: {
        CVE_OP_LK_OPTICAL_FLOW_PYR_T stOpLkOpticalFlowPyr;

        ret = copy_from_user(&stOpLkOpticalFlowPyr, (void *)arg,
                             sizeof(CVE_OP_LK_OPTICAL_FLOW_PYR_T));
        if (ret) {
            goto exit;
        }
        ret = cve_lk_optical_flow_pyr(&stOpLkOpticalFlowPyr);
        if (ret) {
            goto exit;
        }
        ret =
            copy_to_user((void *)arg, &stOpLkOpticalFlowPyr, sizeof(CVE_OP_LK_OPTICAL_FLOW_PYR_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_BUILD_LK_OPTICAL_FLOW_PYR: {
        CVE_OP_BUILD_LK_OPTICAL_FLOW_PYR_T stOpBuildLkOpticalFlowPyr;

        ret = copy_from_user(&stOpBuildLkOpticalFlowPyr, (void *)arg,
                             sizeof(CVE_OP_BUILD_LK_OPTICAL_FLOW_PYR_T));
        if (ret) {
            goto exit;
        }
        ret = cve_build_lk_optical_flow_pyr(&stOpBuildLkOpticalFlowPyr);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpBuildLkOpticalFlowPyr,
                           sizeof(CVE_OP_BUILD_LK_OPTICAL_FLOW_PYR_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_NORM_GRAD: {
        CVE_OP_NORM_GRAD_T stOpNormGrad;

        ret = copy_from_user(&stOpNormGrad, (void *)arg, sizeof(CVE_OP_NORM_GRAD_T));
        if (ret) {
            goto exit;
        }
        ret = cve_norm_grad(&stOpNormGrad);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpNormGrad, sizeof(CVE_OP_NORM_GRAD_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_ST_CANDI_CORNER: {
        CVE_OP_ST_CANDI_CORNER_T stOpSTCandiCorner;

        ret = copy_from_user(&stOpSTCandiCorner, (void *)arg, sizeof(CVE_OP_ST_CANDI_CORNER_T));
        if (ret) {
            goto exit;
        }
        ret = cve_st_candi_corner(&stOpSTCandiCorner);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpSTCandiCorner, sizeof(CVE_OP_ST_CANDI_CORNER_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_MAP: {
        CVE_OP_MAP_T stOpMap;

        ret = copy_from_user(&stOpMap, (void *)arg, sizeof(CVE_OP_MAP_T));
        if (ret) {
            goto exit;
        }
        ret = cve_map(&stOpMap);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpMap, sizeof(CVE_OP_MAP_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_ERODE: {
        CVE_OP_ERODE_T stOpErode;

        ret = copy_from_user(&stOpErode, (void *)arg, sizeof(CVE_OP_ERODE_T));
        if (ret) {
            goto exit;
        }
        ret = cve_erode(&stOpErode);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpErode, sizeof(CVE_OP_ERODE_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_NCC: {
        CVE_OP_NCC_T stOpNcc;

        ret = copy_from_user(&stOpNcc, (void *)arg, sizeof(CVE_OP_NCC_T));
        if (ret) {
            goto exit;
        }
        ret = cve_ncc(&stOpNcc);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpNcc, sizeof(CVE_OP_NCC_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_EQUALIZE_HIST: {
        CVE_OP_EQUALIZE_HIST_T stOpEqualizeHist;

        ret = copy_from_user(&stOpEqualizeHist, (void *)arg, sizeof(CVE_OP_EQUALIZE_HIST_T));
        if (ret) {
            goto exit;
        }
        ret = cve_equalize_hist(&stOpEqualizeHist);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpEqualizeHist, sizeof(CVE_OP_EQUALIZE_HIST_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_CCL: {
        CVE_OP_CCL_T stOpCcl;

        ret = copy_from_user(&stOpCcl, (void *)arg, sizeof(CVE_OP_CCL_T));
        if (ret) {
            goto exit;
        }
        ret = cve_ccl(&stOpCcl);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpCcl, sizeof(CVE_OP_CCL_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_GMM: {
        CVE_OP_GMM_T stOpGmm;

        ret = copy_from_user(&stOpGmm, (void *)arg, sizeof(CVE_OP_GMM_T));
        if (ret) {
            goto exit;
        }
        ret = cve_gmm(&stOpGmm);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpGmm, sizeof(CVE_OP_GMM_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_THRESH_S16: {
        CVE_OP_THRESH_S16_T stOpThreshS16;

        ret = copy_from_user(&stOpThreshS16, (void *)arg, sizeof(CVE_OP_THRESH_S16_T));
        if (ret) {
            goto exit;
        }
        ret = cve_thresh_s16(&stOpThreshS16);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpThreshS16, sizeof(CVE_OP_THRESH_S16_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_16BIT_TO_8BIT: {
        CVE_OP_16BIT_TO_8BIT_T stOp16bitTo8bit;

        ret = copy_from_user(&stOp16bitTo8bit, (void *)arg, sizeof(CVE_OP_16BIT_TO_8BIT_T));
        if (ret) {
            goto exit;
        }
        ret = cve_16_to_8(&stOp16bitTo8bit);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOp16bitTo8bit, sizeof(CVE_OP_16BIT_TO_8BIT_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_THRESH_U16: {
        CVE_OP_THRESH_U16_T stOpThreshU16;

        ret = copy_from_user(&stOpThreshU16, (void *)arg, sizeof(CVE_OP_THRESH_U16_T));
        if (ret) {
            goto exit;
        }
        ret = cve_thresh_u16(&stOpThreshU16);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpThreshU16, sizeof(CVE_OP_THRESH_U16_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_LBP: {
        CVE_OP_LBP_T stOpLbp;

        ret = copy_from_user(&stOpLbp, (void *)arg, sizeof(CVE_OP_LBP_T));
        if (ret) {
            goto exit;
        }
        ret = cve_lbp(&stOpLbp);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpLbp, sizeof(CVE_OP_LBP_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_FILTER: {
        CVE_OP_FILTER_T stOpFilter;

        ret = copy_from_user(&stOpFilter, (void *)arg, sizeof(CVE_OP_FILTER_T));
        if (ret) {
            goto exit;
        }
        ret = cve_filter(&stOpFilter);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpFilter, sizeof(CVE_OP_FILTER_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_INTEG: {
        CVE_OP_INTEG_T stOpInteg;

        ret = copy_from_user(&stOpInteg, (void *)arg, sizeof(CVE_OP_INTEG_T));
        if (ret) {
            goto exit;
        }
        ret = cve_integ(&stOpInteg);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpInteg, sizeof(CVE_OP_INTEG_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_CSC: {
        CVE_OP_CSC_T stOpCsc;

        ret = copy_from_user(&stOpCsc, (void *)arg, sizeof(CVE_OP_CSC_T));
        if (ret) {
            goto exit;
        }
        ret = cve_csc(&stOpCsc);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpCsc, sizeof(CVE_OP_CSC_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_FILTER_AND_CSC: {
        CVE_OP_FILTER_AND_CSC_T stOpFilterCsc;

        ret = copy_from_user(&stOpFilterCsc, (void *)arg, sizeof(CVE_OP_FILTER_AND_CSC_T));
        if (ret) {
            goto exit;
        }
        ret = cve_filter_and_csc(&stOpFilterCsc);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpFilterCsc, sizeof(CVE_OP_FILTER_AND_CSC_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_ORD_STAT_FILTER: {
        CVE_OP_ORD_STAT_FILTER_T stOpOrdStatFilter;

        ret = copy_from_user(&stOpOrdStatFilter, (void *)arg, sizeof(CVE_OP_ORD_STAT_FILTER_T));
        if (ret) {
            goto exit;
        }
        ret = cve_ord_stat_filter(&stOpOrdStatFilter);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpOrdStatFilter, sizeof(CVE_OP_ORD_STAT_FILTER_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_THRESH: {
        CVE_OP_THRESH_T stOpThresh;

        ret = copy_from_user(&stOpThresh, (void *)arg, sizeof(CVE_OP_THRESH_T));
        if (ret) {
            goto exit;
        }
        ret = cve_thresh(&stOpThresh);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpThresh, sizeof(CVE_OP_THRESH_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_DMA: {
        CVE_OP_DMA_T stOpDMA;

        ret = copy_from_user(&stOpDMA, (void *)arg, sizeof(CVE_OP_DMA_T));
        if (ret) {
            goto exit;
        }
        ret = cve_dma(&stOpDMA);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpDMA, sizeof(CVE_OP_DMA_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_LUMA_STAT: {
        CVE_OP_LUAM_STAT_ARRAY_T stOpLuamStat;

        ret = copy_from_user(&stOpLuamStat, (void *)arg, sizeof(CVE_OP_LUAM_STAT_ARRAY_T));
        if (ret) {
            goto exit;
        }
        ret = cve_luma_stat(&stOpLuamStat);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpLuamStat, sizeof(CVE_OP_LUAM_STAT_ARRAY_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_AND: {
        CVE_OP_AND_T stOpAnd;

        ret = copy_from_user(&stOpAnd, (void *)arg, sizeof(CVE_OP_AND_T));
        if (ret) {
            goto exit;
        }
        ret = cve_and(&stOpAnd);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpAnd, sizeof(CVE_OP_AND_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_OR: {
        CVE_OP_OR_T stOpOr;

        ret = copy_from_user(&stOpOr, (void *)arg, sizeof(CVE_OP_OR_T));
        if (ret) {
            goto exit;
        }
        ret = cve_or(&stOpOr);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpOr, sizeof(CVE_OP_OR_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_XOR: {
        CVE_OP_XOR_T stOpXor;

        ret = copy_from_user(&stOpXor, (void *)arg, sizeof(CVE_OP_XOR_T));
        if (ret) {
            goto exit;
        }
        ret = cve_xor(&stOpXor);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpXor, sizeof(CVE_OP_XOR_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_SUB: {
        CVE_OP_SUB_T stOpSub;

        ret = copy_from_user(&stOpSub, (void *)arg, sizeof(CVE_OP_SUB_T));
        if (ret) {
            goto exit;
        }
        ret = cve_sub(&stOpSub);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpSub, sizeof(CVE_OP_SUB_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_ADD: {
        CVE_OP_ADD_T stOpAdd;

        ret = copy_from_user(&stOpAdd, (void *)arg, sizeof(CVE_OP_ADD_T));
        if (ret) {
            goto exit;
        }
        ret = cve_add(&stOpAdd);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpAdd, sizeof(CVE_OP_ADD_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_HIST: {
        CVE_OP_HIST_T stOpHist;

        ret = copy_from_user(&stOpHist, (void *)arg, sizeof(CVE_OP_HIST_T));
        if (ret) {
            goto exit;
        }
        ret = cve_hist(&stOpHist);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpHist, sizeof(CVE_OP_HIST_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_MAG_AND_ANG: {
        CVE_OP_MAG_AND_ANG_T stOpMagAndAng;

        ret = copy_from_user(&stOpMagAndAng, (void *)arg, sizeof(CVE_OP_MAG_AND_ANG_T));
        if (ret) {
            goto exit;
        }
        ret = cve_mag_and_ang(&stOpMagAndAng);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpMagAndAng, sizeof(CVE_OP_MAG_AND_ANG_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_MATCH_BG_MODEL: {
        CVE_OP_MATCH_BG_MODEL_T stOpMatchBgModel;

        ret = copy_from_user(&stOpMatchBgModel, (void *)arg, sizeof(CVE_OP_MATCH_BG_MODEL_T));
        if (ret) {
            goto exit;
        }
        ret = cve_match_bg_model(&stOpMatchBgModel);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpMatchBgModel, sizeof(CVE_OP_MATCH_BG_MODEL_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_UPDATE_BG_MODEL: {
        CVE_OP_UPDATE_BG_MODEL_T stOpUpdateBgModel;

        ret = copy_from_user(&stOpUpdateBgModel, (void *)arg, sizeof(CVE_OP_UPDATE_BG_MODEL_T));
        if (ret) {
            goto exit;
        }
        ret = cve_update_bg_model(&stOpUpdateBgModel);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpUpdateBgModel, sizeof(CVE_OP_UPDATE_BG_MODEL_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_TOF: {
        CVE_OP_TOF_T stOpTof;

        ret = copy_from_user(&stOpTof, (void *)arg, sizeof(CVE_OP_TOF_T));
        if (ret) {
            goto exit;
        }
        ret = cve_tof(&stOpTof);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpTof, sizeof(CVE_OP_TOF_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    case CVE_OP_QUERY: {
        CVE_OP_QUERY_T stOpQuery;

        ret = copy_from_user(&stOpQuery, (void *)arg, sizeof(CVE_OP_QUERY_T));
        if (ret) {
            goto exit;
        }
        ret = cve_query(&stOpQuery);
        if (ret) {
            goto exit;
        }
        ret = copy_to_user((void *)arg, &stOpQuery, sizeof(CVE_OP_QUERY_T));
        if (ret) {
            goto exit;
        }
        break;
    }
    default: {
        CVE_ERR_TRACE("cve can't support the op %08X\n", cmd);
        return AML_ERR_CVE_NOT_SUPPORT;
    }
    }
exit:
    atomic_dec_return(&cve_user_ref);

    return ret;
}

#ifdef CONFIG_COMPAT
static long cve_compat_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
    return cve_ioctl(file, cmd, arg);
}
#endif

static struct file_operations g_cve_fops = {
    .owner = THIS_MODULE,
    .open = cve_open,
    .release = cve_close,
    .unlocked_ioctl = cve_ioctl,
#ifdef CONFIG_COMPAT
    .compat_ioctl = cve_compat_ioctl,
#endif
};
static int cve_create_cmd_desc(cve_cq_desc_t *cq_desc)
{
    unsigned int cnt = 0;

    cq_desc->cmd_descs =
        (cve_cmd_desc_t *)kmalloc(sizeof(cve_cmd_desc_t) * cq_desc->total_cmd_max, GFP_KERNEL);
    if (cq_desc->cmd_descs == NULL) {
        CVE_ERR_TRACE("cmd descriptor alloc failed\n");
        return AML_ERR_CVE_NOMEM;
    }
    memset(cq_desc->cmd_descs, 0, sizeof(cve_cmd_desc_t) * cq_desc->total_cmd_max);

    for (cnt = 0; cnt < cq_desc->total_cmd_max; cnt++) {
        cq_desc->cmd_descs[cnt].cveHandle = -1;
        INIT_LIST_HEAD(&cq_desc->cmd_descs[cnt].list);
    }

    return 0;
}

static int cve_create_task_desc(cve_cq_desc_t *cq_desc)
{
    unsigned int cnt = 0;

    cq_desc->task_descs =
        (cve_task_desc_t *)kmalloc(sizeof(cve_task_desc_t) * cq_desc->task_max, GFP_KERNEL);
    if (cq_desc->task_descs == NULL) {
        CVE_ERR_TRACE("task descriptor alloc failed\n");
        return AML_ERR_CVE_NOMEM;
    }
    memset(cq_desc->task_descs, 0, sizeof(cve_task_desc_t) * cq_desc->task_max);
    for (cnt = 0; cnt < cq_desc->task_max; cnt++) {
        cq_desc->task_descs[cnt].task_id = cnt;
        task_descs_init(&cq_desc->task_descs[cnt]);
    }

    return 0;
}

static int cve_cmd_queue_init(cve_context_t *cve_context)
{
    int i = 0;
    int ret = 0;

    cve_context->cve_cq_buffer.size = 2 * CVE_OP_NODE_CMD_SIZE_MAX * cve_node_num;
    ret = cve_ion_alloc_buffer(&cve_context->cve_cq_buffer);
    if (ret) {
        CVE_ERR_TRACE("cve alloc cmd queue buffer error\n");
        return AML_ERR_CVE_NOMEM;
    }
    cve_context->cq_desc[0].phys_start = cve_context->cve_cq_buffer.phys_start;
    cve_context->cq_desc[0].virt_start = cve_context->cve_cq_buffer.virt_start;
    cve_context->cq_desc[1].phys_start =
        cve_context->cq_desc[0].phys_start + CVE_OP_NODE_CMD_SIZE_MAX * cve_node_num;
    cve_context->cq_desc[1].virt_start =
        cve_context->cq_desc[0].virt_start + CVE_OP_NODE_CMD_SIZE_MAX * cve_node_num;
    for (i = 0; i < 2; i++) {
        cve_context->cq_desc[i].length = CVE_OP_NODE_CMD_SIZE_MAX * cve_node_num;
        cve_context->cq_desc[i].total_cmd_max = cve_node_num;
        cve_context->cq_desc[i].task_max = cve_node_num;
        cve_context->cq_desc[i].task_phys_offset = 0;
        cve_context->cq_desc[i].task_virt_offset = 0;
        cve_context->cq_desc[i].cur_cmd_id = 0;
        cve_context->cq_desc[i].end_cmd_id = 0;
        cve_context->cq_desc[i].task_instant = 0;
        cve_create_task_desc(&cve_context->cq_desc[i]);
        cve_context->cq_desc[i].task_descs_create_index = 0;
        cve_context->cq_desc[i].task_descs_invoke_index = 0;
        cve_create_cmd_desc(&cve_context->cq_desc[i]);
        cve_context->cq_desc[i].cmd_descs_index = 0;
    }
    memset((void *)cve_context->cq_desc[0].virt_start, 0,
           2 * CVE_OP_NODE_CMD_SIZE_MAX * cve_node_num);

    return 0;
}

static int cve_cmd_queue_deinit(cve_context_t *cve_context)
{
    int i = 0;

    if (cve_context) {
        for (i = 0; i < 2; i++) {
            if (cve_context->cq_desc[i].task_descs) {
                kfree(cve_context->cq_desc[i].task_descs);
                cve_context->cq_desc[i].task_descs = NULL;
            }
            if (cve_context->cq_desc[i].cmd_descs) {
                kfree(cve_context->cq_desc[i].cmd_descs);
                cve_context->cq_desc[i].cmd_descs = NULL;
            }
            if (cve_context->cq_desc[i].phys_start && cve_context->cq_desc[i].virt_start) {
                cve_context->cq_desc[i].phys_start = 0;
                cve_context->cq_desc[i].virt_start = 0;
            }
        }
    }

    return 0;
}

static int cve_init(void)
{
    int ret = 0;

    cve_context.queue_wait = CVE_STATUS_CQ0;
    cve_context.queue_busy = CVE_STATUS_IDLE;

    init_waitqueue_head(&cve_context.cve_wait);

    ret = cve_cmd_queue_init(&cve_context);
    if (ret) {
        CVE_ERR_TRACE("cve init cmd queue failed!\n");
        goto ERR_CMD_QUEUE_INIT;
    }

    cve_context.cmd_bufs.length = CVE_OP_NODE_CMD_SIZE_MAX * cve_node_num;

    cve_context.cve_cmd_buffer.size = cve_context.cmd_bufs.length;
    ret = cve_ion_alloc_buffer(&cve_context.cve_cmd_buffer);
    if (ret) {
        CVE_ERR_TRACE("cve failed malloc mem!\n");
        ret = AML_ERR_CVE_NOMEM;
        goto ERR_CMD_QUEUE_DEINIT;
    }
    cve_context.cmd_bufs.virt_start = cve_context.cve_cmd_buffer.virt_start;
    cve_context.cmd_bufs.phys_start = cve_context.cve_cmd_buffer.phys_start;
    cve_context.cmd_bufs.cur_index = 0;
    cve_context.cmd_bufs.cmd_size = CVE_OP_NODE_CMD_SIZE_MAX;
    cve_context.cmd_bufs.index_max = cve_node_num;

    sema_init(&cve_context.cmd_bufs.cmd_buf_sema, 1);
    ret = request_threaded_irq(cve_irq, cve_irq_handler, cve_irq_handler_thread, IRQF_SHARED, "CVE",
                               (void *)cve_pdev);
    if (ret) {
        CVE_ERR_TRACE("CVE failed request irq!\n");
        goto ERR_FREE_CMD_BUF;
    }

    return 0;

ERR_FREE_CMD_BUF:
    cve_ion_free_buffer(&cve_context.cve_cmd_buffer);
    cve_context.cmd_bufs.phys_start = 0;
    cve_context.cmd_bufs.virt_start = 0;

ERR_CMD_QUEUE_DEINIT:
    cve_cmd_queue_deinit(&cve_context);
ERR_CMD_QUEUE_INIT:
    return ret;
}

static void cve_runtime_set_power(int enable, void *dev)
{
    int ret = -1;

    if (enable) {
        if (pm_runtime_suspended(dev)) {
            CVE_ERR_TRACE("cve power on\n");
            ret = pm_runtime_get_sync(dev);
            if (ret < 0) {
                CVE_ERR_TRACE("runtime get power error\n");
            }
        }
    } else {
        if (pm_runtime_active(dev)) {
            CVE_ERR_TRACE("cve power off\n");
            ret = pm_runtime_put_sync(dev);
            if (ret < 0) {
                CVE_ERR_TRACE("runtime put power error\n");
            }
        }
    }

    return;
}

static int cve_platform_module_init(void)
{
    int ret = 0;

    if (cve_state == CVE_SYS_STATUS_READY) {
        CVE_INFO_TRACE("cve ready, state = %d!\n", cve_state);
        return 0;
    }

    if (cve_state == CVE_SYS_STATUS_STOP) {
        CVE_ERR_TRACE("cve state = %d!\n", cve_state);
        return AML_ERR_CVE_BUSY;
    }

    ret = cve_init();
    if (ret) {
        CVE_ERR_TRACE("cve platform module init failed!\n");
        return ret;
    }

    cve_reg_init();
    cve_cq0_int_enable();
    cve_cq0_int_clear();
    cve_cq0_function_enable();
#if 0
    cve_sys_timeout_enable(CVE_SYS_TIMEOUT);
    cve_clk_gate_enable();
#endif
    cve_context.run_time_info.cve_cycle.cve_dbg_run_cycles_en = true;
    if (cve_context.run_time_info.cve_cycle.cve_dbg_run_cycles_en == true) {
        cve_op_run_time_cycle_enable();
    }
    cve_context.run_time_info.cve_cycle.cve_ute_stat_en = true;
    if (cve_context.run_time_info.cve_cycle.cve_ute_stat_en == true) {
        cve_ute_stat_enable(800000000);
    }

    cve_state = CVE_SYS_STATUS_READY;
    pm_runtime_enable(cve_pdev);
    cve_runtime_set_power(1, cve_pdev);

    return 0;
}

static void cve_platform_module_exit(void)
{
    if (cve_state != CVE_SYS_STATUS_IDLE) {
        cve_runtime_set_power(1, cve_pdev);
        cve_set_reset();
        free_irq(cve_irq, (void *)cve_pdev);
        cve_cmd_queue_deinit(&cve_context);
        cve_ion_free_buffer(&cve_context.cve_cq_buffer);
        cve_ion_free_buffer(&cve_context.cve_cmd_buffer);
        memset(&cve_context, 0, sizeof(cve_context_t));
#if 0
        cve_sys_timeout_disable();
#endif
        cve_cq0_function_disable();
        cve_cq0_int_disable();
        cve_runtime_set_power(0, cve_pdev);

        cve_state = CVE_SYS_STATUS_IDLE;
    }

    if (pm_runtime_enabled(cve_pdev)) {
        pm_runtime_disable(cve_pdev);
    }

    return;
}

static int aml_cve_probe(struct platform_device *pf_dev)
{
    void __iomem *regs;
    struct clk *clk_gate;
    struct clk *mux_gate = NULL;
    struct clk *mux_sel = NULL;
    int clk_rate = 0;
    int ret = -1;

    memset(&cve_context, 0, sizeof(cve_context_t));
    sprintf(cve_device.name, AML_MOD_CVE);
    cve_device.dev_num = 0;
    ret = register_chrdev(cve_device.dev_num, cve_device.name, &g_cve_fops);
    if (ret <= 0) {
        dev_err(&pf_dev->dev, "register cve device error\n");
        goto module_init_error;
    }
    cve_device.major = ret;
    cve_device.cve_class = class_create(THIS_MODULE, cve_device.name);
    if (IS_ERR(cve_device.cve_class)) {
        dev_err(&pf_dev->dev, "create cve class error\n");
        goto destroy_cve_chrdev;
    }

    cve_device.cve_pdev = device_create(cve_device.cve_class, NULL, MKDEV(cve_device.major, 0),
                                        NULL, cve_device.name);
    if (IS_ERR_OR_NULL(cve_device.cve_pdev)) {
        dev_err(&pf_dev->dev, "create cve device error\n");
        ret = AML_ERR_CVE_NOMEM;
        goto destroy_cve_class;
    }
    cve_irq = platform_get_irq_byname(pf_dev, "cve");
    if (cve_irq <= 0) {
        dev_err(&pf_dev->dev, "cannot find cve IRQ\n");
        goto destroy_cve_device;
    }

    cve_device.mem = platform_get_resource_byname(pf_dev, IORESOURCE_MEM, "cve");
    regs = devm_ioremap_resource(&pf_dev->dev, cve_device.mem);
    if (IS_ERR(regs)) {
        ret = PTR_ERR(regs);
        goto destroy_cve_device;
    }

    cve_regs_map = regs;
    ret = of_property_read_u32(pf_dev->dev.of_node, "clk-rate", &clk_rate);
    if (ret < 0) {
        clk_rate = 800000000;
    }

    mux_gate = devm_clk_get(&pf_dev->dev, "mux_gate");
    if (IS_ERR(mux_gate)) {
        dev_err(&pf_dev->dev, "cannot get cve mux_gate\n");
        ret = PTR_ERR(mux_gate);
        goto destroy_cve_device;
    }
    mux_sel = devm_clk_get(&pf_dev->dev, "mux_sel");
    if (IS_ERR(mux_gate)) {
        dev_err(&pf_dev->dev, "cannot get gdc mux_sel\n");
        ret = PTR_ERR(mux_gate);
        goto destroy_cve_device;
    }
    clk_set_parent(mux_sel, mux_gate);
    clk_gate = devm_clk_get(&pf_dev->dev, "clk_gate");
    if (IS_ERR(clk_gate)) {
        dev_err(&pf_dev->dev, "cannot get cve clk_gate\n");
        ret = PTR_ERR(clk_gate);
        goto destroy_cve_device;
    } else {
        clk_set_rate(clk_gate, clk_rate);
        clk_prepare_enable(clk_gate);
        ret = clk_get_rate(clk_gate);
    }

    cve_pdev = &pf_dev->dev;
    atomic_set(&cve_user_ref, 0);
    spin_lock_init(&cve_spinlock);
    sema_init(&cve_context.cve_sema, 1);
    cve_device.proc_node_entry = proc_create_data("cve", 0644, NULL, &cve_proc_file_ops, NULL);
    if (cve_device.proc_node_entry == NULL) {
        dev_err(&pf_dev->dev, "cve create proc failed!\n");
        ret = -1;
        goto destroy_cve_device;
    }

    ret = cve_platform_module_init();
    if (ret) {
        dev_err(&pf_dev->dev, "init cve module error\n");
        goto destroy_cve_device;
    }
    ret = 0;

    goto module_init_error;

destroy_cve_device:
    device_destroy(cve_device.cve_class, MKDEV(cve_device.major, 0));
destroy_cve_class:
    class_destroy(cve_device.cve_class);
destroy_cve_chrdev:
    unregister_chrdev(cve_device.major, cve_device.name);
module_init_error:

    return ret;
}

static int aml_cve_remove(struct platform_device *pf_dev)
{
    cve_platform_module_exit();
    remove_proc_entry("cve", NULL);
    device_destroy(cve_device.cve_class, MKDEV(cve_device.major, 0));
    class_destroy(cve_device.cve_class);
    unregister_chrdev(cve_device.major, cve_device.name);
    cve_regs_map = NULL;

    return 0;
}

static const struct of_device_id amlogic_cve_dt_match[] = {{
                                                               .compatible = "amlogic, cve",
                                                           },
                                                           {}};

static struct platform_driver aml_cve_driver = {.probe = aml_cve_probe,
                                                .remove = aml_cve_remove,
                                                .driver = {
                                                    .name = "aml_cve",
                                                    .owner = THIS_MODULE,
                                                    .of_match_table = amlogic_cve_dt_match,
                                                }};

static int __init aml_cve_module_init(void)
{
    int ret = -1;

    if (platform_driver_register(&aml_cve_driver) < 0) {
        CVE_ERR_TRACE("register platform driver error\n");
        return ret;
    }
    return 0;
}

static void aml_cve_module_exit(void)
{
    platform_driver_unregister(&aml_cve_driver);
}

module_init(aml_cve_module_init);
module_exit(aml_cve_module_exit);

MODULE_AUTHOR("AMLOGIC");
MODULE_VERSION("1.0");
MODULE_LICENSE("GPL v2");
