/*
 * Copyright (C) 2019-2024 Amlogic, Inc. All rights reserved.
 *
 */

#ifndef __AML_CVE_H__
#define __AML_CVE_H__

#include "aml_comm_cve.h"
#include "aml_common.h"

#define CVE_LUMA_RECT_MAX 64
#define CVE_HIST_NUM 256
#define CVE_MAP_NUM 256
#define CVE_MAX_REGION_NUM 384
#define CVE_ST_MAX_CORNER_NUM 500

/* Crop */
typedef struct {
    AML_BOOL_E enCrop; /*Crop enable*/
    AML_U16 u16X;      /*Crop xstart*/
    AML_U16 u16Y;      /*Crop ystart*/
    AML_U16 u16Width;  /*Crop xsize*/
    AML_U16 u16Height; /*Crop ysize*/
} CVE_CROP_T;

/* mode of the DMA */
typedef enum {
    CVE_DMA_MODE_DIRECT_COPY = 0x0,   /* direct copy */
    CVE_DMA_MODE_INTERVAL_COPY = 0x1, /* interval copy */
    CVE_DMA_MODE_SET_3BYTE = 0x2,     /* 3-byte filling mode */
    CVE_DMA_MODE_SET_8BYTE = 0x3,     /* 8-byte filling mode */
    CVE_DMA_MODE_NOT = 0x4,           /* u8data is reversed */
    CVE_DMA_MODE_BUTT
} CVE_DMA_MODE_E;

/* control parameter of the DMA */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_DMA_MODE_E enMode;
    AML_U64 u64Val;      /* memset mode parameter */
    AML_U8 u8HorSegSize; /* interval-copy,u8HorSegSize divides bytes of each
                                        row,The limit values are 2,3,4,8,16 */

    AML_U8 u8ElemSize; /* interval-copy,copy to the valid bytes before each
                                    segment in the valid row,0<u8ElemSize<u8HorSegSize */

    AML_U8 u8VerSegRows; /* interval-copy,make a row copy from each
                                                u8VerSegRows */
} CVE_DMA_CTRL_T;

/* mode of the LumaStat */
typedef enum {
    CVE_LUMA_STAT_MODE_STEP_0 = 0x0,
    CVE_LUMA_STAT_MODE_STEP_1 = 0x1,
    CVE_LUMA_STAT_MODE_STEP_2 = 0x2,
    CVE_LUMA_STAT_MODE_STEP_3 = 0x3,

    CVE_LUMA_STAT_MODE_BUTT
} CVE_LUMA_STAT_MODE_E;

/* control parameter of the LumaStat */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_LUMA_STAT_MODE_E enMode;
    AML_U8 u8MaxLumaRect; /* 0 <= u8MaxLumaRect  <=  63 */
} CVE_LUMA_STAT_ARRAY_CTRL_T;

/*
 * These parameters need to be set when using the filter operator
 * Change the template filter from 3x3 to 5x5
 */
typedef struct {
    CVE_CROP_T stCrop;
    AML_S8 as8Mask[25]; /* Template parameter filter factor */
    AML_U8 u8Norm;      /* standardized parameter,shift right >> */
} CVE_FILTER_CTRL_T;

/* mode of the CSC */
typedef enum {
    CVE_CSC_MODE_PIC_BT601_YUV2RGB = 0x0,
    CVE_CSC_MODE_PIC_BT601_YUV2BGR = 0x1,
    CVE_CSC_MODE_PIC_BT601_YUV2HSV = 0x2,
    CVE_CSC_MODE_PIC_BT601_YUV2LAB = 0x3,
    CVE_CSC_MODE_PIC_BT601_RGB2YUV = 0x4,
    CVE_CSC_MODE_PIC_BT601_BGR2YUV = 0x5,
    CVE_CSC_MODE_PIC_BT601_RGB2HSV = 0x6,
    CVE_CSC_MODE_PIC_BT601_RGB2LAB = 0x7,
    CVE_CSC_MODE_PIC_BT601_RGB2BGR = 0x8,
    CVE_CSC_MODE_PIC_BT601_BGR2RGB = 0x9,
    CVE_CSC_MODE_PIC_BT601_YVU2RGB = 0xa,
    CVE_CSC_MODE_PIC_BT601_YVU2BGR = 0xb,
    CVE_CSC_MODE_PIC_BT601_YVU2HSV = 0xc,
    CVE_CSC_MODE_PIC_BT601_YVU2LAB = 0xd,
    CVE_CSC_MODE_PIC_BT601_RGB2YVU = 0xe,
    CVE_CSC_MODE_PIC_BT601_BGR2YVU = 0xf,
    CVE_CSC_MODE_PIC_BT601_BGR2HSV = 0x10,
    CVE_CSC_MODE_PIC_BT601_BGR2LAB = 0x11,
    CVE_CSC_MODE_BUTT
} CVE_CSC_MODE_E;

/*
 * These parameters need to be set when using the CSC operator
 * The CSC output format is removed and replaced with output image type
 */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_CSC_MODE_E enMode; /* CSC mode */
} CVE_CSC_CTRL_T;

/*
 * These parameters need to be set when using the filter+CSC operator
 * Control message about the filter and CSC needs to be set
 * Change the template filter from 3x3 to 5x5
 */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_CSC_MODE_E enMode; /* CSC mode */
    AML_S8 as8Mask[25];    /* Template parameter filter factor */
    AML_U8 u8Norm;         /* standardized parameter,shift right >> */
} CVE_FILTER_AND_CSC_CTRL_T;

/* output ctrl of the SOBEL */
typedef enum {
    CVE_SOBEL_OUT_CTRL_BOTH = 0x0, /* horizontal and vertical output */
    CVE_SOBEL_OUT_CTRL_HOR = 0x1,  /* horizontal output */
    CVE_SOBEL_OUT_CTRL_VER = 0x2,  /* vertical output */
    CVE_SOBEL_OUT_CTRL_BUTT
} CVE_SOBEL_OUT_CTRL_E;

/* control parameter of the SOBEL */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_SOBEL_OUT_CTRL_E enOutCtrl; /* The output format */
    AML_S8 as8Mask[25];             /* Template parameter SOBEL factor */
} CVE_SOBEL_CTRL_T;

/* magnitude and angle output type */
typedef enum {
    CVE_MAG_AND_ANG_OUT_CTRL_MAG = 0x0,         /* magnitude output */
    CVE_MAG_AND_ANG_OUT_CTRL_MAG_AND_ANG = 0x1, /* output both magnitude and angle */
    CVE_MAG_AND_ANG_OUT_CTRL_HOG = 0x2,         /* output both magnitude and angle in HOG mode  */
    CVE_MAG_AND_ANG_OUT_CTRL_BUTT
} CVE_MAG_AND_ANG_OUT_CTRL_E;

/* control parameter of the magnitude and angle */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_MAG_AND_ANG_OUT_CTRL_E enOutCtrl;
    AML_U16 u16Thr;
    AML_S8 as8Mask[25]; /* Template parameter magnitude and angle factor */
} CVE_MAG_AND_ANG_CTRL_T;

/* Control parameter of the Dilate */
typedef struct {
    CVE_CROP_T stCrop;
    AML_U8 au8Mask[25]; /* template parameter must be 0 or 255 */
} CVE_DILATE_CTRL_T;

/* Control parameter of the Erode */
typedef struct {
    CVE_CROP_T stCrop;
    AML_U8 au8Mask[25]; /* template parameter must be 0 or 255 */
} CVE_ERODE_CTRL_T;

/* Thresh mode type */
typedef enum {
    CVE_THRESH_MODE_BINARY = 0x0,
    /* outVal = minVal,srcVal <= lowThr;
      outVal = maxVal,srcVal > lowThr */
    CVE_THRESH_MODE_TRUNC = 0x1,
    /* outVal = srcVal,srcVal <= lowThr;
              outVal = maxVal,srcVal > lowThr */
    CVE_THRESH_MODE_TO_MINVAL = 0x2,
    /* outVal = minVal,srcVal <= lowThr;
      outVal = srcVal,srcVal > lowThr */
    CVE_THRESH_MODE_MIN_MID_MAX = 0x3,
    /* outVal = minVal,srcVal <= lowThr;
      outVal = midVal,lowThr < srcVal <= highThr;
              outVal = maxVal,srcVal > highThr */
    CVE_THRESH_MODE_ORI_MID_MAX = 0x4,
    /* outVal = srcVal,srcVal <= lowThr;
      outVal = midVal,lowThr < srcVal <= highThr;
              outVal = maxVal,srcVal > highThr */
    CVE_THRESH_MODE_MIN_MID_ORI = 0x5,
    /* outVal = minVal,srcVal <= lowThr;
      outVal = midVal,lowThr < srcVal <= highThr;
              outVal = srcVal,srcVal > highThr */
    CVE_THRESH_MODE_MIN_ORI_MAX = 0x6,
    /* outVal = minVal,srcVal <= lowThr;
      outVal = srcVal,lowThr < srcVal <= highThr;
              outVal = maxVal,srcVal > highThr */
    CVE_THRESH_MODE_ORI_MID_ORI = 0x7,
    /* outVal = srcVal,srcVal <= lowThr;
      outVal = midVal,lowThr < srcVal <= highThr;
      outVal = srcVal,srcVal > highThr */

    CVE_THRESH_MODE_BUTT
} CVE_THRESH_MODE_E;

/* Control parameters of the Thresh */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_THRESH_MODE_E enMode;
    AML_U8 u8LowThr;  /* Threshold value,usr-defined,0 <= LowThr <= 255 */
    AML_U8 u8HighThr; /* Threshold value,usr-defined;
                                        if (enMode < CVE_THRESH_MODE_MIN_MID_MAX) U8HighThr is not
                         used; else 0 <= u8LowThr <= u8HighThr <= 255 */
    AML_U8 u8MinVal;  /* tri level threshold minimum */
    AML_U8 u8MidVal;  /* tri level threshold middle;
                                                if (enMode < 2) U8MidVal is not used */
    AML_U8 u8MaxVal;  /* tri level threshold maximum */
} CVE_THRESH_CTRL_T;

/* Sub output type */
typedef enum {
    CVE_SUB_MODE_ABS = 0x0,   /* Absolute value of the srcVal difference */
    CVE_SUB_MODE_SHIFT = 0x1, /* Result right shift 1bit to reserve the signed bit
                                                        ,get the output */
    CVE_SUB_MODE_THRESH = 0x2,
    CVE_SUB_MODE_BUTT
} CVE_SUB_MODE_E;

/* Control parameter of the SUB */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_SUB_MODE_E enMode;
    AML_U16 u16ThreshRatio; /* Threshold ratio of the difference */
} CVE_SUB_CTRL_T;

/* Integ output results type */
typedef enum {
    CVE_INTEG_OUT_CTRL_COMBINE = 0x0,
    CVE_INTEG_OUT_CTRL_TUM = 0x1,
    CVE_INTEG_OUT_CTRL_TQSUM = 0x2,
    CVE_INTEG_OUT_CTRL_BUTT
} CVE_INTEG_OUT_CTRL_E;

/* Control parameter of the integ */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_INTEG_OUT_CTRL_E enOutCtrl;
} CVE_INTEG_CTRL_T;

/* Control parameter of the hist */
typedef struct {
    CVE_CROP_T stCrop;
} CVE_HIST_CTRL_T;

/* Thresh_s16 mode type */
typedef enum {
    CVE_THRESH_S16_MODE_S16_TO_S8_MIN_MID_MAX = 0x0,
    CVE_THRESH_S16_MODE_S16_TO_S8_MIN_ORI_MAX = 0x1,
    CVE_THRESH_S16_MODE_S16_TO_U8_MIN_MID_MAX = 0x2,
    CVE_THRESH_S16_MODE_S16_TO_U8_MIN_ORI_MAX = 0x3,

    CVE_THRESH_S16_MODE_BUTT
} CVE_THRESH_S16_MODE_E;

/* Control parameters of the Thresh_s16 */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_THRESH_S16_MODE_E enMode;
    AML_S16 s16LowThr;    /* threshold value ,usr-defined */
    AML_S16 s16HighThr;   /* threshold value ,usr-defined */
    CVE_8BIT_U un8MinVal; /* tri level threshold minimum */
    CVE_8BIT_U un8MidVal; /* tri level threshold midimum */
    CVE_8BIT_U un8MaxVal; /* tri level threshold maximum */
} CVE_THRESH_S16_CTRL_T;

/* Thresh_u16 mode type */
typedef enum {
    CVE_THRESH_U16_MODE_U16_TO_U8_MIN_MID_MAX = 0x0,
    CVE_THRESH_U16_MODE_U16_TO_U8_MIN_ORI_MAX = 0x1,

    CVE_THRESH_U16_MODE_BUTT
} CVE_THRESH_U16_MODE_E;

/* Control parameters of the Thresh_u16 */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_THRESH_U16_MODE_E enMode;
    AML_U16 u16LowThr;  /* threshold value ,usr-defined */
    AML_U16 u16HighThr; /* threshold value ,usr-defined */
    AML_U8 u8MinVal;    /* tri level threshold minimum */
    AML_U8 u8MidVal;    /* tri level threshold midimum */
    AML_U8 u8MaxVal;    /* tri level threshold maximum */
} CVE_THRESH_U16_CTRL_T;

/* 16bit to  8bit mode */
typedef enum {
    CVE_16BIT_TO_8BIT_MODE_S16_TO_S8 = 0x0,
    CVE_16BIT_TO_8BIT_MODE_S16_TO_U8_ABS = 0x1,
    CVE_16BIT_TO_8BIT_MODE_S16_TO_U8_BIAS = 0x2,
    CVE_16BIT_TO_8BIT_MODE_U16_TO_U8 = 0x3,

    CVE_16BIT_TO_8BIT_MODE_BUTT
} CVE_16BIT_TO_8BIT_MODE_E;

/* Control parameters of the 16bit to 8bit */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_16BIT_TO_8BIT_MODE_E enMode;
    AML_U16 u8LineCoefThr;
    AML_S8 s8Bias;
} CVE_16BIT_TO_8BIT_CTRL_T;

/* OrdstaFilter type */
typedef enum {
    CVE_ORD_STAT_FILTER_MODE_MIN = 0x0,
    CVE_ORD_STAT_FILTER_MODE_MEDIAN = 0x1,
    CVE_ORD_STAT_FILTER_MODE_MAX = 0x2,

    CVE_ORD_STAT_FILTER_MODE_BUTT
} CVE_ORD_STAT_FILTER_MODE_E;

/* Control parameters of the OrdstaFilter */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_ORD_STAT_FILTER_MODE_E enMode;

} CVE_ORD_STAT_FILTER_CTRL_T;

/* Map type */
typedef enum {
    CVE_MAP_MODE_U8 = 0x0,
    CVE_MAP_MODE_S16 = 0x1,
    CVE_MAP_MODE_U16 = 0x2,

    CVE_MAP_MODE_BUTT
} CVE_MAP_MODE_E;

/* Control parameter of the Map */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_MAP_MODE_E enMode;
} CVE_MAP_CTRL_T;

/* unsigned 8bit LUT memory of the Map */
typedef struct {
    AML_U8 au8Map[CVE_MAP_NUM];
} CVE_MAP_U8BIT_LUT_MEM_T;

/* unsigned 16bit LUT memory of the Map */
typedef struct {
    AML_U16 au16Map[CVE_MAP_NUM];
} CVE_MAP_U16BIT_LUT_MEM_T;

/* 16bit LUT memory of the Map */
typedef struct {
    AML_S16 as16Map[CVE_MAP_NUM];
} CVE_MAP_S16BIT_LUT_MEM_T;

/* Control member of the Equalizehist */
typedef struct {
    AML_U32 au32Hist[CVE_HIST_NUM];
    AML_U8 au8Map[CVE_MAP_NUM];
} CVE_EQUALIZE_HIST_CTRL_MEM_T;

/* Control parameter of the Equalizehist */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_MEM_INFO_T stMem;
} CVE_EQUALIZE_HIST_CTRL_T;

/* Control parameters of the Add */
typedef struct {
    CVE_CROP_T stCrop;
    AML_U0Q16 u0q16X;
    AML_U0Q16 u0q16Y;
} CVE_ADD_CTRL_T;

/* Control parameters of the And */
typedef struct {
    CVE_CROP_T stCrop;
} CVE_AND_CTRL_T;

/* Control parameters of the Or */
typedef struct {
    CVE_CROP_T stCrop;
} CVE_OR_CTRL_T;

/* Control parameters of the Xor */
typedef struct {
    CVE_CROP_T stCrop;
} CVE_XOR_CTRL_T;

/* NCC dst memory */
typedef struct {
    AML_U64 u64Numerator;
    AML_U64 u64QuadSum1;
    AML_U64 u64QuadSum2;
    AML_U8 u8Reserved[8];
} CVE_NCC_DST_MEM_T;

/* Region factors */
typedef struct {
    AML_U32 u32Area;   /* expressed by the number of pixel */
    AML_U16 u16Left;   /* The left boundary of the rectangle */
    AML_U16 u16Right;  /* The right boundary of the rectangle */
    AML_U16 u16Top;    /* The top boundary of the rectangle */
    AML_U16 u16Bottom; /* The bottom boundary of the rectangle */
    AML_U16 u16label;
} CVE_REGION_T;

/* CCBLOB factors */
typedef struct {
    CVE_REGION_T astRegion[CVE_MAX_REGION_NUM]; /* u32Areae > 0, label = Arrayindex+1 */
    AML_U16 u16RegionNum;                       /* CCL connected component */
} CVE_CCBLOB_T;

/* NCC mode */
typedef enum {
    CVE_NCC_MODE_COM = 0x0,
    CVE_NCC_MODE_SIMILAR = 0x1,

    CVE_NCC_MODE_BUTT
} CVE_NCC_MODE_E;

/* NCC control factors */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_NCC_MODE_E enMode;
    AML_U8 u8Src1Offset;
    AML_U8 u8Src0Offset;
} CVE_NCC_CTRL_T;

/* CCL input mode */
typedef enum {
    CVE_CCL_INPUT_DATA_MODE_U8 = 0x0, /* [0,255] */
    CVE_CCL_INPUT_DATA_MODE_U1 = 0x1, /* {0,1} */

    CVE_CCL_INPUT_DATA_MODE_BUTT
} CVE_CCL_INPUT_DATA_MODE_E;

/* CCL control factor */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_CCL_INPUT_DATA_MODE_E enInputDataMode;
} CVE_CCL_CTRL_T;

/* GMM output mode */
typedef enum {
    CVE_GMM_OUTPUT_MODE_FG = 0x0,        /* output FG */
    CVE_GMM_OUTPUT_MODE_FG_AND_BG = 0x1, /* output FG and BG */

    CVE_GMM_OUTPUT_MODE_BUTT
} CVE_GMM_OUTPUT_MODE_E;

/* GMM sensitivity mode */
typedef enum {
    CVE_GMM_SNS_FACTOR_MODE_GLB = 0x0, /* Global mode */
    CVE_GMM_SNS_FACTOR_MODE_PIX = 0x1, /* Pixel mode */

    CVE_GMM_SNS_FACTOR_MODE_BUTT
} CVE_GMM_SNS_FACTOR_MODE_E;

/* GMM duration update factor mode */
typedef enum {
    CVE_GMM_DURATION_UPDATE_FACTOR_MODE_GLB = 0x0, /* Global mode */
    CVE_GMM_DURATION_UPDATE_FACTOR_MODE_PIX = 0x1, /* Pixel mode */

    CVE_GMM_DURATION_UPDATE_FACTOR_MODE_BUTT
} CVE_GMM_DURATION_UPDATE_FACTOR_MODE_E;

/* GMMM down scale factor mode */
typedef enum {
    CVE_GMM_DOWN_SCALE_MODE_DISABLE = 0x0,
    CVE_GMM_DOWN_SCALE_MODE_2X2 = 0x1,
    CVE_GMM_DOWN_SCALE_MODE_4X4 = 0x2,

    CVE_GMM_DOWN_SCALE_MODE_BUTT
} CVE_GMM_DOWN_SCALE_MODE_E;

typedef enum {
    CVE_GMM_OUTPUT_DATA_MODE_U8 = 0x0, /* [0,255] */
    CVE_GMM_OUTPUT_DATA_MODE_U1 = 0x1, /* {0,1} */

    CVE_GMM_OUTPUT_DATA_MODE_BUTT
} CVE_GMM_OUTPUT_DATA_MODE_E;

/* GMM control factor */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_GMM_OUTPUT_MODE_E enOutputMode;
    CVE_GMM_SNS_FACTOR_MODE_E enSnsFactorMode; /* sensitivity factor mode */
    CVE_GMM_DURATION_UPDATE_FACTOR_MODE_E
    enDurationUpdateFactorMode;                  /* Duration update factor mode */
    CVE_GMM_DOWN_SCALE_MODE_E enDownScaleMode;   /* Down scale mode */
    CVE_GMM_OUTPUT_DATA_MODE_E enOutputDataMode; /* output data mode */
    AML_U10Q6 u10q6NoiseVar;                     /* Initial noise Variance */
    AML_U10Q6 u10q6MaxVar;                       /* Max Variance */
    AML_U10Q6 u10q6MinVar;                       /* Min Variance */
    AML_U0Q16 u0q16LearnRate;                    /* Learn rate */
    AML_U0Q16 u0q16InitWeight;                   /* Initial weight */
    AML_U0Q16 u0q16WeightThr;                    /* Weight threshold */
    AML_U8 u8ModelNum;                           /*  [1,5] */
    AML_U3Q7 u3q7SigmaScale;                     /* Variance scale */
    AML_BOOL_E enFastLearn;                      /* Fast learn switch */
} CVE_GMM_CTRL_T;

/* CannyHysEdge control factor */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_MEM_INFO_T stMem;
    AML_BOOL_E bGauss;
    AML_U16 u16LowThr;
    AML_U16 u16HighThr;
    AML_S8 as8Mask[25];
} CVE_CANNY_HYS_EDGE_CTRL_T;

/* CannyEdge output data mode  */
typedef enum {
    CVE_CANNY_EDGE_OUTPUT_DATA_MODE_U8 = 0x0, /* [0,255] */
    CVE_CANNY_EDGE_OUTPUT_DATA_MODE_U1 = 0x1, /* {0,1} */

    CVE_CANNY_EDGE_OUTPUT_DATA_MODE_BUTT
} CVE_CANNY_EDGE_OUTPUT_DATA_MODE_E;

/* control parameter of the CannyEdge */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_CANNY_EDGE_OUTPUT_DATA_MODE_E enOutputDataMode;
    AML_U32 u32CannyCount;
} CVE_CANNY_EDGE_CTRL_T;

/*  LBP compare mode */
typedef enum {
    CVE_LBP_CMP_MODE_NORMAL = 0x0, /* P(x)-P(center)>= un8BitThr.s8Val, s(x)=1; else s(x)=0 */
    CVE_LBP_CMP_MODE_ABS = 0x1,    /* Abs(P(x)-P(center))>=un8BitThr.u8Val, s(x)=1; else s(x)=0 */

    CVE_LBP_CMP_MODE_BUTT
} CVE_LBP_CMP_MODE_E;

/* LBP control factor */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_LBP_CMP_MODE_E enMode;
    CVE_8BIT_U un8BitThr;
} CVE_LBP_CTRL_T;

/* GradientFilter output format type */
typedef enum {
    CVE_NORM_GRAD_OUT_CTRL_HOR_AND_VER = 0x0,
    CVE_NORM_GRAD_OUT_CTRL_HOR = 0x1,
    CVE_NORM_GRAD_OUT_CTRL_VER = 0x2,
    CVE_NORM_GRAD_OUT_CTRL_COMBINE = 0x3,

    CVE_NORM_GRAD_OUT_CTRL_BUTT
} CVE_NORM_GRAD_OUT_CTRL_E;

/* Control parameters of the GradientFilter */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_NORM_GRAD_OUT_CTRL_E enOutCtrl;
    AML_S8 as8Mask[25];
    AML_U8 u8Norm;
} CVE_NORM_GRAD_CTRL_T;

/* LK Optical flow output mode */
typedef enum {
    CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_NONE = 0, /* none */
    CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_ERR = 1,  /* error */
    CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_BOTH = 2, /* error */

    CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_BUTT
} CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_E;

/* Control parameter of the LK Optical flow */
typedef struct {
    CVE_CROP_T stCrop;
    AML_U8 u8MaxLevel; /* Sub 1 from the number of levels in the pyramid,
                                                corresponding to pyramid layers [1,4]，
                                                （range: [0,3]; default:3） */
} CVE_BUILD_LK_OPTICAL_FLOW_PYR_CTRL_T;

/* Control parameter of the Build LK Optical flow */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_LK_OPTICAL_FLOW_PYR_OUT_MODE_E enOutMode; /* out mode */
    AML_BOOL_E bUseInitFlow;                      /* Use initial flow or not */
    AML_U16 u16PtsNum; /* Feature points number (range: [1,500]; default : 100) */
    AML_U8 u8MaxLevel; /* Sub 1 from the number of levels in the pyramid,
                                                 corresponding to pyramid layers [1,4]，
                                                 （range: [0,3]; default:3） */
    AML_U8 u8IterCnt;  /* Max iteration times (range: [1,20]; default: 10) */
    AML_U0Q8 u0q8Eps;  /* exit criteria:(range: [1,255]; default :10) */
} CVE_LK_OPTICAL_FLOW_PYR_CTRL_T;

/* STCandiCorner output data mode */
typedef enum {
    CVE_ST_CANDI_CORNER_OUTPUT_DATA_MODE_U8 = 0x0, /* [0,255] */
    CVE_ST_CANDI_CORNER_OUTPUT_DATA_MODE_U1 = 0x1, /* {0,1} */

    CVE_ST_CANDI_CORNER_OUTPUT_DATA_MODE_BUTT
} CVE_ST_CANDI_CORNER_OUTPUT_DATA_MODE_E;

/* Control parameter of the  STCandiCorner */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_ST_CANDI_CORNER_OUTPUT_DATA_MODE_E enOutputDataMode;
    CVE_MEM_INFO_T stMem;
} CVE_ST_CANDI_CORNER_CTRL_T;

/* ST Corner struct */
typedef struct {
    CVE_POINT_U16_T astCorner[CVE_ST_MAX_CORNER_NUM];
    AML_U16 u16CornerNum;
} CVE_ST_CORNER_INFO_T;

/* Control parameters of the STCorner */
typedef struct {
    AML_U32 u32CornerNum;
    AML_U16 u32MaxCornerNum;
    AML_U16 u32MinDist;
} CVE_ST_CORNER_CTRL_T;

/* Gradfg mode */
typedef enum {
    CVE_GRAD_FG_MODE_DIRECT = 0x0, /* output = diff = current gradient - background gradient  */
    CVE_GRAD_FG_MODE_THR = 0x1,    /* if(diff < threshold) output = 0;
                                                            else output = diff */
    CVE_GRAD_FG_MODE_THR_FG = 0x2, /* if(foreground != O && diff < threshold) output = diff;
                                              else  output = 0 */
    CVE_GRAD_FG_MODE_FG = 0x3,     /* if(foreground != 0) output = diff;
                                                            else output = 0 */
    CVE_GRAD_FG_MODE_BUTT
} CVE_GRAD_FG_MODE_E;

/* Gradfg ctrl factor */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_GRAD_FG_MODE_E enMode; /* ctrl mode (range: [0,3]; default: 0) */
    AML_U8
    u8MinMagDiff; /* Max gradient and min gradient difference threshold(range : [0,255]; default:
                     100) */
} CVE_GRAD_FG_CTRL_T;

/* Candi background pixels parameters */
typedef struct {
    AML_U8Q4F4 u8q4f4Mean;    /* Candidate background grays value */
    AML_U16 u16StartTime;     /* Candidate background start time */
    AML_U16 u16SumAccessTime; /* Candidate background Cumulative access time */
    AML_U16 u16ShortKeepTime; /* Candidate background hold time */
    AML_U8 u8ChgCond;         /* The time condition is that candidate background
                                                                enter a state of change */
    AML_U8 u8PotenBgLife;     /* Potential background cumulative access time */
} CVE_CANDI_BG_PIX_T;

/* WorkBg pixels parameters */
typedef struct {
    AML_U8Q4F4 u8q4f4Mean; /* Background grays value  0# */
    AML_U16 u16AccTime;    /* Background Cumulative access time */
    AML_U8 u8PreGray;      /* The ray value of the last pixel */
    AML_U5Q3 u5q3DiffThr;  /* Differential threshold */
    AML_U8 u8AccFlag;      /* Background access flag */
    AML_U8 u8BgGray[3];    /* Background grays value  1# to 3# */
} CVE_WORK_BG_PIX_T;

/* BgLife parameters */
typedef struct {
    AML_U8 u8WorkBgLife[3]; /* background vitality 1# to 3# */
    AML_U8 u8CandiBgLife;   /* Candidate background vitality */
} CVE_BG_LIFE_T;

/* BgModel Pixel parameters */
typedef struct {
    CVE_WORK_BG_PIX_T stWorkBgPixel; /* Working background */
    CVE_CANDI_BG_PIX_T stCandiPixel; /* Candidate background */
    CVE_BG_LIFE_T stBgLife;          /* Background vitality */
} CVE_BG_MODEL_PIX_T;

/* FgStat data */
typedef struct {
    AML_U32 u32PixNum;
    AML_U32 u32SumLum;
} CVE_FG_STAT_DATA_T;

/* BgStat data */
typedef struct {
    AML_U32 u32PixNum;
    AML_U32 u32SumLum;
} CVE_BG_STAT_DATA_T;

/* MatchBgModel output mode */
typedef enum {
    CVE_MATCH_BG_MODEL_OUTPUT_MODE_FG = 0x0,                      /* Fg output */
    CVE_MATCH_BG_MODEL_OUTPUT_MODE_FG_AND_BG = 0x1,               /* Fg and Bg output */
    CVE_MATCH_BG_MODEL_OUTPUT_MODE_FG_AND_BG_AND_CURDIFFBG = 0x2, /* Fg ,Bg and curDiffbg output */
    CVE_MATCH_BG_MODEL_OUTPUT_MODE_ALL = 0x3, /* all output (Fg,Bg,curDiffbg,frmDiff) */
    CVE_MATCH_BG_MODEL_OUTPUT_MODE_BUTT
} CVE_MATCH_BG_MODEL_OUTPUT_MODE_E;

/* MatchBgModel down scale mode */
typedef enum {
    CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_DISABLE = 0x0,
    CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_2X2 = 0x1,
    CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_4X4 = 0x2,

    CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_BUTT
} CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_E;

/* Control parameters of the MatchBgModel */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_MATCH_BG_MODEL_OUTPUT_MODE_E enOutputMode; /* output mode (range: [0,2]; default: 0) */
    CVE_MATCH_BG_MODEL_DOWN_SCALE_MODE_E
    enDownScaleMode;      /* Down scale mdoe (range: [0,2]; default: 2) */
    AML_U8Q4 u8q4DistThr; /* Variance threshold (range:[0,4095]; default: 400) */
    AML_U8 u8GrayThr;     /* current frame sub background (range [0,255]； default: 20) */
} CVE_MATCH_BG_MODEL_CTRL_T;

/* updateBgModel down scale mode */
typedef enum {
    CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_DISABLE = 0x0,
    CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_2X2 = 0x1,
    CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_4X4 = 0x2,

    CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_BUTT
} CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_E;

/* Control parameters of the updateBgModel */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_UPDATE_BG_MODEL_DOWN_SCALE_MODE_E
    enDownScaleMode;          /* Down scale mode(range: [0,2];default: 0) */
    AML_U16 u16DelThr;        /* Delete threshold(range: [0,65535];default: 1000) */
    AML_U16 u16FrqThr;        /* Frequency threshold(range: [0,65535];default: 200) */
    AML_U16 u16IntervalThr;   /* Interval threshold(range: [0,65535];default: 40) */
    AML_U0Q16 u0q16LearnRate; /* Learn rate(range: [0,65535];default:1310) */

} CVE_UPDATE_BG_MODEL_CTRL_T;

typedef enum {
    CVE_ANN_MLP_ACTIV_FUNC_IDENTITY = 0x0,
    CVE_ANN_MLP_ACTIV_FUNC_SIGMOID_SYM = 0x1,
    CVE_ANN_MLP_ACTIV_FUNC_GAUSSIAN = 0x2,

    CVE_ANN_MLP_ACTIV_FUNC_BUTT
} CVE_ANN_MLP_ACTIV_FUNC_E;

typedef enum {
    CVE_ANN_MLP_ACCURATE_SRC16_WGT16 = 0x0, /* Input decimal's accurate 16 bit,weight 16bit */
    CVE_ANN_MLP_ACCURATE_SRC14_WGT20 = 0x1, /* Input decimal's accurate 14 bit,weight 20bit */

    CVE_ANN_MLP_ACCURATE_BUTT
} CVE_ANN_MLP_ACCURATE_E;

typedef struct {
    CVE_ANN_MLP_ACTIV_FUNC_E enActivFunc;
    CVE_ANN_MLP_ACCURATE_E enAccurate;
    CVE_MEM_INFO_T stWeight;
    AML_U32 u32TotalWeightSize;

    AML_U16 au16LayerCount[8]; /* There are 8 layers including output and output layers */
    AML_U16 u16MaxCount;       /* MaxCount <= 24 */
    AML_U8 u8LayerNum;         /* 2 < layerNum <= 8 */
    AML_U8 u8Reserved;
} CVE_ANN_MLP_MODEL_T;

typedef enum {
    CVE_SVM_TYPE_C_SVC = 0x0,
    CVE_SVM_TYPE_NU_SVC = 0x1,

    CVE_SVM_TYPE_BUTT
} CVE_SVM_TYPE_E;

typedef enum {
    CVE_SVM_KERNEL_TYPE_LINEAR = 0x0,
    CVE_SVM_KERNEL_TYPE_POLY = 0x1,
    CVE_SVM_KERNEL_TYPE_RBF = 0x2,
    CVE_SVM_KERNEL_TYPE_SIGMOID = 0x3,

    CVE_SVM_KERNEL_TYPE_BUTT
} CVE_SVM_KERNEL_TYPE_E;

typedef struct {
    CVE_SVM_TYPE_E enType;
    CVE_SVM_KERNEL_TYPE_E enKernelType;

    CVE_MEM_INFO_T stSv;    /* SV memory */
    CVE_MEM_INFO_T stDf;    /* DF memory */
    AML_U32 u32TotalDfSize; /* functions coef size in byte */

    AML_U16 u16FeatureDim;
    AML_U16 u16SvTotal;
    AML_U8 u8ClassCount;
} CVE_SVM_MODEL_T;

/* SAD mode */
typedef enum {
    CVE_SAD_MODE_MB_4X4 = 0x0,   /* 4 x 4  */
    CVE_SAD_MODE_MB_8X8 = 0x1,   /* 8 x 8 */
    CVE_SAD_MODE_MB_16X16 = 0x2, /* 16 x 16 */

    CVE_SAD_MODE_BUTT
} CVE_SAD_MODE_E;

/* Control parameters of the SAD */
typedef struct {
    CVE_CROP_T stCrop;
    CVE_SAD_MODE_E enMode;
    AML_U16 u16Thr;  /* output threshold = minVal, srcVal <= u16Thr;
                                                output threshold = maxVal, srcVal > u16Thr */
    AML_U8 u8MinVal; /* Min value */
    AML_U8 u8MaxVal; /* Max value */
} CVE_SAD_CTRL_T;

/* Resize mode */
typedef enum {
    CVE_RESIZE_MODE_LINEAR = 0x0, /* Bilinear interpolation */

    CVE_RESIZE_MODE_BUTT
} CVE_RESIZE_MODE_E;

/* Control parameter of the Resize */
typedef struct {
    CVE_RESIZE_MODE_E enMode;
    CVE_MEM_INFO_T stMem;
    AML_U16 u16Num;
} CVE_RESIZE_CTRL_T;

/* CNN active function mode */
typedef enum {
    CVE_CNN_ACTIV_FUNC_NONE = 0x0,    /* not activation,f(x) = x */
    CVE_CNN_ACTIV_FUNC_RELU = 0x1,    /* f(x) = max(0,x) */
    CVE_CNN_ACTIV_FUNC_SIGMOID = 0x2, /* not support,f(x) = 1/(1+exp(-x)) */

    CVE_CNN_ACTIV_FUNC_BUTT
} CVE_CNN_ACTIV_FUNC_E;

/* CNN poolint mode */
typedef enum {
    CVE_CNN_POOLING_NONE = 0x0, /* not pooling activation */
    CVE_CNN_POOLING_MAX = 0x1,  /* Use the max value of the area in each pooling */
    CVE_CNN_POOLING_AVG = 0x2,  /* Use the average of each pooling area */
    CVE_CNN_POOLING_BUTT
} CVE_CNN_POOLING_E;

/* Control parameters of the CNN conv ReLU pooling layer */
typedef struct {
    CVE_CNN_ACTIV_FUNC_E enActivFunc; /* Activation function type */
    CVE_CNN_POOLING_E enPooling;      /* Pooling method mode */

    AML_U8 u8FeatureMapNum; /* Feature maps num */
    AML_U8 u8KernelSize;    /* only support 3 currently */
    AML_U8 u8ConvStep;      /* only support 1 currently */

    AML_U8 u8PoolSize; /* only support 2 currently */
    AML_U8 u8PoolStep; /* only support 2 currently */
    AML_U8 u8Reserved[3];

} CVE_CNN_CONV_POOLING_T;

/* Control parameters of the CNN fully connect */
typedef struct {
    AML_U16 au16LayerCnt[8]; /* Number of neurons each fully connect layers */
    AML_U16 u16MaxCnt;       /* Max neuron number */
    AML_U8 u8LayerNum;       /* Fully connect layer num */
    AML_U8 u8Reserved;
} CVE_CNN_FULL_CONNECT_T;

/* CNN model */
typedef struct {
    CVE_CNN_CONV_POOLING_T astConvPool[8]; /* Conv ReLU pooling layers */
    CVE_CNN_FULL_CONNECT_T stFullConnect;  /* Fully connected layers */

    CVE_MEM_INFO_T stConvKernelBias; /* Conv ReLU pooling kernels and bias */
    AML_U32 u32ConvKernelBiasSize;   /* Conv ReLU pooling kernels and bias size */

    CVE_MEM_INFO_T stFCLWgtBias; /* Fully connected layers weight and bias */
    AML_U32 u32FCLWgtBiasSize;   /* Fully connected layers weight and bias size */

    AML_U32 u32TotalMemSize; /* Total size of all memory(kernel,weight,bias) */

    CVE_IMAGE_TYPE_E enType; /* The image type of the CNN model */
    AML_U32 u32Width;        /* Image width for the model */
    AML_U32 u32Height;       /* Image height for the model */

    AML_U16 u16ClassCount;     /* Classes num */
    AML_U8 u8ConvPoolLayerNum; /* Conv ReLU pooling layers num */
    AML_U8 u8Reserved;
} CVE_CNN_MODEL_T;

/* Control parameter of the CNN */
typedef struct {
    CVE_MEM_INFO_T stMem; /* Assist memory */
    AML_U32 u32Num;       /* Input image num */
} CVE_CNN_CTRL_T;

/* CNN result */
typedef struct {
    AML_S32 s32ClassIdx;   /* possible Index of the classification */
    AML_S32 s32Confidence; /* confidence of classification */
} CVE_CNN_RESULT_T;

/* Perspective transform algorithm mode */
typedef enum {
    CVE_PERSP_TRANS_ALG_MODE_NR_SIM = 0x0, /* Non reflective similar transformation mode */
    CVE_PERSP_TRANS_ALG_MODE_SIM = 0x1,    /* Reflective similar transformation mode */
    CVE_PERSP_TRANS_ALG_MODE_AFFINE = 0x2, /* Affine transformation mode */

    CVE_PERSP_TRANS_ALG_MODE_BUTT
} CVE_PERSP_TRANS_ALG_MODE_E;

/* Perspective transform point pair */
typedef struct {
    CVE_POINT_U14Q2_T stSrcPoint; /* Src point */
    CVE_POINT_U14Q2_T stDstPoint; /* dst point */
} CVE_PERSP_TRANS_POINT_PAIR_T;

/* Perspective transform CSC mode */
typedef enum {
    CVE_PERSP_TRANS_CSC_MODE_NONE = 0x0,                /* Not use csc */
    CVE_PERSP_TRANS_CSC_MODE_VIDEO_BT601_YUV2RGB = 0x1, /* CSC:YUV2RGB,video transfer mode,
                                                             RGB value range[16,235] */
    CVE_PERSP_TRANS_CSC_MODE_VIDEO_BT709_YUV2RGB = 0x2, /* CSC:YUV2RGB,video transfer mode,
                                                             RGB value range[16,235] */
    CVE_PERSP_TRANS_CSC_MODE_PIC_BT601_YUV2RGB = 0x3,   /* CSC:YUV2RGB,picture transfer mode,
                                                               RGB value range[0,255] */
    CVE_PERSP_TRANS_CSC_MODE_PIC_BT709_YUV2RGB = 0x4,   /* CSC:YUV2RGB,picture transfer mode,
                                                               RGB value range[0,255] */

    CVE_PERSP_TRANS_CSC_MODE_BUTT
} CVE_PERSP_TRANS_CSC_MODE_E;

/* Perspective transform ctrl */
typedef struct {
    CVE_PERSP_TRANS_ALG_MODE_E enAlgMode; /* ALG mode */
    CVE_PERSP_TRANS_CSC_MODE_E enCscMode; /* CSC mode */
    AML_U16 u16RoiNum;                    /* Roi num */
    AML_U16 u16PointPairNum;              /* Point pair num */
} CVE_PERSP_TRANS_CTRL_T;

/* ROI info */
typedef struct {
    CVE_RECT_S24Q8_T stRoi;
    AML_U32 u32RoiId;
} CVE_ROI_INFO_T;

/* Tof fpn mode */
typedef enum {
    CVE_TOF_FPN_MODE_BYPASS = 0x0,
    CVE_TOF_FPN_MODE_FPN = 0x1,
    CVE_TOF_FPN_MODE_FD = 0x2,

    CVE_TOF_FPN_MODE_BUTT
} CVE_TOF_FPN_MODE_E;

/* Control parameter of the Tof */
typedef struct {
    CVE_RAW_MODE_E enRawMode;
    CVE_TOF_FPN_MODE_E enFpnMode;
    AML_BOOL_E bRawShift;
    AML_BOOL_E bBypass;
    AML_BOOL_E bSpa1En;
    AML_BOOL_E bSpa2En;
    AML_S21Q3 as32PCoef[5];
    AML_S16 as16TCoef1[4];
    AML_U16 u16HistXstart;
    AML_U16 u16HistYstart;
    AML_U16 u16HistXend;
    AML_U16 u16HistYend;
    AML_U16 u16Q1HighThr;
    AML_U16 u16Q23HighThr;
    AML_U16 u16IRHighThr;
    AML_U16 u16IRLowThr;
    AML_U16 u16DepthMax;
    AML_U16 u16DepthMin;
    AML_U8 u8SpaNorm;
    AML_U8 u8BadPointNum;
    AML_S8 s8IntTemp;
    AML_S8 s8ExtTemp;
} CVE_TOF_CTRL_T;

#endif /* __AML_CVE_H__ */
