/*
 * Copyright (C) 2017 Amlogic Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "audio_hw_primary"
//#define LOG_NDEBUG 0
#include <math.h>
#include <inttypes.h>
#include <cutils/log.h>
#include <tinyalsa/asoundlib.h>
#include <cutils/properties.h>
#include <audio_utils/channels.h>
#include <audio_route/audio_route.h>
#include <audio_utils/format.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "aml_alsa_mixer.h"
#include "aml_audio_stream.h"
#include "dolby_lib_api.h"
#include "audio_hw_utils.h"
#include "audio_hw_profile.h"
#include "alsa_manager.h"
#include "alsa_device_parser.h"
#include "aml_avsync_tuning.h"
#include "aml_android_utils.h"
#include "audio_format_parse.h"
#include "alsa_config_parameters.h"
#include "audio_hw_ms12.h"
#include "amlAudioMixer.h"
#include "aml_audio_sysfs.h"

#ifdef MS12_V24_ENABLE
#include "audio_hw_ms12_v2.h"
#endif
#define FMT_UPDATE_THRESHOLD_MAX    (10)
#define DOLBY_FMT_UPDATE_THRESHOLD  (5)
#define DTS_FMT_UPDATE_THRESHOLD    (1)
#define INVALID_TYPE                -1

#define HDMIIN_MULTICH_DOWNMIX

/*
 * DAP Speaker Virtualizer
 * -dap_surround_virtualizer    * <2 int> Virtualizer Parameter
 *                                         - virtualizer_mode (0,1,2, def: 1)
 *                                            0:OFF
 *                                            1:ON
 *                                            2:AUTO
 *                                         - surround_boost (0...96, def: 96)
 */
#define MS12_DAP_SPEAKER_VIRTUALIZER_OFF  (0)//Disable Speaker Virtualizer(Disable Dolby Atmos Virtualization).
#define MS12_DAP_SPEAKER_VIRTUALIZER_ON   (1)//Enable Speaker Virtualizer.
#define MS12_DAP_SPEAKER_VIRTUALIZER_AUTO (2)//Enable Dolby Atmos Virtualization.

#define HDMI_HDR_STATUS_NODE        "/sys/class/amhdmitx/amhdmitx0/hdmi_hdr_status"
#define SINK_DV_KEYWORD             "DolbyVision"


#define MINUS_3_DB_IN_FLOAT M_SQRT1_2 // -3dB = 0.70710678

static inline float clamp_float(float value) {
    return fmin(fmax(value, -1.f), 1.f);
}

static audio_format_t ms12_max_support_output_format() {
#ifndef MS12_V24_ENABLE
    return AUDIO_FORMAT_E_AC3;
#else
    return AUDIO_FORMAT_MAT;
#endif
}

int get_file_size(char *name)
{
    struct stat statbuf;
    int ret;

    ret = stat(name, &statbuf);
    if (ret != 0)
        return -1;

    return statbuf.st_size;
}

/*
 *@brief get sink capability
 */
static audio_format_t get_sink_capability (struct aml_audio_device *adev)
{
    struct aml_arc_hdmi_desc *hdmi_desc = &adev->hdmi_descs;

    bool dd_is_support = hdmi_desc->dd_fmt.is_support;
    bool ddp_is_support = hdmi_desc->ddp_fmt.is_support;
    bool mat_is_support = hdmi_desc->mat_fmt.is_support;

    audio_format_t sink_capability = AUDIO_FORMAT_PCM_16_BIT;

    //STB case
    //TV + STB case (BDS)
    //TODO HDMITX+ARC mixed connected case
    //need check active port ???
    if (!adev->is_TV || adev->is_BDS)
    {
        char *cap = NULL;
        /*we should get the real audio cap, so we need it report the correct truehd info*/
        cap = (char *) get_hdmi_sink_cap_new (AUDIO_PARAMETER_STREAM_SUP_FORMATS,0,&(adev->hdmi_descs), true);
        if (cap) {
            /*
             * Dolby MAT 2.0/2.1 has low latency vs Dolby MAT 1.0(TRUEHD inside)
             * Dolby MS12 prefers to output MAT2.0/2.1.
             */
            if ((strstr(cap, "AUDIO_FORMAT_MAT_2_0") != NULL) || (strstr(cap, "AUDIO_FORMAT_MAT_2_1") != NULL)) {
                sink_capability = AUDIO_FORMAT_MAT;
            }
            /*
             * Dolby MAT 1.0(TRUEHD inside) vs DDP+DD
             * Dolby MS12 prefers to output DDP.
             * But set sink as TrueHD, then TrueHD can encoded with MAT encoder in Passthrough mode.
             */
            else if (strstr(cap, "AUDIO_FORMAT_MAT_1_0") != NULL) {
                sink_capability = AUDIO_FORMAT_DOLBY_TRUEHD;
            }
            /*
             * DDP vs DDP
             * Dolby MS12 prefers to output DDP.
             */
            else if (strstr(cap, "AUDIO_FORMAT_E_AC3") != NULL) {
                sink_capability = AUDIO_FORMAT_E_AC3;
            }
            /*
             * DD vs PCM
             * Dolby MS12 prefers to output DD.
             */
            else if (strstr(cap, "AUDIO_FORMAT_AC3") != NULL) {
                sink_capability = AUDIO_FORMAT_AC3;
            }
            ALOGI ("%s mbox+dvb case sink_capability =  %#x\n", __FUNCTION__, sink_capability);
            aml_audio_free(cap);
            cap = NULL;
        }

        dd_is_support = hdmi_desc->dd_fmt.is_support;
        ddp_is_support = hdmi_desc->ddp_fmt.is_support;
        mat_is_support = hdmi_desc->mat_fmt.is_support;

    } else {
        if (mat_is_support || adev->hdmi_descs.mat_fmt.MAT_PCM_48kHz_only) {
            sink_capability = AUDIO_FORMAT_MAT;
            mat_is_support = true;
            hdmi_desc->mat_fmt.is_support = true;
        } else if (ddp_is_support) {
            sink_capability = AUDIO_FORMAT_E_AC3;
        } else if (dd_is_support) {
            sink_capability = AUDIO_FORMAT_AC3;
        }
        ALOGI ("%s dd support %d ddp support %#x\n", __FUNCTION__, dd_is_support, ddp_is_support);
    }

    if (eDolbyMS12Lib == adev->dolby_lib_type) {
        bool b_force_ddp = adev->ms12_force_ddp_out;
        ALOGI("force ddp out =%d", b_force_ddp);
        if (b_force_ddp) {
            if (ddp_is_support) {
                sink_capability = AUDIO_FORMAT_E_AC3;
            } else if (dd_is_support) {
                sink_capability = AUDIO_FORMAT_AC3;
            }
        }
    }

    return sink_capability;
}

static audio_format_t get_sink_dts_capability (struct aml_audio_device *adev)
{
    struct aml_arc_hdmi_desc *hdmi_desc = &adev->hdmi_descs;

    bool dts_is_support = hdmi_desc->dts_fmt.is_support;
    bool dtshd_is_support = hdmi_desc->dtshd_fmt.is_support;

    audio_format_t sink_capability = AUDIO_FORMAT_PCM_16_BIT;

    //STB case
    if (!adev->is_TV)
    {
        char *cap = NULL;
        cap = (char *) get_hdmi_sink_cap_new (AUDIO_PARAMETER_STREAM_SUP_FORMATS,0,&(adev->hdmi_descs), true);
        if (cap) {
            if (adev->hdmi_descs.dts_fmt.is_support) {
                sink_capability = AUDIO_FORMAT_DTS;
            } else if (adev->hdmi_descs.dtshd_fmt.is_support) {
                sink_capability = AUDIO_FORMAT_DTS_HD;
            }
            AM_LOGI("mbox+dvb case sink_capability: %s(%#x)", audioFormat2Str(sink_capability), sink_capability);
            aml_audio_free(cap);
            cap = NULL;
        }
    } else {
        if (dtshd_is_support) {
            sink_capability = AUDIO_FORMAT_DTS_HD;
        } else if (dts_is_support) {
            sink_capability = AUDIO_FORMAT_DTS;
        }
        ALOGI ("%s dts support %d dtshd support %d\n", __FUNCTION__, dts_is_support, dtshd_is_support);
    }
    return sink_capability;
}


static audio_format_t get_sink_mpegh_capability (struct aml_audio_device *adev)
{
    struct aml_arc_hdmi_desc *hdmi_desc = &adev->hdmi_descs;

    bool mpegh_is_support = hdmi_desc->mpegh_fmt.is_support;

    audio_format_t sink_capability = AUDIO_FORMAT_PCM_16_BIT;

    //STB case
    if (!adev->is_TV)
    {
        char *cap = NULL;
        cap = (char *) get_hdmi_sink_cap_new (AUDIO_PARAMETER_STREAM_SUP_FORMATS, 0, &(adev->hdmi_descs), true);
        if (cap) {
            if (adev->hdmi_descs.mpegh_fmt.is_support) {
                sink_capability = (audio_format_t)AUDIO_FORMAT_MPEGH;
            }
            AM_LOGI("mbox+dvb case sink_capability: %s(%#x)", audioFormat2Str(sink_capability), sink_capability);
            aml_audio_free(cap);
            cap = NULL;
        }
    } else {
        if (mpegh_is_support) {
            sink_capability = (audio_format_t)AUDIO_FORMAT_MPEGH;
        }
        ALOGI ("%s mpegh support %d\n", __FUNCTION__, mpegh_is_support);
    }
    return sink_capability;
}

static void get_sink_pcm_capability(struct aml_audio_device *adev)
{
    struct aml_arc_hdmi_desc *hdmi_desc = &adev->hdmi_descs;
    char *cap = NULL;
    hdmi_desc->pcm_fmt.sample_rate_mask = 0;

    cap = (char *) get_hdmi_sink_cap_new (AUDIO_PARAMETER_STREAM_SUP_SAMPLING_RATES, AUDIO_FORMAT_PCM_16_BIT,&(adev->hdmi_descs), true);
    if (cap) {
        /*
         * bit:    6     5     4    3    2    1    0
         * rate: 192  176.4   96  88.2  48  44.1   32
         */
        if (strstr(cap, "32000") != NULL)
            hdmi_desc->pcm_fmt.sample_rate_mask |= (1<<0);
        if (strstr(cap, "44100") != NULL)
            hdmi_desc->pcm_fmt.sample_rate_mask |= (1<<1);
        if (strstr(cap, "48000") != NULL)
            hdmi_desc->pcm_fmt.sample_rate_mask |= (1<<2);
        if (strstr(cap, "88200") != NULL)
            hdmi_desc->pcm_fmt.sample_rate_mask |= (1<<3);
        if (strstr(cap, "96000") != NULL)
            hdmi_desc->pcm_fmt.sample_rate_mask |= (1<<4);
        if (strstr(cap, "176400") != NULL)
            hdmi_desc->pcm_fmt.sample_rate_mask |= (1<<5);
        if (strstr(cap, "192000") != NULL)
            hdmi_desc->pcm_fmt.sample_rate_mask |= (1<<6);

        aml_audio_free(cap);
        cap = NULL;
    }

    ALOGI("pcm_fmt support sample_rate_mask:0x%x", hdmi_desc->pcm_fmt.sample_rate_mask);
}

static unsigned int get_sink_format_max_channels(struct aml_audio_device *adev, audio_format_t sink_format) {
    unsigned int max_channels = 2;
    struct aml_arc_hdmi_desc *hdmi_desc = &adev->hdmi_descs;

    switch (sink_format) {
    case AUDIO_FORMAT_PCM_16_BIT:
        max_channels = hdmi_desc->pcm_fmt.max_channels;
        break;
    case AUDIO_FORMAT_AC3:
        max_channels = hdmi_desc->dd_fmt.max_channels;
        break;
    case AUDIO_FORMAT_E_AC3:
        max_channels = hdmi_desc->ddp_fmt.max_channels;
        break;
    case AUDIO_FORMAT_DTS:
        max_channels = hdmi_desc->dts_fmt.max_channels;
        break;
    case AUDIO_FORMAT_DTS_HD:
        max_channels = hdmi_desc->dtshd_fmt.max_channels;
        break;
    case AUDIO_FORMAT_MAT:
        max_channels = hdmi_desc->mat_fmt.max_channels;
        break;
    default:
        max_channels = 2;
        break;
    }
    if (max_channels == 0) {
        max_channels = 2;
    }
    return max_channels;
}

static bool get_sink_dv_capability()
{
    char buffer[128];
    FILE *fp = NULL;
    bool dv_enable = false;

    memset(buffer, 0, sizeof(buffer));
    fp = fopen(HDMI_HDR_STATUS_NODE, "r");
    if (fp) {
        int read_count = fread((char *)buffer, 1, sizeof(buffer)-1, fp);
        if (ferror(fp)) {
            ALOGE("%s : fread has IO wrong", __func__);
       }
        fclose(fp);
    }
    ALOGI("%s : %s = %s", __func__, HDMI_HDR_STATUS_NODE, buffer);

    if (strstr(buffer, SINK_DV_KEYWORD) != NULL) {
        dv_enable = true;
    }

    ALOGI("%s : dv enable %d", __func__, dv_enable);
    return dv_enable;
}

bool is_sink_support_dolby_passthrough(audio_format_t sink_capability)
{
    return sink_capability == AUDIO_FORMAT_MAT ||
        sink_capability == AUDIO_FORMAT_E_AC3 ||
        sink_capability == AUDIO_FORMAT_AC3;
}

/*
 *1. source format includes these formats:
 *        AUDIO_FORMAT_PCM_16_BIT = 0x1u
 *        AUDIO_FORMAT_AC3 =    0x09000000u
 *        AUDIO_FORMAT_E_AC3 =  0x0A000000u
 *        AUDIO_FORMAT_DTS =    0x0B000000u
 *        AUDIO_FORMAT_DTS_HD = 0x0C000000u
 *        AUDIO_FORMAT_AC4 =    0x22000000u
 *        AUDIO_FORMAT_MAT =    0x24000000u
 *
 *2. if the source format can output directly as PCM format or as IEC61937 format,
 *   btw, AUDIO_FORMAT_PCM_16_BIT < AUDIO_FORMAT_AC3 < AUDIO_FORMAT_MAT is true.
 *   we can use the min(a, b) to get an suitable output format.
 *3. if source format is AUDIO_FORMAT_AC4, we can not use the min(a,b) to get the
 *   suitable format but use the sink device max capability format.
 */
static audio_format_t get_suitable_output_format(struct aml_stream_out *out,
        audio_format_t source_format, audio_format_t sink_format)
{
    audio_format_t output_format;
    if (IS_EXTERNAL_DECODER_SUPPORT_FORMAT(source_format)) {
        output_format = MIN(source_format, sink_format);
        /*
         * if source: AUDIO_FORMAT_DOLBY_TRUEHD and sink: AUDIO_FORMAT_MAT
         * use the AUDIO_FORMAT_MAT as output format(from IEC 61937-1).
         */
        output_format = (output_format != AUDIO_FORMAT_DOLBY_TRUEHD) ? output_format: AUDIO_FORMAT_MAT;
    } else {
        output_format = sink_format;
    }
    if ((out->hal_rate == 32000 || out->hal_rate == 128000) && output_format == AUDIO_FORMAT_E_AC3) {
        output_format = AUDIO_FORMAT_AC3;
    }
    return output_format;
}

/*
 * When turn on the Automatic function to play the dolby audio in low speed,the TV might
 * can't decode.So if the play mode was Automatic and the output speed was in not equal
 * 1.0f, set the ret_format to AUDIO_FORMAT_PCM_16_BIT to send the PCM data to spdif.
 */
static audio_format_t reconfig_optical_audio_format(struct aml_stream_out *aml_out,
        audio_format_t org_optical_format)
{
    audio_format_t ret_format = org_optical_format;

    if (aml_out == NULL)
        return org_optical_format;

    if (aml_out->output_speed != 1.0f && aml_out->output_speed != 0.0f) {
        ALOGI("change to micro speed need reconfig optical audio format to PCM");
        ret_format = AUDIO_FORMAT_PCM_16_BIT;
    }

    return ret_format;
}

/*
 *@brief get sink format by logic min(source format / digital format / sink capability)
 * For Speaker/Headphone output, sink format keep PCM-16bits
 * For optical output, min(dd, source format, digital format)
 * For HDMI_ARC output
 *      1.digital format is PCM, sink format is PCM-16bits
 *      2.digital format is dd, sink format is min (source format,  AUDIO_FORMAT_AC3)
 *      3.digital format is auto, sink format is min (source format, digital format)
 */
void get_sink_format(struct audio_stream_out *stream)
{
   if (stream == NULL) {
        ALOGE("stream NULL");
        return;
    }
    struct aml_stream_out *aml_out = (struct aml_stream_out *) stream;
    struct aml_audio_device *adev = aml_out->dev;
    /*set default value for sink_audio_format/optical_audio_format*/
    audio_format_t sink_audio_format = AUDIO_FORMAT_PCM_16_BIT;
    audio_format_t optical_audio_format = AUDIO_FORMAT_PCM_16_BIT;

    audio_format_t sink_capability = get_sink_capability(adev);
    audio_format_t sink_dts_capability = get_sink_dts_capability(adev);
    audio_format_t sink_mpegh_capability = get_sink_mpegh_capability(adev);
    audio_format_t source_format = aml_out->hal_internal_format;

    get_sink_pcm_capability(adev);

    adev->bDVEnable = get_sink_dv_capability();

    if (adev->out_device & AUDIO_DEVICE_OUT_ALL_A2DP || adev->out_device & AUDIO_DEVICE_OUT_ALL_USB) {
        ALOGD("get_sink_format: a2dp and usb set to pcm");
        adev->sink_format = AUDIO_FORMAT_PCM_16_BIT;
        adev->sink_capability = AUDIO_FORMAT_PCM_16_BIT;
        adev->optical_format = AUDIO_FORMAT_PCM_16_BIT;
        aml_out->dual_output_flag = false;
        adev->sink_max_channels = 2;
        return;
    }

    /*when device is HDMI_ARC*/
    AM_LOGI("out:%p cur_out_devices:%#x format:%s(%#x) digital_mode(%s) Sink format:%s(%#x)", aml_out,
          adev->cur_out_devices, audioFormat2Str(aml_out->hal_internal_format), aml_out->hal_internal_format,
          digitalAudioModeType2Str(adev->digital_audio_mode), audioFormat2Str(sink_capability), sink_capability);

    if ((source_format != AUDIO_FORMAT_PCM_16_BIT) && \
        (source_format != AUDIO_FORMAT_AC3) && \
        (source_format != AUDIO_FORMAT_E_AC3) && \
        (source_format != AUDIO_FORMAT_MAT) && \
        (source_format != AUDIO_FORMAT_AC4) && \
        (source_format != AUDIO_FORMAT_DTS) &&
        (source_format != AUDIO_FORMAT_DTS_HD) && \
        (source_format != AUDIO_FORMAT_DOLBY_TRUEHD) && \
        (source_format != AUDIO_FORMAT_AAC) && \
        (source_format != AUDIO_FORMAT_AAC_LATM) && \
        (source_format != AUDIO_FORMAT_HE_AAC_V1) && \
        (source_format != AUDIO_FORMAT_HE_AAC_V2) && \
        (source_format != AUDIO_FORMAT_MPEGH)) {
        /*unsupport format [dts-hd/true-hd]*/
        ALOGI("%s() source format %#x change to %#x", __FUNCTION__, source_format, AUDIO_FORMAT_PCM_16_BIT);
        source_format = AUDIO_FORMAT_PCM_16_BIT;
    }
    adev->sink_capability = sink_capability;

    // "adev->digital_audio_mode" is the UI selection item.
    // "adev->active_outport" was set when HDMI ARC cable plug in/off
    // condition 1: ARC port, single output.
    // condition 2: for STB case with dolby-ms12 libs
    // condition 3: T7 BDS with HDMITX case
    if ((adev->cur_out_devices & AUDIO_DEVICE_OUT_HDMI_ARC) != 0 || !adev->is_TV || adev->is_BDS) {
        struct audio_board_config *bd_config = &adev->board_config;
        ALOGI("%s() HDMI ARC or mbox + dvb case", __FUNCTION__);
        switch (adev->digital_audio_mode) {
        case AML_DIGITAL_AUDIO_MODE_PCM:
            sink_audio_format = AUDIO_FORMAT_PCM_16_BIT;
            optical_audio_format = sink_audio_format;
            break;
        case AML_DIGITAL_AUDIO_MODE_DD:
            sink_audio_format = AUDIO_FORMAT_AC3;
            optical_audio_format = sink_audio_format;
            break;
        case AML_DIGITAL_AUDIO_MODE_AUTO:
            if (is_dts_format(source_format)) {
                sink_audio_format = MIN(source_format, sink_dts_capability);
            } else {
                sink_audio_format = get_suitable_output_format(aml_out, source_format, sink_capability);
            }
            if (eDolbyMS12Lib == adev->dolby_lib_type && !is_dts_format(source_format)) {
                sink_audio_format = MIN(ms12_max_support_output_format(), sink_capability);
            }
            optical_audio_format = sink_audio_format;

            /*if the sink device only support pcm, we check whether we can output dd or dts to spdif,
             *For arc case, we can't support pcm and dd dual output, so we limit it to non arc case
             */
            if (bd_config->spdif_independent && ((adev->cur_out_devices & AUDIO_DEVICE_OUT_HDMI_ARC) == 0)) {
                if (sink_audio_format == AUDIO_FORMAT_PCM_16_BIT) {
                    if (is_dts_format(source_format)) {
                        optical_audio_format = MIN(source_format, AUDIO_FORMAT_DTS);
                    } else {
                        if (eDolbyMS12Lib == adev->dolby_lib_type) {
                            optical_audio_format = AUDIO_FORMAT_AC3;
                        } else {
                            optical_audio_format = MIN(source_format, AUDIO_FORMAT_AC3);
                        }
                    }
                }
            }

            optical_audio_format = reconfig_optical_audio_format(aml_out, optical_audio_format);
            break;
        case AML_DIGITAL_AUDIO_MODE_BYPASS:
            if (is_dts_format(source_format)) {
                sink_audio_format = MIN(source_format, sink_dts_capability);
            } else if (is_mpegh_format(source_format)) {
                sink_audio_format = sink_mpegh_capability;
            } else {
                sink_audio_format = get_suitable_output_format(aml_out, source_format, sink_capability);
            }
            optical_audio_format = sink_audio_format;
            /*if the sink device only support pcm, we check whether we can output dd or dts to spdif*/
            if (bd_config->spdif_independent && ((adev->cur_out_devices & AUDIO_DEVICE_OUT_HDMI_ARC) == 0)) {
                if (sink_audio_format == AUDIO_FORMAT_PCM_16_BIT) {
                    if (is_dts_format(source_format)) {
                        optical_audio_format = MIN(source_format, AUDIO_FORMAT_DTS);
                    } else {
                        optical_audio_format = MIN(source_format, AUDIO_FORMAT_AC3);
                    }
                }
            }

            break;
        default:
            sink_audio_format = AUDIO_FORMAT_PCM_16_BIT;
            optical_audio_format = sink_audio_format;
            break;
        }
    }
    /*when device is SPEAKER/HEADPHONE*/
    else {
        ALOGI("%s() SPEAKER/HEADPHONE case", __FUNCTION__);
        switch (adev->digital_audio_mode) {
        case AML_DIGITAL_AUDIO_MODE_PCM:
            sink_audio_format = AUDIO_FORMAT_PCM_16_BIT;
            optical_audio_format = sink_audio_format;
            break;
        case AML_DIGITAL_AUDIO_MODE_DD:
            sink_audio_format = AUDIO_FORMAT_PCM_16_BIT;
            optical_audio_format = AUDIO_FORMAT_AC3;
            break;
        case AML_DIGITAL_AUDIO_MODE_AUTO:
            sink_audio_format = AUDIO_FORMAT_PCM_16_BIT;
            optical_audio_format = (source_format != AUDIO_FORMAT_DTS && source_format != AUDIO_FORMAT_DTS_HD)
                                   ? MIN(source_format, AUDIO_FORMAT_AC3)
                                   : AUDIO_FORMAT_DTS;

            if (eDolbyMS12Lib == adev->dolby_lib_type && !is_dts_format(source_format)) {
                optical_audio_format = AUDIO_FORMAT_AC3;
            }

            optical_audio_format = reconfig_optical_audio_format(aml_out, optical_audio_format);
            break;
        case AML_DIGITAL_AUDIO_MODE_BYPASS:
           sink_audio_format = AUDIO_FORMAT_PCM_16_BIT;
           if (is_dts_format(source_format)) {
               optical_audio_format = MIN(source_format, AUDIO_FORMAT_DTS);
           } else {
               optical_audio_format = MIN(source_format, AUDIO_FORMAT_AC3);
           }
           break;
        default:
            sink_audio_format = AUDIO_FORMAT_PCM_16_BIT;
            optical_audio_format = sink_audio_format;
            break;
        }
    }
    if (adev->sink_format != sink_audio_format) {
        adev->sink_format_changed = true;
    }
    adev->sink_format = sink_audio_format;
    adev->optical_format = optical_audio_format;
    adev->sink_max_channels = get_sink_format_max_channels(adev, adev->sink_format);

    /* set the dual output format flag */
    if (adev->sink_format != adev->optical_format) {
        aml_out->dual_output_flag = true;
    } else {
        aml_out->dual_output_flag = false;
    }
    AM_LOGI("sink_format:%s(%#x) max channel:%d optical_format:%s(%#x) dual_output %d",
           audioFormat2Str(adev->sink_format), adev->sink_format, adev->sink_max_channels,
           audioFormat2Str(adev->optical_format), adev->optical_format, aml_out->dual_output_flag);
    return ;
}

bool is_hdmi_in_stable_hw (struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *aml_dev = in->dev;
    struct aml_audio_patch *audio_patch = aml_dev->audio_patch;
    audio_type_parse_t *audio_type_status = (audio_type_parse_t *)audio_patch->audio_parse_para;
    int type = 0;
    int stable = 0;
    int tl1_chip = check_chip_name("tl1", 3, &aml_dev->alsa_mixer);

    stable = aml_mixer_ctrl_get_int (&aml_dev->alsa_mixer, AML_MIXER_ID_HDMI_IN_AUDIO_STABLE);
    if (!stable) {
        ALOGV("%s() amixer %s get %d\n", __func__, "HDMIIN audio stable", stable);
        return false;
    }
    return true;
}

bool is_hdmi_in_sample_rate_changed(struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *aml_dev = in->dev;
    int last_hdmi_in_samplerate = in->hdmi_in_samplerate;

    int samplerate = aml_mixer_ctrl_get_int(&aml_dev->alsa_mixer, AML_MIXER_ID_HDMI_IN_SAMPLERATE);
    if (last_hdmi_in_samplerate != samplerate) {
        ALOGD("hdmi in samplerate changes from %d to %d",last_hdmi_in_samplerate, samplerate);
        in->hdmi_in_samplerate = samplerate;
        return true;
    }
    return false;
}
bool is_hdmi_in_hw_format_change(struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *aml_dev = in->dev;
    struct aml_audio_patch *audio_patch = aml_dev->audio_patch;
    audio_type_parse_t *audio_type_status = (audio_type_parse_t *)audio_patch->audio_parse_para;
    int tl1_chip = check_chip_name("tl1", 3, &aml_dev->alsa_mixer);
    int type = 0;
    bool ret = false;

    /* TL1 do not use HDMIIN_AUDIO_TYPE */
    if (audio_type_status != NULL && audio_type_status->soft_parser != 1 && !tl1_chip) {
        type = aml_mixer_ctrl_get_int (&aml_dev->alsa_mixer, AML_MIXER_ID_HDMIIN_AUDIO_TYPE);
        if ((type != INVALID_TYPE) && (type != in->spdif_fmt_hw)) {
            ALOGD ("%s(), in type changed from %d to %d", __func__, in->spdif_fmt_hw, type);
            ret = true;
        }
        in->spdif_fmt_hw = type;
    }
    return ret;
}

bool is_HBR_stream(struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *aml_dev = in->dev;
    bool ret = false;

    if (aml_dev->in_device & AUDIO_DEVICE_IN_HDMI && aml_dev->audio_patch) {
        struct aml_audio_patch *audio_patch = aml_dev->audio_patch;
        audio_type_parse_t *audio_type_status = (audio_type_parse_t *)audio_patch->audio_parse_para;

        if (audio_patch && audio_type_status && audio_type_status->soft_parser != 1) {
            if (in->last_audio_packet_type == AUDIO_PACKET_HBR) {
                ret = true;
            }
        }
    }
    return ret;
}

bool is_dual_output_stream(struct audio_stream_out *stream)
{
    struct aml_stream_out *aml_out = (struct aml_stream_out *)stream;
    if (aml_out == NULL)
        return 0;

    return aml_out->dual_output_flag;
}

bool is_hdmi_in_stable_sw (struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *aml_dev = in->dev;
    struct aml_audio_patch *patch = aml_dev->audio_patch;
    audio_format_t fmt;

    /* now, only hdmiin->(spk, hp, arc) cases init the soft parser thread
     * TODO: init hdmiin->mix soft parser too
     */
    if (!patch)
        return true;

    fmt = audio_parse_get_audio_type (patch->audio_parse_para);
    if (fmt != in->spdif_fmt_sw) {
        ALOGD ("%s(), in type changed from %#x to %#x", __func__, in->spdif_fmt_sw, fmt);
        in->spdif_fmt_sw = fmt;
        return false;
    }

    return true;
}


void  release_audio_stream(struct audio_stream_out *stream)
{
    struct aml_stream_out *aml_out = (struct aml_stream_out *)stream;
    aml_audio_free(stream);
}
bool is_atv_in_stable_hw (struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *aml_dev = in->dev;
    int type = 0;
    int stable = 0;

    stable = aml_mixer_ctrl_get_int (&aml_dev->alsa_mixer, AML_MIXER_ID_ATV_IN_AUDIO_STABLE);
    if (!stable)
        return false;

    return true;
}
bool is_av_in_stable_hw(struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *)stream;
    struct aml_audio_device *aml_dev = in->dev;
    int type = 0;
    int stable = 0;

    stable = aml_mixer_ctrl_get_int (&aml_dev->alsa_mixer, AML_MIXER_ID_AV_IN_AUDIO_STABLE);
    if (!stable)
        return false;

    return true;
}

bool is_spdif_in_stable_hw (struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *aml_dev = in->dev;
    int type = 0;

    if (is_earc_descrpt())
        type = eArcIn_audio_format_detection(&aml_dev->alsa_mixer);
    else
        type = aml_mixer_ctrl_get_int (&aml_dev->alsa_mixer, AML_MIXER_ID_SPDIFIN_AUDIO_TYPE);
    if (type != in->spdif_fmt_hw) {
        ALOGI ("%s(), in type changed from %d to %d", __func__, in->spdif_fmt_hw, type);
        in->spdif_fmt_hw = type;
        return false;
    }

    return true;
}

bool check_digital_in_stream_signal(struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *aml_dev = in->dev;
    struct aml_audio_patch *patch = aml_dev->audio_patch;
    audio_type_parse_t *audio_type_status = (audio_type_parse_t *)patch->audio_parse_para;
    enum audio_type cur_audio_type = LPCM;

    /* parse thread may have exited ,add the code to avoid NULL point visit*/
    if (audio_type_status == NULL)  {
        return true;
    }

    if (audio_type_status->soft_parser != 1) {
        if (in->spdif_fmt_hw == SPDIFIN_AUDIO_TYPE_PAUSE) {
            ALOGV("%s(), hw detect iec61937 PAUSE packet, mute input", __func__);
            return false;
        }
    } else {
        cur_audio_type = audio_parse_get_audio_type_direct(patch->audio_parse_para);
        if (cur_audio_type == PAUSE || cur_audio_type == MUTE) {
            ALOGV("%s(), soft parser iec61937 %s packet, mute input", __func__,
                cur_audio_type == PAUSE ? "PAUSE" : "MUTE");
            return false;
        }
    }

    return true;
}

int set_audio_source(struct aml_mixer_handle *mixer_handle,
        enum input_source audio_source, bool is_auge)
{
    int src = audio_source;

    if (is_auge) {
        switch (audio_source) {
        case LINEIN:
            src = TDMIN_A;
            break;
        case ATV:
            src = FRATV;
            break;
        case HDMIIN:
            src = FRHDMIRX;
            break;
        case ARCIN:
            src = EARCRX_DMAC;
            break;
        case SPDIFIN:
            src = SPDIFIN_AUGE;
            break;
        default:
            ALOGW("%s(), src: %d not support", __func__, src);
            src = FRHDMIRX;
            break;
        }
    }

    return aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_AUDIO_IN_SRC, src);
}

int set_resample_source(struct aml_mixer_handle *mixer_handle, enum ResampleSource source)
{
    return aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_HW_RESAMPLE_SOURCE, source);
}

int set_spdifin_pao(struct aml_mixer_handle *mixer_handle,int enable)
{
    return aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_SPDIFIN_PAO, enable);
}

int get_spdifin_samplerate(struct aml_mixer_handle *mixer_handle)
{
    int index = aml_mixer_ctrl_get_int(mixer_handle, AML_MIXER_ID_SPDIF_IN_SAMPLERATE);

    return index;
}

int get_hdmiin_samplerate(struct aml_mixer_handle *mixer_handle)
{
    int stable = 0;

    stable = aml_mixer_ctrl_get_int(mixer_handle, AML_MIXER_ID_HDMI_IN_AUDIO_STABLE);
    if (!stable) {
        return -1;
    }

    return aml_mixer_ctrl_get_int(mixer_handle, AML_MIXER_ID_HDMI_IN_SAMPLERATE);
}

int get_hdmiin_channel(struct aml_mixer_handle *mixer_handle)
{
    int stable = 0;
    int channel_index = 0;

    stable = aml_mixer_ctrl_get_int(mixer_handle, AML_MIXER_ID_HDMI_IN_AUDIO_STABLE);
    if (!stable) {
        return -1;
    }

    /*hdmirx audio support: N/A, 2, 3, 4, 5, 6, 7, 8*/
    channel_index = aml_mixer_ctrl_get_int(mixer_handle, AML_MIXER_ID_HDMI_IN_CHANNELS);
    if (channel_index == 0) {
        return 0;
    }
    else if (channel_index != 7) {
        return 2;
    }
    else {
        return 8;
    }
}

hdmiin_audio_packet_t get_hdmiin_audio_packet(struct aml_mixer_handle *mixer_handle)
{
    int audio_packet = 0;
    audio_packet = aml_mixer_ctrl_get_int(mixer_handle,AML_MIXER_ID_HDMIIN_AUDIO_PACKET);
    if (audio_packet < 0) {
        return AUDIO_PACKET_NONE;
    }
    return (hdmiin_audio_packet_t)audio_packet;
}

int get_HW_resample(struct aml_mixer_handle *mixer_handle)
{
    return aml_mixer_ctrl_get_int(mixer_handle, AML_MIXER_ID_HW_RESAMPLE_ENABLE);
}

int enable_HW_resample(struct aml_mixer_handle *mixer_handle, int enable_sr)
{
    if (enable_sr == 0)
        aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_HW_RESAMPLE_ENABLE, HW_RESAMPLE_DISABLE);
    else
        aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_HW_RESAMPLE_ENABLE, enable_sr);
    return 0;
}

bool Stop_watch(struct timespec start_ts, int64_t time) {
    struct timespec end_ts;
    int64_t start_ms, end_ms;
    int64_t interval_ms;

    clock_gettime (CLOCK_MONOTONIC, &end_ts);
    start_ms = start_ts.tv_sec * 1000LL +
               start_ts.tv_nsec / 1000000LL;
    end_ms = end_ts.tv_sec * 1000LL +
             end_ts.tv_nsec / 1000000LL;
    interval_ms = end_ms - start_ms;
    if (interval_ms < time) {
        return true;
    }
    return false;
}

bool signal_status_check(audio_devices_t in_device, int *mute_time,
                        struct audio_stream_in *stream) {

    struct aml_stream_in *in = (struct aml_stream_in *) stream;
    struct aml_audio_device *adev = in->dev;
    hdmiin_audio_packet_t last_audio_packet = in->last_audio_packet_type;
    bool is_audio_packet_changed = false;

    hdmiin_audio_packet_t cur_audio_packet = get_hdmiin_audio_packet(&adev->alsa_mixer);

    is_audio_packet_changed = (((cur_audio_packet == AUDIO_PACKET_AUDS) || (cur_audio_packet == AUDIO_PACKET_HBR)) &&
                               (last_audio_packet != cur_audio_packet));

    if (in_device & AUDIO_DEVICE_IN_HDMI) {
        bool hw_stable = is_hdmi_in_stable_hw(stream);
        if ((!hw_stable) || is_audio_packet_changed) {
            ALOGV("%s() hdmi in hw unstable\n", __func__);
            *mute_time = 300;
            in->last_audio_packet_type = cur_audio_packet;
            return false;
        }
    }
    if ((in_device & AUDIO_DEVICE_IN_TV_TUNER) &&
            !is_atv_in_stable_hw (stream)) {
        *mute_time = 1000;
        return false;
    }
    if (((in_device & AUDIO_DEVICE_IN_SPDIF) ||
            (in_device & AUDIO_DEVICE_IN_HDMI_ARC)) &&
            !is_spdif_in_stable_hw(stream)) {
        *mute_time = 1000;
        return false;
    }
    if ((in_device & AUDIO_DEVICE_IN_LINE) &&
            !is_av_in_stable_hw(stream)) {
       *mute_time = 1500;
       return false;
    }
    return true;
}

bool check_tv_stream_signal(struct audio_stream_in *stream)
{
    struct aml_stream_in *in = (struct aml_stream_in *)stream;
    struct aml_audio_device *adev = in->dev;
    struct aml_audio_patch* patch = adev->audio_patch;
    int in_mute = 0, parental_mute = 0;
    bool stable = true;
    stable = signal_status_check(adev->in_device, &in->mute_mdelay, stream);
    if (!stable) {
        if (in->mute_log_cntr == 0)
            ALOGI("%s: audio is unstable, mute channel", __func__);
        if (in->mute_log_cntr++ >= 100)
            in->mute_log_cntr = 0;
        clock_gettime(CLOCK_MONOTONIC, &in->mute_start_ts);
        in->mute_flag = true;
    }
    if (in->mute_flag) {
        in_mute = Stop_watch(in->mute_start_ts, in->mute_mdelay);
        if (!in_mute) {
            ALOGI("%s: unmute audio since audio signal is stable", __func__);
            /* The data of ALSA has not been read for a long time in the muted state,
             * resulting in the accumulation of data. So, cache of capture needs to be cleared.
             */
            if (!(in->device & AUDIO_DEVICE_IN_HDMI_ARC || in->device & AUDIO_DEVICE_IN_SPDIF))
                pcm_stop(in->pcm);
            in->mute_log_cntr = 0;
            in->mute_flag = false;
        }
    }

    if (adev->parental_control_av_mute && (adev->active_inport == INPORT_TUNER || adev->active_inport == INPORT_LINEIN))
        parental_mute = 1;

    /*if need mute input source, don't read data from hardware anymore*/
    if (in_mute || parental_mute) {

        /* when audio is unstable, start avsync*/
        if (patch && in_mute) {
            if (!(in->device & AUDIO_DEVICE_IN_HDMI_ARC || in->device & AUDIO_DEVICE_IN_SPDIF))
                patch->need_do_avsync = true;
            patch->input_signal_stable = false;
            adev->mute_start = true;
        }
        return false;
    } else {
        if (patch) {
            patch->input_signal_stable = true;
        }
    }
    return true;
}

const char *write_func_strs[MIXER_WRITE_FUNC_MAX] = {
    "OUT_WRITE_NEW",
    "MIXER_AUX_BUFFER_WRITE_SM",
    "MIXER_MAIN_BUFFER_WRITE_SM,"
    "MIXER_MMAP_BUFFER_WRITE_SM"
    "MIXER_AUX_BUFFER_WRITE",
    "MIXER_MAIN_BUFFER_WRITE",
    "MIXER_APP_BUFFER_WRITE",
    "PROCESS_BUFFER_WRITE,"
};

const char *write_func_to_str(enum stream_write_func func)
{
    return write_func_strs[func];
}

/*this should match with stream_status_t in audio_hw.h*/
const char *stream_status_2_string[STREAM_STATUS_MAX] = {
    "_STANDBY",
    "_HW_WRITING",
    "_MIXING",
    "_PAUSED"
};


void aml_stream_out_info_print(struct aml_stream_out *aml_out)
{
    struct timespec current_timestamp;
    clock_gettime(CLOCK_MONOTONIC, &current_timestamp);
    int64_t time_diff = calc_time_interval_us(&aml_out->last_info_timestamp, &current_timestamp);
    ALOGV("%s time_diff %"PRIu64" ", __func__, time_diff);
    if (time_diff >= (TIME_DIFF_THRESHOLD * USEC_PER_SEC) || aml_out->jitter_ms > 100) {
        char * stream_type = audio_is_linear_pcm(aml_out->hal_format) ? "pcm" : "raw";
        char * sync_mode   = aml_out->hw_sync_mode ? "tunnel" : "non tunnel";
        int64_t cur_time_nanos = (long long)aml_out->last_timestamp_reported.tv_sec * NSEC_PER_SEC + (long long)aml_out->last_timestamp_reported.tv_nsec;
        ALOGI("[audio_stream_out,stream_id:%p] time_diff[%"PRIu64"]us stream_type[%s] sync_mode[%s] input_size[%"PRIu64"]bytes, last_position[%"PRIu64"], last_time[%"PRIu64"]us, delay[%d], jitter[%"PRIu64"]ms",
            aml_out,
            time_diff,
            stream_type,
            sync_mode,
            aml_out->input_bytes_size,
            aml_out->last_frame_reported,
            cur_time_nanos / NSEC_PER_USEC,
            aml_out->audio_delay,
            aml_out->jitter_ms
            );
        aml_out->last_info_timestamp = current_timestamp;
    }
    return;
}

void aml_stream_out_dump(struct aml_stream_out *aml_out, int fd)
{
    if (aml_out) {
        dprintf(fd, "\t\t-usecase: %s\n", usecase2Str(aml_out->usecase));
        dprintf(fd, "\t\t-out device: %#x\n", aml_out->out_device);
        dprintf(fd, "\t\t-is tv source stream: %s\n", aml_out->is_tv_src_stream?"true":"false");
        dprintf(fd, "\t\t-stream status:%d %s\n", aml_out->stream_status, stream_status_2_string[aml_out->stream_status]);
        dprintf(fd, "\t\t-standby: %s\n", aml_out->standby?"true":"false");
        if (aml_out->is_normal_pcm) {
            dprintf(fd, "\t\t-normal pcm: %s\n",
                write_func_to_str(aml_out->write_func));
        }

        uint64_t frames;
        struct timespec timestamp;
        aml_out->stream.get_presentation_position((const struct audio_stream_out *)aml_out, &frames, &timestamp);
        dprintf(fd, "\t\t-presentation_position::%" PRIu64 "   | sec:%ld  nsec:%ld\n", frames, timestamp.tv_sec, timestamp.tv_nsec);
    }
}

void aml_adev_stream_out_dump(struct aml_audio_device *aml_dev, int fd) {
    dprintf(fd, "\n-------------[AML_HAL] StreamOut --------------------------------\n");
    dprintf(fd, "[AML_HAL]    usecase_masks: %#x\n", aml_dev->usecase_masks);
    dprintf(fd, "[AML_HAL]    stream outs:\n");
    for (int i = 0; i < STREAM_USECASE_MAX ; i++) {
        struct aml_stream_out *aml_out = aml_dev->active_outputs[i];
        if (aml_out) {
            dprintf(fd, "\tout: %d, pointer: %p\n", i, aml_out);
            aml_stream_out_dump(aml_out, fd);
        }
    }
}

static void aml_audio_port_config_dump(struct audio_port_config *port_config, int fd)
{
    if (port_config == NULL)
        return;

    dprintf(fd, "\t-id(%d), role(%s), type(%s)\n", port_config->id, audioPortRole2Str(port_config->role), audioPortType2Str(port_config->type));
    switch (port_config->type) {
    case AUDIO_PORT_TYPE_DEVICE:
        dprintf(fd, "\t-port device: type(%#x) addr(%s)\n",
               port_config->ext.device.type, port_config->ext.device.address);
        break;
    case AUDIO_PORT_TYPE_MIX:
        dprintf(fd, "\t-port mix: io handle(%d)\n", port_config->ext.mix.handle);
        break;
    default:
        break;
    }
}

static void aml_audio_patch_dump(struct audio_patch *patch, int fd)
{
    int i = 0;

    dprintf(fd, " handle %d\n", patch->id);
    for (i = 0; i < patch->num_sources; i++) {
        dprintf(fd, "    [src  %d]\n", i);
        aml_audio_port_config_dump(&patch->sources[i], fd);
    }

    for (i = 0; i < patch->num_sinks; i++) {
        dprintf(fd, "    [sink %d]\n", i);
        aml_audio_port_config_dump(&patch->sinks[i], fd);
    }
}

static void aml_audio_patches_dump(struct aml_audio_device* aml_dev, int fd)
{
    struct audio_patch_set *patch_set = NULL;
    struct audio_patch *patch = NULL;
    struct listnode *node = NULL;
    int i = 0;

    dprintf(fd, "\n-------------[AML_HAL] Registered Audio Patches------------------\n");
    list_for_each(node, &aml_dev->patch_list) {
        dprintf(fd, "  patch %d:", i);
        patch_set = node_to_item (node, struct audio_patch_set, list);
        if (patch_set)
            aml_audio_patch_dump(&patch_set->audio_patch, fd);

        i++;
    }
}

int aml_dev_dump_latency(struct aml_audio_device *aml_dev, int fd)
{
    struct aml_stream_in *in = aml_dev->active_input;
    struct aml_audio_patch *patch = aml_dev->audio_patch;

    dprintf(fd, "\n-------------[AML_HAL] audio patch Latency-----------------------\n");

    if (patch) {
        aml_dev_sample_audio_path_latency(aml_dev, NULL);
        dprintf(fd, "[AML_HAL]      audio patch latency         : %6d ms\n", patch->audio_latency.ringbuffer_latency);
        dprintf(fd, "[AML_HAL]      audio spk tuning latency    : %6d ms\n", patch->audio_latency.user_tune_latency);
        dprintf(fd, "[AML_HAL]      MS12 buffer latency         : %6d ms\n", patch->audio_latency.ms12_latency);
        dprintf(fd, "[AML_HAL]      alsa out hw i2s latency     : %6d ms\n", patch->audio_latency.alsa_i2s_out_latency);
        dprintf(fd, "[AML_HAL]      alsa out hw spdif latency   : %6d ms\n", patch->audio_latency.alsa_spdif_out_latency);
        dprintf(fd, "[AML_HAL]      alsa in hw latency          : %6d ms\n\n", patch->audio_latency.alsa_in_latency);
        dprintf(fd, "[AML_HAL]      audio total latency         :%6d ms\n", patch->audio_latency.total_latency);

        int v_ltcy = aml_dev_sample_video_path_latency(patch);
        if (v_ltcy > 0) {
            dprintf(fd, "[AML_HAL]      video path total latency    : %6d ms\n", v_ltcy);
        } else {
            dprintf(fd, "[AML_HAL]      video path total latency    : N/A\n");
        }
    }
    return 0;
}

static void audio_patch_dump(struct aml_audio_device* aml_dev, int fd)
{
    struct aml_audio_patch *pstPatch = aml_dev->audio_patch;
    if (NULL == pstPatch) {
        dprintf(fd, "\n-------------[AML_HAL] audio device patch [not create]-----------\n");
        return;
    }
    dprintf(fd, "\n-------------[AML_HAL] audio device patch [%p]---------------\n", pstPatch);
    if (pstPatch->aml_ringbuffer.size != 0) {
        uint32_t u32FreeBuffer = get_buffer_write_space(&pstPatch->aml_ringbuffer);
        dprintf(fd, "[AML_HAL]      RingBuf   size: %10d Byte|  UnusedBuf:%10d Byte(%d%%)\n",
        pstPatch->aml_ringbuffer.size, u32FreeBuffer, u32FreeBuffer* 100 / pstPatch->aml_ringbuffer.size);
    } else {
        dprintf(fd, "[AML_HAL]      patch  RingBuf    : buffer size is 0\n");
    }
    if (pstPatch->audio_parse_para) {
        int s32AudioType = ((audio_type_parse_t*)pstPatch->audio_parse_para)->audio_type;
        dprintf(fd, "[AML_HAL]      Hal audio Type: [0x%x]%-10s| Src Format:%#10x\n", s32AudioType,
            audio_type_convert_to_string(s32AudioType),  pstPatch->aformat);
    }

    dprintf(fd, "[AML_HAL]      IN_SRC        : %#10x     | OUT_SRC   :%#10x\n", pstPatch->input_src, pstPatch->output_src);
    dprintf(fd, "[AML_HAL]      IN_Format     : %#10x     | OUT_Format:%#10x\n", pstPatch->aformat, pstPatch->out_format);
    dprintf(fd, "[AML_HAL]      sink format: %#x\n", aml_dev->sink_format);
    if ((aml_dev->cur_out_devices & AUDIO_DEVICE_OUT_HDMI_ARC) != 0) {
        struct aml_arc_hdmi_desc *hdmi_desc = &aml_dev->hdmi_descs;
        bool dd_is_support = hdmi_desc->dd_fmt.is_support;
        bool ddp_is_support = hdmi_desc->ddp_fmt.is_support;
        bool mat_is_support = hdmi_desc->mat_fmt.is_support;

        dprintf(fd, "[AML_HAL]      -dd: %d, ddp: %d, mat: %d\n",
                dd_is_support, ddp_is_support, mat_is_support);
    }
}

void aml_adev_audio_patch_dump(struct aml_audio_device* aml_dev, int fd)
{
    //android registered audio patch info dump
    aml_audio_patches_dump(aml_dev, fd);
    //aml audio device patch detail dump
    audio_patch_dump(aml_dev, fd);
}

void aml_alsa_device_status_dump(struct aml_audio_device* aml_dev, int fd)
{
    dprintf(fd, "\n-------------[AML_HAL]  ALSA devices status ---------------------\n");
    bool stream_using = false;
    /* StreamOut using alsa devices list */
    for (int i = 0; i < ALSA_DEVICE_CNT; i++) {
        pthread_mutex_lock(&aml_dev->lock);
        pthread_mutex_lock(&aml_dev->alsa_pcm_lock);
        struct pcm *pcm_1 = aml_dev->pcm_handle[i];
        void *alsa_handle = aml_dev->alsa_handle[i];
        if (!pcm_1 && !alsa_handle) {
            pthread_mutex_unlock(&aml_dev->alsa_pcm_lock);
            pthread_mutex_unlock(&aml_dev->lock);
            continue;
        }

        if (!stream_using) {
            dprintf(fd, "  [AML_HAL] StreamOut using PCM list:\n");
            stream_using = true;
        }

        if (pcm_1) {
            aml_alsa_pcm_info_dump(pcm_1, fd);
        }

        if (alsa_handle) {
            struct pcm* pcm_2 = (struct pcm*) get_internal_pcm(alsa_handle);
            aml_alsa_pcm_info_dump(pcm_2, fd);
        }
        pthread_mutex_unlock(&aml_dev->alsa_pcm_lock);
        pthread_mutex_unlock(&aml_dev->lock);
    }
    if (!stream_using) {
        dprintf(fd, "  [AML_HAL] StreamOut using PCM list: None!\n");
    }

    /* Mixer outport using alsa devices list */
    dprintf(fd, "  [AML_HAL] mixer using PCM list:\n");
    mixer_using_alsa_device_dump(fd, aml_dev);

    /* StreamIn using alsa devices list */
    pthread_mutex_lock(&aml_dev->lock);
    stream_using = false;
    if (aml_dev->active_input) {
        struct pcm *pcm = aml_dev->active_input->pcm;
        if (pcm) {
            stream_using = true;
            dprintf(fd, "  [AML_HAL] StreamIn using PCM list:\n");
            aml_alsa_pcm_info_dump(pcm, fd);
        }
    }
    if (!stream_using) {
       dprintf(fd, "  [AML_HAL] StreamIn using PCM list: None!\n");
    }
    pthread_mutex_unlock(&aml_dev->lock);
}


void aml_decoder_info_dump(struct aml_audio_device *adev, int fd)
{
    dprintf(fd, "\n-------------[AML_HAL] licence decoder --------------------------\n");
    dprintf(fd, "[AML_HAL]    dolby_lib: %d\n", adev->dolby_lib_type);
    dprintf(fd, "[AML_HAL]    build ms12 version: %d\n", adev->support_ms12_version);
    dprintf(fd, "[AML_HAL]    MS12 library size:\n");
    dprintf(fd, "             \t-V2 Encrypted: %d\n", get_file_size("/oem/lib/ms12/libdolbyms12.so"));
    dprintf(fd, "             \t-V2 Decrypted: %d\n", get_file_size("/odm/lib/ms12/libdolbyms12.so"));
    dprintf(fd, "             \t-V1 Encrypted: %d\n", get_file_size("/oem/lib/libdolbyms12.so"));
    dprintf(fd, "             \t-V1 Decrypted: %d\n", get_file_size("/odm/lib/libdolbyms12.so"));
    dprintf(fd, "[AML_HAL]    DDP library size:\n");
    dprintf(fd, "             \t-lib32: %d\n", get_file_size("/odm/lib/libHwAudio_dcvdec.so"));
    dprintf(fd, "             \t-lib64: %d\n", get_file_size("/odm/lib64/libHwAudio_dcvdec.so"));
    dprintf(fd, "[AML_HAL]    DTS library size:\n");
    dprintf(fd, "             \t-lib32: %d\n", get_file_size("/odm/lib/libHwAudio_dtshd.so"));
    dprintf(fd, "             \t-lib64: %d\n", get_file_size("/odm/lib64/libHwAudio_dtshd.so"));
}

static void print_enum(struct mixer_ctl *ctl, int fd)
{
    unsigned int num_enums;
    unsigned int i;
    unsigned int value;
    const char *string;

    num_enums = mixer_ctl_get_num_enums(ctl);
    value = mixer_ctl_get_value(ctl, 0);

    for (i = 0; i < num_enums; i++) {
        string = mixer_ctl_get_enum_string(ctl, i);
        dprintf(fd, "%s%s, ", value == i ? "> " : "", string);
    }
}

static void print_control_values(struct mixer_ctl *control, int fd)
{
    enum mixer_ctl_type type;
    unsigned int num_values;
    unsigned int i;
    int min, max;
    int ret;
    char *buf = NULL;

    type = mixer_ctl_get_type(control);
    num_values = mixer_ctl_get_num_values(control);

    if ((type == MIXER_CTL_TYPE_BYTE) && (num_values > 0)) {
        buf = calloc(1, num_values);
        if (buf == NULL) {
            ALOGE("Failed to alloc mem for bytes %u", num_values);
            return;
        }

        ret = mixer_ctl_get_array(control, buf, num_values);
        if (ret < 0) {
            ALOGE("Failed to mixer_ctl_get_array");
            free(buf);
            return;
        }
    }

    for (i = 0; i < num_values; i++) {
        switch (type)
        {
        case MIXER_CTL_TYPE_INT:
            dprintf(fd,"%d", mixer_ctl_get_value(control, i));
            break;
        case MIXER_CTL_TYPE_BOOL:
            dprintf(fd,"%s", mixer_ctl_get_value(control, i) ? "On" : "Off");
            break;
        case MIXER_CTL_TYPE_ENUM:
            print_enum(control, fd);
            break;
        case MIXER_CTL_TYPE_BYTE:
            dprintf(fd,"%02hhx", buf[i]);
            break;
        default:
            dprintf(fd,"unknown");
            break;
        };
        if ((i + 1) < num_values) {
           dprintf(fd, ", ");
        }
    }

    if (type == MIXER_CTL_TYPE_INT) {
        min = mixer_ctl_get_range_min(control);
        max = mixer_ctl_get_range_max(control);
        dprintf(fd, " (range %d->%d)", min, max);
    }

    free(buf);
}

void aml_alsa_mixer_status_dump(struct aml_audio_device *adev, int fd)
{
    dprintf(fd, "\n-------------[AML_HAL] ALSA mixer status ------------------------\n");

    struct mixer_ctl *ctl;
    const char *name, *type;
    unsigned int num_ctls, num_values;
    unsigned int i;
    struct aml_mixer_handle *aml_mixer = &adev->alsa_mixer;

    if (!aml_mixer->pMixer) {
        ALOGW("%s() Warning! mixer = NULL!, return!", __func__);
        return;
    }

    num_ctls = mixer_get_num_ctls(aml_mixer->pMixer);

    dprintf(fd,"Number of controls: %u\n", num_ctls);

    dprintf(fd,"ctl\ttype\tnum\t%-40svalue\n", "name");

    for (i = 0; i < num_ctls; i++) {
        ctl = mixer_get_ctl(aml_mixer->pMixer, i);  //ask one mixer_ctrl

        name = mixer_ctl_get_name(ctl);
        type = mixer_ctl_get_type_string(ctl);
        num_values = mixer_ctl_get_num_values(ctl);
        dprintf(fd, "%u\t%s\t%u\t%-40s", i, type, num_values, name);

        pthread_mutex_lock(&aml_mixer->lock);
        print_control_values(ctl, fd);
        pthread_mutex_unlock(&aml_mixer->lock);
        dprintf(fd, "\n");
    }
}


bool is_use_spdifb(struct aml_stream_out *out) {
    struct aml_audio_device *adev = out->dev;
    /*this patch is for DCV DDP noise.
    **DCV have two kinds of mode, DCV decoder and passthrough.
    **the dolby_decode_enable is 0 when DCV passthrough.
    **so here should remove the dolby_decode_enable judgment.
    */
    if (eDolbyDcvLib == adev->dolby_lib_type /*&& adev->dolby_decode_enable*/ &&
        (out->hal_format == AUDIO_FORMAT_E_AC3 || out->hal_internal_format == AUDIO_FORMAT_E_AC3 ||
        (out->need_convert && out->hal_internal_format == AUDIO_FORMAT_AC3))) {
        /*dual spdif we need convert
          or non dual spdif, we need check audio setting and optical format*/
        if (adev->dual_spdif_support) {
            out->dual_spdif = true;
        }
        if (out->dual_spdif && ((adev->digital_audio_mode == AML_DIGITAL_AUDIO_MODE_AUTO) &&
            adev->optical_format == AUDIO_FORMAT_E_AC3) &&
            out->hal_rate != 32000) {
            return true;
        }
    }

    return false;
}

bool is_dolby_ms12_support_compression_format(audio_format_t format)
{

#ifdef MS12_V24_ENABLE
    if (format == AUDIO_FORMAT_HE_AAC_V1 ||
        format == AUDIO_FORMAT_HE_AAC_V2 ||
        format == AUDIO_FORMAT_AAC ||
        format == AUDIO_FORMAT_AAC_LATM)  {
        if (property_get_bool("ro.vendor.audio.use.ms12heaac", false)) {
            return true;
        }
    }
#endif

    return (format == AUDIO_FORMAT_AC3 ||
            format == AUDIO_FORMAT_E_AC3 ||
            format == AUDIO_FORMAT_E_AC3_JOC ||
            format == AUDIO_FORMAT_DOLBY_TRUEHD ||
            format == AUDIO_FORMAT_AC4 ||
            format == AUDIO_FORMAT_MAT);
}

bool is_dolby_ddp_support_compression_format(audio_format_t format)
{
    return (format == AUDIO_FORMAT_AC3 ||
            format == AUDIO_FORMAT_E_AC3 ||
            format == AUDIO_FORMAT_E_AC3_JOC);
}

bool is_direct_stream_and_pcm_format(struct aml_stream_out *out)
{
    return audio_is_linear_pcm(out->hal_internal_format) && (out->flags & AUDIO_OUTPUT_FLAG_DIRECT);
}

bool is_mmap_stream_and_pcm_format(struct aml_stream_out *out)
{
    return audio_is_linear_pcm(out->hal_internal_format) && (out->flags & AUDIO_OUTPUT_FLAG_MMAP_NOIRQ);
}

void get_audio_indicator(struct aml_audio_device *dev, char *temp_buf) {
    struct aml_audio_device *adev = (struct aml_audio_device *) dev;

    if (adev->audio_hal_info.update_type == TYPE_PCM)
        sprintf (temp_buf, "audioindicator=");
    else if (adev->audio_hal_info.update_type == TYPE_AC3)
        sprintf (temp_buf, "audioindicator=Dolby AC3");
    else if (adev->audio_hal_info.update_type == TYPE_EAC3)
        sprintf (temp_buf, "audioindicator=Dolby EAC3");
    else if (adev->audio_hal_info.update_type == TYPE_AC4)
        sprintf (temp_buf, "audioindicator=Dolby AC4");
    else if (adev->audio_hal_info.update_type == TYPE_MAT)
        sprintf (temp_buf, "audioindicator=Dolby MAT");
    else if (adev->audio_hal_info.update_type == TYPE_TRUE_HD)
        sprintf (temp_buf, "audioindicator=Dolby THD");
    else if (adev->audio_hal_info.update_type == TYPE_DDP_ATMOS)
        sprintf (temp_buf, "audioindicator=Dolby EAC3,Dolby Atmos");
    else if (adev->audio_hal_info.update_type == TYPE_TRUE_HD_ATMOS)
        sprintf (temp_buf, "audioindicator=Dolby THD,Dolby Atmos");
    else if (adev->audio_hal_info.update_type == TYPE_MAT_ATMOS)
        sprintf (temp_buf, "audioindicator=Dolby MAT,Dolby Atmos");
    else if (adev->audio_hal_info.update_type == TYPE_AC4_ATMOS)
        sprintf (temp_buf, "audioindicator=Dolby AC4,Dolby Atmos");
    else if (adev->audio_hal_info.update_type == TYPE_DTS)
        sprintf (temp_buf, "audioindicator=DTS");
    else if (adev->audio_hal_info.update_type == TYPE_DTS_HD_MA)
        sprintf (temp_buf, "audioindicator=DTS HD");
    else if (adev->audio_hal_info.update_type == TYPE_DDP_ATMOS_PROMPT_ON_ATMOS)
        sprintf (temp_buf, "audioindicator=Dolby EAC3");
    else if (adev->audio_hal_info.update_type == TYPE_TRUE_HD_ATMOS_PROMPT_ON_ATMOS)
        sprintf (temp_buf, "audioindicator=Dolby THD");
    else if (adev->audio_hal_info.update_type == TYPE_MAT_ATMOS_PROMPT_ON_ATMOS)
        sprintf (temp_buf, "audioindicator=Dolby MAT");
    else if (adev->audio_hal_info.update_type == TYPE_AC4_ATMOS_PROMPT_ON_ATMOS)
        sprintf (temp_buf, "audioindicator=Dolby AC4");

    ALOGI("%s(), [%s]", __func__, temp_buf);
}

static int update_audio_hal_info(struct aml_audio_device *adev, audio_format_t format, int atmos_flag)
{
    struct dolby_ms12_desc *ms12 = &(adev->ms12);
    int update_type = get_codec_type(format);
    int update_threshold = DOLBY_FMT_UPDATE_THRESHOLD;
    int cur_aml_dap_surround_virtualizer = dolby_ms12_get_dap_surround_virtualizer();

    if (is_dolby_ms12_support_compression_format(format)) {
        update_threshold = DOLBY_FMT_UPDATE_THRESHOLD;
    } else if (is_dts_format(format)) {
        update_threshold = DTS_FMT_UPDATE_THRESHOLD;
    }

    /* avoid the value out-of-bounds */
    if (adev->audio_hal_info.update_cnt < FMT_UPDATE_THRESHOLD_MAX) {
        adev->audio_hal_info.update_cnt++;
    }

    /* Check whether the update_type is stable or not as bellow. */
    bool is_virt_updated_off_vs_on_auto = (!!adev->audio_hal_info.aml_dap_surround_virtualizer != !!cur_aml_dap_surround_virtualizer);
    bool is_virt_updated_for_aml_atmos = (atmos_flag && is_virt_updated_off_vs_on_auto);

    if ((format != adev->audio_hal_info.format) ||
        (atmos_flag != adev->audio_hal_info.is_dolby_atmos) ||
        is_virt_updated_for_aml_atmos) {
        adev->audio_hal_info.update_cnt = 0;
    }

    /* @audio_format_t does not include dts_express.
     * So we get the format(dts_express especially) from the decoder(@dts_hd.stream_type).
     * @dts_hd.stream_type is updated after decoding at least one frame.
     */
    if (is_dts_format(format)) {
        if (adev->dts_hd.stream_type <= 0 /*TYPE_PCM*/) {
            adev->audio_hal_info.update_cnt = 0;
        }

        update_type = adev->dts_hd.stream_type;
        if (update_type != adev->audio_hal_info.update_type) {
            adev->audio_hal_info.update_cnt = 0;
        }
    }

    bool is_dolby_atmos_off = (MS12_DAP_SPEAKER_VIRTUALIZER_OFF == cur_aml_dap_surround_virtualizer);
    if (atmos_flag == 1) {
        if (format == AUDIO_FORMAT_E_AC3)
            update_type = (is_dolby_atmos_off) ? TYPE_DDP_ATMOS_PROMPT_ON_ATMOS : TYPE_DDP_ATMOS;
        else if (format == AUDIO_FORMAT_DOLBY_TRUEHD)
            update_type = (is_dolby_atmos_off) ? TYPE_TRUE_HD_ATMOS_PROMPT_ON_ATMOS : TYPE_TRUE_HD_ATMOS;
        else if (format == AUDIO_FORMAT_MAT)
            update_type = (is_dolby_atmos_off) ? TYPE_MAT_ATMOS_PROMPT_ON_ATMOS : TYPE_MAT_ATMOS;
        else if (format == AUDIO_FORMAT_AC4)
            update_type = (is_dolby_atmos_off) ? TYPE_AC4_ATMOS_PROMPT_ON_ATMOS : TYPE_AC4_ATMOS;
    }

    ALOGV("%s() update_cnt %d format %#x vs hal_internal_format %#x  atmos_flag %d vs is_dolby_atmos %d update_type %d\n",
        __FUNCTION__, adev->audio_hal_info.update_cnt, format, adev->audio_hal_info.format,
        atmos_flag, adev->audio_hal_info.is_dolby_atmos, update_type);

    adev->audio_hal_info.format = format;
    adev->audio_hal_info.is_dolby_atmos = atmos_flag;
    adev->audio_hal_info.update_type = update_type;
    adev->audio_hal_info.aml_dap_surround_virtualizer = cur_aml_dap_surround_virtualizer;

    if (adev->audio_hal_info.update_cnt == update_threshold) {

        if ((format == AUDIO_FORMAT_DTS || format == AUDIO_FORMAT_DTS_HD) && adev->dts_hd.is_headphone_x) {
            aml_mixer_ctrl_set_int(&adev->alsa_mixer, AML_MIXER_ID_AUDIO_HAL_FORMAT, TYPE_DTS_HP);
        }
        aml_mixer_ctrl_set_int(&adev->alsa_mixer, AML_MIXER_ID_AUDIO_HAL_FORMAT, update_type);
        ALOGD("%s() audio hal format change to %x, atmos flag = %d, is_dolby_atmos = %d, dts_hp_x = %d, update_type = %d is_dolby_atmos_off = %d\n",
            __FUNCTION__, adev->audio_hal_info.format, adev->audio_hal_info.is_dolby_atmos, adev->ms12.is_dolby_atmos,
            adev->dts_hd.is_headphone_x, adev->audio_hal_info.update_type, is_dolby_atmos_off);
        ALOGD("%s() cur_out_devices %#x, dap_bypass_enable = %d, is_ms12_tuning_dat = %d, dolby_ms12_enable = %d, output_config = %#x\n",
            __FUNCTION__, adev->cur_out_devices, adev->ms12.dap_bypass_enable, adev->is_ms12_tuning_dat, ms12->dolby_ms12_enable, ms12->output_config);
    }

    return 0;
}

void update_audio_format(struct aml_audio_device *adev, audio_format_t format)
{
    int atmos_flag = 0;
    int update_type = TYPE_PCM;
    bool is_dolby_active = dolby_stream_active(adev);
    bool is_dolby_format = is_dolby_ms12_support_compression_format(format);
    bool is_dts_active = dts_stream_active(adev);
    /*
     *for dolby & pcm case or dolby case
     *to update the dolby stream's format
     */
    if (is_dolby_active && is_dolby_format) {
        if (eDolbyMS12Lib == adev->dolby_lib_type) {
            atmos_flag = adev->ms12.is_dolby_atmos;
        }

#ifdef MS12_V24_ENABLE
        /* when DAP is not in audio postprocessing, there is no ATMOS Experience. */
        if (is_audio_postprocessing_add_dolbyms12_dap(adev) == 0) {
            atmos_flag = 0;
        }
#else
        atmos_flag = 0; /* Todo: MS12 V1, how to indicate the Dolby ATMOS? */
#endif

        update_audio_hal_info(adev, format, atmos_flag);
    }
    /*
     *to update the audio format for other cases
     *DTS-format / DTS format & Mixer-PCM
     *only Mixer-PCM
     */
    else if (!is_dolby_active && !is_dolby_format) {

        /* if there is DTS/DTS alive, only update DTS/DTS-HD */
        if (is_dts_active) {
            if (is_dts_format(format)) {
                update_audio_hal_info(adev, format, false);
            }
            /*else case is PCM format, will ignore that PCM*/
        }

        /* there is none-dolby or none-dts alive, then update PCM format*/
        if (!is_dts_active) {
            update_audio_hal_info(adev, format, false);
        }
    }
    /*
     * **Dolby stream is active, and get the Mixer-PCM case steam format,
     * **we should ignore this Mixer-PCM update request.
     * else if (is_dolby_active && !is_dolby_format) {
     * }
     * **If Dolby steam is not active, the available format is LPCM or DTS
     * **The following case do not exit at all **
     //else //(!is_dolby_active && is_dolby_format) {
     * }
     */
}

int audio_route_set_hdmi_arc_mute(struct aml_mixer_handle *mixer_handle, int enable)
{
    int extern_arc = 0;

    if (check_chip_name("t5", 3, mixer_handle) || check_chip_name("t5d", 3, mixer_handle))
        extern_arc = 1;
    if (extern_arc) {
        return aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_SPDIF_MUTE, enable);
    } else {
       if (is_earc_descrpt())
           return aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_ARC_EARC_SPDIFOUT_REG_MUTE, enable);
       else
           return aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_HDMI_ARC_AUDIO_ENABLE, !enable);
    }
}

int audio_route_set_spdif_mute(struct aml_mixer_handle *mixer_handle, int enable)
{
    int extern_arc = 0;

    if (check_chip_name("t5", 3, mixer_handle) || check_chip_name("t5d", 3, mixer_handle))
        extern_arc = 1;

    if (extern_arc) {
        return aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_SPDIF_B_MUTE, enable);
    } else {
        return aml_mixer_ctrl_set_int(mixer_handle, AML_MIXER_ID_SPDIF_MUTE, enable);
    }
}

void audio_route_set_speaker_mute(struct aml_audio_device* aml_dev, int enable)
{
    if (aml_dev == NULL) {
        return;
    }

    if (enable) {
        //Need reset fading status, keep alsa mixer and audio route same status.
        audio_route_apply_path(aml_dev->ar, "speaker_fadein");
        audio_route_update_mixer(aml_dev->ar);
        audio_route_apply_path(aml_dev->ar, "speaker_fadeout");
    } else {
        audio_route_apply_path(aml_dev->ar, "speaker_fadein");
    }
    audio_route_update_mixer(aml_dev->ar);
    // Need time to fedaout
    if (enable) {
        aml_audio_sleep(15000);
    }

    return;
}

void audio_route_set_speaker_mute_l(struct aml_audio_device* aml_dev, int enable)
{
    if (aml_dev == NULL) {
        return;
    }

    if (enable) {
        //Need reset fading status, keep alsa mixer and audio route same status.
        audio_route_apply_path(aml_dev->ar, "speaker_fadein");
        audio_route_update_mixer(aml_dev->ar);
        audio_route_apply_path(aml_dev->ar, "speaker_fadeout");
    } else {
        audio_route_apply_path(aml_dev->ar, "speaker_fadein");
    }
    audio_route_update_mixer(aml_dev->ar);

    return;
}

void audio_raw_data_continuous_check(struct aml_audio_device *aml_dev, audio_type_parse_t *status, char *buffer, int size)
{
    audio_type_parse_t *audio_type_status = status;
    struct aml_audio_patch* patch = aml_dev->audio_patch;

    if (!audio_type_status || !aml_dev->audio_patch) {
        return;
    }

    int sync_word_offset = find_61937_sync_word(buffer, size);
    if (sync_word_offset >= 0) {
        patch->sync_offset = sync_word_offset;
        if (patch->start_mute) {
            audio_route_set_speaker_mute_l(aml_dev, false);
            patch->start_mute = false;
            patch->mdelay = 0;
        }
        if (patch->read_size > 0) {
            patch->read_size = 0;
        }
        if (!patch->read_size) {
            patch->read_size = size;
            patch->read_size -= sync_word_offset;
        }
    } else if (patch->sync_offset >= 0) {
        if ((patch->read_size < audio_type_status->package_size) && ((patch->read_size + size) > audio_type_status->package_size)) {
            audio_route_set_speaker_mute_l(aml_dev, true);
            clock_gettime(CLOCK_MONOTONIC, &patch->start_ts);
            patch->start_mute = true;
            patch->read_size = 0;
            patch->mdelay = DEFAULT_PLAYBACK_PERIOD_SIZE * DEFAULT_PLAYBACK_PERIOD_CNT / (MM_FULL_POWER_SAMPLING_RATE / 1000);
        } else {
            if (patch->start_mute) {
                int flag = Stop_watch(patch->start_ts, patch->mdelay);
                if (!flag) {
                    patch->sync_offset = -1;
                    patch->start_mute = false;
                    audio_route_set_speaker_mute_l(aml_dev, false);
                }
            } else {
                patch->read_size += size;
            }
        }
    }
}

int reconfig_read_param_through_hdmiin(struct aml_audio_device *aml_dev,
                                       struct aml_stream_in *stream_in,
                                       ring_buffer_t *ringbuffer, int buffer_size)
{
    int ring_buffer_size = buffer_size;
    int last_channel_count = 2;
    int s32Ret = 0;
    bool is_channel_changed = false;
    bool is_audio_packet_changed = false;
    hdmiin_audio_packet_t last_audio_packet = AUDIO_PACKET_AUDS;
    int period_size = 0;
    int buf_size = 0;

    if (!aml_dev || !stream_in) {
        ALOGE("%s line %d aml_dev %p stream_in %p\n", __func__, __LINE__, aml_dev, stream_in);
        return -1;
    }

    /* check game mode change and reconfig input */
    if (aml_dev->mode_reconfig_in) {
        int play_buffer_size = DEFAULT_PLAYBACK_PERIOD_SIZE * PLAYBACK_PERIOD_COUNT;

        if (is_game_mode(aml_dev)) {
            period_size = LOW_LATENCY_CAPTURE_PERIOD_SIZE;
            buf_size = 2 * 4 * LOW_LATENCY_PLAYBACK_PERIOD_SIZE;
        } else {
            period_size = DEFAULT_CAPTURE_PERIOD_SIZE;
            buf_size = 4 * 2 * 2 * play_buffer_size * PATCH_PERIOD_COUNT;
        }
        ALOGD("%s(), game pic mode %d, period size %d",
                __func__, is_game_mode(aml_dev), period_size);
        stream_in->config.period_size = period_size;
        if (!stream_in->standby) {
            do_input_standby(stream_in);
        }
        s32Ret = start_input_stream(stream_in);
        stream_in->standby = 0;
        if (s32Ret < 0) {
            ALOGE("[%s:%d] start input stream failed! ret:%#x", __func__, __LINE__, s32Ret);
        }

        if (ringbuffer) {
            ring_buffer_reset_size(ringbuffer, buf_size);
        }

        aml_dev->mode_reconfig_in = false;
    }

    last_channel_count = stream_in->config.channels;
    last_audio_packet = stream_in->audio_packet_type;

    hdmiin_audio_packet_t cur_audio_packet = get_hdmiin_audio_packet(&aml_dev->alsa_mixer);
    int current_channel = get_hdmiin_channel(&aml_dev->alsa_mixer);

    /* only audio type is normal pcm audio, change the input channel */
    if (cur_audio_packet == AUDIO_PACKET_AUDS) {
        is_channel_changed = ((current_channel > 0) && last_channel_count != current_channel);
    }
    is_audio_packet_changed = (((cur_audio_packet == AUDIO_PACKET_AUDS) || (cur_audio_packet == AUDIO_PACKET_HBR)) &&
                               (last_audio_packet != cur_audio_packet));
    //reconfig input stream and buffer when HBR and AUDS audio switching or channel num changed
    if ((is_channel_changed) || is_audio_packet_changed) {
        int period_size = 0;
        int buf_size = 0;
        int channel = 2;
        bool bSpdifin_PAO = false;

        ALOGI("HDMI Format Switch [audio_packet pre:%d->cur:%d changed:%d] [channel pre:%d->cur:%d changed:%d]",
            last_audio_packet, cur_audio_packet, is_audio_packet_changed, last_channel_count, current_channel, is_channel_changed);
        if (cur_audio_packet == AUDIO_PACKET_HBR) {
            // if it is high bitrate bitstream, use PAO and increase the buffer size
            bSpdifin_PAO = true;
            period_size = DEFAULT_CAPTURE_PERIOD_SIZE * 4;
            // increase the buffer size
            buf_size = ring_buffer_size * 8;
            channel = 8;
            //if (check_chip_name("t3", 2, &aml_dev->alsa_mixer)) {
            //    channel = 2;    // T3's HDMIRX IP does not support 8CH in design.
            //}
        } else if (cur_audio_packet == AUDIO_PACKET_AUDS) {
            bSpdifin_PAO = false;
            period_size = DEFAULT_CAPTURE_PERIOD_SIZE;
            // reset to original one
            buf_size = ring_buffer_size;
            channel = current_channel;
        }

        if (ringbuffer) {
            ring_buffer_reset_size(ringbuffer, buf_size);
        }
        stream_in->config.period_size = period_size;
        stream_in->config.channels = channel;
        if (!stream_in->standby) {
            do_input_standby(stream_in);
        }
        s32Ret = start_input_stream(stream_in);
        stream_in->standby = 0;
        if (s32Ret < 0) {
            ALOGE("[%s:%d] start input stream failed! ret:%#x", __func__, __LINE__, s32Ret);
        }
        stream_in->audio_packet_type = cur_audio_packet;
        return 0;
    } else {
        return -1;
    }
}

int stream_check_reconfig_param(struct audio_stream_out *stream)
{
    struct aml_stream_out *out = (struct aml_stream_out *)stream;
    struct aml_audio_device *adev = out->dev;
    struct dolby_ms12_desc *ms12 = &(adev->ms12);
    struct audio_board_config *bd_config = &adev->board_config;
    int period_size = 0;

    if (adev->mode_reconfig_out) {
        ALOGD("%s(), game mode reconfig out", __func__);
        if (ms12->dolby_ms12_enable && !is_bypass_dolbyms12(stream)) {
            get_hardware_config_parameters(&(adev->ms12_config),
                AUDIO_FORMAT_PCM_16_BIT,
                bd_config->default_alsa_ch,
                ms12->output_samplerate,
                out->is_tv_platform, continuous_mode(adev),
                is_game_mode(adev));

            adev->mode_reconfig_ms12 = true;
        }
        alsa_out_reconfig_params(stream);
        adev->mode_reconfig_out = false;
    }
    return 0;
}

int update_sink_format_after_hotplug(struct aml_audio_device *adev)
{
    struct audio_stream_out *stream = NULL;
    /* raw stream is at high priority */
    if (adev->active_outputs[STREAM_RAW_DIRECT]) {
        stream = (struct audio_stream_out *)adev->active_outputs[STREAM_RAW_DIRECT];
    }
    else if (adev->active_outputs[STREAM_RAW_HWSYNC]) {
        stream = (struct audio_stream_out *)adev->active_outputs[STREAM_RAW_HWSYNC];
    }
    /* pcm direct/hwsync stream is at medium priority */
    else if (adev->active_outputs[STREAM_PCM_HWSYNC]) {
        stream = (struct audio_stream_out *)adev->active_outputs[STREAM_PCM_HWSYNC];
    }
    else if (adev->active_outputs[STREAM_PCM_DIRECT]) {
        stream = (struct audio_stream_out *)adev->active_outputs[STREAM_PCM_DIRECT];
    }
    /* pcm stream from mixer is at lower priority */
    else if (adev->active_outputs[STREAM_PCM_NORMAL]){
        stream = (struct audio_stream_out *)adev->active_outputs[STREAM_PCM_NORMAL];
    }


    if (stream) {
        ALOGD("%s() active stream %p\n", __FUNCTION__, stream);
        get_sink_format(stream);
    }
    else {
        if (adev->ms12_out && continuous_mode(adev)) {
            ALOGD("%s() active stream is ms12_out %p\n", __FUNCTION__, adev->ms12_out);
            get_sink_format((struct audio_stream_out *)adev->ms12_out);
        }
        else {
            ALOGD("%s() active stream %p ms12_out %p\n", __FUNCTION__, stream, adev->ms12_out);
        }
    }

    return 0;
}

/*----------------------------------------------------------------------------
 * Downmix_foldFrom7Point1()
 *----------------------------------------------------------------------------
 * Purpose:
 * downmix a 7.1 signal to stereo
 *
 * Inputs:
 *  pSrc       7.1 audio samples to downmix
 *  numFrames  the number of 7.1 frames to downmix
 *  accumulate whether to mix (when true) the result of the downmix with the contents of pDst,
 *               or overwrite pDst (when false)
 *
 * Outputs:
 *  pDst       downmixed stereo audio samples
 *
 *----------------------------------------------------------------------------
 */
void Downmix_foldFrom7Point1(float *pSrc, float *pDst, size_t numFrames, bool accumulate) {
    float lt, rt, centerPlusLfeContrib; // samples in Q19.12 format
    // sample at index 0 is FL
    // sample at index 1 is FR
    // sample at index 2 is FC
    // sample at index 3 is LFE
    // sample at index 4 is RL
    // sample at index 5 is RR
    // sample at index 6 is SL
    // sample at index 7 is SR
    // code is mostly duplicated between the two values of accumulate to avoid repeating the test
    // for every sample
    if (accumulate) {
        while (numFrames) {
            // centerPlusLfeContrib = FC(-3dB) + LFE(-3dB)
            centerPlusLfeContrib = (pSrc[2] * MINUS_3_DB_IN_FLOAT)
                    + (pSrc[3] * MINUS_3_DB_IN_FLOAT);
            // FL + centerPlusLfeContrib + SL + RL
            lt = pSrc[0] + centerPlusLfeContrib + pSrc[6] + pSrc[4];
            // FR + centerPlusLfeContrib + SR + RR
            rt = pSrc[1] + centerPlusLfeContrib + pSrc[7] + pSrc[5];
            //accumulate in destination
            pDst[0] = clamp_float(pDst[0] + (lt / 2.0f));
            pDst[1] = clamp_float(pDst[1] + (rt / 2.0f));
            pSrc += 8;
            pDst += 2;
            numFrames--;
        }
    } else { // same code as above but without adding and clamping pDst[i] to itself
        while (numFrames) {
            // centerPlusLfeContrib = FC(-3dB) + LFE(-3dB)
            centerPlusLfeContrib = (pSrc[2] * MINUS_3_DB_IN_FLOAT)
                    + (pSrc[3] * MINUS_3_DB_IN_FLOAT);
            // FL + centerPlusLfeContrib + SL + RL
            lt = pSrc[0] + centerPlusLfeContrib + pSrc[6] + pSrc[4];
            // FR + centerPlusLfeContrib + SR + RR
            rt = pSrc[1] + centerPlusLfeContrib + pSrc[7] + pSrc[5];
            // store in destination
            pDst[0] = clamp_float(lt / 2.0f); // differs from when accumulate is true above
            pDst[1] = clamp_float(rt / 2.0f); // differs from when accumulate is true above
            pSrc += 8;
            pDst += 2;
            numFrames--;
        }
    }
}

/* expand channels or contract channels*/
int input_stream_channels_adjust(struct audio_stream_in *stream, void* buffer, size_t bytes)
{
    struct aml_stream_in *in = (struct aml_stream_in *)stream;
    int ret = -1;

    if (!in || !bytes)
        return ret;

    int channel_count = audio_channel_count_from_in_mask(in->hal_channel_mask);

    if (!channel_count)
        return ret;

    size_t read_bytes = in->config.channels * bytes / channel_count;
    if (!in->input_tmp_buffer || in->input_tmp_buffer_size < read_bytes) {
        in->input_tmp_buffer = aml_audio_realloc(in->input_tmp_buffer, read_bytes * 2);
        if (!in->input_tmp_buffer) {
            AM_LOGE("aml_audio_realloc is fail");
            return ret;
        }
        in->input_tmp_buffer_size = read_bytes;
    }

    ret = aml_alsa_input_read(stream, in->input_tmp_buffer, read_bytes);
    if (in->config.format == PCM_FORMAT_S16_LE) {
#ifdef HDMIIN_MULTICH_DOWNMIX
        int samples = read_bytes / 2;
        int output_samples = bytes / 2;
        memcpy_by_audio_format(in->input_tmp_buffer, AUDIO_FORMAT_PCM_FLOAT,
            in->input_tmp_buffer, AUDIO_FORMAT_PCM_16_BIT, samples);
        Downmix_foldFrom7Point1((float *)in->input_tmp_buffer,
            (float *)in->input_tmp_buffer, samples >> 3, false);
        memcpy_by_audio_format(buffer, AUDIO_FORMAT_PCM_16_BIT,
            in->input_tmp_buffer, AUDIO_FORMAT_PCM_FLOAT, output_samples);
#else
        adjust_channels(in->input_tmp_buffer, in->config.channels,
            buffer, channel_count, 2, read_bytes);
#endif
    } else if (in->config.format == PCM_FORMAT_S32_LE) {
#ifdef HDMIIN_MULTICH_DOWNMIX
        int samples = read_bytes / 4;
        int output_samples = bytes / 4;
        memcpy_by_audio_format(in->input_tmp_buffer, AUDIO_FORMAT_PCM_FLOAT,
            in->input_tmp_buffer, AUDIO_FORMAT_PCM_32_BIT, samples);
        Downmix_foldFrom7Point1((float *)in->input_tmp_buffer,
            (float *)in->input_tmp_buffer, samples >> 3, false);
        memcpy_by_audio_format(buffer, AUDIO_FORMAT_PCM_32_BIT,
            in->input_tmp_buffer, AUDIO_FORMAT_PCM_FLOAT, output_samples);
#else
        adjust_channels(in->input_tmp_buffer, in->config.channels,
            buffer, channel_count, 4, read_bytes);
#endif
    }

   return ret;
}

void create_tvin_buffer(struct aml_audio_patch *patch)
{
    int ret = ring_buffer_init(&patch->tvin_ringbuffer, 4 * 48 * 64);
    AM_LOGI("ring_buffer_init size:%d, ret=%d", 4 * 48 * 64, ret);
    if (ret == 0) {
        patch->tvin_buffer_inited = 1;
    }

}

void release_tvin_buffer(struct aml_audio_patch *patch)
{
    if (patch->tvin_buffer_inited == 1) {
        patch->tvin_buffer_inited = 0;
        ring_buffer_release(&(patch->tvin_ringbuffer));
    }
}

uint32_t tv_in_write(struct audio_stream_out *stream, const void* buffer, size_t bytes)
{
    R_CHECK_POINTER_LEGAL(bytes, stream, "");
    R_CHECK_POINTER_LEGAL(bytes, buffer, "");
    struct aml_stream_out *out = (struct aml_stream_out *) stream;
    struct aml_audio_device *adev = out->dev;
    struct aml_audio_patch *patch = adev->audio_patch;
    R_CHECK_POINTER_LEGAL(bytes, patch, "");
    if (bytes == 0 || patch->tvin_buffer_inited != 1) {
        return bytes;
    }

    uint32_t bytes_written = 0;
    uint32_t wr_size = bytes;
    uint32_t full_count = 0;
    while (bytes_written < bytes) {
        uint32_t sent = ring_buffer_write(&patch->tvin_ringbuffer, (uint8_t *)buffer + bytes_written, bytes - bytes_written, UNCOVER_WRITE);
        AM_LOGV("need_write:%zu, actual sent:%d, bytes:%zu", (bytes - bytes_written), sent, bytes);
        bytes_written += sent;
        if (bytes_written == bytes) {
            if (adev->debug_flag) {
                AM_LOGD("write finished, bytes:%zu, timeout:%d ms", bytes, 5 * full_count);
            }
            return bytes;
        }
        full_count++;
        if (full_count >= 20) {
            AM_LOGW("write data timeout 100ms, need write:%zu, bytes_written:%d, reset buffer", bytes, bytes_written);
            ring_buffer_reset(&patch->tvin_ringbuffer);
            return bytes_written;
        }
        usleep(5000);
    }
    return bytes;
}

uint32_t tv_in_read(struct audio_stream_in *stream, void* buffer, size_t bytes)
{
    R_CHECK_POINTER_LEGAL(bytes, stream, "");
    R_CHECK_POINTER_LEGAL(bytes, buffer, "");
    struct aml_stream_in *in = (struct aml_stream_in *)stream;
    struct aml_audio_device *adev = in->dev;
    struct aml_audio_patch *patch = adev->audio_patch;
    R_CHECK_POINTER_LEGAL(bytes, patch, "");
    if (bytes == 0 || patch->tvin_buffer_inited != 1) {
        memset(buffer, 0, bytes);
        return bytes;
    }

    uint32_t read_bytes = 0;
    uint32_t nodata_count = 0;
    while (read_bytes < bytes) {
        uint32_t ret = ring_buffer_read(&patch->tvin_ringbuffer, (uint8_t *)buffer + read_bytes, bytes - read_bytes);
        AM_LOGV("need_write:%zu, actual sent:%d, bytes:%zu", (bytes - read_bytes), ret, bytes);
        read_bytes += ret;
        if (read_bytes == bytes) {
            if (adev->debug_flag) {
                int available = get_buffer_read_space(&patch->tvin_ringbuffer);
                AM_LOGD("read finished, bytes:%zu, timeout:%d ms, available:%d", bytes, 5 * nodata_count, available);
            }
            return bytes;
        }
        nodata_count++;
        if (nodata_count >= 20) {
            AM_LOGW("read data timeout 100ms, need:%zu, read_bytes:%d", bytes, read_bytes);
            return read_bytes;
        }
        usleep(5000);
    }
    return bytes;
}

int set_tv_source_switch_parameters(struct audio_hw_device *dev, struct str_parms *parms)
{
    struct aml_audio_device *adev = (struct aml_audio_device *)dev;
    int ret = -1;
    char value[64] = {'\0'};

    /*----ATV <-> DTV switch----*/
    ret = str_parms_get_str(parms, "hal_param_tuner_in", value, sizeof(value));
    // tuner_in=atv: tuner_in=dtv
    if (ret >= 0 && adev->is_TV) {
        if (strncmp(value, "dtv", 3) == 0) {
#ifdef ENABLE_DVB_PATCH
            // no audio patching in dtv
            if (adev->audio_patching && (adev->patch_src == SRC_ATV)) {
                // this is to handle atv->dtv case
                ALOGI("%s, atv->dtv", __func__);
                ret = release_patch(adev);
                if (!ret) {
                    adev->audio_patching = 0;
                }
            }
            ALOGI("%s, now the audio patch src is %s, the audio_patching is %d ", __func__,
                patchSrc2Str(adev->patch_src), adev->audio_patching);

            if ((adev->patch_src == SRC_DTV) && adev->audio_patching) {
                ALOGI("[audiohal_kpi] %s dtv patch exit do nothing\n ", __func__);
            } else {
                ALOGI("[audiohal_kpi] %s, now create the dtv patch now\n ", __func__);
                adev->patch_src = SRC_DTV;
                if (eDolbyMS12Lib == adev->dolby_lib_type) {
                    bool set_ms12_non_continuous = true;
                    get_dolby_ms12_cleanup(&adev->ms12, set_ms12_non_continuous);
                    adev->exiting_ms12 = 1;
                    clock_gettime(CLOCK_MONOTONIC, &adev->ms12_exiting_start);
                    if (adev->active_outputs[STREAM_PCM_NORMAL] != NULL)
                        usecase_change_validate_l(adev->active_outputs[STREAM_PCM_NORMAL], true);
                }


                ret = create_dtv_patch(dev, AUDIO_DEVICE_IN_TV_TUNER, AUDIO_DEVICE_OUT_SPEAKER);
                if (ret == 0) {
                    adev->audio_patching = 1;
                }
                ALOGI("[audiohal_kpi] %s, now end create dtv patch the audio_patching is %d ", __func__, adev->audio_patching);
            }
#endif
        } else if (strncmp(value, "atv", 3) == 0) {
#ifdef ENABLE_DVB_PATCH
            // need create patching
            if ((adev->patch_src == SRC_DTV) && adev->audio_patching) {
                ALOGI("[audiohal_kpi] %s, release dtv patching", __func__);
                ret = release_dtv_patch(adev);
                if (!ret) {
                    adev->audio_patching = 0;
                }
            }
#endif
            if (eDolbyMS12Lib == adev->dolby_lib_type && adev->continuous_audio_mode) {
                ALOGI("In ATV exit MS12 continuous mode");
                bool set_ms12_non_continuous = true;
                get_dolby_ms12_cleanup(&adev->ms12, set_ms12_non_continuous);
                adev->exiting_ms12 = 1;
                clock_gettime(CLOCK_MONOTONIC, &adev->ms12_exiting_start);
                if (adev->active_outputs[STREAM_PCM_NORMAL] != NULL)
                    usecase_change_validate_l(adev->active_outputs[STREAM_PCM_NORMAL], true);
            }

            if (!adev->audio_patching) {
                ALOGI("[audiohal_kpi] %s, create atv patching", __func__);
                set_audio_source(&adev->alsa_mixer,
                        ATV, alsa_device_is_auge());
                ret = create_patch (dev, AUDIO_DEVICE_IN_TV_TUNER, AUDIO_DEVICE_OUT_SPEAKER);
                // audio_patching ok, mark the patching status
                if (ret == 0) {
                    adev->audio_patching = 1;
                }
            }
            adev->patch_src = SRC_ATV;
        } else if (strncmp(value, "broadband", 9) == 0) {
#ifdef ENABLE_DVB_PATCH
            if ((adev->patch_src == SRC_DTV) && adev->audio_patching) {
                ALOGI("[audiohal_kpi] %s, release dtv patching", __func__);
                ret = release_dtv_patch(adev);
                if (!ret) {
                    adev->audio_patching = 0;
                }
            }
            adev->patch_src == SRC_INVAL;
            if (eDolbyMS12Lib == adev->dolby_lib_type) {
                get_dolby_ms12_cleanup(&adev->ms12, false);
                /*continuous mode is using in ms12 prepare, we should lock it*/
                pthread_mutex_lock(&adev->ms12.lock);
                adev->continuous_audio_mode = 1;
                pthread_mutex_unlock(&adev->ms12.lock);
                ALOGI("%s restore continuous_audio_mode=%d", __func__, adev->continuous_audio_mode);
            }
#endif
        }
        goto exit;
    }

    /*----HDMIIN <-> LINEIN switch----*/
    ret = str_parms_get_str(parms, "audio", value, sizeof(value));
    if (ret >= 0) {
        /*
         * This is a work around when plug in HDMI-DVI connector
         * first time application only recognize it as HDMI input device
         * then it can know it's DVI in, and then send "audio=linein" message to audio hal
         */
        struct audio_patch *pAudPatchTmp = NULL;
        if (strncmp(value, "linein", 6) == 0) {

            get_audio_patch_by_src_dev(dev, AUDIO_DEVICE_IN_HDMI, &pAudPatchTmp);
            if (pAudPatchTmp == NULL) {
                ALOGE("%s,There is no audio patch using HDMI as input", __func__);
                goto exit;
            }
            if (pAudPatchTmp->sources[0].ext.device.type != AUDIO_DEVICE_IN_HDMI) {
                ALOGE("%s, pAudPatchTmp->sources[0].ext.device.type != AUDIO_DEVICE_IN_HDMI", __func__);
                goto exit;
            }

            // dev->dev (example: HDMI in-> speaker out)
            if (pAudPatchTmp->sources[0].type == AUDIO_PORT_TYPE_DEVICE
                && pAudPatchTmp->sinks[0].type == AUDIO_PORT_TYPE_DEVICE) {
                // This "adev->audio_patch" will be created in create_patch() function
                if (adev->audio_patch && (adev->patch_src == SRC_HDMIIN)) {
                    ALOGI("%s, create hdmi-dvi patching dev->dev", __func__);
                    release_patch(adev);
                    aml_audio_input_routing(dev, INPORT_LINEIN);
                    create_patch(dev, AUDIO_DEVICE_IN_LINE, pAudPatchTmp->sinks[0].ext.device.type);
                }
            }

            adev->patch_src = SRC_LINEIN;
            adev->active_inport = INPORT_LINEIN;
            pAudPatchTmp->sources[0].ext.device.type = AUDIO_DEVICE_IN_LINE;
            set_audio_source(&adev->alsa_mixer, LINEIN, alsa_device_is_auge());
        } else if (strncmp(value, "hdmi", 4) == 0 && adev->audio_patch) {

            get_audio_patch_by_src_dev(dev, AUDIO_DEVICE_IN_LINE, &pAudPatchTmp);
            if (pAudPatchTmp == NULL) {
                ALOGE("%s,There is no audio patch using LINEIN as input", __func__);
                goto exit;
            }
            if (pAudPatchTmp->sources[0].ext.device.type != AUDIO_DEVICE_IN_LINE) {
                ALOGE("%s, pAudPatchTmp->sources[0].ext.device.type != AUDIO_DEVICE_IN_HDMI", __func__);
                goto exit;
            }

            // dev->dev (example: LINE in -> speaker out)
            if (pAudPatchTmp->sources[0].type == AUDIO_PORT_TYPE_DEVICE
                && pAudPatchTmp->sinks[0].type == AUDIO_PORT_TYPE_DEVICE) {
                // This "adev->audio_patch" will be created in create_patch() function
                if (adev->audio_patch && (adev->patch_src == SRC_LINEIN)) {
                    ALOGI("%s, create dvi-hdmi patching dev->dev", __func__);
                    release_patch(adev);
                    aml_audio_input_routing(dev, INPORT_HDMIIN);
                    create_patch(dev, AUDIO_DEVICE_IN_HDMI, pAudPatchTmp->sinks[0].ext.device.type);
                }
            }

            adev->patch_src = SRC_HDMIIN;
            adev->active_inport = INPORT_HDMIIN;
            pAudPatchTmp->sources[0].ext.device.type = AUDIO_DEVICE_IN_HDMI;
            set_audio_source(&adev->alsa_mixer, HDMIIN, alsa_device_is_auge());
        }
        goto exit;
    }

exit:
    return ret;
}

int set_hdmiin_audio_mode(struct aml_mixer_handle *mixer_handle, char *mode)
{
    if (mode == NULL || strlen(mode) > 5)
        return -EINVAL;

    return aml_mixer_ctrl_set_str(mixer_handle,
            AML_MIXER_ID_HDMIIN_AUDIO_MODE, mode);
}

enum hdmiin_audio_mode get_hdmiin_audio_mode(struct aml_mixer_handle *mixer_handle)
{
    return (enum hdmiin_audio_mode)aml_mixer_ctrl_get_int(mixer_handle,
            AML_MIXER_ID_HDMIIN_AUDIO_MODE);
}

int aml_audio_earctx_get_type(struct aml_audio_device *adev)
{
    int attend_type = 0;

    attend_type = aml_mixer_ctrl_get_int(&adev->alsa_mixer, AML_MIXER_ID_EARC_TX_ATTENDED_TYPE);
    return attend_type;
}

int aml_audio_earc_get_latency(struct aml_audio_device *adev)
{
    int latency = 0;

    latency = aml_mixer_ctrl_get_int(&adev->alsa_mixer, AML_MIXER_ID_EARC_TX_LATENCY);
    return latency;
}

void tv_do_ease_out(struct aml_audio_device *aml_dev)
{
    int duration_ms = 0;
    if (aml_dev && aml_dev->audio_ease) {
        float vol_now = aml_audio_ease_get_current_volume(aml_dev->audio_ease);
        if (vol_now == 0.0f) {
            ALOGI("%s(),vol_now %f skip fade out", __func__, vol_now);
        } else {
            ALOGI("%s(), vol_now %f do fade out", __func__, vol_now);
            if (aml_dev->is_TV) {
                if (eDolbyMS12Lib == aml_dev->dolby_lib_type) {
                    duration_ms = property_get_int32("vendor.media.audio.dtv.fadeout.us", MS12_AUDIO_FADEOUT_TV_DURATION_US) / 1000;
                } else {
                    duration_ms = property_get_int32("vendor.media.audio.dtv.fadeout.us", AUDIO_FADEOUT_TV_DURATION_US) / 1000;
                }
            } else {
                duration_ms = property_get_int32("vendor.media.audio.dtv.fadeout.us", AUDIO_FADEOUT_STB_DURATION_US) / 1000;
            }
            if (eDolbyMS12Lib == aml_dev->dolby_lib_type) {
                aml_dev->ms12.do_easing = true;
                ALOGI("%s()  %d ms doing easing out", __func__, duration_ms);
                set_ms12_main_audio_mute(&aml_dev->ms12, true, duration_ms);
                usleep(2 * duration_ms * 1000);
                aml_dev->ms12.do_easing = false;
            } else {
                start_ease_out(aml_dev->audio_ease, aml_dev->is_TV, duration_ms / 2);
                usleep(duration_ms * 1000);
            }
        }
    }
}

bool is_game_mode(struct aml_audio_device *aml_dev)
{
    if (aml_dev->patch_src != SRC_HDMIIN ||
        !aml_dev->audio_patch ||
        (is_audio_patch_valid(aml_dev) && aml_dev->audio_patch && (aml_dev->audio_patch->input_src != AUDIO_DEVICE_IN_HDMI ||
        aml_dev->audio_patch->IEC61937_format == true))) {
        return false;
    }

    return (is_audio_patch_valid(aml_dev) && aml_dev->audio_patch && aml_dev->audio_patch->pic_mode == PQ_GAME);
}

void aml_check_pic_mode(struct aml_audio_patch *patch)
{
    struct aml_audio_device *aml_dev = NULL;
    if (!patch || patch->input_src != AUDIO_DEVICE_IN_HDMI) {
        return;
    }
    aml_dev = (struct aml_audio_device *)patch->dev;

    if (aml_dev->pic_mode == PQ_GAME && patch->mode_reconfig_flag == true) {
        ALOGD("%s(), IEC61937 data, reconfig audio path", __func__);
        aml_dev->mode_reconfig_in = true;
        aml_dev->mode_reconfig_out = true;
        patch->mode_reconfig_flag = false;
        return;
    }

    /* in PCM data case, picture mode setting changed */
    if (patch->IEC61937_format == false && patch->pic_mode != aml_dev->pic_mode) {
        ALOGD("%s(), pic mode changes from %d to %d", __func__, patch->pic_mode, aml_dev->pic_mode);
        aml_dev->mode_reconfig_in = true;
        aml_dev->mode_reconfig_out = true;
        patch->pic_mode = aml_dev->pic_mode;
    }

}

