/*
 * Copyright (C) 2021 Amlogic Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "aml_audio_ms12_render"
//#define LOG_NDEBUG 0

#include <unistd.h>
#include <errno.h>
#include <pthread.h>
#include <cutils/log.h>

#include <inttypes.h>
#include "audio_hw.h"
#include "audio_hw_utils.h"
#ifdef ENABLE_DVB_PATCH
#include "audio_dtv_utils.h"
#endif
#include "dolby_lib_api.h"
#include "aml_volume_utils.h"
#include "audio_hw_ms12.h"
#include "aml_audio_timer.h"
#include "alsa_config_parameters.h"
#include <aml_android_utils.h>
#include "audio_hw_ms12_common.h"
#include "aml_audio_ms12_sync.h"
#include <cutils/properties.h>
#include <fcntl.h>

#define MS12_MAIN_WRITE_LOOP_THRESHOLD                  (2000)
#define AUDIO_IEC61937_FRAME_SIZE 4
#define MS12_TRUNK_SIZE                                 (1024)
#define SECOND_2_PTS (90000) // 1s = 90000 (pts)

#ifdef ENABLE_DVB_PATCH
extern unsigned long decoder_apts_lookup(unsigned int offset);
extern int32_t PtsServ_ioctl(int32_t PServerDevId, int32_t PServerCmd, uint64_t param);

/*now th latency api is just used for DTV doing avsync by using mediasync */
int aml_audio_get_cur_ms12_latency(struct audio_stream_out *stream) {

    struct aml_stream_out *aml_out = (struct aml_stream_out *) stream;
    struct aml_audio_device *adev = aml_out->dev;
    struct dolby_ms12_desc *ms12 = &(adev->ms12);
    struct aml_audio_patch *patch = adev->audio_patch;
    aml_demux_audiopara_t *demux_info = (aml_demux_audiopara_t *)patch->demux_info;
    int ms12_latencyms = 0;

    uint64_t inputnode_consumed = dolby_ms12_get_main_bytes_consumed(stream);
    uint64_t frames_generated = dolby_ms12_get_main_pcm_generated(stream);
    if (is_dolby_ms12_support_compression_format(aml_out->hal_internal_format)) {
        /*for ms12 dual_decoder_support input node latency can not be calculated, AC4 and MAT frame is not 32ms*/
        if ((demux_info && demux_info->dual_decoder_support) ||
            aml_out->hal_internal_format == AUDIO_FORMAT_AC4 ||
            aml_out->hal_internal_format == AUDIO_FORMAT_MAT) {
            ms12_latencyms = (frames_generated - ms12->master_pcm_frames) / 48;
       } else {
            ms12_latencyms = (ms12->ms12_main_input_size - inputnode_consumed) / aml_out->ddp_frame_size * 32 + (frames_generated - ms12->master_pcm_frames) / 48;
       }
    } else {
        ms12_latencyms = ((ms12->ms12_main_input_size - inputnode_consumed ) / (4 * aml_out->output_speed) + frames_generated /aml_out->output_speed - ms12->master_pcm_frames) / 48;
    }
    if (adev->debug_flag)
        ALOGI("aml_out->output_speed %f ms12_latencyms %d  ms12_main_input_size %" PRId64 " inputnode_consumed %" PRId64 " frames_generated %" PRId64 " master_pcm_frames %" PRId64 "",
        aml_out->output_speed, ms12_latencyms, ms12->ms12_main_input_size, inputnode_consumed,frames_generated, ms12->master_pcm_frames);
    return ms12_latencyms;

}
#endif


int aml_audio_ms12_process_wrapper(struct audio_stream_out *stream, const void *write_buf, size_t write_bytes)

{
    struct aml_stream_out *aml_out = (struct aml_stream_out *) stream;
    struct aml_audio_device *adev = aml_out->dev;
    int return_bytes = write_bytes;
    int ret = 0;
    int total_write = 0;
    void *buffer = (void *)write_buf;
    struct aml_audio_patch *patch = adev->audio_patch;
    int write_retry =0;
    size_t used_size = 0;
    void *output_buffer = NULL;
    size_t output_buffer_bytes = 0;
    int ms12_write_failed = 0;
    int consume_size = 0,remain_size = 0,ms12_threshold_size = 256;
    unsigned long long all_pcm_len1 = 0;
    unsigned long long all_pcm_len2 = 0;
    unsigned long long all_zero_len = 0;
    struct dolby_ms12_desc *ms12 = &(adev->ms12);
    audio_format_t output_format = get_output_format (stream);

    if (adev->debug_flag) {
        ALOGD("%s:%d hal_format:%#x, output_format:0x%x, sink_format:0x%x do_easing %d",
            __func__, __LINE__, aml_out->hal_format, output_format, adev->sink_format,ms12->do_easing);
    }

    if ((adev->patch_src == SRC_HDMIIN ||
            adev->patch_src == SRC_SPDIFIN ||
            adev->patch_src == SRC_LINEIN ||
            adev->patch_src == SRC_ATV ||
            adev->patch_src == SRC_DTV ||
            adev->patch_src == SRC_ARCIN) && patch) {

        if (patch->need_do_avsync) {
            if (!ms12->is_muted) {
                set_ms12_main_audio_mute(ms12, true, 0);
            }
        } else {
            if (adev->tv_mute) {
                if (!ms12->is_muted) {
                    set_ms12_main_audio_mute(ms12, true, 0);
                }
            } else {
                if (!ms12->do_easing) {
                    if (ms12->is_muted) {
                        ALOGI("ms12 render easing in using %d ms ",MS12_AUDIO_FADEIN_TV_DURATION_US / 1000);
                        set_ms12_main_audio_mute(ms12, false, MS12_AUDIO_FADEIN_TV_DURATION_US / 1000);
                    }
                }
            }
        }
    }

    remain_size = dolby_ms12_get_main_buffer_avail(NULL);
    dolby_ms12_get_pcm_output_size(&all_pcm_len1, &all_zero_len);

    if (is_bypass_dolbyms12(stream)) {
        if (adev->debug_flag) {
            ALOGI("%s passthrough dolbyms12, format %#x\n", __func__, aml_out->hal_format);
        }
        output_format = aml_out->hal_internal_format;
        if (audio_hal_data_processing (stream, write_buf, write_bytes, &output_buffer, &output_buffer_bytes, output_format) == 0)
            hw_write (stream, output_buffer, output_buffer_bytes, output_format);
    } else {
        /*not continuous mode, we use sink gain control the volume*/
        if (!continuous_mode(adev)) {
            /* non-TV device, here the dtv set the dolby ms12's volume*/
            dtv_set_ms12_volume_on_non_TV_device(aml_out);

            /*when it is non continuous mode, we bypass data here*/
            dolby_ms12_bypass_process(stream, buffer, write_bytes);

            /* This code is the True Passthrough(MS12) method. */
            //if (is_ms12_passthrough(stream)) {
            //    ALOGD("%s only passthrough, do not goto dolbyms12!\n", __func__);
            //    return return_bytes;
            //}
        }
        /*begin to write, clear the total write*/
        total_write = 0;
re_write:
        if (adev->debug_flag) {
            ALOGI("%s dolby_ms12_main_process before write_bytes %zu!\n", __func__, write_bytes);
        }
#ifdef ENABLE_DVB_PATCH
        bool dtv_stream_flag = patch && (adev->patch_src  == SRC_DTV) && aml_out->is_tv_src_stream;
        if (dtv_stream_flag && patch->output_thread_exit) {
            return return_bytes;
        }
#endif
        used_size = 0;
        ret = dolby_ms12_main_process(stream, (char*)write_buf + total_write, write_bytes, &used_size);
        if (ret == 0) {
            if (adev->debug_flag) {
                ALOGI("%s dolby_ms12_main_process return %d, return used_size %zu!\n", __FUNCTION__, ret, used_size);
            }
            if (used_size < write_bytes && write_retry < MS12_MAIN_WRITE_LOOP_THRESHOLD) {
                if (adev->debug_flag) {
                    ALOGI("%s dolby_ms12_main_process used  %zu,write total %zu,left %zu\n", __FUNCTION__, used_size, write_bytes, write_bytes - used_size);
                }
                total_write += used_size;
                write_bytes -= used_size;
                /*if ms12 doesn't consume any data, we need sleep*/
                if (used_size == 0) {
                    aml_audio_sleep(1000);
                }
                if (adev->debug_flag >= 2) {
                    ALOGI("%s sleep 1ms\n", __FUNCTION__);
                }
                write_retry++;
                if (adev->ms12.dolby_ms12_enable) {
                    goto re_write;
                }
            }
            if (write_retry >= MS12_MAIN_WRITE_LOOP_THRESHOLD) {
                ALOGE("%s main write retry time output,left %zu", __func__, write_bytes);
                //bytes -= write_bytes;
                ms12_write_failed = 1;
            }
        } else {
            ALOGE("%s dolby_ms12_main_process failed %d", __func__, ret);
        }
    }

    int size = dolby_ms12_get_main_buffer_avail(NULL);
    dolby_ms12_get_pcm_output_size(&all_pcm_len2, &all_zero_len);

    return return_bytes;

}

static void aml_audio_ms12_init_pts_param(struct dolby_ms12_desc *ms12, uint64_t first_pts)
{
    if (!ms12)
        return;
    ms12->first_in_frame_pts = first_pts;
    ms12->last_synced_frame_pts = -1;
    ms12->out_synced_frame_count = 0;
    ALOGI("first_in_frame_pts  %" PRIu64 " ms" , ms12->first_in_frame_pts / 90);
}

static int aml_audio_ms12_process(struct audio_stream_out *stream, const void *write_buf, size_t write_bytes) {
    struct aml_stream_out *aml_out = (struct aml_stream_out *) stream;
    struct aml_audio_device *adev = aml_out->dev;
    int return_bytes = write_bytes;
    struct aml_audio_patch *patch = adev->audio_patch;
    bool need_separate_frame  = false;
    int ret = 0;

    /*
     * case 1 for local ddp 44.1khz, now the input size is too big,
     * the passthrough output will be blocked by the decoded pcm output,
     * so we need feed it with small trunk to fix this issue
     *
     *
     * case 2 SWPL-66107
     * for dtv passthrough case, sometimes it has big input size,
     * we need separate it to small trunk
     *
     * todo, we need add a parser for such case
     */
    if (!adev->continuous_audio_mode && !patch) {
        need_separate_frame = true;
    } else if (patch && (adev->patch_src == SRC_DTV) && (AML_DIGITAL_AUDIO_MODE_BYPASS == adev->digital_audio_mode)) {
        need_separate_frame = true;
    }
    if (need_separate_frame) {
        size_t left_bytes = write_bytes;
        size_t used_bytes = 0;
        int process_size = 0;
        /*
         * Reason:
         * After enable the amlogic_truehd encoded by the dolby mat encoder, passthrough the Dolby MS12 pipeline.
         * Found the process_bytes(MS12_TRUNK_SIZE 1024Bytes) can lead the alsa underrun.
         *
         * Solution:
         * After send all the truehd to ms12, sound is smooth. If dolby truehd occur underrun in passthrough mode,
         * please take care of the value of process_size(aml_audio_ms12_render: bytes).
         *
         * Issue:
         * SWPL-60957: passthrough TrueHD format in Movie player.
         */
        int process_bytes = (aml_out->hal_format == AUDIO_FORMAT_DOLBY_TRUEHD) ? (write_bytes) : MS12_TRUNK_SIZE;
        while (1) {
            process_size = left_bytes > process_bytes ? process_bytes : left_bytes;
            ret = aml_audio_ms12_process_wrapper(stream, (char *)write_buf + used_bytes, process_size);
            if (ret <= 0) {
                break;
            }
            used_bytes += process_size;
            left_bytes -= process_size;
            if (left_bytes <= 0) {
                break;
            }
        }
    } else {
        aml_audio_ms12_process_wrapper(stream, write_buf, write_bytes);
    }
    return return_bytes;
}

int aml_audio_ms12_render(struct audio_stream_out *stream, const void *buffer, size_t bytes)
{
    int ret = -1;
    int dec_used_size = 0;
    int used_size = 0;
    int left_bytes = 0;
    struct aml_stream_out *aml_out = (struct aml_stream_out *) stream;
    struct aml_audio_device *adev = aml_out->dev;
    int return_bytes = bytes;
    struct aml_audio_patch *patch = adev->audio_patch;
    void *output_buffer = NULL;
    size_t output_buffer_bytes = 0;
    int out_frames = 0;
    int ms12_delayms = 0;
    int alsa_latency = 0;
    int force_setting_delayms = 0;
    bool bypass_aml_dec = false;
    struct dolby_ms12_desc *ms12 = &(adev->ms12);
    checkout_pts_offset checkout_pts;
#ifdef ENABLE_DVB_PATCH
    bool dtv_stream_flag = patch && (adev->patch_src == SRC_DTV) && aml_out->is_tv_src_stream;
    bool do_sync_flag = dtv_stream_flag && patch && patch->skip_amadec_flag && patch->dtvsync->sync_type == DTVSYNC_MEDIASYNC;
#endif


    /*
     * define the bypass_aml_dec by audio format
     * 1. AC3/E-AC3/E-AC3_JOC/AC4/TrueHD/MAT
     * 2. multi-pcm(5.1 or 7.1 but not stereo)
     * 1+2, can go through ms12 processing
     */
    if (is_dolby_ms12_support_compression_format(aml_out->hal_internal_format)
        || is_multi_channel_pcm(stream)) {
        bypass_aml_dec = true;
    }

    if (bypass_aml_dec) {
#ifdef ENABLE_DVB_PATCH
        /*
         * DTV instance, should get the APTS and send the APTS + Audio Data together.
         */
        if (do_sync_flag) {
            if (patch->skip_amadec_flag) {
                if (patch->cur_package) {
                    if (patch->cur_package->pts == 0) {
                        patch->cur_package->pts = decoder_apts_lookup((unsigned int)patch->decoder_offset);
                    }
                }
            }
        }

        /* audio data/apts, we send the APTS at first*/
        if (ms12 && patch && patch->cur_package) {
            uint64_t  decoder_base = ms12->dtv_decoder_offset_base;
            uint64_t  decoder_offset = patch->decoder_offset;
            if (decoder_base != 0 && decoder_offset >= decoder_base) {
                decoder_offset -= decoder_base;
            }
            if (adev->debug_flag) {
                ALOGI("%s dolby pts %" PRIu64 " decoder_base =%" PRIu64 " decoder_offset =%" PRIu64 "", __func__, patch->cur_package->pts, decoder_base, decoder_offset);
            }
            if (patch->cur_package->pts != ULLONG_MAX && patch->cur_package->pts != DTVSYNC_INVALID_PTS) {
                set_ms12_main_audio_pts(ms12, patch->cur_package->pts, decoder_offset);
            } else {
                set_ms12_main_audio_pts(ms12,  patch->dtvsync->out_end_apts, decoder_offset);
            }
            /* to init the pts information */
            if (patch->decoder_offset == 0) {
                aml_audio_ms12_init_pts_param(ms12, patch->cur_package->pts);
            }
        }
#endif
        /* audio data/apts, then we send the audio data*/
        aml_audio_ms12_process(stream, buffer, bytes);
#ifdef ENABLE_DVB_PATCH
        if (patch && patch->decoder_offset == 0) {
            aml_demux_audiopara_t *demux_info = (aml_demux_audiopara_t *)patch->demux_info;
           /*after ms12 process one frame, set mix level and ad vol level */
            if (demux_info && demux_info->dual_decoder_support) {
                pthread_mutex_lock(&ms12->lock);
                dolby_ms12_set_user_control_value_for_mixing_main_and_associated_audio(demux_info->mixing_level);
                set_ms12_ad_mixing_level(ms12, demux_info->mixing_level);
                set_ms12_ad_vol(ms12, demux_info->advol_level);
                /*coverity[double_unlock]*/
                pthread_mutex_unlock(&ms12->lock);
            }
        }
#endif

    } else {
        if (aml_out->aml_dec == NULL) {
            config_output(stream, true);
        }
        aml_dec_t *aml_dec = aml_out->aml_dec;

#ifdef ENABLE_DVB_PATCH
        if (do_sync_flag && aml_dec) {
            if(patch->skip_amadec_flag) {
                if (patch->cur_package) {
                    if (!adev->is_multi_demux && patch->singleDmxNonTunnelMode) {
                        checkout_pts.offset = patch->decoder_offset;
                        ALOGV("offset:%" PRId64 "\n", checkout_pts.offset);
                        if (patch->PServerDev != -1) {
                            PtsServ_ioctl(patch->PServerDev, PTSSERVER_IOC_CHECKOUT_APTS, (unsigned long)&checkout_pts);
                        }
                        // aml_dec->in_frame_pts = decoder_apts_lookup((unsigned int)patch->decoder_offset);
                        aml_dec->in_frame_pts = checkout_pts.pts_90k;
                        if (!aml_dec->in_frame_pts)
                           aml_dec->in_frame_pts = aml_dec->out_frame_pts;
                        ALOGV("in_frame_pts:%" PRId64 " PtsServ_checkout_pts64:%" PRId64 "\n",aml_dec->in_frame_pts, checkout_pts.pts_64);
                    } else {
                        if (patch->cur_package->pts != DTVSYNC_INVALID_PTS) {
                            if (patch->cur_package->pts != 0) {
                                aml_dec->in_frame_pts = patch->cur_package->pts;
                                aml_dec->out_frame_pts = 0;
                            }
                        } else {
                            aml_dec->in_frame_pts = aml_dec->out_frame_pts;
                          }
                        }
                    } else
                        ALOGW("cur_package null !!!");
            } else {
                aml_dec->in_frame_pts = decoder_apts_lookup((unsigned int)patch->decoder_offset);
            }
        }
#endif
        if (aml_dec) {
            dec_data_info_t * dec_pcm_data = &aml_dec->dec_pcm_data;
            dec_data_info_t * dec_raw_data = &aml_dec->dec_raw_data;
            dec_data_info_t * raw_in_data  = &aml_dec->raw_in_data;
            left_bytes = bytes;
            do {
                if (adev->debug_flag)
                    ALOGI("left_bytes %d dec_used_size %d", left_bytes, dec_used_size);
                ret = aml_decoder_process(aml_dec, (unsigned char *)buffer + dec_used_size, left_bytes, &used_size);

                if (ret < 0) {
                    ALOGV("aml_decoder_process error");
                    return return_bytes;
                }
                left_bytes -= used_size;
                dec_used_size += used_size;
                ALOGV("%s() ret =%d pcm len =%d raw len=%d", __func__, ret, dec_pcm_data->data_len, dec_raw_data->data_len);
                // write pcm data
                if (dec_pcm_data->data_len > 0) {
                    void  *dec_data = (void *)dec_pcm_data->buf;
#ifdef ENABLE_DVB_PATCH
                    if (dtv_stream_flag && adev->start_mute_flag == 1) {
                        memset(dec_pcm_data->buf, 0, dec_pcm_data->data_len);
                    }
#endif
                    if (dec_pcm_data->data_sr > 0) {
                        aml_out->config.rate = dec_pcm_data->data_sr;
                    }
                    if (patch) {
                        patch->sample_rate = dec_pcm_data->data_sr;
                    }
                    if (dec_pcm_data->data_sr != OUTPUT_ALSA_SAMPLERATE ) {
                         ret = aml_audio_resample_process_wrapper(&aml_out->resample_handle, dec_pcm_data->buf, dec_pcm_data->data_len, dec_pcm_data->data_sr, dec_pcm_data->data_ch);
                         if (ret != 0) {
                             ALOGI("aml_audio_resample_process_wrapper failed");
                         } else {
                             dec_data = aml_out->resample_handle->resample_buffer;
                             dec_pcm_data->data_len = aml_out->resample_handle->resample_size;
                         }
                    }
#ifdef ENABLE_DVB_PATCH
                    if (dtv_stream_flag)
                        patch->dtv_pcm_wrote += dec_pcm_data->data_len;
                    aml_dec->out_frame_pts = aml_dec->in_frame_pts + (90 * out_frames /(dec_pcm_data->data_sr / 1000));
                    out_frames += dec_pcm_data->data_len /( 2 * dec_pcm_data->data_ch);
                    if (get_debug_value(AML_DEBUG_AUDIOHAL_AUT)) {
                        ALOGI("pes_pts: %" PRIx64 ", frame_pts: %" PRIx64 ", pcm[len:%d, pcm_dur:%dms, total_dur:%dms].",\
                            aml_dec->in_frame_pts, aml_dec->out_frame_pts, dec_pcm_data->data_len,\
                            dec_pcm_data->data_len * 1000 /( 2 * dec_pcm_data->data_ch * dec_pcm_data->data_sr),\
                            out_frames /(dec_pcm_data->data_sr / 1000));
                    }

                    //aml_audio_dump_audio_bitstreams("/data/mixing_data.raw", dec_data, dec_pcm_data->data_len);
                    /* audio data/apts, we send the APTS at first*/
                    if (ms12 && aml_dec) {
                        /*Fixme, how to get the right apts(long long unsigned int) and bytes_offset*/
                        //ALOGV("%s non-dolby pts %llu decoder_offset %llu", __func__, ms12->ms12_main_input_size/4/48, ms12->ms12_main_input_size);
                        //set_ms12_main_audio_pts(ms12, ms12->ms12_main_input_size * 90000 / 192 /* bytes_per_sample(4) plus sr(48 kHz)*/, ms12->ms12_main_input_size);
                    }
#endif
                    /* audio data/apts, then we send the audio data*/
                    aml_audio_ms12_process_wrapper(stream, dec_data, dec_pcm_data->data_len);
#ifdef ENABLE_DVB_PATCH
                    if (do_sync_flag) {
                        if (patch->output_thread_exit) {
                            break;
                        }
                        ms12_delayms = aml_audio_get_cur_ms12_latency(stream);
                        alsa_latency = 90 *(out_get_alsa_latency_frames(stream)  * 1000) / aml_out->config.rate;
                        int tune_latency = aml_audio_dtv_get_ms12_latency(stream) * SECOND_2_PTS / dec_pcm_data->data_sr;

                        if (adev->bHDMIARCon) {
                            force_setting_delayms = aml_getprop_int(PROPERTY_LOCAL_PASSTHROUGH_LATENCY);
                        }

                        if (patch->dtvsync->sync_type == DTVSYNC_MEDIASYNC) {
                            if (aml_out->alsa_status_changed) {
                                ALOGI("aml_out->alsa_running_status %d", aml_out->alsa_running_status);
                                aml_dtvsync_setParameter(patch->dtvsync, MEDIASYNC_KEY_ALSAREADY, &aml_out->alsa_running_status);
                                aml_out->alsa_status_changed = false;
                            }
                            patch->dtvsync->cur_outapts = aml_dec->out_frame_pts - ms12_delayms * 90 - alsa_latency + force_setting_delayms * 90 + tune_latency;
                            if (get_debug_value(AML_DEBUG_AUDIOHAL_AUT)) {
                                ALOGI("frame_pts:%" PRIx64 ", output_pts:%" PRIx64 ", latency:%" PRId64 " ms, tune_latency:%d ms.",\
                                    aml_dec->out_frame_pts, patch->dtvsync->cur_outapts,\
                                    (aml_dec->out_frame_pts - patch->dtvsync->cur_outapts) / 90, tune_latency / 90);
                            }

                            if (adev->debug_flag)
                                ALOGI("patch->dtvsync->cur_outapts %" PRId64 ", ms12_delayms:%d ms, alsa_latency:%d ms, tune_latency:%d ms",\
                                       patch->dtvsync->cur_outapts, ms12_delayms, alsa_latency/90, tune_latency / 90);
                            if (aml_out->dtvsync_enable)
                                aml_dtvsync_ms12_get_policy(stream);
                        }
                    }
#endif
                }

            } while ((left_bytes > 0) || aml_dec->fragment_left_size);
        }

    }

#ifdef ENABLE_DVB_PATCH
    /*
     *if main&associate dolby input, decoder_offset should only add main data size.
     *if main dolby input, decoder_offset should add main data size.
     */
    if (patch && patch->cur_package && patch->skip_amadec_flag && patch->demux_info) {
        aml_demux_audiopara_t *demux_info = (aml_demux_audiopara_t *)patch->demux_info;
        if (!demux_info->dual_decoder_support) {
             /*we have separated the input data, then we need use input size to calculate pts*/
             if (is_aac_format(patch->aformat)) {
                patch->decoder_offset += bytes;
             } else {
                patch->decoder_offset += patch->cur_package->size;
             }
        } else {
             if (patch->aformat == AUDIO_FORMAT_HE_AAC_V1 ||
                 patch->aformat == AUDIO_FORMAT_AAC_LATM ||
                 patch->aformat == AUDIO_FORMAT_AAC ||
                 patch->aformat == AUDIO_FORMAT_MP3 ||
                 patch->aformat == AUDIO_FORMAT_MP2) {
                 patch->decoder_offset += patch->cur_package->size;
             } else {
                 patch->decoder_offset += patch->cur_package->split_frame_size;
             }
        }
    }
#endif
    return return_bytes;
}




