/*
 * Copyright (C) 2017 Amlogic Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG  "a2dp_hal"
//#define LOG_NDEBUG 0
#include <system/audio.h>
#include <cinttypes>
#include <cutils/log.h>
#include <cutils/properties.h>
#include <android-base/strings.h>
#include <audio_utils/primitives.h>

#include "BluetoothAudioSession.h"

#include "a2dp_hal.h"
#include "a2dp_hw.h"
#include "aml_audio_resampler.h"


extern "C" {
#include "audio_hw_utils.h"
#include "aml_audio_stream.h"
#include "aml_audio_timer.h"
}

using ::android::bluetooth::audio::BluetoothAudioPortOut;
using ::android::bluetooth::audio::BluetoothAudioSession;
using ::android::bluetooth::audio::BluetoothAudioSessionInstance;
using ::android::hardware::bluetooth::audio::V2_0::SessionType;

#define A2DP_RING_BUFFER_DELAY_TIME_MS              (64)
#define A2DP_SEND_DATA_TIMEOUT_RESET_MS             (300)
#define A2DP_WAIT_STATE_DELAY_TIME_US               (5000)
#define A2DP_WRITE_DATE_TIME_OUT_MS                 (64)
#define A2DP_LATENCY_INVALID_NS                     (NSEC_PER_SEC)
#define DEFAULT_A2DP_LATENCY_NS                     (100 * NSEC_PER_MSEC) // Default delay to use when BT device does not report a delay
#define A2DP_STATIC_DELAY_MS                        (0) // Additional device-specific delay
#define AUDIO_HAL_FIXED_CFG_CHANNEL                 (AUDIO_CHANNEL_OUT_STEREO)
#define AUDIO_HAL_FIXED_CFG_FORMAT                  (AUDIO_FORMAT_PCM_16_BIT)
#define AUDIO_HAL_FIXED_FRAME_SIZE                  (audio_channel_count_from_out_mask(AUDIO_HAL_FIXED_CFG_CHANNEL) * \
                                                        audio_bytes_per_sample(AUDIO_HAL_FIXED_CFG_FORMAT))
#define A2DP_TEST_AUDIO_FILE_PATH                  "/data/a2dp_test.wav"
#define A2DP_TEST_AUDIO_FILE_PROP                  "vendor.media.audiohal.a2dp.test"
#define A2DP_TEST_AUDIO_CHECK_MUTE_PROP            "vendor.media.audiohal.a2dp.checkmute"


struct aml_a2dp_hal {
    BluetoothAudioPortOut a2dphw;
    audio_config config;
    aml_audio_resample_t *resample;
    int64_t last_write_time;
    uint64_t mute_time;
    char * buff_conv_format;
    size_t buff_size_conv_format;
    BluetoothStreamState state;
    uint64_t a2dp_latency;
    bool is_sending_data;
    bool exit_out_monitor_thread;
    pthread_t out_monitor_thread_id;
    pthread_mutex_t out_monitor_thread_mutex;
    pthread_cond_t out_monitor_thread_cond;
};
static int a2dp_out_standby(struct aml_audio_device *adev);
static void a2dp_notify_monitor(aml_a2dp_hal *hal, bool is_sending);

std::unordered_map<std::string, std::string> ParseAudioParams(const std::string& params) {
    std::vector<std::string> segments = android::base::Split(params, ";");
    std::unordered_map<std::string, std::string> params_map;
    for (const auto& segment : segments) {
        if (segment.length() == 0) {
            continue;
        }
        std::vector<std::string> kv = android::base::Split(segment, "=");
        if (kv[0].empty()) {
            //AM_LOGD("Invalid audio parameter: ", segment.char());
            continue;
        }
        params_map[kv[0]] = (kv.size() > 1 ? kv[1] : "");
    }
    return params_map;
}

static bool a2dp_wait_status(const char *caller, struct aml_a2dp_hal *hal) {
    hal->state = hal->a2dphw.GetState();
    int retry = 0;
    // max timeout 1s.
    while (retry < 200) {
        if ((hal->state != BluetoothStreamState::STARTING) && (hal->state != BluetoothStreamState::SUSPENDING)) {
            if (retry > 0) {
                AM_LOGI("(%s) wait for state change to successd. waited: %d ms cur state:%s", caller,
                    retry * A2DP_WAIT_STATE_DELAY_TIME_US / 1000, a2dpStatus2String(hal->state));
            }
            return true;
        }
        usleep(A2DP_WAIT_STATE_DELAY_TIME_US);
        a2dp_notify_monitor(hal, false);
        retry++;
        // Warning log once every 200ms after timeout.
        if (retry % 40 == 0) {
            AM_LOGW("(%s) timeout: %d ms, cur state:%s, contine waiting >>>>>>", caller,
                retry * A2DP_WAIT_STATE_DELAY_TIME_US / 1000, a2dpStatus2String(hal->state));
        }
        hal->state = hal->a2dphw.GetState();
    }
    AM_LOGE("(%s) waiting for state change failed, timeout: %d ms, cur state:%s", caller,
        retry * A2DP_WAIT_STATE_DELAY_TIME_US / 1000, a2dpStatus2String(hal->state));
    return false;
}

static void dump_a2dp_output_data(aml_a2dp_hal *hal, const void *buffer, size_t size) {
    if (getprop_bool("vendor.media.audiohal.a2dpdump")) {
        char acFilePathStr[ENUM_TYPE_STR_MAX_LEN];
        size_t out_per_sample_byte = audio_bytes_per_sample(hal->config.format);
        size_t out_channel_byte = audio_channel_count_from_out_mask(hal->config.channel_mask);
        sprintf(acFilePathStr, "/data/audio/a2dp_%0.1fK_%zuC_%zuB.pcm", hal->config.sample_rate/1000.0, out_channel_byte, out_per_sample_byte);
        aml_audio_dump_audio_bitstreams(acFilePathStr, buffer, size);
    }
}

static void a2dp_notify_monitor(aml_a2dp_hal *hal, bool is_sending = false) {
    pthread_mutex_lock(&hal->out_monitor_thread_mutex);
    hal->is_sending_data = is_sending;
    pthread_cond_signal(&hal->out_monitor_thread_cond);
    pthread_mutex_unlock(&hal->out_monitor_thread_mutex);
}

static void *a2dp_out_monitor_thread(void *arg) {
    struct aml_audio_device *adev = (struct aml_audio_device *)arg;
    aml_a2dp_hal *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    struct timespec next_time;
    uint32_t timeout_ms = 0;
    bool is_standby = true;
    int ret = 0;
    AM_LOGI("Start monitoring the write rate+++");
    uint64_t time_ns;
    while (hal->exit_out_monitor_thread == false) {
        pthread_mutex_lock(&hal->out_monitor_thread_mutex);
        if (is_standby) {
            ret = pthread_cond_wait(&hal->out_monitor_thread_cond, &hal->out_monitor_thread_mutex);
        } else {
            timeout_ms = A2DP_RING_BUFFER_DELAY_TIME_MS; // 64 ms
            if (hal->is_sending_data) {
                timeout_ms = A2DP_SEND_DATA_TIMEOUT_RESET_MS; // 300ms
            }
            /* Here is an empirical value 64ms, when each write interval is greater than this value, we think standby BT,
             * needed to reduce power consumption.
             */
            time_ns = aml_audio_get_systime_ns();
            next_time = aml_audio_ns_to_time(time_ns + timeout_ms * NSEC_PER_MSEC);
            ret = pthread_cond_timedwait(&hal->out_monitor_thread_cond, &hal->out_monitor_thread_mutex, &next_time);
        }

        if (timeout_ms == A2DP_SEND_DATA_TIMEOUT_RESET_MS) {
            AM_LOGV("send bt elapsed time: %" PRIu64 " ms", (aml_audio_get_systime_ns() - time_ns) / NSEC_PER_MSEC);
        }
        pthread_mutex_unlock(&hal->out_monitor_thread_mutex);
        if (ret == ETIMEDOUT && hal->exit_out_monitor_thread == false) {
            if (timeout_ms == A2DP_SEND_DATA_TIMEOUT_RESET_MS) {
                AM_LOGW("send BT stack timeout %dms, need standby, cur_state:%s", timeout_ms, a2dpStatus2String(hal->state));
            } else {
                AM_LOGI("audio write timeout %dms, need standby, cur_state:%s", timeout_ms, a2dpStatus2String(hal->state));
            }
            a2dp_out_standby(adev);
            is_standby = true;
        } else {
            is_standby = false;
        }
    }
    AM_LOGI("Exit the monitor---");
    return NULL;
}

int a2dp_out_open(struct aml_audio_device *adev) {
    struct aml_a2dp_hal *hal = NULL;
    struct timespec ts = {.tv_sec = 0, .tv_nsec = 0};
    pthread_mutex_lock(&adev->a2dp_lock);

    if (adev->a2dp_hal != NULL) {
        AM_LOGW("already open");
        pthread_mutex_unlock(&adev->a2dp_lock);
        return 0;
    }
    hal = new aml_a2dp_hal;
    if (hal == NULL) {
        AM_LOGE("new BluetoothAudioPortOut fail");
        pthread_mutex_unlock(&adev->a2dp_lock);
        return -1;
    }
    hal->resample = NULL;
    hal->buff_conv_format = NULL;
    hal->buff_size_conv_format = 0;
    hal->state = BluetoothStreamState::UNKNOWN;
    hal->a2dp_latency = A2DP_LATENCY_INVALID_NS;
    if (!hal->a2dphw.SetUp(AUDIO_DEVICE_OUT_BLUETOOTH_A2DP)) {
        AM_LOGE("BluetoothAudioPortOut setup fail");
        pthread_mutex_unlock(&adev->a2dp_lock);
        delete hal;
        return -1;
    }
    if (!hal->a2dphw.LoadAudioConfig(&hal->config)) {
        AM_LOGE("LoadAudioConfig fail");
    }
    if (hal->config.channel_mask == AUDIO_CHANNEL_OUT_MONO)
        hal->a2dphw.ForcePcmStereoToMono(true);
    clock_gettime(CLOCK_MONOTONIC, &ts);
    hal->mute_time = ts.tv_sec * 1000000LL + ts.tv_nsec / 1000;
    hal->mute_time += 1000000LL; // mute for 1s
    adev->a2dp_hal = (void*)hal;
    pthread_mutex_unlock(&adev->a2dp_lock);

    pthread_condattr_t condattr;
    pthread_mutex_init(&hal->out_monitor_thread_mutex, NULL);
    pthread_condattr_init(&condattr);
    pthread_condattr_setclock(&condattr, CLOCK_MONOTONIC);
    pthread_cond_init(&hal->out_monitor_thread_cond, &condattr);
    pthread_condattr_destroy(&condattr);
    hal->exit_out_monitor_thread = false;
    hal->is_sending_data = false;
    int pthread_ret = pthread_create(&hal->out_monitor_thread_id, NULL, &a2dp_out_monitor_thread, adev);
    if (pthread_ret != 0) {
        AM_LOGE("pthread_create fail");
        return -1;
    }
    AM_LOGI("Rx param rate:%d, bytes_per_sample:%zu, ch:%d", hal->config.sample_rate,
        audio_bytes_per_sample(hal->config.format), audio_channel_count_from_out_mask(hal->config.channel_mask));
    return 0;
}

int a2dp_out_close(struct aml_audio_device *adev) {
    pthread_mutex_lock(&adev->a2dp_lock);
    struct aml_a2dp_hal *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    if (hal == NULL) {
        AM_LOGW("a2dp hw is already closed.");
        pthread_mutex_unlock(&adev->a2dp_lock);
        return -1;
    }

    /*coverity[sleep]*/
    a2dp_wait_status(__func__, hal);
    hal->exit_out_monitor_thread = true;
    a2dp_notify_monitor(hal);
    pthread_join(hal->out_monitor_thread_id, NULL);
    pthread_cond_destroy(&hal->out_monitor_thread_cond);
    pthread_mutex_destroy(&hal->out_monitor_thread_mutex);
    adev->a2dp_hal = NULL;
    AM_LOGI("");
    hal->a2dphw.Stop();
    hal->a2dphw.TearDown();
    if (hal->resample) {
        aml_audio_resample_close(hal->resample);
        hal->resample = NULL;
    }
    if (hal->buff_conv_format)
        aml_audio_free(hal->buff_conv_format);
    pthread_mutex_unlock(&adev->a2dp_lock);
    delete hal;
    return 0;
}

static int a2dp_out_resume_l(aml_audio_device *adev) {
    struct aml_a2dp_hal *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    AM_LOGI("start resume... cur status:%s", a2dpStatus2String(hal->state));
    a2dp_wait_status(__func__, hal);
    if (hal->state == BluetoothStreamState::STARTED) {
        AM_LOGI("A2dp already resumed. status:%s", a2dpStatus2String(hal->state));
        return 0;
    } else if (hal->state == BluetoothStreamState::STANDBY) {
        if (hal->a2dphw.Start()) {
            BluetoothStreamState cur_status = hal->a2dphw.GetState();
            AM_LOGI("status: %s -> %s Resume in progress...", a2dpStatus2String(hal->state), a2dpStatus2String(cur_status));
            hal->state = cur_status;
            return 0;
        } else {
            AM_LOGW("Start fail. state:%s", a2dpStatus2String(hal->a2dphw.GetState()));
            return -1;
        }
    }
    AM_LOGW("cur state:%s error, can't resume", a2dpStatus2String(hal->state));
    return -1;
}

static int a2dp_out_resume(struct aml_audio_device *adev) {
    pthread_mutex_lock(&adev->a2dp_lock);
    struct aml_a2dp_hal *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    if (hal == NULL) {
        AM_LOGW("a2dp has been released.");
        pthread_mutex_unlock(&adev->a2dp_lock);
        return -1;
    }
    /*coverity[sleep]*/
    int32_t ret = a2dp_out_resume_l(adev);
    pthread_mutex_unlock(&adev->a2dp_lock);
    return ret;
}

static int a2dp_out_standby_l(struct aml_audio_device *adev) {
    struct aml_a2dp_hal *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    AM_LOGI("start standby... cur status:%s", a2dpStatus2String(hal->state));
    a2dp_wait_status(__func__, hal);
    if (hal->state == BluetoothStreamState::STANDBY) {
        AM_LOGI("A2dp already standby. status:%s", a2dpStatus2String(hal->state));
        return 0;
    } else if (hal->state == BluetoothStreamState::STARTED) {
        if (hal->a2dphw.Suspend()) {
            BluetoothStreamState cur_status = hal->a2dphw.GetState();
            AM_LOGI("status: %s -> %s Standby in progress...", a2dpStatus2String(hal->state), a2dpStatus2String(cur_status));
            hal->state = cur_status;
            return 0;
        } else {
            AM_LOGW("Suspend fail. state:%s", a2dpStatus2String(hal->a2dphw.GetState()));
            return -1;
        }
    }
    AM_LOGW("cur state:%s error, can't standby", a2dpStatus2String(hal->state));
    return -1;
}

static int a2dp_out_standby(struct aml_audio_device *adev) {
    pthread_mutex_lock(&adev->a2dp_lock);
    struct aml_a2dp_hal *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    if (hal == NULL) {
        AM_LOGW("a2dp has been released.");
        pthread_mutex_unlock(&adev->a2dp_lock);
        return -1;
    }
    /*coverity[sleep]*/
    int32_t ret = a2dp_out_standby_l(adev);
    pthread_mutex_unlock(&adev->a2dp_lock);
    return ret;
}

static bool a2dp_state_process(struct aml_audio_device *adev, audio_config_base_t *config, size_t cur_frames) {
    aml_a2dp_hal            *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    BluetoothStreamState    cur_state = hal->a2dphw.GetState();
    const int64_t           cur_write_time_us = aml_audio_get_systime();
    bool                    prepared = false;

    const int64_t write_delta_time_us = cur_write_time_us - hal->last_write_time;
    int64_t data_delta_time_us = (int64_t)(cur_frames * USEC_PER_SEC / config->sample_rate) - write_delta_time_us;
    hal->last_write_time = cur_write_time_us;
    if (hal->state != cur_state) {
        AM_LOGI("a2dp state changed: %s -> %s",  a2dpStatus2String(hal->state), a2dpStatus2String(cur_state));
        hal->state = cur_state;
    }
    if (adev->debug_flag) {
        AM_LOGD("cur_state:%s, frames:%zu, gap:%" PRId64 " ms", a2dpStatus2String(cur_state), cur_frames, write_delta_time_us / 1000);
    }

    if (cur_state == BluetoothStreamState::STARTING) {
        if (data_delta_time_us > 0) {
            if (adev->debug_flag) {
                AM_LOGD("write too fast, need sleep:%" PRId64 " ms", data_delta_time_us / USEC_PER_MSEC);
            }
            usleep(data_delta_time_us);
        }
        AM_LOGI("a2dp state is %s",  a2dpStatus2String(cur_state));
    } else if (cur_state == BluetoothStreamState::STARTED) {
         if (adev->audio_patch && adev->tv_mute) {
            /* tv_mute for atv switch channel */
            AM_LOGI("tv_mute:%d, start standby", adev->tv_mute);
            a2dp_out_standby_l(adev);
        } else {
            prepared = true;
        }
    } else if (cur_state == BluetoothStreamState::DISABLED) {
        // TODO: A2DP is disconnected. do nothing.
    } else {
        struct aml_audio_patch *patch = adev->audio_patch;
        if (!(adev->tv_mute && patch)) {
            a2dp_out_resume_l(adev);
        }
        // a2dp_out_resume maybe cause over 100ms, so set last_write_time after resume,
        // otherwise, the gap would always over 64ms, and always standby in dtv
        hal->last_write_time = aml_audio_get_systime();
    }
    return prepared;
}

static ssize_t a2dp_in_data_process(aml_a2dp_hal *hal, audio_config_base_t *config, const void *buffer, size_t bytes) {
    size_t frames = 0;
    int realloc_ret = 0;
    if (config->channel_mask == AUDIO_CHANNEL_OUT_7POINT1 && config->format == AUDIO_FORMAT_PCM_32_BIT) {
        frames = bytes / (4 * 8);
        realloc_ret = aml_audio_check_and_realloc((void **)&hal->buff_conv_format, &hal->buff_size_conv_format, frames * 4);
        if (realloc_ret != 0) {
            AM_LOGE("aml_audio_check_and_realloc fail");
            return -1;
        }
        int16_t *tmp_buffer = (int16_t *)hal->buff_conv_format;
        int32_t *tmp_buffer_8ch = (int32_t *)buffer;
        for (int i=0; i<frames; i++) {
            tmp_buffer[2 * i]       = (tmp_buffer_8ch[8 *  i] >> 16);
            tmp_buffer[2 * i + 1]   = (tmp_buffer_8ch[8 * i + 1] >> 16);
        }
    } else if (config->channel_mask == AUDIO_CHANNEL_OUT_STEREO && config->format == AUDIO_FORMAT_PCM_16_BIT) {
        frames = bytes / AUDIO_HAL_FIXED_FRAME_SIZE;
        realloc_ret = aml_audio_check_and_realloc((void **)&hal->buff_conv_format, &hal->buff_size_conv_format, bytes);
        if (realloc_ret != 0) {
            AM_LOGE("aml_audio_check_and_realloc fail");
            return -1;
        }
        memcpy(hal->buff_conv_format, buffer, bytes);
    } else {
        AM_LOGW("not support param, channel_cnt:%d, format:%#x",
            audio_channel_count_from_out_mask(config->channel_mask), config->format);
        return -1;
    }

    const int64_t cur_write_time_us = aml_audio_get_systime();
    if (hal->mute_time > 0) {
        if (hal->mute_time > cur_write_time_us) {
            memset((void*)buffer, 0, bytes);
        } else {
            hal->mute_time = 0;
        }
    }
    return frames;
}

static ssize_t a2dp_data_resample_process(aml_a2dp_hal *hal, audio_config_base_t *input_cfg,
    const void *buffer, size_t in_frames, const void **output_buffer) {
    int out_frames = in_frames;
    *output_buffer = buffer;
    if (input_cfg->sample_rate != hal->config.sample_rate) {
        size_t in_frame_size = AUDIO_HAL_FIXED_FRAME_SIZE;
        /* The resampled frames may be large than the theoretical value.
         * So, there is an extra 32 bytes allocated to prevent overflows.
         */
        int resample_out_buffer_size = in_frames * hal->config.sample_rate * in_frame_size / input_cfg->sample_rate + 32;
        if (hal->resample == NULL || hal->resample->resample_config.input_sr != input_cfg->sample_rate) {
            audio_resample_config_t resample_cfg;
            resample_cfg.aformat   = AUDIO_HAL_FIXED_CFG_FORMAT;
            resample_cfg.channels  = audio_channel_count_from_out_mask(AUDIO_HAL_FIXED_CFG_CHANNEL);
            resample_cfg.input_sr  = input_cfg->sample_rate;
            resample_cfg.output_sr = hal->config.sample_rate;
            if (hal->resample == NULL) {
                AM_LOGI("resample init format:%#x, ch:%d", resample_cfg.aformat, resample_cfg.channels);
                int ret = aml_audio_resample_init(&hal->resample, AML_AUDIO_SIMPLE_RESAMPLE, &resample_cfg);
                R_CHECK_RET(ret, "Resampler is failed initialization !!!");
            } else {
                AM_LOGI("resample reconfig, input_sr changed %d -> %d", hal->resample->resample_config.input_sr, input_cfg->sample_rate);
                memcpy(&hal->resample->resample_config, &resample_cfg, sizeof(audio_resample_config_t));
                aml_audio_resample_reset(hal->resample);
            }
        }
        aml_audio_resample_process(hal->resample, (void *)buffer, in_frames * in_frame_size);
        if (in_frame_size > 0)
            out_frames = hal->resample->resample_size / in_frame_size;
        *output_buffer = hal->resample->resample_buffer;
    }
    return out_frames;
}

static size_t a2dp_out_data_process(aml_a2dp_hal *hal, audio_config_base_t *config __unused,
    const void *buffer, size_t in_frames, const void **output_buffer) {
    size_t out_size = in_frames * AUDIO_HAL_FIXED_FRAME_SIZE;
    if (hal->config.channel_mask == AUDIO_CHANNEL_OUT_MONO) {
        int16_t *tmp_buffer = (int16_t *)buffer;
        for (int i=0; i<in_frames; i++) {
            tmp_buffer[i] = tmp_buffer[2 * i];
        }
        out_size = in_frames * 1 * audio_bytes_per_sample(AUDIO_HAL_FIXED_CFG_FORMAT);
    } else if (hal->config.channel_mask == AUDIO_CHANNEL_OUT_STEREO) {
        /* 2channel do nothing*/
    } else {
        AM_LOGW("not support a2dp output channel_cnt:%#x",
            audio_channel_count_from_out_mask(AUDIO_HAL_FIXED_CFG_CHANNEL));
        return 0;
    }

    size_t out_per_sample_byte = audio_bytes_per_sample(hal->config.format);
    size_t out_channel_byte = audio_channel_count_from_out_mask(hal->config.channel_mask);
    out_size = out_per_sample_byte * out_channel_byte * in_frames;
    if (hal->config.format != AUDIO_FORMAT_PCM_16_BIT) {
        int realloc_ret = aml_audio_check_and_realloc((void **)&hal->buff_conv_format, &hal->buff_size_conv_format, out_size);
        if (realloc_ret != 0) {
            AM_LOGE("aml_audio_check_and_realloc fail");
            return 0;
        }
        R_CHECK_RET(0, "realloc buff_conv_format size:%zu fail", out_size);
        if (hal->config.format == AUDIO_FORMAT_PCM_32_BIT) {
            memcpy_to_i32_from_i16((int32_t *)hal->buff_conv_format, (int16_t *)buffer, in_frames * out_channel_byte);
        } else if (hal->config.format == AUDIO_FORMAT_PCM_24_BIT_PACKED) {
            memcpy_to_p24_from_i16((uint8_t *)hal->buff_conv_format, (int16_t *)buffer, in_frames * out_channel_byte);
        } else {
            AM_LOGW("not support a2dp output format:%#x", hal->config.format);
            return 0;
        }
        *output_buffer = hal->buff_conv_format;
    }
    return out_size;
}

static ssize_t a2dp_out_write_l(struct aml_audio_device *adev, audio_config_base_t *config, const void* buffer, size_t bytes) {
    aml_a2dp_hal *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    static long int test_file_position = 0;
    size_t wr_size = 0;
    const void *wr_buff = NULL;
    ssize_t cur_frames = 0;
    ssize_t resample_frames = 0;
    size_t bytes_written = 0;
    size_t sent = 0;

    if (adev->a2dp_hal == NULL) {
        if (adev->debug_flag) {
            AM_LOGW("a2dp_hal is null pointer");
        }
        return bytes;
    }

    cur_frames = a2dp_in_data_process(hal, config, buffer, bytes);
    if (cur_frames < 0) {
        return bytes;
    }

    if (!a2dp_state_process(adev, config, cur_frames)) {
        a2dp_notify_monitor(hal);
        return bytes;
    }

    // For debug a2dp data. 48Khz, 2channel, 2byte.
    aml_audio_read_audio_data_by_file(A2DP_TEST_AUDIO_FILE_PATH, A2DP_TEST_AUDIO_FILE_PROP,
        hal->buff_conv_format, cur_frames * AUDIO_HAL_FIXED_FRAME_SIZE, &test_file_position);

    resample_frames = a2dp_data_resample_process(hal, config, hal->buff_conv_format, cur_frames, &wr_buff);
    if (resample_frames < 0) {
        return bytes;
    }

    wr_size = a2dp_out_data_process(hal, config, wr_buff, resample_frames, &wr_buff);
    if (wr_size == 0) {
        return bytes;
    }

    if (adev->patch_src == SRC_DTV && adev->parental_control_av_mute) {
        memset((void*)wr_buff, 0, wr_size);
    }
    dump_a2dp_output_data(hal, wr_buff, wr_size);
    uint64_t write_enter_time_us = aml_audio_get_systime();
    while (bytes_written < wr_size) {
        size_t need_write = wr_size - bytes_written;
        if (getprop_bool(A2DP_TEST_AUDIO_CHECK_MUTE_PROP)) {
            check_audio_level("a2dp_check", (char *)wr_buff + bytes_written, need_write);
        }
        a2dp_notify_monitor(hal, true);
        uint64_t write_start_time_us = aml_audio_get_systime();
        sent = hal->a2dphw.WriteData((char *)wr_buff + bytes_written, need_write);
        uint64_t write_stop_time_us = aml_audio_get_systime();
        uint64_t write_data_time_ms = (write_stop_time_us - write_start_time_us) / USEC_PER_MSEC;
        if (adev->debug_flag || write_data_time_ms > 40) {
            /* Debug the time of write data to policy and the time of the write_data */
            uint64_t data_time_ms = need_write / (hal->config.sample_rate * audio_bytes_per_sample(hal->config.format)
                * audio_channel_count_from_out_mask(hal->config.channel_mask) / MSEC_PER_SEC);
            AM_LOGD("write:%zu sent:%zu total:%zu write_time: %" PRIu64 " ms data_time: %" PRIu64 " ms",
                need_write, sent, wr_size, write_data_time_ms, data_time_ms);
        }
        a2dp_notify_monitor(hal);
        bytes_written += sent;
        /* The cache of BT stack is about 40ms data, and exit from writing data
         * after timeout of 64ms here. */
        if (bytes_written < wr_size &&
            (write_stop_time_us - write_enter_time_us) > A2DP_WRITE_DATE_TIME_OUT_MS * USEC_PER_MSEC) {
            AM_LOGW("WriteData timeout: %" PRIu64 " ms, quit now.", (write_stop_time_us - write_enter_time_us) / USEC_PER_MSEC);
            break;
        }
    }
    return bytes;
}

ssize_t a2dp_out_write(struct aml_audio_device *adev, audio_config_base_t *config, const void* buffer, size_t bytes) {
    size_t in_frame_size = audio_channel_count_from_out_mask(config->channel_mask) * audio_bytes_per_sample(config->format);
    uint32_t one_ms_data = in_frame_size * config->sample_rate / 1000;
    uint32_t date_len_ms = bytes / one_ms_data;
    const uint32_t period_time_ms = 32;
    const uint32_t period_time_size = one_ms_data * period_time_ms;

    if (bytes == 0) {
        AM_LOGW("bytes is 0");
        return -1;
    }
    R_CHECK_POINTER_LEGAL(-1, config, "");
    R_CHECK_POINTER_LEGAL(-1, buffer, "");

    uint32_t written_size = 0;
    pthread_mutex_lock(&adev->a2dp_lock);
    while (bytes > written_size) {
        uint32_t remain_size = bytes - written_size;
        size_t sent = remain_size;
        if (remain_size > period_time_ms * one_ms_data) {
            sent = period_time_size;
        }
        /*coverity[sleep]*/
        a2dp_out_write_l(adev, config, (char *)buffer + written_size, sent);
        AM_LOGV("written_size:%d, remain_size:%d, sent:%zu", written_size, remain_size, sent);
        written_size += sent;
    }
    pthread_mutex_unlock(&adev->a2dp_lock);
    return written_size;
}

uint32_t a2dp_out_get_latency(struct aml_audio_device *adev) {
    uint64_t remote_delay_report_ns = 0;
    pthread_mutex_lock(&adev->a2dp_lock);
    struct aml_a2dp_hal * hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    if (!hal) {
        pthread_mutex_unlock(&adev->a2dp_lock);
        return DEFAULT_A2DP_LATENCY_NS / NSEC_PER_MSEC;
    }
    /* Some BT devices(eg: Xiaomi Air2) will change the latency after the connection is successful,
     * causing Youtube playback fail. */
    if (hal->a2dp_latency == A2DP_LATENCY_INVALID_NS) {
        std::shared_ptr<BluetoothAudioSession> session_ptr =
            BluetoothAudioSessionInstance::GetSessionInstance(SessionType::A2DP_SOFTWARE_ENCODING_DATAPATH);
        bool success = session_ptr->GetPresentationPosition(&remote_delay_report_ns, nullptr, nullptr);
        if (!success || remote_delay_report_ns == 0) {
            hal->a2dp_latency = DEFAULT_A2DP_LATENCY_NS;
        } else {
            hal->a2dp_latency = remote_delay_report_ns;
        }
        AM_LOGI("success:%d report_latency:%" PRIu64" ms, latency:%" PRIu64" ms", success,
            remote_delay_report_ns / NSEC_PER_MSEC, hal->a2dp_latency / NSEC_PER_MSEC);
    }
    remote_delay_report_ns = hal->a2dp_latency;
    pthread_mutex_unlock(&adev->a2dp_lock);
    return static_cast<uint32_t>(remote_delay_report_ns / NSEC_PER_MSEC + A2DP_STATIC_DELAY_MS);
}

int a2dp_out_get_status(struct aml_audio_device *adev) {
    struct aml_a2dp_hal * hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    if (!hal) {
        AM_LOGW("a2dp_hal is null");
        return -1;
    }
    return (int)hal->state;
}

int a2dp_out_set_parameters(struct aml_audio_device *adev, const char *kvpairs) {
    struct aml_a2dp_hal * hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    R_CHECK_POINTER_LEGAL(-1, hal, "a2dp hw is released");

    std::unordered_map<std::string, std::string> params = ParseAudioParams(kvpairs);
    if (params.empty())
        return 0;

    if (params.find("A2dpSuspended") != params.end()) {
        if (params["A2dpSuspended"] == "true") {
            if (hal->a2dphw.GetState() != BluetoothStreamState::DISABLED)
                hal->a2dphw.Stop();
        } else {
            if (hal->a2dphw.GetState() == BluetoothStreamState::DISABLED)
                hal->a2dphw.SetState(BluetoothStreamState::STANDBY);
        }
    }
    if (params["closing"] == "true") {
        if (hal->a2dphw.GetState() != BluetoothStreamState::DISABLED)
            hal->a2dphw.Stop();
    }
    return 0;
}

int a2dp_hal_dump(struct aml_audio_device *adev, int fd) {
    struct aml_a2dp_hal *hal = (struct aml_a2dp_hal *)adev->a2dp_hal;
    if (hal) {
        dprintf(fd, "------------ [AM_HAL][A2DP] -------------------------------------\n");
        dprintf(fd, "-[AML_HAL]      out_rate      : %10d     | out_ch    :%10d\n", hal->config.sample_rate, audio_channel_count_from_out_mask(hal->config.channel_mask));
        dprintf(fd, "-[AML_HAL]      out_format    : %#10x     | cur_state :%10s\n", hal->config.format, a2dpStatus2String(hal->a2dphw.GetState()));
        dprintf(fd, "-[AML_HAL]      a2dp_latency  : %" PRIu64" ms\n", hal->a2dp_latency / NSEC_PER_MSEC);
        aml_audio_resample_t *resample = hal->resample;
        if (resample) {
            audio_resample_config_t *config = &resample->resample_config;
            dprintf(fd, "-[AML_HAL] resample in_sr     : %10d     | out_sr    :%10d\n", config->input_sr, config->output_sr);
            dprintf(fd, "-[AML_HAL] resample ch        : %10d     | type      :%10d\n", config->channels, resample->resample_type);
        }
    }
    return 0;
}


