/******************************************************************************
 *
 *  Copyright (C) 2020-2021 SeekWave Technology
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 *
 ******************************************************************************/

#include <pthread.h>
#include <errno.h>
#include <utils/Log.h>
#include <sys/inotify.h>
#include <sys/epoll.h>

#include <string.h>
#include <unistd.h>
#include "skw_common.h"
#include "skw_ext.h"

pthread_t skw_ext_inotify_thread_id;
char skw_ext_inotify_thread_running = FALSE;

extern void scomm_vendor_write_wakeup_adv_enable();


/*
struct inotify_event {
   int      wd;       // Watch descriptor
   uint32_t mask;     // Mask of events
   uint32_t cookie;   // Unique cookie associating related  events (for rename(2))
   uint32_t len;      // Size of name field
   char     name[];   // Optional null-terminated name
};
*/

/*******************************************************************************
**
** Function        skw_ext_inotify_thread
**
** Description     inotify thread function
**
** Returns         void *
**
*******************************************************************************/
static void *skw_ext_inotify_thread(void *arg)
{
    int fd = -1;
    int wd = -1;
    int read_len = 0;
    int event_pos = 0;
    int event_size = 0;
    char buffer[2049] = {0};
    struct inotify_event *event = NULL;

    int epfd = epoll_create(256);
    struct epoll_event ev;
    struct epoll_event events[20];

    SKW_UNUSED(arg);
    fd = inotify_init1(IN_NONBLOCK);
    if(fd < 0)
    {
        ALOGE("inotify_init1 failed: %s", strerror(errno));
        return NULL;
    }
    wd = inotify_add_watch(fd, "/dev", IN_CREATE);
    if (wd < 0)
    {
        ALOGE("inotify_add_watch fail: %s\n", strerror(errno));
        return NULL;
    }

    ev.data.fd = fd;
    ev.events = EPOLLIN | EPOLLET;

    epoll_ctl(epfd, EPOLL_CTL_ADD, fd, &ev);


    while(skw_ext_inotify_thread_running)
    {
        int nfds = epoll_wait(epfd, events, 20, 500);
        for (int i = 0; i < nfds; ++i)
        {
            if (events[i].data.fd != fd)
            {
                continue;
            }
            read_len = read(fd, buffer, 2048);

            //SKWBT_LOG("select read_len:%d, nfds:%d", read_len, nfds);

            if(read_len <= 0)
            {
                continue;
            }
            event_pos = 0;

            while(read_len >= (int)sizeof(struct inotify_event))//may receive multiple events
            {
                event = (struct inotify_event *)(buffer + event_pos);
                if(event->len)
                {
                    if(event->mask & IN_CREATE)
                    {
                        SKWBT_LOG("recv command msg mask:0x%X name:%s", event->mask, event->name);
                        if (((event->mask & IN_ISDIR) == 0) && (strcmp(event->name, "shutdown") == 0))
                        {
                            scomm_vendor_write_wakeup_adv_enable();

                            SKWBT_LOG("received shutdown command");
                            goto thread_exit;
                        }
                    }
                    else
                    {
                        SKWBT_LOG("othre event, mask:0x%X, name: %s", event->mask, event->name);
                    }
                }

                event_size = sizeof(struct inotify_event) + event->len;
                read_len -= event_size;
                event_pos += event_size;
            }

        }
    }

thread_exit:
    inotify_rm_watch(fd, wd);
    close(fd);
    skw_ext_inotify_thread_running = FALSE;
    return NULL;
}

/*******************************************************************************
**
** Function        skw_ext_inotify_thread_init
**
** Description     inotify thread init
**
** Returns         None
**
*******************************************************************************/
void skw_ext_inotify_thread_init(void)
{
#if BLE_ADV_WAKEUP_ENABLE
    pthread_attr_t thread_attr;
    pthread_attr_init(&thread_attr);
    pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_JOINABLE);
    skw_ext_inotify_thread_running = FALSE;
    if(pthread_create(&skw_ext_inotify_thread_id, &thread_attr, skw_ext_inotify_thread, NULL) != 0)
    {
        ALOGE("%s pthread_create : %s", __func__, strerror(errno));
        return ;
    }
    skw_ext_inotify_thread_running = TRUE;
#else
    SKW_UNUSED(skw_ext_inotify_thread);
#endif
}

/*******************************************************************************
**
** Function        skw_ext_inotify_thread_exit
**
** Description     inotify thread exit
**
** Returns         None
**
*******************************************************************************/
void skw_ext_inotify_thread_exit(void)
{
    if(skw_ext_inotify_thread_running && (skw_ext_inotify_thread_id != -1))
    {
        skw_ext_inotify_thread_running = FALSE;
        pthread_join(skw_ext_inotify_thread_id, NULL);
    }
}