/* Copyright Statement:
 *
 * This software/firmware and related documentation ("MediaTek Software") are
 * protected under relevant copyright laws. The information contained herein is
 * confidential and proprietary to MediaTek Inc. and/or its licensors. Without
 * the prior written permission of MediaTek inc. and/or its licensors, any
 * reproduction, modification, use or disclosure of MediaTek Software, and
 * information contained herein, in whole or in part, shall be strictly
 * prohibited.
 *
 * MediaTek Inc. (C) 2014. All rights reserved.
 *
 * BY OPENING THIS FILE, RECEIVER HEREBY UNEQUIVOCALLY ACKNOWLEDGES AND AGREES
 * THAT THE SOFTWARE/FIRMWARE AND ITS DOCUMENTATIONS ("MEDIATEK SOFTWARE")
 * RECEIVED FROM MEDIATEK AND/OR ITS REPRESENTATIVES ARE PROVIDED TO RECEIVER
 * ON AN "AS-IS" BASIS ONLY. MEDIATEK EXPRESSLY DISCLAIMS ANY AND ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NONINFRINGEMENT. NEITHER DOES MEDIATEK PROVIDE ANY WARRANTY WHATSOEVER WITH
 * RESPECT TO THE SOFTWARE OF ANY THIRD PARTY WHICH MAY BE USED BY,
 * INCORPORATED IN, OR SUPPLIED WITH THE MEDIATEK SOFTWARE, AND RECEIVER AGREES
 * TO LOOK ONLY TO SUCH THIRD PARTY FOR ANY WARRANTY CLAIM RELATING THERETO.
 * RECEIVER EXPRESSLY ACKNOWLEDGES THAT IT IS RECEIVER'S SOLE RESPONSIBILITY TO
 * OBTAIN FROM ANY THIRD PARTY ALL PROPER LICENSES CONTAINED IN MEDIATEK
 * SOFTWARE. MEDIATEK SHALL ALSO NOT BE RESPONSIBLE FOR ANY MEDIATEK SOFTWARE
 * RELEASES MADE TO RECEIVER'S SPECIFICATION OR TO CONFORM TO A PARTICULAR
 * STANDARD OR OPEN FORUM. RECEIVER'S SOLE AND EXCLUSIVE REMEDY AND MEDIATEK'S
 * ENTIRE AND CUMULATIVE LIABILITY WITH RESPECT TO THE MEDIATEK SOFTWARE
 * RELEASED HEREUNDER WILL BE, AT MEDIATEK'S OPTION, TO REVISE OR REPLACE THE
 * MEDIATEK SOFTWARE AT ISSUE, OR REFUND ANY SOFTWARE LICENSE FEES OR SERVICE
 * CHARGE PAID BY RECEIVER TO MEDIATEK FOR SUCH MEDIATEK SOFTWARE AT ISSUE.
 *
 * The following software/firmware and/or related documentation ("MediaTek
 * Software") have been modified by MediaTek Inc. All revisions are subject to
 * any receiver's applicable license agreements with MediaTek Inc.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sched.h>
#include <pthread.h>
#include <sys/prctl.h>
#include <sys/syscall.h>
#include <sys/time.h>
#if defined(MTK_LINUX)
//#include "osi/include/properties.h"
#else
#include <cutils/properties.h>
#endif
#include "bt_mtk.h"

#ifndef SUPPORT_BLE_AUDIO_BY_ADD_ACL_HEADER
#define SUPPORT_BLE_AUDIO_BY_ADD_ACL_HEADER 1
#endif

#if (defined(MTK_VENDOR_OPCODE) && (MTK_VENDOR_OPCODE == TRUE))
#include <sys/ioctl.h>

#define BTMTK_IOCTL_MAGIC 'k'

#define BTMTK_IOCTL_STEREO_GET_CLK _IOR(BTMTK_IOCTL_MAGIC, 1, void *)
#define BTMTK_IOCTL_STEREO_SET_PARA _IOW(BTMTK_IOCTL_MAGIC, 2, void *)
#endif

#define BTPROTO_HCI 1
#define HCI_CHANNEL_USER 1
#define HCI_BUF_SIZE (4 * 1024)
enum {
    FD_FOR_STACK = 0,
    FD_FOR_VENDOR = 1,
    FD_FOR_DRV = 2,
    FD_NUM = 3,
};
#define RETRY_COUNT         50
#define CUST_BT_SERIAL_PORT "/dev/stpbt"
#define HW_ERR_CODE_BROKEN_PIPE 0xFF

#if __GLIBC__ || (defined(ENABLE_MUSL_LIBC) && (ENABLE_MUSL_LIBC == TRUE))
pid_t gettid(void) { return syscall(SYS_gettid); }
#endif

#ifdef MTK_LINUX_FETCH_ADDRESS
#define HCI_SET_BD_ADDRESS_OP_CODE 0xFC1A
#endif

/**************************************************************************
 *                  G L O B A L   V A R I A B L E S                       *
***************************************************************************/
bt_vendor_callbacks_t *bt_vnd_cbacks = NULL;
static int  bt_fd = -1;
static int tx_quit = 0;
static int rx_quit = 0;
static int sock_fd[FD_NUM] = {-1, -1, -1};
static pthread_t mtk_tx_thread = -1;
static pthread_t mtk_rx_thread = -1;
static pthread_mutex_t tx_mutex;
static pthread_mutex_t rx_mutex;

struct sockaddr_hci {
    sa_family_t    hci_family;
    unsigned short hci_dev;
    unsigned short hci_channel;
};

#define SHOW_RAW(p, l, fmt, ...)                        \
    do {    \
        int cnt_ = 0;    \
        int len_ = (l <= HCI_SNOOP_MAX_BUF_SIZE ? l : HCI_SNOOP_MAX_BUF_SIZE);    \
        uint8_t raw[HCI_SNOOP_MAX_BUF_SIZE * 3 + 10];    \
        const unsigned char *ptr = p;    \
        for (cnt_ = 0; cnt_ < len_; ++cnt_)    \
            (void)snprintf(raw+3*cnt_, 4, "%02X ", ptr[cnt_]);    \
        raw[3*cnt_] = '\0';    \
        if (l <= HCI_SNOOP_MAX_BUF_SIZE) {    \
            LOG_ERR(fmt"%s\n", ##__VA_ARGS__, raw);    \
        } else {    \
            LOG_ERR(fmt"%s (prtail)\n", ##__VA_ARGS__, raw);    \
        }    \
    } while (0)

#define HCI_SNOOP_ENTRY_NUM    30
#define HCI_SNOOP_BUF_SIZE    32
#define HCI_SNOOP_MAX_BUF_SIZE    66
#define HCI_SNOOP_TS_STR_LEN    32

struct hci_snoop {
    uint8_t buf[HCI_SNOOP_ENTRY_NUM][HCI_SNOOP_MAX_BUF_SIZE];
    uint8_t len[HCI_SNOOP_ENTRY_NUM];
    uint16_t actual_len[HCI_SNOOP_ENTRY_NUM];
    char timestamp[HCI_SNOOP_ENTRY_NUM][HCI_SNOOP_TS_STR_LEN];
    uint8_t index;
};

enum {
    HCI_SNOOP_TYPE_CMD = 0,
    HCI_SNOOP_TYPE_EVT,
    HCI_SNOOP_TYPE_ADV_EVT,
    HCI_SNOOP_TYPE_NOCP_EVT,
    HCI_SNOOP_TYPE_TX_ACL,
    HCI_SNOOP_TYPE_RX_ACL,
    HCI_SNOOP_TYPE_TX_ISO,
    HCI_SNOOP_TYPE_RX_ISO,
    HCI_SNOOP_TYPE_MAX
};

static struct hci_snoop snoop[HCI_SNOOP_TYPE_MAX];

/**************************************************************************
 *              F U N C T I O N   D E C L A R A T I O N S                 *
***************************************************************************/

/**************************************************************************
 *                          F U N C T I O N S                             *
***************************************************************************/

BOOL is_memzero(unsigned char *buf, int size)
{
    int i;
    for (i = 0; i < size; i++) {
        if (*(buf+i) != 0) return FALSE;
    }
    return TRUE;
}

/* Register callback functions to libbt-hci.so */
void set_callbacks(const bt_vendor_callbacks_t *p_cb)
{
    bt_vnd_cbacks = (bt_vendor_callbacks_t*)p_cb;
}

/* Cleanup callback functions previously registered */
void clean_callbacks(void)
{
    bt_vnd_cbacks = NULL;
}

static void get_time_str(char *ts_str, uint8_t ts_len)
{
    struct timeval tv;
    struct tm *ltime;
    struct tm ltime_data;
    char buffer[HCI_SNOOP_TS_STR_LEN] = {0};

    memset((void*)&ltime_data, 0, sizeof(ltime_data));
    gettimeofday(&tv, NULL);
    ltime = localtime_r(&tv.tv_sec, &ltime_data);
    if (ltime == NULL)
        return;
    memset(ts_str, 0, sizeof(ts_len));
    (void)strftime(buffer, sizeof(buffer), "%Y%m%d-%H%M%S", ltime);
    (void)snprintf(ts_str, ts_len, "%s.%06ld", buffer, tv.tv_usec);
}

static void btmtk_hci_snoop_print(void)
{
    uint8_t counter, index, snoop_index;
    char *snoop_str[HCI_SNOOP_TYPE_MAX] = {"Command", "Event", "ADV Event", "NOCP Event", "TX ACL", "RX ACL", "TX ISO", "RX ISO"};

    for (snoop_index = 0; snoop_index < HCI_SNOOP_TYPE_MAX; snoop_index++) {
        LOG_ERR("HCI %s Dump: Using A5 A5 to separator the head 32 bytes and the tail 32 bytes data", snoop_str[snoop_index]);
        if (snoop[snoop_index].index >= (HCI_SNOOP_ENTRY_NUM - 1))
            index = 0;
        else
            index = snoop[snoop_index].index + 1;
        for (counter = 0; counter < HCI_SNOOP_ENTRY_NUM; counter++) {
            if (snoop[snoop_index].len[index] > 0)
                SHOW_RAW(snoop[snoop_index].buf[index], snoop[snoop_index].len[index],
                    "time(%s)-act_len(%d)-len(%d):", snoop[snoop_index].timestamp[index],
                    snoop[snoop_index].actual_len[index], snoop[snoop_index].len[index]);
            index++;
            if (index >= HCI_SNOOP_ENTRY_NUM)
                index = 0;
        }
    }
}

static void btmtk_hci_snoop_save(unsigned int type, uint8_t *buf, unsigned int len)
{
    unsigned int copy_len = HCI_SNOOP_BUF_SIZE;
    unsigned int copy_tail_len = HCI_SNOOP_BUF_SIZE;
    uint8_t separator_char[2] = {0xA5, 0xA5};
    uint8_t *copy_tail_buf;

    if (!buf || len == 0 || type >= HCI_SNOOP_TYPE_MAX) {
        LOG_ERR("invalid parameters!");
        return;
    }

    if (snoop[type].index < HCI_SNOOP_ENTRY_NUM) {
        if (len < HCI_SNOOP_BUF_SIZE) {
            copy_len = len;
            copy_tail_len = 0;
        } else if (len > HCI_SNOOP_BUF_SIZE && len <= HCI_SNOOP_BUF_SIZE * 2)
            copy_tail_len = len - copy_len;

        snoop[type].len[snoop[type].index] = copy_len & 0xff;
        snoop[type].actual_len[snoop[type].index] = len & 0xffff;
        get_time_str(snoop[type].timestamp[snoop[type].index], HCI_SNOOP_TS_STR_LEN);
        memset(snoop[type].buf[snoop[type].index], 0, HCI_SNOOP_MAX_BUF_SIZE);
        memcpy(snoop[type].buf[snoop[type].index], buf, copy_len & 0xff);
        /* save less then 32 bytes data in the buffer tail, using A5 A5 to
         * separator the head 32 bytes data and the tail 32 bytes data
         */
        if (copy_tail_len > 0) {
            copy_tail_buf = buf + len - copy_tail_len;
            snoop[type].len[snoop[type].index] +=
                (copy_tail_len + 2) & 0xff;
            memcpy(snoop[type].buf[snoop[type].index] + copy_len, separator_char,
                2);
            memcpy(snoop[type].buf[snoop[type].index] + copy_len + 2,
                copy_tail_buf, copy_tail_len);
        }

        if (snoop[type].index == 0)
            snoop[type].index = HCI_SNOOP_ENTRY_NUM;
        snoop[type].index--;
    }
}

static void mtk_set_tx_thread_quit(void)
{
    LOG_WAN("enter\n");
    (void)pthread_mutex_lock(&tx_mutex);
    LOG_WAN("lock tx mutex, tx_quit = %d\n", tx_quit);
    tx_quit = 1;
    LOG_WAN("unlock tx mutex, tx_quit = %d\n", tx_quit);
    pthread_mutex_unlock(&tx_mutex);
    LOG_WAN("exit\n");
}
static int mtk_is_tx_thread_quit(void)
{
    int quit = 0;
    (void)pthread_mutex_lock(&tx_mutex);
    quit = tx_quit;
    pthread_mutex_unlock(&tx_mutex);
    if (quit == 1) {
        LOG_WAN("quit(%d)\n", quit);
    }
    return quit;
}

static void mtk_set_rx_thread_quit(void)
{
    LOG_WAN("enter\n");
    (void)pthread_mutex_lock(&rx_mutex);
    LOG_WAN("lock rx mutex, rx_quit = %d\n", rx_quit);
    rx_quit = 1;
    LOG_WAN("unlock rx mutex, rx_quit = %d\n", rx_quit);
    pthread_mutex_unlock(&rx_mutex);
    LOG_WAN("exit\n");
}
static int mtk_is_rx_thread_quit(void)
{
    int quit = 0;
    (void)pthread_mutex_lock(&rx_mutex);
    quit = rx_quit;
    pthread_mutex_unlock(&rx_mutex);
    if (quit == 1) {
        LOG_WAN("quit(%d)\n", quit);
    }
    return quit;
}

static ssize_t mtk_read_data(int fd, uint8_t *buf, size_t size)
{
    ssize_t nRead = 0;

    if (fd < 0) {
        LOG_ERR("File descriptor in bad state\n");
        return -EIO;
    }

    if (!buf || !size) {
        LOG_ERR("Invalid argument buf:%p, size:%d\n", buf, (int)size);
        return -EINVAL;
    }

    //nRead = read(fd, buf, size);
    nRead = recv(fd, buf, size, 0);
    if (nRead <= 0) {
        if (errno == EINTR || errno == EAGAIN || errno == EWOULDBLOCK) {
            LOG_WARNNO(errno, "nRead = %ld\n", nRead);
            return 0;
        } else {
            LOG_ERRORNO(errno, "%s(%d)\n", __FUNCTION__, __LINE__);
            return -errno;
        }
    }
    //LOG_DBG("read data from fd: %d, nRead = %d\n", fd, nRead);
    //SHOW_RAW(nRead, buf);
    return nRead;
}

static ssize_t mtk_write_data(int fd, uint8_t *buf, size_t len)
{
    ssize_t ret = 0;
    size_t bytesToWrite = len;
    int ms_timeout = 1000;

    if (fd < 0) {
        LOG_ERR("File descriptor in bad state\n");
        return -EIO;
    }

    if (!buf || !len) {
        LOG_ERR("Invalid argument buf:%p, size:%d\n", buf, (int)len);
        return -EINVAL;
    }

    //LOG_WAN("write data to fd: %d, len = %d\n", fd, len);
    //SHOW_RAW(len, buf);
    while (bytesToWrite > 0) {
        //ret = write(fd, buf, bytesToWrite);
         ret = send(fd, buf, bytesToWrite, MSG_DONTWAIT);
        if (ret == -1) {
            if (errno != EAGAIN && errno != EWOULDBLOCK) {
                LOG_WARNNO(errno, "");
                return -1;
            }

            if (ms_timeout >= 100) {
                (void)usleep(100 * 1000);
                ms_timeout -= 100;
                continue;
            }
            LOG_WAN("write timeout exceeded, sent %zu bytes", len - bytesToWrite);
            return -1;
        }
        bytesToWrite -= ret;
        buf += ret;
    }

    //LOG_WAN("write data actual len = %d\n", len - bytesToWrite);
    return (len - bytesToWrite);
}

static int mtk_unpack_and_write_host_data(int fd, uint8_t *buf, size_t total_len)
{
    ssize_t write_len = 0;
    uint8_t *cur = NULL;
    uint8_t *next = buf;
    uint16_t pkt_len = 0;
    unsigned int type = 0;

    if (fd < 0 || !buf) {
        LOG_ERR("Invalid argument buf:%p, fd :%d\n", buf, fd);
        return -EINVAL;
    }

    if (total_len < HCI_MIN_FRAME_SIZE) {
        LOG_ERR("Read data from host failed, total_len = %lu\n", total_len);
        return -EINVAL;
    }

    if (buf[0] != HCI_COMMAND_PKT &&
        buf[0] != HCI_ACLDATA_PKT &&
        buf[0] != HCI_SCODATA_PKT &&
        buf[0] != HCI_ISO_PKT) {
        SHOW_RAW(buf, total_len, "type is invalid, len = %lu - \n", total_len);
        return -EINVAL;
    }
    /* maybe two or more acl packets in socket buffer,
       * the read_len will be larger than HCI_MAX_FRAME_SIZE,
       * we need to unpack the data from the buf
       */
#if 0
    if (total_len > HCI_MAX_FRAME_SIZE) {
        LOG_WAN("buf len maybe include two or more packets leng, len = %d\n", total_len);
    }
#endif

    while (next < (buf + total_len)) {
        cur = next;
        if (cur[0] == HCI_COMMAND_PKT && (cur + HCI_MIN_FRAME_SIZE) <= (buf + total_len)) {
            pkt_len = cur[3] + HCI_MIN_FRAME_SIZE;
            type = HCI_SNOOP_TYPE_CMD;
        }
        else if (cur[0] == HCI_ACLDATA_PKT && (cur + ACL_DATA_HEADER_SIZE) <= (buf + total_len)) {
            pkt_len = cur[3] + ((cur[4] << 8) & 0xff00) + ACL_DATA_HEADER_SIZE;
            type = HCI_SNOOP_TYPE_TX_ACL;
        }
        else if (cur[0] == HCI_ISO_PKT && (cur + ISO_DATA_HEADER_SIZE) <= (buf + total_len)) {
            pkt_len = cur[3] + ((cur[4] << 8) & 0x3f00) + ISO_DATA_HEADER_SIZE;
            type = HCI_SNOOP_TYPE_TX_ISO;
        }
        else if (cur[0] == HCI_SCODATA_PKT) {
            LOG_WAN("type is sco = %d, don't send to driver, discard data!\n", HCI_SCODATA_PKT);
            return 0;
        }
        else
            return cur - buf;

        if (cur + pkt_len > (buf + total_len))
            return cur - buf;

#if SUPPORT_BLE_AUDIO_BY_ADD_ACL_HEADER
        if (cur[0] == HCI_ISO_PKT)
        {
            /* Add a ACL header for kernel 4.x:
           * Old format: 05 xx xx xx xx
           * New format: 02 yy yy yy yy xx xx xx xx
           */
            pkt_len += ACL_DATA_HEADER_SIZE - 1;
            cur -= (ACL_DATA_HEADER_SIZE - 1);
            cur[0] = HCI_ACLDATA_PKT;
            cur[1] = 0x00;
            cur[2] = 0x44;
            cur[3] = ((pkt_len - ACL_DATA_HEADER_SIZE) & 0x00ff);
            cur[4] = ((pkt_len - ACL_DATA_HEADER_SIZE) >> 8);
        }
#endif
        next = cur + pkt_len;
        btmtk_hci_snoop_save(type, cur, pkt_len);
        if (cur[0] == HCI_COMMAND_PKT && cur[1] == 0x5B && cur[2] == 0xFD)
            btmtk_hci_snoop_print();
        write_len = mtk_write_data(fd, cur, pkt_len);
        if (write_len <= 0) {
            LOG_ERR("Write data to controller failed\n");
            return -EIO;
        }
    }
    return 0;
}

static void mtk_close_sock_fd(void)
{
    if (bt_fd >= 0) {
        LOG_WAN("close bt_fd = %d!\n", bt_fd);
        close(bt_fd);
        bt_fd = -1;
        sock_fd[FD_FOR_STACK] = -1;
    }

    if (sock_fd[FD_FOR_VENDOR] >= 0) {
        LOG_WAN("close sock_fd[VENDOR] = %d!\n", sock_fd[FD_FOR_VENDOR]);
        close(sock_fd[FD_FOR_VENDOR]);
        sock_fd[FD_FOR_VENDOR] = -1;
    }

    if (sock_fd[FD_FOR_DRV] >= 0) {
        LOG_WAN("close sock_fd[DRV] = %d!\n", sock_fd[FD_FOR_DRV]);
        close(sock_fd[FD_FOR_DRV]);
        sock_fd[FD_FOR_DRV] = -1;
    }
}

void mtk_vendor_lib_send_hw_err_to_host(int write_fd)
{
    uint8_t hwerr_event[] = { 0x04, 0x10, 0x01, HW_ERR_CODE_BROKEN_PIPE };
    ssize_t write_len = 0;

    LOG_WAN("Send hw err to host");
    write_len = mtk_write_data(write_fd, hwerr_event, sizeof(hwerr_event));
    if (write_len <= 0)
        LOG_ERR("Write hw err to host failed, write_len = %ld\n", write_len);
}

static void *mtk_vendor_lib_tx_thread_main(void *arg)
{
    int *fd = (int *)arg;
    uint8_t buf[HCI_BUF_SIZE] = {0};
    ssize_t read_len = 0;
    unsigned int head = 0, rd_pos = 0;
    int ret = 0, result;
    fd_set read_fds;
    struct timeval tv;
    struct sched_param rt_params;
    memset(&rt_params, 0, sizeof(rt_params));

    rt_params.sched_priority = 20;
    if (sched_setscheduler(gettid(), SCHED_FIFO, &rt_params)) {
#if defined(ENABLE_MUSL_LIBC) && (ENABLE_MUSL_LIBC == TRUE)
        LOG_ERR("unable to set SCHED_FIFO for pid %d, tid %p\n",
#else
        LOG_ERR("unable to set SCHED_FIFO for pid %d, tid %lx\n",
#endif
          getpid(), pthread_self());
    }

    prctl(PR_SET_NAME, "bt_vendor_tx", 0, 0, 0);

#if defined(ENABLE_MUSL_LIBC) && (ENABLE_MUSL_LIBC == TRUE)
    LOG_WAN("enter, fd[STACK] = %d, fd[VENDOR] = %d, fd[DRV] = %d, tx_quit = %d, tid %p\n",
#else
    LOG_WAN("enter, fd[STACK] = %d, fd[VENDOR] = %d, fd[DRV] = %d, tx_quit = %d, tid %lx\n",
#endif
        fd[FD_FOR_STACK], fd[FD_FOR_VENDOR], fd[FD_FOR_DRV], mtk_is_tx_thread_quit(), pthread_self());
#if SUPPORT_BLE_AUDIO_BY_ADD_ACL_HEADER
    head = ISO_DATA_HEADER_SIZE;
#endif
    rd_pos = head;

    /* Process the data from BT host, BT host write to fd0, vendor lib read from  fd1, write to fd2 */
    while (!mtk_is_tx_thread_quit() && fd[FD_FOR_VENDOR] >= 0 && fd[FD_FOR_DRV] >= 0) {
        FD_ZERO(&read_fds);
        FD_SET(fd[FD_FOR_VENDOR], &read_fds);

        /* select() will clear tv, it must be re-config */
        tv.tv_sec = 0; /* SECOND */
        tv.tv_usec = 200000; /* USECOND */
        result = select(fd[FD_FOR_VENDOR] + 1, &read_fds, NULL, NULL, &tv);
        if (result == 0) {
            continue;
        }

        if (result < 0) {
            LOG_WARNNO(errno, "select failed ");
            continue;
        }

        if (FD_ISSET(fd[FD_FOR_VENDOR], &read_fds)) {
            memset(&buf[rd_pos], 0, sizeof(buf) - rd_pos);
            read_len = mtk_read_data(fd[FD_FOR_VENDOR], &buf[rd_pos], sizeof(buf) - rd_pos);
            if (read_len <= 0) {
                LOG_ERR("Read data from host failed, read_len = %ld\n", read_len);
                break;
            }

            ret = mtk_unpack_and_write_host_data(fd[FD_FOR_DRV], &buf[head], read_len + (rd_pos - head));
            if (ret < 0) {
                LOG_ERR("Write data to controller failed\n");
                /* rd_pos = head; */
                break;
            } else if (ret > 0) {
                memcpy(&buf[head], &buf[head + ret], sizeof(buf) - (head + ret));
                rd_pos = sizeof(buf) - ret;
            } else {
                rd_pos = head;
            }
        }
    }

    mtk_tx_thread = -1;
    LOG_WAN("exit\n");
    return NULL;
}

static void *mtk_vendor_lib_rx_thread_main(void *arg)
{
    int *fd = (int *)arg;
    uint8_t buf[HCI_BUF_SIZE] = {0};
    ssize_t read_len = 0;
    ssize_t write_len = 0;
    ssize_t data_len = 0;
    // Make watching thread RT.
    struct sched_param rt_params;
    memset(&rt_params, 0, sizeof(rt_params));
    int result;
    struct timeval tv;
    unsigned int type = 0;

    rt_params.sched_priority = 20;
    if (sched_setscheduler(gettid(), SCHED_FIFO, &rt_params)) {
#if defined(ENABLE_MUSL_LIBC) && (ENABLE_MUSL_LIBC == TRUE)
        LOG_ERR("unable to set SCHED_FIFO for pid %d, tid %p\n",
#else
        LOG_ERR("unable to set SCHED_FIFO for pid %d, tid %lx\n",
#endif
          getpid(), pthread_self());
    }

    prctl(PR_SET_NAME, "bt_vendor_rx", 0, 0, 0);
#if defined(ENABLE_MUSL_LIBC) && (ENABLE_MUSL_LIBC == TRUE)
    LOG_WAN("enter, fd[STACK] = %d, fd[VENDOR] = %d, fd[DRV] = %d, rx_quit = %d, tid %p\n",
#else
    LOG_WAN("enter, fd[STACK] = %d, fd[VENDOR] = %d, fd[DRV] = %d, rx_quit = %d, tid %lx\n",
#endif
        fd[FD_FOR_STACK], fd[FD_FOR_VENDOR], fd[FD_FOR_DRV], mtk_is_rx_thread_quit(), pthread_self());

    /* Process the data from BT controller, vendor lib read from fd2, write to fd1, BT host read from fd0 */
    while (!mtk_is_rx_thread_quit() && fd[FD_FOR_VENDOR] >= 0 && fd[FD_FOR_DRV] >= 0) {
        fd_set read_fds;
        FD_ZERO(&read_fds);
        memset(buf, 0, sizeof(buf));

        if (fd[FD_FOR_DRV] >= 0)
            FD_SET(fd[FD_FOR_DRV], &read_fds);

        tv.tv_sec = 0; /* SECOND */
        tv.tv_usec = 200000; /* USECOND */
        result = select(fd[FD_FOR_DRV] + 1, &read_fds, NULL, NULL, &tv);
        if (result == 0) {
            //LOG_WAN("select timeout\n");
            continue;
        }

        if (result < 0) {
            LOG_WARNNO(errno, "select failed ");
            continue;
        }

        if (FD_ISSET(fd[FD_FOR_DRV], &read_fds)) {
            read_len = mtk_read_data(fd[FD_FOR_DRV], buf, HCI_BUF_SIZE);
            if (read_len == 0) {
                LOG_WAN("read_len = %ld\n", read_len);
                continue;
            } else if (read_len < 0) {
                LOG_ERR("Read data from controller failed, read_len:%ld\n", read_len);
                if (read_len == -EPIPE) {
                    //mtk_vendor_lib_process_reset();
                    mtk_vendor_lib_send_hw_err_to_host(fd[FD_FOR_VENDOR]);
                }
                break;
            }

            if (buf[0] == HCI_EVENT_PKT) {
                if (buf[1] == 0x3E)
                    type = HCI_SNOOP_TYPE_ADV_EVT;
                else if (buf[1] == 0x13)
                    type = HCI_SNOOP_TYPE_NOCP_EVT;
                else
                    type = HCI_SNOOP_TYPE_EVT;
            } else if (buf[0] == HCI_ACLDATA_PKT) {
                type = HCI_SNOOP_TYPE_RX_ACL;

#if SUPPORT_BLE_AUDIO_BY_ADD_ACL_HEADER
                if (buf[1] == 0x00 && buf[2] == 0x44) {
                    /* it's for ble iso, remove speicific header
                     * 02 00 44 len len + payload to 05 + payload
                     */
                    data_len = buf[3] + ((buf[4] << 8) & 0xff00);
                    buf[0] = HCI_ISO_PKT;
                    memcpy(&buf[1], &buf[5], data_len);
                    memset(&buf[0] + data_len + 1, 0, HCI_BUF_SIZE - data_len - 1);
                    read_len = data_len + 1;
                    type = HCI_SNOOP_TYPE_RX_ISO;
                }
#endif
            } else if (buf[0] == HCI_ISO_PKT) {
                type = HCI_SNOOP_TYPE_RX_ISO;
            } else if (buf[0] == HCI_SCODATA_PKT) {
                LOG_WAN("type is sco = %d, don't send to host, discard data!\n", HCI_SCODATA_PKT);
                continue;
            } else {
                SHOW_RAW(buf, read_len, "type is invalid, len = %ld - \n", read_len);
                break;
            }

            btmtk_hci_snoop_save(type, buf, read_len);
            write_len = mtk_write_data(fd[FD_FOR_VENDOR], buf, read_len);
            if (write_len <= 0) {
                LOG_ERR("Write data to host failed\n");
                break;
            }
        }
    }

    mtk_rx_thread = -1;
    LOG_WAN("exit\n");
    return NULL;
}

int mtk_vendor_lib_setsockopt(void)
{
    struct timeval tv;
    int get_size = 0;
    socklen_t sock_len = sizeof(get_size);
    int i = 0;
    int ret = -1;

    tv.tv_sec = 1; /* SECOND */
    tv.tv_usec = 0; /* USECOND */

    for (i = 0; i < FD_NUM; i++) {

        /*getsockopt socket buffer size*/
        ret = getsockopt(sock_fd[i], SOL_SOCKET, SO_RCVBUF, (char*)&get_size, &sock_len);
        if (ret < 0) {
            LOG_ERRORNO(errno, "setsockopt sock_fd[%d] SO_RCVBUF failed ", i);
            return ret;
        }
        LOG_WAN("getsockopt sock_fd[%d] SO_RCVBUF get_size (%d)\n", i, get_size / 1024);

        ret = getsockopt(sock_fd[i], SOL_SOCKET, SO_SNDBUF, (char*)&get_size, &sock_len);
        if (ret < 0) {
            LOG_ERRORNO(errno, "setsockopt sock_fd[%d] SO_SNDBUF failed ", i);
            return ret;
        }
        LOG_WAN("getsockopt sock_fd[%d] SO_SNDBUF get_size (%d)\n", i, get_size / 1024);

        ret = setsockopt(sock_fd[i], SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));
        if (ret < 0) {
            LOG_ERRORNO(errno, "setsockopt sock_fd[%d] SO_RCVTIMEO faild ", i);
            return ret;
        }

        ret = setsockopt(sock_fd[i], SOL_SOCKET, SO_SNDTIMEO, &tv, sizeof(tv));
        if (ret < 0) {
            LOG_ERRORNO(errno, "setsockopt sock_fd[%d] SO_SNDTIMEO failed ", i);
            return ret;
        }
    }

    return ret;
}

static int open_stpbt(void)
{
    return open(CUST_BT_SERIAL_PORT, O_RDWR | O_NOCTTY | O_NONBLOCK | O_CLOEXEC);
}

static int open_hci(void)
{
    struct sockaddr_hci addr;
    pthread_attr_t attr;
    int ret = 0;

    /* create socket to comunicate with kernel hci driver*/
    sock_fd[FD_FOR_DRV] = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
    if (sock_fd[FD_FOR_DRV] < 0) {
        LOG_ERRORNO(errno, "socket create error : ");
        return -1;
    }

    memset(&addr, 0, sizeof(addr));
    addr.hci_family = AF_BLUETOOTH;
    addr.hci_dev = 0;
    addr.hci_channel = HCI_CHANNEL_USER;
    if(bind(sock_fd[FD_FOR_DRV], (struct sockaddr *) &addr, sizeof(addr)) < 0) {
        LOG_ERRORNO(errno, "bind error ");
        close(sock_fd[FD_FOR_DRV]);
        return -1;
    }

    /*create socketpair*/
    ret = socketpair(AF_LOCAL, SOCK_STREAM, 0, sock_fd);
    if (ret == -1) {
        LOG_ERRORNO(errno, "create socketpair fail: ");
        close(sock_fd[FD_FOR_DRV]);
        return -1;
    }

    rx_quit = 0;
    tx_quit = 0;
    LOG_WAN("create socket start, fd[STACK] = %d, fd[VENDOR] = %d, fd[DRV] = %d\n",
            sock_fd[FD_FOR_STACK], sock_fd[FD_FOR_VENDOR], sock_fd[FD_FOR_DRV]);

    ret = mtk_vendor_lib_setsockopt();
    if (ret < 0) {
        LOG_ERR("mtk_vendor_lib_setsockopt error, ret(%d)\n", ret);
        mtk_close_sock_fd();
        return -1;
    }

    (void)pthread_mutex_init(&tx_mutex, NULL);
    (void)pthread_mutex_init(&rx_mutex, NULL);

    /* create thread to read/write from/to sockpair*/
    (void)pthread_attr_init(&attr);
    (void)pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    ret = pthread_create(&mtk_tx_thread, &attr, mtk_vendor_lib_tx_thread_main, (void *)sock_fd);
    if (ret != 0) {
        LOG_ERRORNO(errno, "pthread_create tx thread fail : ");
        mtk_close_sock_fd();
        return -1;
    }
    LOG_WAN("create mtk_vendor_lib_tx_thread_main success!\n");

    ret = pthread_create(&mtk_rx_thread, &attr, mtk_vendor_lib_rx_thread_main, (void *)sock_fd);
    if (ret != 0) {
        LOG_ERRORNO(errno, "pthread_create rx thread fail : ");
        mtk_close_sock_fd();
        return -1;
    }
    LOG_WAN("create mtk_vendor_lib_rx_thread_main success!\n");
    return sock_fd[FD_FOR_STACK];
}

/* Initialize UART port */
int init_uart(void)
{
    int retry = RETRY_COUNT;

    LOG_TRC();
    if (bt_fd >= 0) {
        LOG_WAN("Previous serial port is not closed\n");
        close_uart();
    }

    while (retry > 0) {
        bt_fd = open_hci();
        if (bt_fd >= 0) {
            LOG_WAN("open hci success");
            return bt_fd;
        }

        bt_fd = open_stpbt();
        if (bt_fd >= 0) {
            LOG_WAN("open stpbt success");
            return bt_fd;
        }

        LOG_WAN("Can't open DRV port, Retry...errno = %d\n", errno);
        (void)usleep(200000);/*200ms*/
        retry--;
    }

    return -1;
}

/* Close UART port previously opened */
void close_uart(void)
{
    LOG_TRC();
    mtk_set_tx_thread_quit();
    mtk_set_rx_thread_quit();

    while (1) {
        if (mtk_tx_thread != -1 || mtk_rx_thread != -1) {
#if defined(ENABLE_MUSL_LIBC) && (ENABLE_MUSL_LIBC == TRUE)
            LOG_WAN("wait tx/rx thread exit, mtk_tx_thread = %p, mtk_rx_thread = %p!\n",
#else
            LOG_WAN("wait tx/rx thread exit, mtk_tx_thread = %lx, mtk_rx_thread = %lx!\n",
#endif
                mtk_tx_thread, mtk_rx_thread);
            (void)usleep(200000);
        }
        else
            break;
    }

    mtk_close_sock_fd();
    (void)pthread_mutex_destroy(&tx_mutex);
    (void)pthread_mutex_destroy(&rx_mutex);
}

#ifdef MTK_LINUX_FETCH_ADDRESS
/** callback function for xmit_cb() */
static VOID xmit_complete_cb(VOID *p_evt)
{
#define HCE_COMMAND_COMPLETE 0x0E
    HC_BT_HDR *p_buf = (HC_BT_HDR *)p_evt;
    uint8_t event = 0;
    uint8_t len = 0;
    uint16_t opcode = 0;
    uint8_t status = 0;

    if (p_buf == NULL) {
        LOG_ERR("Incorrect parameter - p_evt!!!");
        return;
    }

    LOG_TRC();
    if (p_buf->data[0] != HCE_COMMAND_COMPLETE) {
        int i = 0;

        for (i = 0; i < p_buf->len; i++)
            LOG_WAN("p_buf[%d] = %02X", i, p_buf->data[i]);
        return;
    }

    // Expect this is command complete event
    event = p_buf->data[0];
    len = p_buf->data[1];
    opcode = *(uint16_t *)&p_buf->data[3];
    status = p_buf->data[5];
    LOG_DBG("Command_Complete OPCode: %04X, LEN: %02X, Status: %02X",
            opcode, len, status);
    return;
}

/** Set Bluetooth local address */
static bool bd_set_local_bdaddr(uint8_t *addr)
{
#define OPCODE_LEN 2
#define PARAM_SIZE_LEN 1
#define BD_ADDR_LEN 6
    uint16_t opcode = HCI_SET_BD_ADDRESS_OP_CODE;
    HC_BT_HDR *p_buf = NULL;
    uint8_t *p = NULL;
    int i = 0;

    if (bt_vnd_cbacks == NULL) {
        LOG_ERR("No HIDL interface callbacks!!!");
        return false;
    }
    if (addr == NULL) {
        LOG_ERR("No BD address!!!");
        return false;
    }
    LOG_TRC();

    p_buf = (HC_BT_HDR *)bt_vnd_cbacks->alloc(BT_HC_HDR_SIZE + OPCODE_LEN
            + PARAM_SIZE_LEN + BD_ADDR_LEN);
    if (p_buf == NULL) {
        LOG_ERR("Allocation fail!!!");
        return false;
    }

    p_buf->event = MSG_STACK_TO_HC_HCI_CMD;
    p_buf->len = OPCODE_LEN + PARAM_SIZE_LEN + BD_ADDR_LEN;
    p_buf->offset = 0;
    p_buf->layer_specific = 0;

    p = (uint8_t *)(p_buf + 1);
    memcpy(p, &opcode, OPCODE_LEN); // opcode
    p += 2;
    *p++ = 6; // len
    // reverse the address from host
    *p++ = addr[5];
    *p++ = addr[4];
    *p++ = addr[3];
    *p++ = addr[2];
    *p++ = addr[1];
    *p++ = addr[0];

    bt_vnd_cbacks->xmit_cb(opcode, p_buf, xmit_complete_cb);
    // p_buf will free by xmit_cb.
    return true;
}
#endif

/* MTK specific chip initialize process */
#ifdef MTK_LINUX_FETCH_ADDRESS
int vendor_fw_cfg(uint8_t *bdaddr)
#else
int vendor_fw_cfg(void)
#endif
{
#ifdef MTK_LINUX_FETCH_ADDRESS
    bd_set_local_bdaddr(bdaddr);
#endif
    LOG_TRC();
    if (bt_vnd_cbacks) {
        LOG_ERR("bt_vnd_cbacks fwcfg_cb done\n");
        bt_vnd_cbacks->fwcfg_cb(BT_VND_OP_RESULT_SUCCESS);
    }
    return 0;
}

/* MTK specific SCO/PCM configuration */
int vendor_sco_cfg(void)
{
    LOG_TRC();
    if (bt_vnd_cbacks) {
        LOG_ERR("bt_vnd_cbacks scocfg_cb done\n");
        bt_vnd_cbacks->scocfg_cb(BT_VND_OP_RESULT_SUCCESS);
    }
    return 0;
}

void vendor_op_lmp_set_mode(void)
{
    if (bt_vnd_cbacks) {
        bt_vnd_cbacks->lpm_cb(BT_VND_OP_RESULT_SUCCESS);
    }
}

/* MTK specific deinitialize process */
int mtk_prepare_off(void)
{
    /*
    * On KK, BlueDroid adds BT_VND_OP_EPILOG procedure when BT disable:
    *   - 1. BT_VND_OP_EPILOG;
    *   - 2. In vendor epilog_cb, send EXIT event to bt_hc_worker_thread;
    *   - 3. Wait for bt_hc_worker_thread exit;
    *   - 4. userial close;
    *   - 5. vendor cleanup;
    *   - 6. Set power off.
    * On L, the disable flow is modified as below:
    *   - 1. userial Rx thread exit;
    *   - 2. BT_VND_OP_EPILOG;
    *   - 3. Write reactor->event_fd to trigger bt_hc_worker_thread exit
    *        (not wait to vendor epilog_cb and do nothing in epilog_cb);
    *   - 4. Wait for bt_hc_worker_thread exit;
    *   - 5. userial close;
    *   - 6. Set power off;
    *   - 7. vendor cleanup.
    *
    * It seems BlueDroid does not expect Tx/Rx interaction with chip during
    * BT_VND_OP_EPILOG procedure, and also does not need to do it in a new
    * thread context (NE may occur in __pthread_start if bt_hc_worker_thread
    * has already exited).
    * So BT_VND_OP_EPILOG procedure may be not for chip deinitialization,
    * do nothing, just notify success.
    *
    * [FIXME!!]How to do if chip deinit is needed?
    */
    //return (BT_DeinitDevice() == TRUE ? 0 : -1);
    if (bt_vnd_cbacks) {
        bt_vnd_cbacks->epilog_cb(BT_VND_OP_RESULT_SUCCESS);
    }
    return 0;
}

/* Cleanup driver resources, e.g thread exit */
void clean_resource(void)
{
    BT_Cleanup();
}

#if (defined(MTK_VENDOR_OPCODE) && (MTK_VENDOR_OPCODE == TRUE))
void mtk_enable_btsysclk(void *param)
{
    if (ioctl(bt_fd, BTMTK_IOCTL_STEREO_SET_PARA, param) < 0) {
        LOG_ERRORNO(errno, "Set mtk_enable_btsysclk error, ");
        return;
    }
}

void mtk_read_btsysclk(void *param)
{
    if (ioctl(bt_fd, BTMTK_IOCTL_STEREO_GET_CLK, param) < 0) {
        LOG_ERRORNO(errno, "Set mtk_read_btsysclk error, ");
        return;
    }
}
#endif

VOID BT_Cleanup(VOID)
{
    if (SIG_ERR == signal(SIGRTMIN, SIG_DFL)) {
        LOG_ERR("Restore signal handler fails errno(%d)\n", errno);
    }
    return;
}

