#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/skbuff.h>
#include <linux/usb.h>
#include <linux/cdev.h>
#include <linux/ioctl.h>
#include <linux/io.h>
#include <linux/firmware.h>
#include <linux/vmalloc.h>
#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/reboot.h>
#include <linux/time.h>
#include <linux/kthread.h>
#include <linux/semaphore.h>
#include <linux/poll.h>
#include <linux/platform_device.h>

#include "bt_fucode_w2.h"
#include "aml_bt_usb_w2.h"

//static int reg_config_complete = 0;
static int amlbt_sdio_major;
static struct cdev amlbt_sdio_cdev;
static int amlbt_sdio_devs = 1;
static struct class *amlbt_sdio_class;
static struct device *amlbt_sdio_dev;
static unsigned int rf_num = -1;


#define AML_BT_VERSION  (0x20230202)
#define AML_BT_PRODUCTION_TOOLS
#define CONFIG_BLUEDROID        1 /* bleuz 0, bluedroid 1 */
#define INDEPENDENT_USB			0

#define AML_USB_DEBUG			0
#define AML_RW_DEBUG 			0
#define AML_BT_RW_DEBUG         0
#define AML_BT_ROM_CHECK        0
#define REG_DEV_RESET           0xf03058
#define REG_PMU_POWER_CFG       0xf03040
#define	REG_RAM_PD_SHUTDWONW_SW 0xf03050

#define BIT_PHY                 1
#define BIT_MAC                 (1 << 1)
#define BIT_CPU                 (1 << 2)
#define DEV_RESET_SW            16
#define DEV_RESET_HW            0
#define BIT_RF_NUM              28

#if AML_USB_DEBUG
    #define AMLBT_DBG(fmt, arg...) printk(KERN_INFO "aml_btusb: " fmt "\n" , ## arg)
#else
    #define AMLBT_DBG(fmt, arg...)
#endif

#if AML_RW_DEBUG
    #define AMLRW_DBG(fmt, arg...) printk(KERN_INFO "aml_btusb: " fmt "\n" , ## arg)
#else
    #define AMLRW_DBG(fmt, arg...)
#endif

#define FAMLIY_TYPE_IS_W1(x)        ((AMLBT_PD_ID_FAMILY & x) == AMLBT_FAMLIY_W1)
#define FAMLIY_TYPE_IS_W1U(x)       ((AMLBT_PD_ID_FAMILY & x) == AMLBT_FAMLIY_W1U)
#define FAMLIY_TYPE_IS_W2(x)        ((AMLBT_PD_ID_FAMILY & x) == AMLBT_FAMLIY_W2)
#define INTF_TYPE_IS_SDIO(x)        ((AMLBT_PD_ID_INTF & x) == AMLBT_INTF_SDIO)
#define INTF_TYPE_IS_USB(x)         ((AMLBT_PD_ID_INTF & x) == AMLBT_INTF_USB)
#define INTF_TYPE_IS_PCIE(x)        ((AMLBT_PD_ID_INTF & x) == AMLBT_INTF_PCIE)

#define TYPE_RETURN_STR(type) \
    case type:                \
    return #type;

const char *amlbt_famliy_intf(int type)
{
    switch (type)
    {
            TYPE_RETURN_STR(AMLBT_FAMLIY_W1)
            TYPE_RETURN_STR(AMLBT_FAMLIY_W1U)
            TYPE_RETURN_STR(AMLBT_FAMLIY_W2)
            TYPE_RETURN_STR(AMLBT_INTF_SDIO)
            TYPE_RETURN_STR(AMLBT_INTF_USB)
            TYPE_RETURN_STR(AMLBT_INTF_PCIE)
        default:
            break;
    }

    return "unknow type";
}

static unsigned int amlbt_if_type = AMLBT_TRANS_UNKNOW;
static unsigned char *usb_buf = NULL;
static unsigned char *sdio_buf = NULL;
static int evt_state = 0;
static unsigned char *type = NULL;
static unsigned char *p_acl_buf = NULL;
static unsigned char download_fw = 0;
static unsigned char download_flag = 0;
static unsigned char download_end = 0;
static unsigned int iccm_base_addr = 0;
static unsigned int dccm_base_addr = 0;
static unsigned int close_state = 0;

static unsigned char cmd[4][2] = {{0xf2, 0xfe}, {0xf1, 0xfe}, {0xf0, 0xfe}, {0xf3, 0xfe}};
static unsigned char cmd_cpt[64] = {0x04, 0x0e, 0x04, 0x01, 0x98, 0xfc, 0x00};
static volatile unsigned char cmd_index = 0xff;
static unsigned char dw_state = 0;
static unsigned char *read_buff = NULL;
static unsigned char *type_buff = NULL;

extern struct auc_hif_ops g_auc_hif_ops;
extern struct usb_device *g_udev;
extern int auc_send_cmd_ep1(unsigned int addr, unsigned int len);

struct completion usb_completion;
static unsigned int fw_cmd_w = 0;
static unsigned int fw_cmd_r = 0;


#define ICCM_RAM_BASE           (0x000000)
#define DCCM_RAM_BASE           (0xd00000)

#define CMD_FIFO_SIZE           128
#define DATA_FIFO_SIZE          16*1024
#define EVT_FIFO_SIZE           4*1024
#define TYPE_FIFO_SIZE          1024

static struct task_struct *check_fw_rx_stask = NULL;
static struct semaphore read_rx_sem;
static int fw_data_flag;

static dev_t bt_devid; /* bt char device number */
static struct cdev bt_char_dev; /* bt character device structure */
static struct class *bt_char_class; /* device class for usb char driver */

static gdsl_fifo_t *g_cmd_fifo = 0;
static gdsl_fifo_t *g_event_fifo = 0;
static gdsl_fifo_t *g_rx_fifo = 0;
static gdsl_tx_q_t *g_tx_q = 0;
static gdsl_fifo_t *g_rx_type_fifo = 0;
static gdsl_fifo_t *g_fw_data_fifo = 0;
static gdsl_fifo_t *g_fw_evt_fifo = 0;
static gdsl_fifo_t *g_fw_type_fifo = 0;
static gdsl_fifo_t *g_lib_cmd_fifo = 0;

static unsigned char *g_lib_cmd_buff = NULL;
static unsigned char *g_fw_data_buf = NULL;
static unsigned char *g_fw_evt_buf  = NULL;

static unsigned int gdsl_fifo_is_valid(gdsl_fifo_t *p_fifo, unsigned int len);
static int amlbt_usb_check_fw_rx(void *data);
static ssize_t amlbt_usb_read_fw_data(void);

DECLARE_WAIT_QUEUE_HEAD(poll_amlbt_queue);

void amlbt_buff_init(void)
{
    unsigned int i = 0;
    unsigned int buff_size = 0;
    unsigned int index = 0;
    unsigned int buf_size[] =
    {
        TYPE_FIFO_SIZE,             //type
        HCI_MAX_FRAME_SIZE,         //p_acl_buf
        HI_USB_RX_Q_LEN,            //read_buff
        HI_USB_RX_TYPE_FIFO_LEN,    //type_buff
        DATA_FIFO_SIZE,             //hci data buff
        EVT_FIFO_SIZE,              //hci event buff
        CMD_FIFO_SIZE,              //g_lib_cmd_buff
    };

    for (i = 0; i < sizeof(buf_size)/sizeof(buf_size[0]); i++)
    {
        buff_size += buf_size[i];
    }
    printk("%s size %d, usb_buf %#x \n", __func__, buff_size, (unsigned long)usb_buf);
    if (usb_buf == NULL)
    {
        usb_buf = kzalloc(buff_size, GFP_DMA|GFP_ATOMIC);
        if (usb_buf == NULL)
        {
            printk("%s kzalloc falied! \n", __func__);
            return ;
        }
        type        = &usb_buf[index];
        index       += buf_size[0];
        p_acl_buf   = &usb_buf[index];
        index       += buf_size[1];
        read_buff   = &usb_buf[index];
        index       += buf_size[2];
        type_buff   = &usb_buf[index];
        index       += buf_size[3];
        g_fw_data_buf = &usb_buf[index];
        index       += buf_size[4];
        g_fw_evt_buf = &usb_buf[index];
        index       += buf_size[5];
        g_lib_cmd_buff = &usb_buf[index];
    }
}

void amlbt_buff_deinit(void)
{
    kfree(usb_buf);
    usb_buf         = NULL;
    type            = NULL;
    p_acl_buf       = NULL;
    read_buff       = NULL;
    type_buff       = NULL;
    g_fw_data_buf   = NULL;
    g_fw_evt_buf    = NULL;
    g_lib_cmd_buff  = NULL;
}

static gdsl_fifo_t *gdsl_fifo_init(unsigned int len, unsigned char *base_addr)
{
    gdsl_fifo_t *p_fifo = (gdsl_fifo_t *)kzalloc(sizeof(gdsl_fifo_t), GFP_DMA|GFP_ATOMIC);

    AMLBT_DBG("%s \n", __func__);

    if (p_fifo)
    {
        memset(p_fifo, 0, sizeof(gdsl_fifo_t));
        p_fifo->w = 0;
        p_fifo->r = 0;
        p_fifo->base_addr = base_addr;
        p_fifo->size = len;
    }

    return p_fifo;
}

static void gdsl_fifo_deinit(gdsl_fifo_t *p_fifo)
{
    if (p_fifo == NULL)
    {
        return ;
    }

    kfree(p_fifo);
}

unsigned int gdsl_fifo_is_valid(gdsl_fifo_t *p_fifo, unsigned int len)
{
    unsigned int unused = 0;

    AMLBT_DBG("func gdsl_fifo_is_valid \n");
    AMLBT_DBG("p_fifo->w:%#lx , p_fifo->r:%#lx\n", (unsigned long)p_fifo->w, (unsigned long)p_fifo->r);

    if (p_fifo->w > p_fifo->r)
    {
        unused = p_fifo->size - (p_fifo->w - p_fifo->r) - 4;
    }
    else if (p_fifo->w == p_fifo->r)
    {
        unused = p_fifo->size - 4;
    }
    else
    {
        unused = p_fifo->r - p_fifo->w - 4;
    }

    AMLBT_DBG("unused:%d, len:%d\n", unused, len);

    return (unused >= len) ? GDSL_ERR_SPACE_VALID : GDSL_ERR_SPACE_INVALID;
}

unsigned int gdsl_fifo_copy_data(gdsl_fifo_t *p_fifo, unsigned char *data, unsigned int len)
{
    unsigned int ret = 0;
    unsigned int offset = 0;

    //len = ((len + 3) & 0xFFFFFFFC);
    ret = gdsl_fifo_is_valid(p_fifo, len);

    if (ret == GDSL_ERR_SPACE_VALID)
    {
        offset = p_fifo->size - (unsigned int)(p_fifo->w - p_fifo->base_addr);
        if (len < offset)
        {
            memcpy(p_fifo->w, data, len);
            p_fifo->w += len;
        }
        else
        {
            memcpy(p_fifo->w, data, offset);
            p_fifo->w = p_fifo->base_addr;
            memcpy(p_fifo->w, &data[offset], (len - offset));
            p_fifo->w += (len - offset);
        }
    }
    else
    {
        printk("drv_gdsl_send_data no space!!!\n");
    }
    return ret;
}

unsigned int gdsl_fifo_write_data_by_ep(gdsl_fifo_t *p_fifo, unsigned char *data, unsigned int len, unsigned int ep)
{
    unsigned int ret = 0;
    unsigned int offset = 0;

    AMLBT_DBG("%s len:%d\n", __func__, len);

    len = ((len + 3) & 0xFFFFFFFC);
    ret = gdsl_fifo_is_valid(p_fifo, len);

    if (ret == GDSL_ERR_SPACE_VALID)
    {
        offset = p_fifo->size - (unsigned int)(p_fifo->w - p_fifo->base_addr);
        if (len < offset)
        {
            g_auc_hif_ops.hi_write_sram_for_bt(data, p_fifo->w, len, ep);
            p_fifo->w += len;
        }
        else
        {
            g_auc_hif_ops.hi_write_sram_for_bt(data, p_fifo->w, offset, ep);
            p_fifo->w = p_fifo->base_addr;
            g_auc_hif_ops.hi_write_sram_for_bt(&data[offset], p_fifo->w, (len - offset), ep);
            p_fifo->w += (len - offset);
        }
    }
    else
    {
        printk("gdsl_send_data no space!!!\n");
    }
    return ret;
}


unsigned int gdsl_fifo_used_size(gdsl_fifo_t *p_fifo)
{
    if (p_fifo->w == p_fifo->r)
    {
        return 0;
    }

    if (p_fifo->w > p_fifo->r)
    {
        return (p_fifo->w - p_fifo->r);
    }

    if (p_fifo->w < p_fifo->r)
    {
        return (p_fifo->size - (p_fifo->r - p_fifo->w));
    }

    return 0;
}

#if 0
unsigned int gdsl_fifo_update_w(gdsl_fifo_t *p_fifo, unsigned int len)
{
    unsigned int ret = 0;
    unsigned int offset = 0;

    AMLBT_DBG("%s len:%d\n", __func__, len);

    len = ((len + 3) & 0xFFFFFFFC);
    ret = gdsl_fifo_is_valid(p_fifo, len);

    if (ret == GDSL_ERR_SPACE_VALID)
    {
        offset = p_fifo->size - (unsigned int)(p_fifo->w - p_fifo->base_addr);
        if (len < offset)
        {
            p_fifo->w += len;
        }
        else
        {
            p_fifo->w = p_fifo->base_addr;
            p_fifo->w += (len - offset);
        }
    }
    else
    {
        printk("gdsl_fifo_update_w gdsl_send_data no space!!!\n");
    }
    return ret;
}

#endif

unsigned int gdsl_fifo_update_r(gdsl_fifo_t *p_fifo, unsigned int len)
{
    unsigned int offset = 0;
    unsigned int read_len = 0;
    unsigned char *p_end = 0;

    //printk("%s p_fifo->w %#x, p_fifo->r %#x\n", __func__, (unsigned long)p_fifo->w, (unsigned long)p_fifo->r);
    //printk("%s len %d\n", __func__, len);

    if (p_fifo->w == p_fifo->r)
    {
        printk("%s no data!!!\n", __func__);
        return 0;
    }

    if (p_fifo->w > p_fifo->r)
    {
        read_len = (unsigned int)(p_fifo->w - p_fifo->r);
        if (len <= read_len)
        {
            read_len = len;
        }
        //printk("%s read len A %d\n", __func__, read_len);
        p_fifo->r += read_len;
    }
    else
    {
        p_end = (p_fifo->base_addr + p_fifo->size);
        AMLBT_DBG("%s w %#x, r %#x\n", __func__, (unsigned long)p_fifo->w, (unsigned long)p_fifo->r);
        AMLBT_DBG("%s read p_end %#x\n", __func__, (unsigned long)p_end);
        offset = (unsigned int)(p_end - p_fifo->r);
        read_len = offset;
        if (len < offset)
        {
            p_fifo->r += len;
            read_len = len;
            AMLBT_DBG("%s 111 len %#x \n", __func__, len);
        }
        else
        {
            p_fifo->r = p_fifo->base_addr;
            read_len += (len - offset);
            p_fifo->r += (len - offset);
            AMLBT_DBG("%s 222 len %#x \n", __func__, len);
        }
        //printk("%s read len B %#x \n", __func__, read_len);
    }

    //printk("%s actual len %#x \n", __func__, read_len);

    return read_len;
}

unsigned int gdsl_fifo_get_data(gdsl_fifo_t *p_fifo, unsigned char *buff, unsigned int len)
{
    unsigned int offset = 0;
    unsigned int read_len = 0;
    unsigned char *p_end = 0;

    //printk("%s p_fifo->w %#x, p_fifo->r %#x\n", __func__, (unsigned long)p_fifo->w, (unsigned long)p_fifo->r);
    //printk("%s len %d\n", __func__, len);

    if (p_fifo->w == p_fifo->r)
    {
        printk("%s no data!!!\n", __func__);
        return 0;
    }

    if (p_fifo->w > p_fifo->r)
    {
        read_len = (unsigned int)(p_fifo->w - p_fifo->r);
        if (len <= read_len)
        {
            read_len = len;
        }
        //printk("%s read len A %d\n", __func__, read_len);
        memcpy(buff, p_fifo->r, read_len);
        p_fifo->r += read_len;
    }
    else
    {
        p_end = (p_fifo->base_addr + p_fifo->size);
        AMLBT_DBG("%s w %#x, r %#x\n", __func__, (unsigned long)p_fifo->w, (unsigned long)p_fifo->r);
        AMLBT_DBG("%s read p_end %#x\n", __func__, (unsigned long)p_end);
        offset = (unsigned int)(p_end - p_fifo->r);
        read_len = offset;
        if (len < offset)
        {
            memcpy(buff, p_fifo->r, len);
            p_fifo->r += len;
            read_len = len;
            AMLBT_DBG("%s 111 len %#x \n", __func__, len);
        }
        else
        {
            memcpy(buff, p_fifo->r, offset);
            p_fifo->r = p_fifo->base_addr;
            memcpy(&buff[offset], p_fifo->r, len - offset);
            read_len += (len - offset);
            p_fifo->r += (len - offset);
            AMLBT_DBG("%s 222 len %#x \n", __func__, len);
        }
        AMLBT_DBG("%s read len B %#x \n", __func__, read_len);
    }

    AMLBT_DBG("%s actual len %#x \n", __func__, read_len);

    return read_len;
}

unsigned int gdsl_read_data_by_ep(gdsl_fifo_t *p_fifo, unsigned char *data, unsigned int len, unsigned int ep)
{
    unsigned int offset = 0;
    unsigned int read_len = 0;
    unsigned int remain_len = 0;
    unsigned char *p_end = 0;

    AMLBT_DBG("%s p_fifo->w %#lx, p_fifo->r %#lx\n", __func__, (unsigned long)p_fifo->w, (unsigned long)p_fifo->r);
    AMLBT_DBG("%s len %d\n", __func__, len);

    if (p_fifo->w == p_fifo->r)
    {
        printk("%s no data!!!\n", __func__);
        return 0;
    }

    if (p_fifo->w > p_fifo->r)
    {
        read_len = (unsigned int)(p_fifo->w - p_fifo->r);
        if (len <= read_len)
        {
            read_len = len;
        }
        AMLBT_DBG("%s read len A %d\n", __func__, read_len);
        g_auc_hif_ops.hi_read_sram_for_bt(data, p_fifo->r, read_len, ep);
        p_fifo->r += read_len;
    }
    else
    {
        p_end = (p_fifo->base_addr + p_fifo->size);
        AMLBT_DBG("%s w %#x, r %#x\n", __func__, (unsigned long)p_fifo->w, (unsigned long)p_fifo->r);
        AMLBT_DBG("%s read p_end %#x\n", __func__, (unsigned long)p_end);
        offset = (unsigned int)(p_end - p_fifo->r);
        if (len < offset)
        {
            g_auc_hif_ops.hi_read_sram_for_bt(data, p_fifo->r, len, ep);
            p_fifo->r += len;
            read_len = len;
            AMLBT_DBG("%s 111 len %#x \n", __func__, len);
        }
        else
        {
            AMLBT_DBG("r %#x offset %#x\n", (unsigned long)p_fifo->r, offset);
            g_auc_hif_ops.hi_read_sram_for_bt(data, p_fifo->r, offset, ep);
            p_fifo->r = p_fifo->base_addr;
            read_len = offset;
            remain_len = (p_fifo->w - p_fifo->r);
            if (((len - offset) != 0) && (remain_len != 0))
            {
                if ((len - offset) > remain_len)
                {
                    AMLBT_DBG("r1 %#x len %#x\n", (unsigned long)p_fifo->r, remain_len);
                    g_auc_hif_ops.hi_read_sram_for_bt(&data[offset], p_fifo->r, remain_len, ep);
                    read_len += remain_len;
                    p_fifo->r += remain_len;
                }
                else
                {
                    AMLBT_DBG("r2 %#x len %#x\n", p_fifo->r, remain_len);
                    g_auc_hif_ops.hi_read_sram_for_bt(&data[offset], p_fifo->r, len - offset, ep);
                    read_len += (len - offset);
                    p_fifo->r += (len - offset);
                }
            }
            AMLBT_DBG("%s 222 len %#x \n", __func__, len);
        }
        AMLBT_DBG("%s read len B %#x \n", __func__, read_len);
    }

    AMLBT_DBG("%s actual len %#x \n", __func__, read_len);

    return read_len;
}


unsigned int amlbt_usb_get_tx_prio(gdsl_tx_q_t *p_fifo, unsigned int acl_handle)
{
    unsigned int prio = 0;
    unsigned int i = 0;
    unsigned int find = 0;

    for (i = 0; i < WF_SRAM_TX_Q_NUM; i++)
    {
        if (p_fifo[i].tx_q_dev_index == acl_handle && p_fifo[i].tx_q_status == GDSL_TX_Q_USED)
        {
            if (p_fifo[i].tx_q_prio >= prio)
            {
                prio = p_fifo[i].tx_q_prio;
                find = 1;
            }
        }
    }

    if (!find)
    {
        prio = 7;
    }

    return prio;
}

void amlbt_usb_update_tx_q(gdsl_tx_q_t *p_fifo)
{
    unsigned int i = 0, j = 0;
    unsigned int acl_handle = 0;
    unsigned int tx_q_status[WF_SRAM_TX_Q_NUM] = {0};
    //unsigned int changed = 0;
    //unsigned int tx_q_info[WF_SRAM_TX_Q_NUM * 3] = {0};

    //AMLBT_DBG("up tx\n");

    AMLBT_DBG("up q\n");

    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_read_sram_for_bt((unsigned char *)tx_q_status, (unsigned char *)p_fifo[0].tx_q_status_addr,
                                          sizeof(tx_q_status), USB_EP2);
    }
    AMLBT_DBG("[%#x,%#x,%#x,%#x,%#x,%#x,%#x,%#x]", tx_q_status[0], tx_q_status[1], tx_q_status[2],
              tx_q_status[3], tx_q_status[4], tx_q_status[5], tx_q_status[6], tx_q_status[7]);

    for (i = 0; i < WF_SRAM_TX_Q_NUM; i++)
    {
        //tx_q_status = g_auc_hif_ops.bt_hi_read_word((unsigned long)p_fifo[i].tx_q_status_addr);

        if (tx_q_status[i] == GDSL_TX_Q_COMPLETE)
        {
            acl_handle = p_fifo[i].tx_q_dev_index;
            AMLBT_DBG("up:%#x,%#x\n", i, p_fifo[i].tx_q_prio);
            p_fifo[i].tx_q_dev_index = 0;
            p_fifo[i].tx_q_status = GDSL_TX_Q_UNUSED;
            p_fifo[i].tx_q_prio = (WF_SRAM_TX_Q_NUM - 1);
            if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
            {
                g_auc_hif_ops.hi_write_word_for_bt((*(unsigned int*)(&g_tx_q[i].tx_q_dev_index_addr)), g_tx_q[i].tx_q_dev_index, USB_EP2);
                g_auc_hif_ops.hi_write_word_for_bt((*(unsigned int*)(&g_tx_q[i].tx_q_prio_addr)), g_tx_q[i].tx_q_prio, USB_EP2);
                g_auc_hif_ops.hi_write_word_for_bt((*(unsigned int*)(&g_tx_q[i].tx_q_status_addr)), g_tx_q[i].tx_q_status, USB_EP2);
            }

            for (j = 0; j < WF_SRAM_TX_Q_NUM; j++)
            {
                if (p_fifo[j].tx_q_dev_index == acl_handle)
                {
                    if (p_fifo[j].tx_q_status == GDSL_TX_Q_USED && p_fifo[j].tx_q_prio)
                    {
                        p_fifo[j].tx_q_prio--;
                        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
                        {
                            g_auc_hif_ops.hi_write_word_for_bt((*(unsigned int*)(&g_tx_q[j].tx_q_prio_addr)), g_tx_q[j].tx_q_prio, USB_EP2);
                        }
                        AMLBT_DBG("dec:%#x,%#x,%#x\n", j, p_fifo[j].tx_q_prio, p_fifo[j].tx_q_status);
                    }
                }
            }
        }
    }
}


unsigned int amlbt_usb_get_tx_q(gdsl_tx_q_t *p_fifo, unsigned int acl_handle)
{
    unsigned int prio = 0xff;
    unsigned int i = 0;
    //unsigned int find = 0;
    unsigned int index = WF_SRAM_TX_Q_NUM;

    for (i = 0; i < WF_SRAM_TX_Q_NUM; i++)
    {
        if (p_fifo[i].tx_q_dev_index == acl_handle && p_fifo[i].tx_q_status == GDSL_TX_Q_USED)
        {
            if (prio == 0xff)
            {
                prio = p_fifo[i].tx_q_prio;
                index = i;
            }
            else if (p_fifo[i].tx_q_prio < prio)
            {
                prio = p_fifo[i].tx_q_prio;
                //find = 1;
                index = i;
            }
        }
    }

    return index;
}

void amlbt_usb_rx_type_fifo_init(void)
{
    if (g_rx_type_fifo == 0)
    {
        g_rx_type_fifo = gdsl_fifo_init(RX_TYPE_FIFO_LEN, (unsigned char *)WF_SRAM_RX_TYPE_FIFO_ADDR);

        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            //update read pointer
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_TYPE_FIFO_R_ADDR, (unsigned int)(unsigned long)g_rx_type_fifo->r,
                                               USB_EP2);
            //update write pointer
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_TYPE_FIFO_W_ADDR, (unsigned int)(unsigned long)g_rx_type_fifo->w,
                                               USB_EP2);
        }
        g_rx_type_fifo->r = (unsigned char *)(unsigned long)(WF_SRAM_RX_TYPE_FIFO_ADDR);
    }
}

void amlbt_usb_rx_type_fifo_deinit(void)
{
    if (g_rx_type_fifo)
    {
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_TYPE_FIFO_R_ADDR, 0, USB_EP2);
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_TYPE_FIFO_W_ADDR, 0, USB_EP2);
        }
        gdsl_fifo_deinit(g_rx_type_fifo);
        g_rx_type_fifo = 0;
    }
}

void amlbt_usb_hci_cmd_fifo_init(void)
{
    if (g_cmd_fifo == 0)
    {
        g_cmd_fifo = gdsl_fifo_init(WF_SRAM_CMD_LEN, (unsigned char *)(WF_SRAM_CMD_Q_ADDR));

        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_CMD_FIFO_R_ADDR, (unsigned int)(unsigned long)g_cmd_fifo->r, USB_EP2);
            AMLBT_DBG("cmd fifo init r: %#lx\n", (unsigned long)g_cmd_fifo->r);
            //update write pointer
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_CMD_FIFO_W_ADDR, (unsigned int)(unsigned long)g_cmd_fifo->w, USB_EP2);
        }
        AMLBT_DBG("cmd fifo init w : %#lx\n", (unsigned long)g_cmd_fifo->w);
        g_cmd_fifo->w = (unsigned char *)(unsigned long)(WF_SRAM_CMD_Q_ADDR);
    }
    AMLBT_DBG("%s end \n", __func__);
}

void amlbt_usb_hci_cmd_fifo_deinit(void)
{
    AMLBT_DBG("%s \n", __func__);
    if (g_cmd_fifo != 0)
    {
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_CMD_FIFO_R_ADDR, 0, USB_EP2);
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_CMD_FIFO_W_ADDR, 0, USB_EP2);
        }
        gdsl_fifo_deinit(g_cmd_fifo);
        g_cmd_fifo = 0;
    }
}

void amlbt_usb_hci_tx_data_init(void)
{
    unsigned int i = 0;
    unsigned tx_info[WF_SRAM_TX_Q_NUM * 3] = {0};

    AMLBT_DBG("%s \n", __func__);

    if (g_tx_q == 0)
    {
        g_tx_q = (gdsl_tx_q_t *)kzalloc(sizeof(gdsl_tx_q_t) * WF_SRAM_TX_Q_NUM, GFP_KERNEL);
    }

    for (i = 0; i < WF_SRAM_TX_Q_NUM; i++)
    {
        g_tx_q[i].tx_q_addr = (unsigned char *)(unsigned long)(WF_SRAM_TX_Q_ADDR + i * TX_Q_LEN);
        g_tx_q[i].tx_q_status_addr = (unsigned int *)(unsigned long)(WF_SRAM_TX_Q_STATUS_ADDR + i * 4);
        g_tx_q[i].tx_q_prio_addr = (unsigned int *)(unsigned long)(WF_SRAM_TX_Q_PRIO_ADDR + i * 4);
        g_tx_q[i].tx_q_dev_index_addr = (unsigned int *)(unsigned long)(WF_SRAM_TX_Q_INDEX_ADDR + i * 4);

        g_tx_q[i].tx_q_dev_index = 0;
        g_tx_q[i].tx_q_prio = (WF_SRAM_TX_Q_NUM - 1);
        g_tx_q[i].tx_q_status = GDSL_TX_Q_UNUSED;
        tx_info[i] = g_tx_q[i].tx_q_status;
        tx_info[i + 8] = g_tx_q[i].tx_q_dev_index;
        tx_info[i + 16] = g_tx_q[i].tx_q_prio;
    }

    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_write_sram_for_bt((unsigned char *)tx_info,
                                           (unsigned char *)WF_SRAM_TX_Q_STATUS_ADDR, sizeof(tx_info),
                                           USB_EP2);
    }
    AMLBT_DBG("%s end \n", __func__);
}

void amlbt_usb_hci_tx_data_deinit(void)
{
    unsigned tx_info[WF_SRAM_TX_Q_NUM * 3] = {0};

    AMLBT_DBG("%s \n", __func__);

    if (g_tx_q)
    {
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_auc_hif_ops.hi_write_sram_for_bt((unsigned char *)tx_info,
                                               (unsigned char *)WF_SRAM_TX_Q_STATUS_ADDR, sizeof(tx_info),
                                               USB_EP2);
        }
        kfree(g_tx_q);
        g_tx_q = 0;
    }
    AMLBT_DBG("%s end \n", __func__);
}

void amlbt_usb_hci_evt_fifo_init(void)
{
    AMLBT_DBG("%s \n", __func__);

    if (g_event_fifo == 0)
    {
        g_event_fifo = gdsl_fifo_init(WF_SRAM_EVENT_LEN, (unsigned char *)(WF_SRAM_EVENT_Q_ADDR));
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_EVT_FIFO_R_ADDR, (unsigned int)(unsigned long)g_event_fifo->r, USB_EP2);
            AMLBT_DBG("event fifo init r: %#lx\n", (unsigned long)g_event_fifo->r);
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_EVT_FIFO_W_ADDR, (unsigned int)(unsigned long)g_event_fifo->w, USB_EP2);
        }
        AMLBT_DBG("event fifo init w : %#lx\n", (unsigned long)g_event_fifo->w);
        g_event_fifo->r = (unsigned char *)(unsigned long)(WF_SRAM_EVENT_Q_ADDR);
    }
    AMLBT_DBG("%s end \n", __func__);
}

void amlbt_usb_hci_evt_fifo_deinit(void)
{
    AMLBT_DBG("%s \n", __func__);
    if (g_event_fifo != 0)
    {
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_EVT_FIFO_R_ADDR, 0, USB_EP2);
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_EVT_FIFO_W_ADDR, 0, USB_EP2);
        }
        gdsl_fifo_deinit(g_event_fifo);
        g_event_fifo = 0;
    }
    AMLBT_DBG("%s end \n", __func__);
}

void amlbt_usb_fw_recv_fifo_init(void)
{
    AMLBT_DBG("%s \n", __func__);

    if (g_rx_fifo == 0)
    {
        g_rx_fifo = gdsl_fifo_init(WF_SRAM_RX_FIFO_LEN, (unsigned char *)(WF_SRAM_RX_Q_FIFO_ADDR));
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_FIFO_R_ADDR, (unsigned int)(unsigned long)g_rx_fifo->r, USB_EP2);
            AMLBT_DBG("recv fifo init r: %#lx\n", (unsigned long)g_rx_fifo->r);
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_FIFO_W_ADDR, (unsigned int)(unsigned long)g_rx_fifo->w, USB_EP2);
        }
        AMLBT_DBG("recv fifo init w : %#lx\n", (unsigned long)g_rx_fifo->w);
        g_rx_fifo->r = (unsigned char *)(unsigned long)(WF_SRAM_RX_Q_FIFO_ADDR);
    }
    AMLBT_DBG("%s end \n", __func__);
}

void amlbt_usb_fw_recv_fifo_deinit(void)
{
    AMLBT_DBG("%s \n", __func__);
    if (g_rx_fifo != 0)
    {
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_FIFO_R_ADDR, 0, USB_EP2);
            g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_FIFO_W_ADDR, 0, USB_EP2);
        }
        gdsl_fifo_deinit(g_rx_fifo);
        g_rx_fifo = 0;
    }
}

static void amlbt_usb_send_hci_cmd(unsigned char *data, unsigned int len)
{
    AMLBT_DBG("%s, len %d \n", __func__, len);

    if (g_cmd_fifo == NULL)
    {
        printk("%s: bt_usb_hci_cmd_fifo NULL!!!!\n", __func__);
        return ;
    }

    len = ((len + 3) & 0xFFFFFFFC);//Keep 4 bytes aligned

    AMLBT_DBG("%s, Actual length %d \n", __func__, len);
    // step 1: Update the command FIFO read pointer
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_cmd_fifo->r = (unsigned char *)(unsigned long)g_auc_hif_ops.hi_read_word_for_bt(WF_SRAM_CMD_FIFO_R_ADDR, USB_EP2);
    }
    g_cmd_fifo->r += WF_SRAM_CMD_Q_ADDR;
    // step 2: Check the command FIFO space

    //step 3: Write HCI commands to WiFi SRAM
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        gdsl_fifo_write_data_by_ep(g_cmd_fifo, data, len, USB_EP2);
    }
    //step 4: Update the write pointer and write to WiFi SRAM

    //	AMLBT_DBG("before write:r:%#lx, w:%#lx\n", r, w);

    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_CMD_FIFO_W_ADDR,
                                           ((unsigned long)g_cmd_fifo->w - WF_SRAM_CMD_Q_ADDR) & 0xfff, USB_EP2);
    }
    AMLBT_DBG("len %#x:w %#lx, r %#lx\n", len, (unsigned long)g_cmd_fifo->w, (unsigned long)g_cmd_fifo->r);

#if AML_BT_RW_DEBUG
    AMLBT_DBG("cmd:r:%#lx, w:%#lx\n", (unsigned long)g_cmd_fifo->r, (unsigned long)g_cmd_fifo->w);
#endif
    AMLBT_DBG("after write: r:%#lx, w:%#lx\n", (unsigned long)g_cmd_fifo->r, (unsigned long)g_cmd_fifo->w);
}

static void amlbt_usb_send_hci_data(unsigned char *data, unsigned int len)
{
    unsigned int i = 0;
    unsigned int acl_handle = (((data[1] << 8) | data[0]) & 0xfff);
    unsigned int prio = 0;
    //    unsigned int sbc_cnt = 0;
    //    unsigned int j = 0;
    //    unsigned int offset = 0;
    //unsigned int tx_q_info[3 * WF_SRAM_TX_Q_NUM] = {0};
    //unsigned int tx_q_status[WF_SRAM_TX_Q_NUM] = {0};
    //	AMLBT_DBG("s d r:%#x, w:%#x\n", g_tx_fifo->r, g_tx_fifo->w);
    AMLBT_DBG("%s, len:%d\n", __func__, len);

    amlbt_usb_update_tx_q(g_tx_q);

    for (i = 0; i < WF_SRAM_TX_Q_NUM; i++)
    {
        if (g_tx_q[i].tx_q_status == GDSL_TX_Q_UNUSED)
        {
            break;
        }
    }

    if (i == WF_SRAM_TX_Q_NUM)
    {
        printk("%s: hci data space invalid!!!! \n", __func__);
        //gdsl_tx_unlock();
        return ;
    }

    AMLBT_DBG("%s idle queue index : %d, handle:%#x\n", __func__, i, acl_handle);

    prio = amlbt_usb_get_tx_prio(g_tx_q, acl_handle);

    g_tx_q[i].tx_q_prio = (++prio & 7);
    g_tx_q[i].tx_q_dev_index = acl_handle;
    g_tx_q[i].tx_q_status = GDSL_TX_Q_USED;

    AMLBT_DBG("D(%#x):%#x,%#x,%#x\n", i, (unsigned long)g_tx_q[i].tx_q_dev_index,
              (unsigned long)g_tx_q[i].tx_q_prio, len);

    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_write_sram_for_bt(data, g_tx_q[i].tx_q_addr, len, USB_EP2);
    }

    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_write_word_for_bt(*((unsigned int*)&g_tx_q[i].tx_q_dev_index_addr), g_tx_q[i].tx_q_dev_index, USB_EP2);
        g_auc_hif_ops.hi_write_word_for_bt(*((unsigned int*)&g_tx_q[i].tx_q_prio_addr), g_tx_q[i].tx_q_prio, USB_EP2);
        g_auc_hif_ops.hi_write_word_for_bt(*((unsigned int*)&g_tx_q[i].tx_q_status_addr), g_tx_q[i].tx_q_status, USB_EP2);
    }

    AMLBT_DBG("%s, Actual length:%d\n", __func__, len);

#if AML_BT_RW_DEBUG
    printk("((%#x,%#x,%#x,%#x,%#x) \n", i, g_tx_q[i].tx_q_status, g_tx_q[i].tx_q_prio, len, g_tx_q[i].tx_q_dev_index);
#endif
}


static unsigned int amlbt_usb_recv_hci_event(unsigned char *buff, unsigned int cnt)
{
    unsigned int len = 0;
    unsigned int i = 0;

    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_event_fifo->w = (unsigned char *)(unsigned long)g_auc_hif_ops.hi_read_word_for_bt(WF_SRAM_EVT_FIFO_W_ADDR, USB_EP2);
    }
    g_event_fifo->w += (WF_SRAM_EVENT_Q_ADDR);

    AMLBT_DBG("%s\n", __func__);
#if AML_BT_RW_DEBUG
    printk("r:%#lx,w:%#lx\n", (unsigned long)g_event_fifo->r, (unsigned long)g_event_fifo->w);
#endif
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        len = gdsl_read_data_by_ep(g_event_fifo, buff, cnt, USB_EP1);
    }
    AMLBT_DBG("read event fifo len %d\n", len);
    if (len)
    {
        AMLBT_DBG("event data:\n");
        for (i = 0; i < len; i++)
        {
            AMLBT_DBG("%#x|", buff[i]);
        }
    }

    return len;
}

void amlbt_usb_fifo_init(void)
{
    unsigned int st_reg = 0;

    AMLBT_DBG("%s\n", __func__);

    //g_auc_hif_ops.bt_hi_write_word((unsigned int)0x00a0d0e4, 0x8000007f);
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        st_reg = g_auc_hif_ops.hi_read_word_for_bt(WF_SRAM_FW_DRIVER_STATUS_ADDR, USB_EP2);
        st_reg |= WF_SRAM_FD_INIT_FLAG;
        g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_FW_DRIVER_STATUS_ADDR, st_reg, USB_EP2);
    }
    amlbt_buff_init();
    amlbt_usb_rx_type_fifo_init();
    amlbt_usb_hci_cmd_fifo_init();
    amlbt_usb_hci_tx_data_init();
    amlbt_usb_hci_evt_fifo_init();
    amlbt_usb_fw_recv_fifo_init();
    //memset(BT_fwICCM, 0, BT_ICCM_SIZE);
    //memset(BT_fwDCCM, 0, BT_DCCM_SIZE);
    memset(type, 0, TYPE_FIFO_SIZE);

    g_fw_data_fifo = gdsl_fifo_init(DATA_FIFO_SIZE, g_fw_data_buf);
    g_fw_data_fifo->w = g_fw_data_buf;
    g_fw_data_fifo->r = g_fw_data_buf;
    g_fw_evt_fifo = gdsl_fifo_init(EVT_FIFO_SIZE, g_fw_evt_buf);
    g_fw_evt_fifo->w = g_fw_evt_buf;
    g_fw_evt_fifo->r = g_fw_evt_buf;

    g_fw_type_fifo = gdsl_fifo_init(TYPE_FIFO_SIZE, type);
    g_fw_type_fifo->w = type;
    g_fw_type_fifo->r = type;

    st_reg &= ~(WF_SRAM_FD_INIT_FLAG);
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_FW_DRIVER_STATUS_ADDR, st_reg, USB_EP2);
    }
}

void amlbt_usb_fifo_deinit(void)
{
    unsigned int st_reg = 0;

    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        st_reg = g_auc_hif_ops.hi_read_word_for_bt(WF_SRAM_FW_DRIVER_STATUS_ADDR, USB_EP2);
        st_reg |= WF_SRAM_FD_INIT_FLAG;
        g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_FW_DRIVER_STATUS_ADDR, st_reg, USB_EP2);
    }
    amlbt_usb_rx_type_fifo_deinit();
    amlbt_usb_hci_cmd_fifo_deinit();
    amlbt_usb_hci_tx_data_deinit();
    amlbt_usb_hci_evt_fifo_deinit();
    amlbt_usb_fw_recv_fifo_deinit();
    gdsl_fifo_deinit(g_fw_data_fifo);
    gdsl_fifo_deinit(g_fw_evt_fifo);
    gdsl_fifo_deinit(g_fw_type_fifo);
    g_fw_data_fifo = 0;
    g_fw_evt_fifo = 0;
    g_fw_type_fifo = 0;


    st_reg &= ~(WF_SRAM_FD_INIT_FLAG);
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_FW_DRIVER_STATUS_ADDR, st_reg, USB_EP2);
    }
}

static void amlbt_usb_init(void)
{
    AMLBT_DBG("%s\n", __func__);
    amlbt_usb_fifo_init();
    printk("%s set the semaphore\n", __func__);
    printk("%s start read fw thread\n", __func__);
    sema_init(&read_rx_sem, 0);
    fw_data_flag = 0;
    check_fw_rx_stask = kthread_run(amlbt_usb_check_fw_rx, NULL, "check_fw_rx_thread");
    if (!check_fw_rx_stask)
    {
        printk("start read fw stask fail\n");
        return;
    }
    AMLBT_DBG("%s end\n", __func__);
}

static void amlbt_usb_deinit(void)
{
    AMLBT_DBG("%s\n", __func__);
    if (check_fw_rx_stask)
    {
        up(&read_rx_sem);
        kthread_stop(check_fw_rx_stask);
        check_fw_rx_stask = NULL;
    }
    amlbt_usb_fifo_deinit();
    AMLBT_DBG("%s end\n", __func__);
}

static void amlbt_usb_reset(void)
{
    memset(p_acl_buf, 0, HCI_MAX_FRAME_SIZE);
    memset(type, 0, TYPE_FIFO_SIZE);
    evt_state = 0;
}

void amlbt_usb_firmware_check(void)
{
    unsigned int offset = 0;
    unsigned int st_reg = 0;
    unsigned int iccm_base_addr = BT_ICCM_AHB_BASE + BT_ICCM_ROM_LEN;
    unsigned int dccm_base_addr = BT_DCCM_AHB_BASE;
    unsigned int *p_check = NULL;
    uint32_t fw_iccmLen = 0;
    uint8_t *fw_iccmBuf = NULL;
    uint32_t fw_dccmLen = 0;
    uint8_t *fw_dccmBuf = NULL;

    fw_iccmLen = BT_ICCM_SIZE / 256;
    fw_iccmBuf = BT_fwICCM;
    fw_dccmLen = BT_DCCM_SIZE / 256;
    fw_dccmBuf = BT_fwDCCM;

    printk("iccm check:\n");

    iccm_base_addr = BT_ICCM_AHB_BASE + BT_ICCM_ROM_LEN;
    p_check = (unsigned int *)fw_iccmBuf;
    for (offset = 0; offset < (fw_iccmLen / 4); offset++)
    {
        printk("%s -s check %#x\n", __func__, iccm_base_addr);
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            st_reg = g_auc_hif_ops.hi_read_word_for_bt(iccm_base_addr, USB_EP2);
        }
        iccm_base_addr += 4;
        if (st_reg != *p_check)
        {
            printk("iccm download data:%#x, raw data:%#x\n", st_reg, *p_check);
            printk("iccm no match, offset = %#x!\n", offset);
            break;
        }
        p_check++;
    }
    printk("iccm check size : %#x\n", offset);

    if (offset == (fw_iccmLen / 4))
    {
        printk("iccm check pass\n");
    }

    printk("dccm check:\n");

    dccm_base_addr = BT_DCCM_AHB_BASE;
    p_check = (unsigned int *)fw_dccmBuf;
    for (offset = 0; offset < fw_dccmLen / 4; offset++)
    {
        if (amlbt_if_type == AMLBT_TRANS_W2_USB)
        {
            st_reg = g_auc_hif_ops.hi_read_word_for_bt(dccm_base_addr, USB_EP2);
        }
        dccm_base_addr += 4;
        if (st_reg != *p_check)
        {
            printk("dccm download data:%#x, raw data:%#x\n", st_reg, *p_check);
            printk("dccm no match!\n");
            break;
        }
        p_check++;
    }
    printk("dccm check size : %#x\n", offset);
    if (offset == fw_dccmLen / 4)
    {
        printk("dccm check pass\n");
    }

}

#if AML_BT_ROM_CHECK
void amlbt_usb_rom_check(void)
{
    unsigned int offset = 0;
    unsigned long addr = 0;

    for (offset = 0; offset < 256 * 1024; offset += 512)
    {
        addr = (BT_ICCM_AHB_BASE + offset);
        g_auc_hif_ops.hi_read_sram_for_bt(read_buff, (unsigned char *)addr,
                                          512, USB_EP2);
        if (memcmp(read_buff, &bt_rom_code[offset], 512))
        {
            printk("amlbt_usb_rom_check fail,%#x \n", offset);
            printk("[%#x,%#x,%#x,%#x] \n", read_buff[0], read_buff[1], read_buff[2], read_buff[3]);
            printk("[%#x,%#x,%#x,%#x] \n", bt_rom_code[offset], bt_rom_code[offset + 1],
                   bt_rom_code[offset + 2], bt_rom_code[offset + 3]);
            return ;
        }
    }

    printk("amlbt_usb_rom_check pass,%#x \n", offset);
}
#endif

void amlbt_usb_write_firmware(unsigned char *buf, unsigned int len, unsigned int addr)
{
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_write_sram_for_bt(buf, (unsigned char *)(unsigned long)(addr), len, USB_EP2);
    }
}

void amlbt_usb_download_firmware(void)
{
    unsigned int offset = 0;
    unsigned int remain_len = 0;
    unsigned int iccm_base_addr = BT_ICCM_AHB_BASE + BT_ICCM_ROM_LEN;
    unsigned int dccm_base_addr = BT_DCCM_AHB_BASE;
    unsigned int download_size = 0;
    uint32_t fw_iccmLen = 0;
    uint8_t *fw_iccmBuf = NULL;
    uint32_t fw_dccmLen = 0;
    uint8_t *fw_dccmBuf = NULL;

    fw_iccmLen = BT_ICCM_SIZE;
    fw_iccmBuf = BT_fwICCM;
    fw_dccmLen = BT_DCCM_SIZE;
    fw_dccmBuf = BT_fwDCCM;

    download_size = fw_iccmLen;

    //to do download bt fw
    printk("bt_usb_download_firmware:iccm size %#x\n", download_size);
    //g_auc_hif_ops.bt_hi_write_word(0xf03050, 1);    //ram power down rg_ram_pd_shutdown_sw
    //g_auc_hif_ops.bt_hi_write_word(REG_DEV_RESET, 0);    //pmu down

    //if (amlbt_if_type == AMLBT_TRANS_W2_USB)
    //{
    //    g_auc_hif_ops.hi_write_word_for_bt(REG_DEV_RESET, 0x007, USB_EP2);    //pmu up
    //}

    //g_auc_hif_ops.bt_hi_write_word(0xf03050, 0);    //ram power up
    remain_len = (download_size - offset);
    //printk("bt_usb_download_firmware:remain_len %#x\n", remain_len);
    while (offset < download_size)
    {
        if (remain_len < WF_SRAM_FW_DOWNLOAD_SIZE)
        {
            //printk("bt_usb_download_firmware iccm1 offset %#x\n", offset);
            amlbt_usb_write_firmware((unsigned char *)&fw_iccmBuf[offset], remain_len, iccm_base_addr);
            offset += remain_len;
            iccm_base_addr += remain_len;
            AMLBT_DBG("bt_usb_download_firmware iccm1 offset %#x, write_len %#x\n", offset, write_len);
        }
        else
        {
            amlbt_usb_write_firmware((unsigned char *)&fw_iccmBuf[offset], WF_SRAM_FW_DOWNLOAD_SIZE, iccm_base_addr);
            offset += WF_SRAM_FW_DOWNLOAD_SIZE;
            remain_len -= WF_SRAM_FW_DOWNLOAD_SIZE;
            iccm_base_addr += WF_SRAM_FW_DOWNLOAD_SIZE;
            AMLBT_DBG("bt_usb_download_firmware iccm2 offset %#x, write_len %#x\n", offset, write_len);
        }
        //printk("bt_usb_download_firmware iccm remain_len %#x\n", remain_len);
    }

    download_size = fw_dccmLen;

    //to do download bt fw
    //printk("bt_usb_download_firmware:dccm size %#x\n", download_size);
    offset = 0;
    remain_len = download_size;
    while (offset < download_size)
    {
        if (remain_len < WF_SRAM_FW_DOWNLOAD_SIZE)
        {
            amlbt_usb_write_firmware((unsigned char *)&fw_dccmBuf[offset], remain_len, dccm_base_addr);
            offset += remain_len;
            dccm_base_addr += remain_len;
        }
        else
        {
            amlbt_usb_write_firmware((unsigned char *)&fw_dccmBuf[offset], WF_SRAM_FW_DOWNLOAD_SIZE, dccm_base_addr);
            offset += WF_SRAM_FW_DOWNLOAD_SIZE;
            remain_len -= WF_SRAM_FW_DOWNLOAD_SIZE;
            dccm_base_addr += WF_SRAM_FW_DOWNLOAD_SIZE;
        }

        printk("bt_usb_download_firmware dccm remain_len %#x \n", remain_len);
    }
    //amlbt_usb_firmware_check();
}

static int amlbt_usb_char_open(struct inode *inode_p, struct file *file_p)
{
    int rf_num= 0;
    printk("%s, %#x \n", __func__, AML_BT_VERSION);
    if (download_fw)
    {
#ifndef AML_BT_PRODUCTION_TOOLS
        download_fw = 0;
        printk("%s, download_fw %d\n", __func__, download_fw);
#else
        amlbt_usb_reset();
        amlbt_usb_init();
        download_fw = 1;
        download_end = 0;
        download_flag = 1;
        fw_cmd_w = 0;
        fw_cmd_r = 0;
        rf_num = ((g_auc_hif_ops.hi_read_word_for_bt(REG_PMU_POWER_CFG, USB_EP2) >> BIT_RF_NUM) & 0x03);
        printk("%s set rf num %#x", __func__, rf_num);
#endif
    }
    close_state = 0;
    init_completion(&usb_completion);
    return nonseekable_open(inode_p, file_p);
}
static void amlbt_usb_char_deinit(void);

static int amlbt_usb_char_close(struct inode *inode_p, struct file *file_p)
{
    printk("%s $$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$ %#x\n", __func__, download_fw);

    if (g_event_fifo != 0)
    {
        printk("event w:%p,r:%p\n", g_event_fifo->w, g_event_fifo->r);
    }

    if (download_fw)
    {
        amlbt_usb_deinit();
    }

    if (close_state == 2)
    {
        amlbt_usb_char_deinit();
    }
    close_state = 0;
    return 0;
}

static ssize_t amlbt_usb_char_read_fw(struct file *file_p,
                                      char __user *buf_p,
                                      size_t count,
                                      loff_t *pos_p)
{
    unsigned char cmd_opcode[2] = {0};
    //unsigned int n  = 0;
    static unsigned int fw_r_state = 0;
    static unsigned int fw_r_index = 0;

#if AML_BT_RW_DEBUG
    //printk("R FW:%#x", count);
#endif

    if (fw_r_state == 0)
    {
        while (g_lib_cmd_fifo->w == g_lib_cmd_fifo->r)
        {
            wait_for_completion(&usb_completion);
        }
        gdsl_fifo_get_data(g_lib_cmd_fifo, cmd_opcode, 2);
        cmd_cpt[4] = cmd_opcode[0];
        cmd_cpt[5] = cmd_opcode[1];
    }

    //AMLRW_DBG("RE:\n");
    //for (n = 0; n < 7; n++)
    {
        //    AMLRW_DBG("%#x|", cmd_cpt[n]);
    }
    //AMLRW_DBG("\n");

    if (copy_to_user(buf_p, &cmd_cpt[fw_r_index], count))
    {
        return -EFAULT;
    }

    switch (fw_r_state)
    {
        case 0:
            fw_r_state = 1;
            fw_r_index += count;
            break;
        case 1:
            fw_r_state = 2;
            fw_r_index += count;
            break;
        case 2:
        {
            fw_r_state = 0;
            fw_r_index = 0;
            cmd_index = 0xff;
            fw_cmd_r++;
            if (download_end)
            {
                amlbt_usb_reset();
                amlbt_usb_init();
                download_fw = 1;
                download_end = 0;
                download_flag = 1;
                printk("%s end \n", __func__);
            }
        }
        break;
    }

    return count;
}

int amlbt_usb_check_fw_rx(void *data)
{
    bool pkt_type;
    bool fw_data;
    bool fw_evt;

    while (!kthread_should_stop())
    {
        if (close_state)
        {
            printk(" %s R CLOSE\n", __func__);
            check_fw_rx_stask = NULL;
            return 0;
        }
        fw_evt = g_fw_evt_fifo->w != g_fw_evt_fifo->r;
        fw_data = g_fw_data_fifo->w != g_fw_data_fifo->r;
        if (fw_data || fw_evt)
        {
            fw_data_flag = 1;
            wake_up_interruptible(&poll_amlbt_queue);
            while (down_interruptible(&read_rx_sem) != 0);
            continue;
        }
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            g_rx_type_fifo->w = (unsigned char *)(unsigned long)g_auc_hif_ops.hi_read_word_for_bt(WF_SRAM_RX_TYPE_FIFO_W_ADDR,
                                USB_EP2);
        }
        g_rx_type_fifo->w += (WF_SRAM_RX_TYPE_FIFO_ADDR);
        pkt_type = g_rx_type_fifo->w != g_rx_type_fifo->r;
        if (pkt_type)
        {
            fw_data_flag = 1;
            wake_up_interruptible(&poll_amlbt_queue);
            while (down_interruptible(&read_rx_sem) != 0);
        }
        usleep_range(3000, 3000);
    }
    printk("%s exit read fw rx thread\n", __func__);
    return 0;
}

static ssize_t amlbt_usb_read_fw_data(void)
{
    unsigned int i = 0;
    unsigned int read_len = 0;
    unsigned int data_index = 0;
    unsigned long tmp = 0;
    unsigned int type_size = 0;
    unsigned int data_size = 0;
    unsigned int fw_data = 0;
    unsigned int fw_evt = 0;

    fw_data = gdsl_fifo_used_size(g_fw_data_fifo);
    fw_evt = gdsl_fifo_used_size(g_fw_evt_fifo);
    //printk("%s fw_data %d fw_evt %d \n", __func__, fw_data, fw_evt);
    if (fw_data || fw_evt)
    {
        return 0;
    }

    memset(read_buff, 0, RX_Q_LEN);
    memset(type_buff, 0, RX_TYPE_FIFO_LEN);

#if AML_BT_RW_DEBUG
    printk("%s (%#x,%#x)\n", __func__, (unsigned int)g_rx_type_fifo->w, (unsigned int)g_rx_type_fifo->r);
#endif
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        type_size = gdsl_read_data_by_ep(g_rx_type_fifo, type_buff, RX_TYPE_FIFO_LEN, USB_EP2);
    }
    AMLBT_DBG("R SIZE:%d\n", type_size);
    //printk("%s R SIZE:%d \n", __func__, type_size);
    if (type_size == 0)
    {
        printk("read type fifo err!!\n");
        return -EFAULT;
    }
    gdsl_fifo_copy_data(g_fw_type_fifo, type_buff, type_size);
#if AML_BT_RW_DEBUG
    printk("%s TYPE:[%#x,%#x,%#x,%#x]\n", __func__, type_buff[0],
           type_buff[4], type_buff[8], type_buff[12]);
#endif
    if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
    {
        g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_TYPE_FIFO_R_ADDR,
                                           ((unsigned int)(unsigned long)g_rx_type_fifo->r - WF_SRAM_RX_TYPE_FIFO_ADDR) & 0x1fff,
                                           USB_EP2);
    }

    for (i = 0; i < (type_size / 4); i++)
    {
        memset(read_buff, 0, RX_Q_LEN);
        if (type_buff[i * 4] == HCI_EVENT_PKT)
        {
            amlbt_usb_recv_hci_event(read_buff, 4);
#if AML_BT_RW_DEBUG
            printk("HEAD:[%#x,%#x,%#x,%#x]\n", read_buff[0],
                   read_buff[1], read_buff[2], read_buff[3]);
#endif
            read_len = read_buff[2];
            read_len -= 1;
            read_len = ((read_len + 3) & 0xFFFFFFFC);
            amlbt_usb_recv_hci_event(&read_buff[4], read_len);
            gdsl_fifo_copy_data(g_fw_evt_fifo, read_buff, read_buff[2] + 3);
            //printk("read 1 r:%#x, w:%#x\n", (unsigned int)g_fw_evt_fifo->r, (unsigned int)g_fw_evt_fifo->w);
            //printk("{1 %#x|%#x|%#x|%#x}\n", g_fw_evt_fifo->r[0],g_fw_evt_fifo->r[1],g_fw_evt_fifo->r[2],g_fw_evt_fifo->r[3]);
            tmp = (unsigned long)g_event_fifo->r;
            tmp = ((tmp + 3) & 0xFFFFFFFC);
            g_event_fifo->r = (unsigned char *)tmp;
            if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
            {
                g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_EVT_FIFO_R_ADDR,
                                                   ((unsigned int)(unsigned long)g_event_fifo->r - WF_SRAM_EVENT_Q_ADDR) & 0xfff,
                                                   USB_EP2);
            }
        }
        else if (type_buff[i * 4] == HCI_ACLDATA_PKT)
        {
            if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
            {
                g_rx_fifo->w = (unsigned char *)(unsigned long)g_auc_hif_ops.hi_read_word_for_bt(WF_SRAM_RX_FIFO_W_ADDR, USB_EP2);
            }
            g_rx_fifo->w += (WF_SRAM_RX_Q_FIFO_ADDR);

#if AML_BT_RW_DEBUG
            printk("%s acl data r:%#lx, w:%#lx\n", __func__, (unsigned long)g_rx_fifo->r, (unsigned long)g_rx_fifo->w);
#endif
            while (g_rx_fifo->r == g_rx_fifo->w)
            {
                if (close_state)
                {
                    printk("R CLOSE 2\n");
                    //close_state = 0;
                    return -EFAULT;
                }
                if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
                {
                    g_rx_fifo->w = (unsigned char *)(unsigned long)g_auc_hif_ops.hi_read_word_for_bt(WF_SRAM_RX_FIFO_W_ADDR, USB_EP2);
                }
                g_rx_fifo->w += (WF_SRAM_RX_Q_FIFO_ADDR);
                AMLBT_DBG("rf2 r %#x, w %#x\n", (unsigned long)g_rx_fifo->r, (unsigned long)g_rx_fifo->w);
            }
            if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
            {
                data_size = gdsl_read_data_by_ep(g_rx_fifo, (unsigned char *)&data_index, 4, USB_EP2);
            }
#if AML_BT_RW_DEBUG
            printk("ds:%#x,%#x\n", data_size, data_index);
#endif
            if (data_size > 0)
            {
                if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
                {
                    g_auc_hif_ops.hi_write_word_for_bt(WF_SRAM_RX_FIFO_R_ADDR,
                                                       ((unsigned int)(unsigned long)g_rx_fifo->r - WF_SRAM_RX_Q_FIFO_ADDR) & 0x1f,
                                                       USB_EP2);
#if AML_BT_RW_DEBUG
                    printk("r dh1 %#x\n", i);
#endif

                    g_auc_hif_ops.hi_read_sram_for_bt(&read_buff[0],
                                                      (unsigned char *)(unsigned long)(WF_SRAM_RX_Q_ADDR + data_index * RX_Q_LEN), 8,
                                                      USB_EP2);
                    read_len = ((read_buff[7] << 8) | (read_buff[6]));
                    read_len = ((read_len + 3) & 0xFFFFFFFC);
#if AML_BT_RW_DEBUG
                    printk("!%#x,%#x,%#x,%#x,%#x,%#x,%#x,%#x!\n", read_buff[0],
                           read_buff[1], read_buff[2], read_buff[3], read_buff[4], read_buff[5],
                           read_buff[6], read_buff[7]);
                    printk("r dh %#x\n", read_len);
#endif
                    g_auc_hif_ops.hi_read_sram_for_bt(&read_buff[8],
                                                      (unsigned char *)(unsigned long)(WF_SRAM_RX_Q_ADDR + data_index * RX_Q_LEN + 8), read_len,
                                                      USB_EP2);
#if AML_BT_RW_DEBUG
                    printk("r dh end\n");
#endif

                }
                gdsl_fifo_copy_data(g_fw_data_fifo, read_buff, (((read_buff[7] << 8) | (read_buff[6])) + 8));
#if AML_BT_RW_DEBUG
                printk("HEAD1:[%#x,%#x,%#x,%#x]\n", read_buff[0],
                       read_buff[1], read_buff[2], read_buff[3]);
                printk("HEAD2:[%#x,%#x,%#x,%#x]\n", read_buff[4],
                       read_buff[5], read_buff[6], read_buff[7]);
                printk("HEAD3:[%#x,%#x,%#x,%#x]\n", read_buff[8],
                       read_buff[9], read_buff[10], read_buff[11]);
                printk("HEAD4:[%#x,%#x,%#x,%#x]\n", read_buff[12],
                       read_buff[13], read_buff[14], read_buff[15]);
#endif
            }
            else
            {
                printk("data size err!!\n");
                return -EFAULT;
            }
        }
        else
        {
            printk("type error!\n");
        }
    }
    return 0;
}

static ssize_t amlbt_usb_char_read(struct file *file_p,
                                   char __user *buf_p,
                                   size_t count,
                                   loff_t *pos_p)
{
    ssize_t ret = 0;
    unsigned char close_evt[7] = {0x04, 0x0e, 0x04, 0x01, 0x00, 0x00, 0x00};
    static unsigned int bt_type = 0;
    //AMLBT_DBG("R:%#x, r %#x, w %#x\n", count, (unsigned long)g_host_tx_fifo->r, (unsigned long)g_host_tx_fifo->w);
    AMLBT_DBG("R:%#x\n", count);

    AMLBT_DBG("%s start, count : %ld\n", __func__, count);
    AMLBT_DBG("fw recv fifo r:%#lx, w:%#lx\n", (unsigned long)g_rx_fifo->r, (unsigned long)g_rx_fifo->w);
    AMLBT_DBG("%s, data : %d\n", __func__, data);

    if (!download_fw || (fw_cmd_r < fw_cmd_w))
    {
        return amlbt_usb_char_read_fw(file_p, buf_p, count, pos_p);
    }

    if (close_state)
    {
        printk(" %s R CLOSE\n", __func__);
        //close_state = 0;
        return 0;
    }

    fw_data_flag = 0;
#if AML_BT_RW_DEBUG
    if (amlbt_if_type == AMLBT_TRANS_W2_USB)
    {
        printk("%s evt_state %d R count:%#x \n", __func__, evt_state, count);
    }
#endif

    ret = amlbt_usb_read_fw_data();
    if (ret < 0)
    {
        printk("ret fail \n");
        if (copy_to_user(buf_p, close_evt, 1))
        {
            printk("%s, copy_to_user error \n", __func__);
            return -EFAULT;
        }
        return 1;
    }
    //printk("recv(%#x,%#x) w:%#x,r:%#x\n", type, evt_state,
    //	(unsigned long)g_rx_fifo->w, (unsigned long)g_rx_fifo->r);

    AMLBT_DBG("recv(%#x)\n", evt_state);

    switch (evt_state)
    {
        case 0:					//read type
            evt_state = 1;
            //bt_type = *(unsigned int *)g_fw_type_fifo->r;
            gdsl_fifo_get_data(g_fw_type_fifo, (unsigned char *)&bt_type, sizeof(bt_type));
            if (bt_type == HCI_EVENT_PKT)
            {
#if AML_BT_RW_DEBUG
                printk("evt r:%#x, w:%#x\n", (unsigned int)g_fw_evt_fifo->r, (unsigned int)g_fw_evt_fifo->w);
#endif
                gdsl_fifo_get_data(g_fw_evt_fifo, read_buff, count);
                if (copy_to_user(buf_p, read_buff, count))
                {
                    printk("%s, copy_to_user error \n", __func__);
                    return -EFAULT;
                }
                //gdsl_fifo_update_r(g_fw_evt_fifo, count);
            }
            else if (bt_type == HCI_ACLDATA_PKT)
            {
                gdsl_fifo_get_data(g_fw_data_fifo, read_buff, sizeof(bt_type));
                if (copy_to_user(buf_p, read_buff, count))
                {
                    printk("%s, copy_to_user error \n", __func__);
                    return -EFAULT;
                }
                //gdsl_fifo_update_r(g_fw_data_fifo, sizeof(bt_type));    //data type cost 4 bytes
            }
            //gdsl_fifo_update_r(g_fw_type_fifo, sizeof(bt_type));
#if AML_BT_RW_DEBUG
            printk("RT:%#x\n", bt_type);
#endif
            break;
        case 1:					// read header
            evt_state = 2;
            if (bt_type == HCI_EVENT_PKT)
            {
                //printk("evt 2 r:%#x, w:%#x\n", (unsigned int)g_fw_evt_fifo->r, (unsigned int)g_fw_evt_fifo->w);
                gdsl_fifo_get_data(g_fw_evt_fifo, read_buff, count);
                if (copy_to_user(buf_p, read_buff, count))
                {
                    AMLBT_DBG("%s, copy_to_user error \n", __func__);
                    return -EFAULT;
                }
#if AML_BT_RW_DEBUG
                printk("%#x|%#x|%#x|%#x\n", read_buff[0], read_buff[1],
                       read_buff[2], read_buff[3]);
#endif
                //gdsl_fifo_update_r(g_fw_evt_fifo, count);
            }
            else if (bt_type == HCI_ACLDATA_PKT)
            {
                gdsl_fifo_get_data(g_fw_data_fifo, read_buff, count);
#if AML_BT_RW_DEBUG
                printk("%#x|%#x|%#x|%#x\n", read_buff[0], read_buff[1],
                       read_buff[2], read_buff[3]);
#endif
                if (copy_to_user(buf_p, read_buff, count))
                {
                    printk("%s, copy_to_user error \n", __func__);
                    return -EFAULT;
                }
                //gdsl_fifo_update_r(g_fw_data_fifo, count);
            }
            break;
        case 2:					//read payload
            evt_state = 0;
            if (bt_type == HCI_EVENT_PKT)
            {
                //printk("evt 3 r:%#x, w:%#x\n", (unsigned int)g_fw_evt_fifo->r, (unsigned int)g_fw_evt_fifo->w);
                gdsl_fifo_get_data(g_fw_evt_fifo, read_buff, count);
#if AML_BT_RW_DEBUG
                printk("%#x|%#x|%#x|%#x\n", read_buff[0], read_buff[1],
                       read_buff[2], read_buff[3]);
#endif
                if (copy_to_user(buf_p, read_buff, count))
                {
                    AMLBT_DBG("%s, copy_to_user error \n", __func__);
                    return -EFAULT;
                }
                //gdsl_fifo_update_r(g_fw_evt_fifo, count);
            }
            else if (bt_type == HCI_ACLDATA_PKT)
            {
                unsigned int offset = (*((unsigned int*)&g_fw_data_fifo->r) - *((unsigned int*)&g_fw_data_fifo->base_addr));
                if ((offset + count) >= g_fw_data_fifo->size)
                {
                    gdsl_fifo_get_data(g_fw_data_fifo, read_buff, count);
#if AML_BT_RW_DEBUG
                    printk("%#x|%#x|%#x|%#x|%#x|%#x|%#x|%#x\n", read_buff[0], read_buff[1],
                           read_buff[2], read_buff[3], read_buff[4],
                           read_buff[5], read_buff[6], read_buff[7]);
#endif
                    if (copy_to_user(buf_p, read_buff, count))
                    {
                        AMLBT_DBG("%s, copy_to_user error \n", __func__);
                        return -EFAULT;
                    }
                }
                else
                {
                    if (copy_to_user(buf_p, g_fw_data_fifo->r, count))
                    {
                        AMLBT_DBG("%s, copy_to_user error \n", __func__);
                        return -EFAULT;
                    }
                    gdsl_fifo_update_r(g_fw_data_fifo, count);
                }
            }
            break;
        default:
            printk("%s, evt_state error!!\n", __func__);
            break;
    }
    up(&read_rx_sem);
    AMLBT_DBG("R END\n");

    return count;
}

static ssize_t amlbt_usb_char_write_fw(struct file *file_p,
                                       const char __user *buf_p,
                                       size_t count,
                                       loff_t *pos_p)
{
    unsigned int n = 0;
    unsigned int size = sizeof(cmd) / sizeof(cmd[0]);
    unsigned char len  = 0;
    unsigned int offset = 0;
    unsigned int reg_value = 0;


    AMLBT_DBG("W_FW:%#x\n", count);

    if (count == 1)
    {
        return count;
    }

    memset(p_acl_buf, 0, HCI_MAX_FRAME_SIZE);

    if (copy_from_user(p_acl_buf, buf_p, count))
    {
        AMLBT_DBG("%s: Failed to get data from user space\n", __func__);
        return -EFAULT;
    }

    for (n = 0; n < size; n++)
    {
        if (!memcmp(p_acl_buf, cmd[n], 2))
        {
            cmd_index = n;
            break;
        }
    }

    if (n == size)
    {
        printk("CMD_I:%#x\n", cmd_index);
        for (n = 0; n < 11; n++)
        {
            printk("%#x|", p_acl_buf[n]);
        }
        printk("\n");
        printk("---------------cmd error!------------");
        return -EINVAL;
    }

    gdsl_fifo_copy_data(g_lib_cmd_fifo, p_acl_buf, 2);

    if (p_acl_buf[0] == 0xf3 && p_acl_buf[1] == 0xfe)   //download fw
    {
        len = count - 7;
        offset = ((p_acl_buf[6] << 24) | (p_acl_buf[5] << 16) | (p_acl_buf[4] << 8) | p_acl_buf[3]);
        AMLRW_DBG("%#x,%#x,%#x\n", len, offset, dw_state);
        AMLRW_DBG("%#x,%#x,%#x,%#x\n", p_acl_buf[7], p_acl_buf[8], p_acl_buf[9], p_acl_buf[10]);
        if (offset == BT_ICCM_ROM_LEN)
        {
            if (!download_flag)
            {
                if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
                {
                    //printk("w2 usb write first reg\n");
                    g_auc_hif_ops.hi_write_word_for_bt(0xf03050, 0, USB_EP2);
                    //printk("w2 usb write first reg end\n");
                }
            }
            iccm_base_addr = 0;
            dccm_base_addr = 0;
            dw_state = 0;
            if (BT_fwICCM == NULL)
            {
                BT_fwICCM = vmalloc(BT_ICCM_SIZE);
            }
            if (BT_fwDCCM == NULL)
            {
                BT_fwDCCM = vmalloc(BT_DCCM_SIZE);
            }

            if (BT_fwICCM == NULL || BT_fwDCCM == NULL)
            {
                printk("amlbt_usb_char_write_fw vmalloc err!!\n");
                if (BT_fwICCM != NULL)
                {
                    vfree(BT_fwICCM);
                    BT_fwICCM = NULL;
                }
                if (BT_fwDCCM != NULL)
                {
                    vfree(BT_fwDCCM);
                    BT_fwDCCM = NULL;
                }
                return -EINVAL;
            }
        }

        if (offset == DCCM_RAM_BASE)
        {
            printk("DCCM_RAM_BASE have address!\n");
            dw_state = 1;
        }

        if (dw_state == 0)
        {
            //amlbt_usb_write_firmware(&p_acl_buf[7], len, iccm_base_addr);
            // printk("W S 1:%#x\n", iccm_base_addr);
            memcpy(&BT_fwICCM[iccm_base_addr], &p_acl_buf[7], len);
            //sprintk("W S 2\n");
            iccm_base_addr += len;
        }
        else
        {
            //amlbt_usb_write_firmware(&p_acl_buf[7], len, dccm_base_addr);
            memcpy(&BT_fwDCCM[dccm_base_addr], &p_acl_buf[7], len);
            dccm_base_addr += len;
        }
    }
    else if (p_acl_buf[0] == 0xf1 && p_acl_buf[1] == 0xfe)
    {
        offset = ((p_acl_buf[6] << 24) | (p_acl_buf[5] << 16) | (p_acl_buf[4] << 8) | p_acl_buf[3]);
        reg_value = ((p_acl_buf[10] << 24) | (p_acl_buf[9] << 16) | (p_acl_buf[8] << 8) | p_acl_buf[7]);
        printk("WR:%#x,%#x\n", offset, reg_value);
        if (offset == 0xa70014 && !(reg_value & (1 << 24))) //rf calibration
        {
            //amlbt_usb_firmware_check();
            if (!download_flag)
            {
                amlbt_usb_download_firmware();
                printk("download finished!\n");
                vfree(BT_fwICCM);
                vfree(BT_fwDCCM);
                BT_fwICCM = NULL;
                BT_fwDCCM = NULL;
            }
            printk("W E %#x,%#x\n", iccm_base_addr, dccm_base_addr);
        }
        else if (offset == REG_PMU_POWER_CFG)
        {
            if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
            {
                g_auc_hif_ops.hi_write_word_for_bt(REG_PMU_POWER_CFG, reg_value, USB_EP2); // rg_pmu_power_cfg
            }
        }
    }
    else if (p_acl_buf[0] == 0xf2 && p_acl_buf[1] == 0xfe)
    {
        offset = ((p_acl_buf[6] << 24) | (p_acl_buf[5] << 16) | (p_acl_buf[4] << 8) | p_acl_buf[3]);
        reg_value = ((p_acl_buf[10] << 24) | (p_acl_buf[9] << 16) | (p_acl_buf[8] << 8) | p_acl_buf[7]);
        printk("WC:%#x,%#x\n", offset, reg_value);
        if (offset == REG_DEV_RESET)
        {
            download_end = 1;
            if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
            {
                g_auc_hif_ops.hi_write_word_for_bt(REG_DEV_RESET, (unsigned int)((BIT_CPU | BIT_MAC | BIT_PHY) << DEV_RESET_SW),
                                                   USB_EP2);
                printk("%s end 15:51,bt reg : %#x\n", __func__, g_auc_hif_ops.hi_read_word_for_bt(REG_DEV_RESET, USB_EP2));
            }
        }
    }
#if 0
    else if (p_acl_buf[0] == 0x1a && p_acl_buf[1] == 0xfc)  //HCI_VSC_WRITE_BD_ADDR
    {
        memcpy(bt_addr, &p_acl_buf[3], sizeof(bt_addr));
        printk("ADDR:[%#x|%#x|%#x|%#x|%#x|%#x]",
               bt_addr[0], bt_addr[1], bt_addr[2], bt_addr[3], bt_addr[4], bt_addr[5]);
        download_end = 1;
    }
#endif
    fw_cmd_w++;
    complete(&usb_completion);
    return count;
}

static ssize_t amlbt_usb_char_write(struct file *file_p,
                                    const char __user *buf_p,
                                    size_t count,
                                    loff_t *pos_p)
{
    unsigned int i = 0;
    static unsigned int w_type = 0;
    unsigned char hci_reset[] = {0x03, 0x0C, 0x00};

    AMLBT_DBG("W:%#x\n", count);

    AMLBT_DBG("%s, count:%ld\n", __func__, count);

    if (count > HCI_MAX_FRAME_SIZE)
    {
        printk("count > HCI_MAX_FRAME_SIZE \n");
        return -EINVAL;
    }

    if (!download_fw)
    {
        return amlbt_usb_char_write_fw(file_p, buf_p, count, pos_p);
    }
#if AML_BT_RW_DEBUG
    if (amlbt_if_type == AMLBT_TRANS_W2_USB)
    {
        printk("W:%#x, %#x,%#x,%#x\n", count, download_fw, fw_cmd_r, fw_cmd_w);
    }
#endif
    if (count == 1)	//host write hci type
    {
        get_user(w_type, buf_p);
        AMLBT_DBG("%s:get type %#x\n", __func__, w_type);
        //drv_gdsl_fifo_write_data(g_host_tx_fifo, (unsigned char *)&type, 4);
        //AMLBT_DBG("T:%#x, r %#x, w %#x\n", type, (unsigned long)g_host_tx_fifo->r, (unsigned long)g_host_tx_fifo->w);
#if AML_BT_RW_DEBUG
        printk("T:%#x\n", w_type);
#endif
        //if (w_type == HCI_ACLDATA_PKT)
        {
            //    bt_usb_update_tx_data();
        }
        return count;
    }

    //p_acl_buf = kmalloc(count, GFP_KERNEL);

    memset(p_acl_buf, 0, HCI_MAX_FRAME_SIZE);

    if (copy_from_user(p_acl_buf, buf_p, count))
    {
        AMLBT_DBG("%s: Failed to get data from user space\n", __func__);
        return -EFAULT;
    }

    AMLBT_DBG("%s:get hci raw data:\n", __func__);

#if AML_BT_RW_DEBUG
    if (count > 1 && w_type == HCI_COMMAND_PKT)
    {
        printk("hci cmd:");
        for (i = 0; i < 8; i++)
        {
            printk("%#x|", p_acl_buf[i]);
        }
        printk("\n");
    }
#endif

    AMLBT_DBG("\n");

    if (w_type == HCI_COMMAND_PKT)
    {
        if (count == 0x0f && p_acl_buf[0] == 0x27 && p_acl_buf[1] == 0xfc)   //close
        {
            printk("reset start \n");
            amlbt_usb_send_hci_cmd(hci_reset, sizeof(hci_reset));
            amlbt_usb_read_fw_data();
            printk("reset end \n");
            close_state = 1;
            for (i = 0; i < 8; i++)
            {
                printk("%#x|", p_acl_buf[i]);
            }
            printk("\n");
            printk("W CLOSE\n");
            return count;
        }
        amlbt_usb_send_hci_cmd(p_acl_buf, count);
    }
    else if (w_type == HCI_ACLDATA_PKT)
    {
        amlbt_usb_send_hci_data(p_acl_buf, count);
    }

    AMLBT_DBG("W END\n");

    return count;
}

unsigned int btchr_poll(struct file *file, poll_table *wait)
{
    int mask = 0;

    if (!download_fw)
    {
        return POLLIN | POLLRDNORM;
    }
    poll_wait(file, &poll_amlbt_queue, wait);
    if (fw_data_flag == 1)
    {
        mask |= POLLIN | POLLRDNORM;
    }
    return mask;
}


static struct file_operations amlbt_usb_fops  =
{
    .open = amlbt_usb_char_open,
    .release = amlbt_usb_char_close,
    .read = amlbt_usb_char_read,
    .write = amlbt_usb_char_write,
    .poll = btchr_poll,
    //.unlocked_ioctl = btchr_ioctl,
};


static int amlbt_usb_char_init(void)
{
    int res = 0;
    struct device *dev;

    printk("%s\n", __func__);

    //	init_timer(&r_timer);

    bt_char_class = class_create(THIS_MODULE, AML_BT_CHAR_DEVICE_NAME);
    if (IS_ERR(bt_char_class))
    {
        printk("%s:Failed to create bt char class\n", __func__);
        return PTR_ERR(bt_char_class);
    }

    res = alloc_chrdev_region(&bt_devid, 0, 1, AML_BT_CHAR_DEVICE_NAME);
    if (res < 0)
    {
        printk("%s:Failed to allocate bt char device\n", __func__);
        goto err_alloc;
    }

    dev = device_create(bt_char_class, NULL, bt_devid, NULL, AML_BT_CHAR_DEVICE_NAME);
    if (IS_ERR(dev))
    {
        printk("%s:Failed to create bt char device\n", __func__);
        res = PTR_ERR(dev);
        goto err_create;
    }

    cdev_init(&bt_char_dev, &amlbt_usb_fops);
    res = cdev_add(&bt_char_dev, bt_devid, 1);
    if (res < 0)
    {
        printk("%s:Failed to add bt char device\n", __func__);
        goto err_add;
    }
    //g_cdev = dev;
    printk("%s end", __func__);
    return 0;

err_add:
    device_destroy(bt_char_class, bt_devid);
err_create:
    unregister_chrdev_region(bt_devid, 1);
err_alloc:
    class_destroy(bt_char_class);
    return res;
}

static void amlbt_usb_char_deinit(void)
{
    printk("%s \n", __func__);
    device_destroy(bt_char_class, bt_devid);
    class_destroy(bt_char_class);
    cdev_del(&bt_char_dev);
    unregister_chrdev_region(bt_devid, 1);
}

static void amlbt_dev_release(struct device *dev)
{
    return;
}
#if 0
static void amlbt_cmd_test(void)
{
    memset(BT_fwICCM, 0x55, BT_ICCM_SIZE);
    memset(BT_fwDCCM, 0xaa, BT_DCCM_SIZE);
    //amlbt_usb_download_firmware();
    g_auc_hif_ops.hi_write_sram_for_bt(BT_fwICCM, (unsigned char *)(unsigned long)(0x00300000 + 256 * 1024), 20 * 1024,
                                       USB_EP2);
    g_auc_hif_ops.hi_read_sram_for_bt(BT_fwDCCM, (unsigned char *)(unsigned long)(0x00300000 + 256 * 1024), 20 * 1024,
                                      USB_EP2);

    if (!memcmp(BT_fwICCM, BT_fwDCCM, 20 * 1024))
    {
        printk("check ok!\n");
    }
    else
    {
        printk("check err : %#x,%#x,%#x,%#x \n", BT_fwDCCM[0], BT_fwDCCM[1], BT_fwDCCM[2], BT_fwDCCM[3]);
    }
    memset(BT_fwDCCM, 0xaa, BT_DCCM_SIZE);
    g_auc_hif_ops.hi_write_sram_for_bt(BT_fwICCM, (unsigned char *)(unsigned long)(0x00400000), 20 * 1024, USB_EP2);
    g_auc_hif_ops.hi_read_sram_for_bt(BT_fwDCCM, (unsigned char *)(unsigned long)(0x00400000), 20 * 1024, USB_EP2);
    if (!memcmp(BT_fwICCM, BT_fwDCCM, 20 * 1024))
    {
        printk("check2 ok!\n");
    }
    else
    {
        printk("check2 err : %#x,%#x,%#x,%#x \n", BT_fwDCCM[0], BT_fwDCCM[1], BT_fwDCCM[2], BT_fwDCCM[3]);
    }
}

#endif
static int amlbt_sdio_fops_open(struct inode *inode, struct file *file)
{
    printk("%s bt opened rf num:%#x\n", __func__, rf_num);
    return 0;
}

static int amlbt_sdio_fops_close(struct inode *inode, struct file *file)
{
    printk("%s BT closed\n", __func__);
    return 0;
}

static ssize_t amlbt_sdio_char_write(struct file *file_p,
                                     const char __user *buf_p,
                                     size_t count,
                                     loff_t *pos_p)
{
    unsigned int offset = 0;
    unsigned int reg_value = 0;
    if (copy_from_user(sdio_buf, buf_p, count))
    {
        AMLBT_DBG("%s: Failed to get data from user space\n", __func__);
        return -EFAULT;
    }

    offset = ((sdio_buf[3] << 24) | (sdio_buf[2] << 16) | (sdio_buf[1] << 8) | sdio_buf[0]);
    reg_value = ((sdio_buf[7] << 24) | (sdio_buf[6] << 16) | (sdio_buf[5] << 8) | sdio_buf[4]);
    printk("WR:%#x,%#x\n", offset, reg_value);
    if (offset == REG_PMU_POWER_CFG)
    {
        if (FAMLIY_TYPE_IS_W2(amlbt_if_type))
        {
            rf_num = reg_value >> BIT_RF_NUM;
            printk("BT set rf succeed!\n");
            printk("%s set rf num %#x", __func__, rf_num);
        }
    }
    return count;
}

const struct file_operations amlbt_sdio_fops =
{
    .open       = amlbt_sdio_fops_open,
    .release    = amlbt_sdio_fops_close,
    .write      = amlbt_sdio_char_write,
    .poll       = NULL,
    .unlocked_ioctl = NULL,
    .fasync     = NULL
};

static int amlbt_sdio_init(void)
{
    int ret = 0;
    int cdevErr = 0;

    dev_t devID = MKDEV(amlbt_sdio_major, 0);
    printk("amlbt_sdio_init\n");

    ret = alloc_chrdev_region(&devID, 0, 1, AML_BT_NOTE);
    if (ret)
    {
        pr_err("fail to allocate chrdev\n");
        return ret;
    }

    amlbt_sdio_major = MAJOR(devID);
    printk("major number:%d\n", amlbt_sdio_major);
    cdev_init(&amlbt_sdio_cdev, &amlbt_sdio_fops);
    amlbt_sdio_cdev.owner = THIS_MODULE;

    cdevErr = cdev_add(&amlbt_sdio_cdev, devID, amlbt_sdio_devs);
    if (cdevErr)
        goto error;

    printk("%s driver(major %d) installed.\n",
           "BT_sdiodev", amlbt_sdio_major);

    amlbt_sdio_class = class_create(THIS_MODULE, AML_BT_NOTE);
    if (IS_ERR(amlbt_sdio_class))
    {
        pr_err("class create fail, error code(%ld)\n",
               PTR_ERR(amlbt_sdio_class));
        goto err1;
    }

    amlbt_sdio_dev = device_create(amlbt_sdio_class, NULL, devID, NULL, AML_BT_NOTE);
    if (IS_ERR(amlbt_sdio_dev))
    {
        pr_err("device create fail, error code(%ld)\n",
               PTR_ERR(amlbt_sdio_dev));
        goto err2;
    }

    printk("%s: BT_major %d\n", __func__, amlbt_sdio_major);
    printk("%s: devID %d\n", __func__, devID);

    return 0;

err2:
    if (amlbt_sdio_class)
    {
        class_destroy(amlbt_sdio_class);
        amlbt_sdio_class = NULL;
    }

err1:

error:
    if (cdevErr == 0)
        cdev_del(&amlbt_sdio_cdev);

    if (ret == 0)
        unregister_chrdev_region(devID, amlbt_sdio_devs);

    return -1;
}

static int amlbt_sdio_insmod(void)
{
    int ret = 0;
    printk("BTAML version:%#x\n", AML_BT_VERSION);
    printk("++++++sdio bt driver insmod start.++++++\n");
    ret = amlbt_sdio_init();
    if (ret)
    {
        pr_err("%s: amlbt_sdio_init failed!\n", __func__);
        return ret;
    }
    printk("------sdio bt driver insmod end.------\n");
    return ret;
}

static int amlbt_sdio_exit(void)
{
    dev_t dev = MKDEV(amlbt_sdio_major, 0);

    if (amlbt_sdio_dev)
    {
        device_destroy(amlbt_sdio_class, dev);
        amlbt_sdio_dev = NULL;
    }
    if (amlbt_sdio_class)
    {
        class_destroy(amlbt_sdio_class);
        amlbt_sdio_class = NULL;
    }
    cdev_del(&amlbt_sdio_cdev);

    unregister_chrdev_region(dev, 1);

    printk("%s driver removed.\n", AML_BT_NOTE);
    return 0;
}

static void amlbt_sdio_rmmod(void)
{
    printk("++++++sdio bt driver rmmod start++++++\n");
    amlbt_sdio_exit();
    printk("------sdio bt driver rmmod end------\n");
}

static int amlbt_sdio_probe(struct platform_device *dev)
{
    printk("%s \n", __func__);
    amlbt_sdio_insmod();
    if (sdio_buf == NULL)
    {
        sdio_buf = kzalloc(HCI_MAX_FRAME_SIZE, GFP_DMA|GFP_ATOMIC);
    }
    return 0;
}

static int amlbt_sdio_remove(struct platform_device *dev)
{
    printk("%s \n", __func__);

    amlbt_sdio_rmmod();
    kfree(sdio_buf);
    sdio_buf = NULL;
    return 0;
}
static int amlbt_sdio_suspend(struct platform_device *dev, pm_message_t state)
{
    printk("%s \n", __func__);
    return 0;
}
static int amlbt_sdio_resume(struct platform_device *dev)
{
    printk("%s \n", __func__);
    return 0;
}

static void amlbt_sdio_shutdown(struct platform_device *dev)
{
    printk("%s \n", __func__);
}


static struct platform_device amlbt_sdio_device =
{
    .name    = "sdio_bt",
    .id      = -1,
    .dev     = {
        .release = &amlbt_dev_release,
    }
};

static struct platform_driver amlbt_sdio_driver =
{
    .probe = amlbt_sdio_probe,
    .remove = amlbt_sdio_remove,
    .suspend = amlbt_sdio_suspend,
    .resume = amlbt_sdio_resume,
    .shutdown = amlbt_sdio_shutdown,

    .driver = {
        .name = "sdio_bt",
        .owner = THIS_MODULE,
    },
};

static void amlbt_usb_insmod(void)
{
    // int ret = 0;
    printk("BTAML version:%#x\n", AML_BT_VERSION);
    printk("++++++usb bt driver insmod start.++++++\n");
    printk("------usb bt driver insmod end.------\n");

    //return ret;
}

static void amlbt_usb_rmmod(void)
{
    //if (amlbt_poweron == AML_SDIO_EN)
    {
        printk("++++++usb bt driver rmmod start++++++\n");
        printk("------usb bt driver rmmod end------\n");
    }
}

static int amlbt_usb_probe(struct platform_device *dev)
{
    int err = 0;
    amlbt_usb_insmod();   //load
    //g_auc_hif_ops.bt_hi_write_word((unsigned int)0x00a0d0e4, 0x8000007f);
    //printk("%s, %#x", __func__, g_auc_hif_ops.bt_hi_read_word(0x00a0d0e4));
    printk("%s \n", __func__);
    err = amlbt_usb_char_init();
    if (err < 0)
    {
        /* usb register will go on, even bt char register failed */
        AMLBT_DBG("%s:Failed to register usb char device interfaces\n", __func__);
    }
    download_fw = 0;
    amlbt_buff_init();

    //amlbt_cmd_test();
    memset(g_lib_cmd_buff, 0, CMD_FIFO_SIZE);
    g_lib_cmd_fifo = gdsl_fifo_init(CMD_FIFO_SIZE, g_lib_cmd_buff);
    g_lib_cmd_fifo->w = g_lib_cmd_buff;
    g_lib_cmd_fifo->r = g_lib_cmd_buff;
#if AML_BT_ROM_CHECK
    amlbt_usb_rom_check();
#endif
    printk("%s, end \n", __func__);
    return err;
}

static int amlbt_usb_remove(struct platform_device *dev)
{
    printk("%s\n", __func__);

    amlbt_usb_rmmod();    //unload

    close_state = 2;
    if (g_fw_data_fifo == 0)
    {
        amlbt_usb_char_deinit();
    }
    gdsl_fifo_deinit(g_lib_cmd_fifo);
    g_lib_cmd_fifo = 0;
    amlbt_buff_deinit();
    msleep(500);
    printk("%s end\n", __func__);
    return 0;
}

static int amlbt_usb_suspend(struct platform_device *dev, pm_message_t state)
{
    printk("%s \n", __func__);
    return 0;
}

static int amlbt_usb_resume(struct platform_device *dev)
{
    printk("%s \n", __func__);
    return 0;
}

static void amlbt_usb_shutdown(struct platform_device *dev)
{
    printk("%s \n", __func__);
}

static struct platform_device amlbt_usb_device =
{
    .name    = "aml_btusb",
    .id      = -1,
    .dev     = {
        .release = &amlbt_dev_release,
    }
};

static struct platform_driver amlbt_usb_driver =
{
    .probe = amlbt_usb_probe,
    .remove = amlbt_usb_remove,
    .suspend = amlbt_usb_suspend,
    .resume = amlbt_usb_resume,
    .shutdown = amlbt_usb_shutdown,

    .driver = {
        .name = "aml_btusb",
        .owner = THIS_MODULE,
    },
};

static int amlbt_init(void)
{
    int ret = 0;
    struct platform_device *p_device = NULL;
    struct platform_driver *p_driver = NULL;

    printk("722_1600 %s, type:%d \n", __func__, amlbt_if_type);

    printk("%s, type:%#x inter %s, famliy %s\n", __func__, amlbt_if_type,
           amlbt_famliy_intf((AMLBT_PD_ID_INTF & amlbt_if_type)),
           amlbt_famliy_intf(AMLBT_PD_ID_FAMILY & amlbt_if_type));

    if (INTF_TYPE_IS_USB(amlbt_if_type))        //usb interface
    {
        p_device = &amlbt_usb_device;
        p_driver = &amlbt_usb_driver;
    }
    else if (INTF_TYPE_IS_SDIO(amlbt_if_type) || INTF_TYPE_IS_PCIE(amlbt_if_type))  //sdio interface
    {
        p_device = &amlbt_sdio_device;
        p_driver = &amlbt_sdio_driver;
    }
    else
    {
        printk("%s amlbt_if_type invalid!! \n", __func__);
        return ret;
    }

    printk("%s 1 \n", __func__);

    ret = platform_device_register(p_device);
    if (ret)
    {
        dev_err(&p_device->dev, "platform_device_register failed!\n");
        return ret;
    }
    printk("%s 2 \n", __func__);
    ret = platform_driver_register(p_driver);
    if (ret)
    {
        dev_err(&p_device->dev, "platform_driver_register failed!\n");
        return ret;
    }

    dev_info(&p_device->dev, "Init %d OK!\n", amlbt_if_type);

    return ret;
}

static void amlbt_exit(void)
{
    struct platform_device *p_device = NULL;
    struct platform_driver *p_driver = NULL;

    printk("%s, type:%d \n", __func__, amlbt_if_type);

    if (INTF_TYPE_IS_USB(amlbt_if_type))        //usb interface
    {
        p_device = &amlbt_usb_device;
        p_driver = &amlbt_usb_driver;
    }
    else if (INTF_TYPE_IS_SDIO(amlbt_if_type) || INTF_TYPE_IS_PCIE(amlbt_if_type))  //sdio interface
    {
        p_device = &amlbt_sdio_device;
        p_driver = &amlbt_sdio_driver;
    }
    platform_driver_unregister(p_driver);
    platform_device_unregister(p_device);
}


module_param(amlbt_if_type, uint, S_IRUGO);
module_init(amlbt_init);
module_exit(amlbt_exit);
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("2022-09-06");

